﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>

#include <nns/dbgui/dbgui_Types.h>

#include "dbgui_Renderer.h"
#include "dbgui_Allocator.h"

#include "libnns_dbgui_BuiltinImguiShader.h"

#include <imgui.h>

namespace nns { namespace dbgui {

namespace {

int ClampValue(float value, int min, int max)
{
    int result = static_cast<int>(value);

    if (result < min)
    {
        return min;
    }
    if (result > max)
    {
        return max;
    }
    return result;
}


} // anonymous namespace

Renderer::Renderer()
: m_pSamplerDescriptorPool(nullptr)
, m_SamplerDescriptorPoolIndex(-1)
, m_pTextureViewDescriptorPool(nullptr)
, m_TextureViewDescriptorPoolIndex(-1)
, m_pConstantBufferDescriptorPool(nullptr)
, m_ConstantBufferDescriptorPoolIndex(-1)
, m_ImguiDisplayWidth(0)
, m_ImguiDisplayHeight(0)
, m_pUpdateScissorCallback(nullptr)
, m_pUpdateScissorCallbackUserData(nullptr)
, m_ViewportYAxisDirection(ViewportYAxisDirection_Up)

, m_BufferCount(0)
, m_BufferIndex(0)

, m_ShaderMemoryPool()
, m_pResShaderFile(nullptr)
, m_ResShaderCodeType(nn::gfx::ShaderCodeType_End)

, m_VertexState()
, m_RasterizerState()
, m_DepthStencilState()
, m_BlendState()

, m_FontTexture()
, m_FontTextureView()
, m_FontTextureSampler()

, m_ConstantBuffer()
, m_pIndexBufferArray(nullptr)
, m_pVertexBufferArray(nullptr)

, m_FontTextureDescriptorSlot()
, m_FontTextureSamplerDescriptorSlot()
, m_ConstantBufferDescriptorSlot()
, m_pAllocator(nullptr)
{
}

Renderer::~Renderer()
{
}

void Renderer::Initialize(const InterfaceInfo& info, Allocator* pAllocator)
{
    NN_ASSERT(info.GetBufferCount() > 0);

    m_pAllocator                        = pAllocator;

    m_pSamplerDescriptorPool            = info.GetSamplerDescriptorPool();
    m_SamplerDescriptorPoolIndex        = info.GetSamplerDescriptorPoolIndex();
    m_pTextureViewDescriptorPool        = info.GetTextureViewDescriptorPool();
    m_TextureViewDescriptorPoolIndex    = info.GetTextureViewDescriptorPoolIndex();
    m_pConstantBufferDescriptorPool     = info.GetConstantBufferDescriptorPool();
    m_ConstantBufferDescriptorPoolIndex = info.GetConstantBufferDescriptorPoolIndex();
    m_ImguiDisplayWidth                 = info.GetImguiDisplayWidth();
    m_ImguiDisplayHeight                = info.GetImguiDisplayHeight();

    m_pUpdateScissorCallback            = info.GetUpdateScissorCallback();
    m_pUpdateScissorCallbackUserData    = info.GetUpdateScissorCallbackUserData();

    m_ViewportYAxisDirection            = info.GetViewportYAxisDirection();

    m_BufferCount = info.GetBufferCount();

    m_BufferIndex = 0;
}

void Renderer::Finalize()
{
}

void Renderer::InitializeGfxObjects(nn::gfx::Device* pDevice)
{
    InitializeShader(pDevice);
    InitializeVertexState(pDevice);
    InitializeRasterizerState(pDevice);
    InitializeBlendState(pDevice);
    InitializeDepthStencilState(pDevice);
    InitializeFontTexture(pDevice);
    InitializeBuffers(pDevice);

    SetDescriptorlots();
    UpdateConstantBuffer();
}

void Renderer::FinalizeGfxObjects(nn::gfx::Device* pDevice)
{
    FinalizeBuffers(pDevice);
    FinalizeFontTexture(pDevice);
    FinalizeDepthStencilState(pDevice);
    FinalizeBlendState(pDevice);
    FinalizeRasterizerState(pDevice);
    FinalizeVertexState(pDevice);
    FinalizeShader(pDevice);
}


void Renderer::RenderDrawLists(nn::gfx::CommandBuffer* pCommand, ImDrawData* pDrawData)
{
    NN_ASSERT_NOT_NULL(pCommand);
    NN_ASSERT_NOT_NULL(pDrawData);

    pCommand->InvalidateMemory(
        nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor |
        nn::gfx::GpuAccess_IndexBuffer | nn::gfx::GpuAccess_VertexBuffer |
        nn::gfx::GpuAccess_ConstantBuffer);

    nn::gfx::ResShaderProgram* pResShaderProgram =
        m_pResShaderFile->GetShaderContainer()->GetResShaderProgram(0);
    pCommand->SetShader(pResShaderProgram->GetShader(), nn::gfx::ShaderStageBit_All);

    pCommand->SetRasterizerState(&m_RasterizerState);
    pCommand->SetBlendState(&m_BlendState);
    pCommand->SetDepthStencilState(&m_DepthStencilState);
    pCommand->SetVertexState(&m_VertexState);

    pCommand->SetConstantBuffer(
        ConstantBufferShaderBindingSlot, nn::gfx::ShaderStage_Vertex,
        m_ConstantBufferDescriptorSlot);

    nn::gfx::Buffer* pVertexBuffer = &m_pVertexBufferArray[m_BufferIndex];
    nn::gfx::Buffer* pIndexBuffer = &m_pIndexBufferArray[m_BufferIndex];

    nn::gfx::GpuAddress vertexBufferAddress;
    pVertexBuffer->GetGpuAddress(&vertexBufferAddress);
    size_t requiredVertexBufferSize = pDrawData->TotalVtxCount * sizeof(ImDrawVert);
    NN_ASSERT(requiredVertexBufferSize < VertexBufferSize);
    pCommand->SetVertexBuffer(
        0, vertexBufferAddress,
        sizeof(ImDrawVert), VertexBufferSize);

    size_t requiredIndexBufferSize = pDrawData->TotalIdxCount * sizeof(ImDrawIdx);
    NN_ASSERT(requiredIndexBufferSize < IndexBufferSize);
    nn::gfx::GpuAddress indexBufferAddress;
    pIndexBuffer->GetGpuAddress(&indexBufferAddress);

    ImDrawVert* pVertices = pVertexBuffer->Map<ImDrawVert>();
    ImDrawIdx* pIndices = pIndexBuffer->Map<ImDrawIdx>();

    int vertexOffset = 0;
    int indexOffset = 0;

    ImVec4 scissorRect;
    scissorRect.x = 0.0f;
    scissorRect.y = 0.0f;
    scissorRect.z = static_cast<float>(m_ImguiDisplayWidth);
    scissorRect.w = static_cast<float>(m_ImguiDisplayHeight);
    SetScissor(pCommand, &scissorRect);

    void* pCurrentTextureId = nullptr;
    SetTexture(pCommand, pCurrentTextureId);

    for (int cmdListIndex = 0; cmdListIndex < pDrawData->CmdListsCount; cmdListIndex++)
    {
        const ImDrawList* cmdList = pDrawData->CmdLists[cmdListIndex];
        int vertexCount = cmdList->VtxBuffer.size();
        int indexCount = cmdList->IdxBuffer.size();

        std::memcpy(
            pVertices + vertexOffset,
            cmdList->VtxBuffer.Data,
            sizeof(ImDrawVert) * vertexCount);

        std::memcpy(
            pIndices + indexOffset,
            cmdList->IdxBuffer.Data,
            sizeof(ImDrawIdx) * indexCount);

        for (int cmdBufferIndex = 0; cmdBufferIndex < cmdList->CmdBuffer.size(); cmdBufferIndex++)
        {
            const ImDrawCmd* cmd = &cmdList->CmdBuffer[cmdBufferIndex];

            if (cmd->UserCallback != nullptr)
            {
                cmd->UserCallback(cmdList, cmd);
            }
            else
            {
                if (!TestRectEquals(&scissorRect, &cmd->ClipRect))
                {
                    SetScissor(pCommand, &cmd->ClipRect);
                    scissorRect = cmd->ClipRect;
                }

                ImTextureID drawTextureId = cmd->TextureId;
                if (drawTextureId != pCurrentTextureId)
                {
                    SetTexture(pCommand, drawTextureId);
                    pCurrentTextureId = drawTextureId;
                }

                nn::gfx::GpuAddress drawGpuAddress = indexBufferAddress;
                drawGpuAddress.Offset(sizeof(ImDrawIdx) * indexOffset);

                pCommand->DrawIndexed(
                    nn::gfx::PrimitiveTopology_TriangleList,
                    nn::gfx::IndexFormat_Uint16,
                    drawGpuAddress,
                    cmd->ElemCount,
                    vertexOffset);
            }

            indexOffset += cmd->ElemCount;
        }

        vertexOffset += vertexCount;
    }

    pVertexBuffer->Unmap();
    pIndexBuffer->Unmap();

#if defined(NNS_DBGUI_MEMORYPOOL_CPU_CACHED)
    pVertexBuffer->FlushMappedRange(0, VertexBufferSize);
    pIndexBuffer->FlushMappedRange(0, IndexBufferSize);
#endif

    m_BufferIndex = (m_BufferIndex + 1) % m_BufferCount;
}


size_t Renderer::ComputeGpuPoolMemoryRequired(nn::gfx::Device* pDevice, int bufferCount)
{
    size_t totalSize = 0;
    size_t bufferSize = 0;
    size_t bufferAlignment = 0;
    unsigned char* pPixels = nullptr;
    int width = 0;
    int height = 0;
    ImGui::GetIO().Fonts->GetTexDataAsAlpha8(&pPixels, &width, &height);

    nn::gfx::Texture::InfoType textureInfo;
    textureInfo.SetDefault();
    textureInfo.SetWidth(width);
    textureInfo.SetHeight(height);
    textureInfo.SetMipCount(1);
    textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
    textureInfo.SetTileMode(nn::gfx::TileMode_Linear);
    textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    textureInfo.SetImageFormat(FontTextureFormat);

    bufferAlignment = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, textureInfo);
    bufferSize = nn::gfx::Texture::CalculateMipDataSize(pDevice, textureInfo);
    totalSize = nn::util::align_up(totalSize, bufferAlignment) + bufferSize;


    nn::gfx::Buffer::InfoType constantBufferInfo;
    constantBufferInfo.SetDefault();
    constantBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
    constantBufferInfo.SetSize(ConstantBufferSize);
    bufferAlignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, constantBufferInfo);
    bufferSize = constantBufferInfo.GetSize();
    totalSize = nn::util::align_up(totalSize, bufferAlignment) + bufferSize;

    for (int bufferIndex = 0; bufferIndex < bufferCount; ++bufferIndex)
    {
        nn::gfx::Buffer::InfoType indexBufferInfo;
        indexBufferInfo.SetDefault();
        indexBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_IndexBuffer);
        indexBufferInfo.SetSize(IndexBufferSize);
        bufferAlignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, indexBufferInfo);
        bufferSize = indexBufferInfo.GetSize();
        totalSize = nn::util::align_up(totalSize, bufferAlignment) + bufferSize;

        nn::gfx::Buffer::InfoType vertexBufferInfo;
        vertexBufferInfo.SetDefault();
        vertexBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);
        vertexBufferInfo.SetSize(VertexBufferSize);
        bufferAlignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, vertexBufferInfo);
        bufferSize = vertexBufferInfo.GetSize();
        totalSize = nn::util::align_up(totalSize, bufferAlignment) + bufferSize;
    }

    return totalSize;
}

void Renderer::InitializeShader(nn::gfx::Device* pDevice)
{
    nn::gfx::MemoryPool::InfoType shaderMemoryPoolInfo;
    shaderMemoryPoolInfo.SetDefault();
    shaderMemoryPoolInfo.SetMemoryPoolProperty(
        nn::gfx::MemoryPoolProperty_CpuUncached |
        nn::gfx::MemoryPoolProperty_GpuCached |
        nn::gfx::MemoryPoolProperty_ShaderCode);

    size_t requiredPoolMemorySize = sizeof(g_ImguiShaderData);

    NN_ASSERT(reinterpret_cast<const nn::util::BinaryFileHeader*>(g_ImguiShaderData)->IsSignatureValid(nn::gfx::ResShaderFile::Signature));
    size_t memoryPoolAlignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, shaderMemoryPoolInfo);
    size_t binaryFileAlignment = reinterpret_cast<const nn::util::BinaryFileHeader*>(g_ImguiShaderData)->GetAlignment();
    size_t maxAlignment = std::max(binaryFileAlignment, memoryPoolAlignment);

    size_t memoryPoolBufferSize = nn::util::align_up(
        requiredPoolMemorySize,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, shaderMemoryPoolInfo));

    void* pRawBuffer = m_pAllocator->AllocateMemory(memoryPoolBufferSize, maxAlignment);
    shaderMemoryPoolInfo.SetPoolMemory(pRawBuffer, memoryPoolBufferSize);

    m_ShaderMemoryPool.Initialize(pDevice, shaderMemoryPoolInfo);

    void* pMemoryPoolBuffer = m_ShaderMemoryPool.Map();
    std::memcpy(pMemoryPoolBuffer, g_ImguiShaderData, requiredPoolMemorySize);

    m_pResShaderFile = nn::gfx::ResShaderFile::ResCast(pMemoryPoolBuffer);
    nn::gfx::ResShaderContainer* pContainer = m_pResShaderFile->GetShaderContainer();
    NN_ASSERT_NOT_NULL(pContainer);
    size_t containerOffset = nn::util::BytePtr(m_pResShaderFile).Distance(pContainer);

    pContainer->Initialize(
        pDevice, &m_ShaderMemoryPool, containerOffset, requiredPoolMemorySize);

    NN_ASSERT_EQUAL(pContainer->GetShaderVariationCount(), 2);

    nn::gfx::ShaderCodeType codeType = nn::gfx::ShaderCodeType_Binary;
    nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation(0);
    nn::gfx::ResShaderProgram* pResShaderProgram = pVariation->GetResShaderProgram(codeType);
    nn::gfx::ShaderInitializeResult shaderResult = nn::gfx::ShaderInitializeResult_Success;

    if (pResShaderProgram)
    {
        shaderResult = pResShaderProgram->Initialize(pDevice);
    }

    if ((pResShaderProgram == nullptr) || (shaderResult != nn::gfx::ShaderInitializeResult_Success))
    {
        codeType = nn::gfx::ShaderCodeType_Source;
        pResShaderProgram = pVariation->GetResShaderProgram(codeType);
        NN_ASSERT_NOT_NULL(pResShaderProgram);
        shaderResult = pResShaderProgram->Initialize(pDevice);
        NN_ASSERT_EQUAL(shaderResult, nn::gfx::ShaderInitializeResult_Success);
    }

    for (int variationIndex = 1; variationIndex < pContainer->GetShaderVariationCount(); ++variationIndex)
    {
        pVariation = pContainer->GetResShaderVariation(variationIndex);
        NN_ASSERT_NOT_NULL(pVariation);
        pResShaderProgram = pVariation->GetResShaderProgram(codeType);
        NN_ASSERT_NOT_NULL(pResShaderProgram);
        shaderResult = pResShaderProgram->Initialize(pDevice);
        NN_ASSERT_EQUAL(shaderResult, nn::gfx::ShaderInitializeResult_Success);
        NN_UNUSED(shaderResult);
    }

    m_ResShaderCodeType = codeType;

    m_ShaderMemoryPool.Unmap();
    m_ShaderMemoryPool.FlushMappedRange(0, requiredPoolMemorySize);
}

void Renderer::FinalizeShader(nn::gfx::Device* pDevice)
{
    nn::gfx::ResShaderContainer* pContainer = m_pResShaderFile->GetShaderContainer();

    for (int variationIndex = 0; variationIndex < pContainer->GetShaderVariationCount(); ++variationIndex)
    {
        nn::gfx::ResShaderProgram* pProgram = pContainer->GetResShaderProgram(variationIndex);
        pProgram->Finalize(pDevice);
    }

    pContainer->Finalize(pDevice);
    m_pResShaderFile = nullptr;
    m_ResShaderCodeType = nn::gfx::ShaderCodeType_End;
}

void Renderer::InitializeVertexState(nn::gfx::Device* pDevice)
{
    nn::gfx::VertexState::InfoType info;
    info.SetDefault();
    ptrdiff_t stride = sizeof(ImDrawVert);
    nn::gfx::VertexAttributeStateInfo attribs[3];
    {
        attribs[0].SetDefault();
        attribs[0].SetNamePtr("i_Position");
        attribs[0].SetBufferIndex(0);
        attribs[0].SetFormat(nn::gfx::AttributeFormat_32_32_Float);
        attribs[0].SetOffset(offsetof(ImDrawVert, pos));
    }
    {
        attribs[1].SetDefault();
        attribs[1].SetNamePtr("i_TexCoord");
        attribs[1].SetBufferIndex(0);
        attribs[1].SetFormat(nn::gfx::AttributeFormat_32_32_Float);
        attribs[1].SetOffset(offsetof(ImDrawVert, uv));
    }
    {
        attribs[2].SetDefault();
        attribs[2].SetNamePtr("i_Color");
        attribs[2].SetBufferIndex(0);
        attribs[2].SetFormat(nn::gfx::AttributeFormat_8_8_8_8_Unorm);
        attribs[2].SetOffset(offsetof(ImDrawVert, col));
    }
    nn::gfx::VertexBufferStateInfo buffer;
    {
        buffer.SetDefault();
        buffer.SetStride(stride);
    }
    info.SetVertexAttributeStateInfoArray(attribs, 3);
    info.SetVertexBufferStateInfoArray(&buffer, 1);

    size_t memorySize = nn::gfx::VertexState::GetRequiredMemorySize(info);
    if (memorySize > 0)
    {
        void* pMemory = m_pAllocator->AllocateMemory(
            memorySize, nn::gfx::VertexState::RequiredMemoryInfo_Alignment);
        NN_SDK_ASSERT_NOT_NULL(pMemory);
        m_VertexState.SetMemory(pMemory, memorySize);
    }

    nn::gfx::ShaderCodeType shaderCodeType = m_ResShaderCodeType;
    nn::gfx::ResShaderVariation* pResShaderVariation =
        m_pResShaderFile->GetShaderContainer()->GetResShaderVariation(0);
    nn::gfx::Shader* pVertexShader =
        pResShaderVariation->GetResShaderProgram(shaderCodeType)->GetShader();
    m_VertexState.Initialize(pDevice, info, pVertexShader);
}

void Renderer::FinalizeVertexState(nn::gfx::Device* pDevice)
{
    void* pMemory = m_VertexState.GetMemory();
    m_VertexState.Finalize(pDevice);
    if (pMemory != nullptr)
    {
        m_pAllocator->FreeMemory(pMemory);
    }
}


void Renderer::InitializeRasterizerState(nn::gfx::Device* pDevice)
{
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode(nn::gfx::CullMode_None);
    info.SetPrimitiveTopologyType(nn::gfx::PrimitiveTopologyType_Triangle);
    info.SetScissorEnabled(true);
    info.SetDepthClipEnabled(false);
    m_RasterizerState.Initialize(pDevice, info);
}

void Renderer::FinalizeRasterizerState(nn::gfx::Device* pDevice)
{
    m_RasterizerState.Finalize(pDevice);
}

void Renderer::InitializeBlendState(nn::gfx::Device* pDevice)
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();

    nn::gfx::BlendTargetStateInfo targetInfo;
    targetInfo.SetDefault();
    targetInfo.SetBlendEnabled(true);
    targetInfo.SetSourceColorBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
    targetInfo.SetDestinationColorBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
    targetInfo.SetColorBlendFunction(nn::gfx::BlendFunction_Add);
    targetInfo.SetSourceAlphaBlendFactor(nn::gfx::BlendFactor_SourceAlpha);
    targetInfo.SetDestinationAlphaBlendFactor(nn::gfx::BlendFactor_OneMinusSourceAlpha);
    targetInfo.SetAlphaBlendFunction(nn::gfx::BlendFunction_Add);

    info.SetBlendTargetStateInfoArray(&targetInfo, 1);
    size_t memorySize = nn::gfx::BlendState::GetRequiredMemorySize(info);
    if (memorySize > 0)
    {
        void* pMemory = m_pAllocator->AllocateMemory(
            memorySize, nn::gfx::BlendState::RequiredMemoryInfo_Alignment);
        m_BlendState.SetMemory(pMemory, memorySize);
    }

    m_BlendState.Initialize(pDevice, info);
}

void Renderer::FinalizeBlendState(nn::gfx::Device* pDevice)
{
    void* pMemory = m_BlendState.GetMemory();
    m_BlendState.Finalize(pDevice);
    if (pMemory != nullptr)
    {
        m_pAllocator->FreeMemory(pMemory);
    }
}

void Renderer::InitializeDepthStencilState(nn::gfx::Device* pDevice)
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled(false);
    info.SetDepthWriteEnabled(false);
    m_DepthStencilState.Initialize(pDevice, info);
}

void Renderer::FinalizeDepthStencilState(nn::gfx::Device* pDevice)
{
    m_DepthStencilState.Finalize(pDevice);
}

void Renderer::InitializeFontTexture(nn::gfx::Device* pDevice)
{
    ImGuiIO& io = ImGui::GetIO();
    unsigned char* source;
    int width, height;
    io.Fonts->GetTexDataAsAlpha8(&source, &width, &height);

    nn::gfx::Texture::InfoType textureInfo;
    textureInfo.SetDefault();
    textureInfo.SetWidth(width);
    textureInfo.SetHeight(height);
    textureInfo.SetMipCount(1);
    textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_Texture);
    textureInfo.SetTileMode(nn::gfx::TileMode_Linear);
    textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    textureInfo.SetImageFormat(FontTextureFormat);

    size_t alignment = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, textureInfo);
    size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, textureInfo);

    nn::gfx::MemoryPool* pMemoryPool = m_pAllocator->GetGpuMemoryPool();
    ptrdiff_t memoryPoolOffset = m_pAllocator->AllocateGpuMemoryPool(size, alignment);
    void* pMemoryPoolBuffer = pMemoryPool->Map<uint8_t>() + memoryPoolOffset;
    std::memcpy(pMemoryPoolBuffer, source, width * height);
    pMemoryPool->Unmap();
#if defined(NNS_DBGUI_MEMORYPOOL_CPU_CACHED)
    pMemoryPool->FlushMappedRange(memoryPoolOffset, width * height);
#endif

    m_FontTexture.Initialize(pDevice, textureInfo, pMemoryPool, memoryPoolOffset, size);

    nn::gfx::TextureView::InfoType textureViewInfo;
    textureViewInfo.SetDefault();
    textureViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    textureViewInfo.SetImageFormat(FontTextureFormat);
    textureViewInfo.SetTexturePtr(&m_FontTexture);
    textureViewInfo.SetChannelMapping(
        nn::gfx::ChannelMapping_Red,
        nn::gfx::ChannelMapping_Red,
        nn::gfx::ChannelMapping_Red,
        nn::gfx::ChannelMapping_Red);
    m_FontTextureView.Initialize(pDevice, textureViewInfo);

    nn::gfx::Sampler::InfoType samplerInfo;
    samplerInfo.SetDefault();
    samplerInfo.SetFilterMode(nn::gfx::FilterMode_MinLinear_MagLinear_MipPoint);
    samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_Repeat);
    samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_Repeat);
    samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_Repeat);
    m_FontTextureSampler.Initialize(pDevice, samplerInfo);
}

void Renderer::FinalizeFontTexture(nn::gfx::Device* pDevice)
{
    m_FontTextureSampler.Finalize(pDevice);
    m_FontTextureView.Finalize(pDevice);
    m_FontTexture.Finalize(pDevice);
}

void Renderer::InitializeBuffers(nn::gfx::Device* pDevice)
{
    {
        nn::gfx::Buffer::InfoType constantBufferInfo;
        constantBufferInfo.SetDefault();
        constantBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ConstantBuffer);
        constantBufferInfo.SetSize(ConstantBufferSize);
        ptrdiff_t constantBufferOffset =
            m_pAllocator->AllocateGpuMemoryPool(
                ConstantBufferSize,
                nn::gfx::Buffer::GetBufferAlignment(pDevice, constantBufferInfo));
        m_ConstantBuffer.Initialize(
            pDevice, constantBufferInfo,
            m_pAllocator->GetGpuMemoryPool(), constantBufferOffset, ConstantBufferSize);
    }

    m_pIndexBufferArray = m_pAllocator->AllocateArray<nn::gfx::Buffer>(m_BufferCount);
    m_pVertexBufferArray = m_pAllocator->AllocateArray<nn::gfx::Buffer>(m_BufferCount);

    for (int bufferIndex = 0; bufferIndex < m_BufferCount; ++bufferIndex)
    {
        {
            nn::gfx::Buffer::InfoType indexBufferInfo;
            indexBufferInfo.SetDefault();
            indexBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_IndexBuffer);
            indexBufferInfo.SetSize(IndexBufferSize);
            ptrdiff_t indexBufferOffset =
                m_pAllocator->AllocateGpuMemoryPool(
                    IndexBufferSize,
                    nn::gfx::Buffer::GetBufferAlignment(pDevice, indexBufferInfo));
            m_pIndexBufferArray[bufferIndex].Initialize(
                pDevice, indexBufferInfo,
                m_pAllocator->GetGpuMemoryPool(), indexBufferOffset, IndexBufferSize);
        }

        {
            nn::gfx::Buffer::InfoType vertexBufferInfo;
            vertexBufferInfo.SetDefault();
            vertexBufferInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_VertexBuffer);
            vertexBufferInfo.SetSize(VertexBufferSize);
            ptrdiff_t vertexBufferOffset =
                m_pAllocator->AllocateGpuMemoryPool(
                    VertexBufferSize,
                    nn::gfx::Buffer::GetBufferAlignment(pDevice, vertexBufferInfo));
            m_pVertexBufferArray[bufferIndex].Initialize(
                pDevice, vertexBufferInfo,
                m_pAllocator->GetGpuMemoryPool(), vertexBufferOffset, VertexBufferSize);
        }
    }
}

void Renderer::FinalizeBuffers(nn::gfx::Device* pDevice)
{
    for (int bufferIndex = 0; bufferIndex < m_BufferCount; ++bufferIndex)
    {
        m_pVertexBufferArray[bufferIndex].Finalize(pDevice);
        m_pIndexBufferArray[bufferIndex].Finalize(pDevice);
    }
    m_pAllocator->FreeArray(m_pIndexBufferArray, m_BufferCount);
    m_pIndexBufferArray = nullptr;
    m_pAllocator->FreeArray(m_pVertexBufferArray, m_BufferCount);
    m_pVertexBufferArray = nullptr;

    m_ConstantBuffer.Finalize(pDevice);
}

void Renderer::SetDescriptorlots()
{
    m_pSamplerDescriptorPool->BeginUpdate();
    m_pSamplerDescriptorPool->SetSampler(m_SamplerDescriptorPoolIndex, &m_FontTextureSampler);
    m_pSamplerDescriptorPool->EndUpdate();
    m_pSamplerDescriptorPool->GetDescriptorSlot(&m_FontTextureSamplerDescriptorSlot, m_SamplerDescriptorPoolIndex);

    m_pTextureViewDescriptorPool->BeginUpdate();
    m_pTextureViewDescriptorPool->SetTextureView(m_TextureViewDescriptorPoolIndex, &m_FontTextureView);
    m_pTextureViewDescriptorPool->EndUpdate();
    m_pTextureViewDescriptorPool->GetDescriptorSlot(&m_FontTextureDescriptorSlot, m_TextureViewDescriptorPoolIndex);

    nn::gfx::GpuAddress constantBufferGpuAddress;
    m_ConstantBuffer.GetGpuAddress(&constantBufferGpuAddress);
    m_pConstantBufferDescriptorPool->BeginUpdate();
    m_pConstantBufferDescriptorPool->SetBufferView(
        m_ConstantBufferDescriptorPoolIndex,
        constantBufferGpuAddress, ConstantBufferSize);
    m_pConstantBufferDescriptorPool->EndUpdate();
    m_pConstantBufferDescriptorPool->GetDescriptorSlot(&m_ConstantBufferDescriptorSlot, m_ConstantBufferDescriptorPoolIndex);
}

void Renderer::UpdateConstantBuffer()
{
    float imguiDisplayWidthAsFloat = static_cast<float>(m_ImguiDisplayWidth);
    float imguiDisplayHeightAsFloat = static_cast<float>(m_ImguiDisplayHeight);

    float* pConstantBuffer = m_ConstantBuffer.Map<float>();

    float viewportScaleX = 2.0f / imguiDisplayWidthAsFloat;
    float viewportScaleY = 2.0f  / imguiDisplayHeightAsFloat;
    float viewportOffsetX = -1.0f;
    float viewportOffsetY = -1.0f;

    if (m_ViewportYAxisDirection == ViewportYAxisDirection_Up)
    {
        viewportScaleY = -viewportScaleY;
        viewportOffsetY = -viewportOffsetY;
    }

    pConstantBuffer[0] = viewportScaleX;
    pConstantBuffer[1] = viewportScaleY;
    pConstantBuffer[2] = viewportOffsetX;
    pConstantBuffer[3] = viewportOffsetY;

    m_ConstantBuffer.Unmap();
#if defined(NNS_DBGUI_MEMORYPOOL_CPU_CACHED)
    m_ConstantBuffer.FlushMappedRange(0, ConstantBufferSize);
#endif
}

bool Renderer::TestRectEquals(const ImVec4* pRect0, const ImVec4* pRect1)
{
    return
        (pRect0->x == pRect1->x)
        && (pRect0->y == pRect1->y)
        && (pRect0->z == pRect1->z)
        && (pRect0->w == pRect1->w);
}

void Renderer::SetScissor(nn::gfx::CommandBuffer* pCommandBuffer, const ImVec4* pScissor)
{
    int scissorX = ClampValue(pScissor->x, 0, m_ImguiDisplayWidth);
    int scissorY = ClampValue(pScissor->y, 0, m_ImguiDisplayHeight);
    int scissorWidth = ClampValue(pScissor->z - scissorX, 0, m_ImguiDisplayWidth - scissorX);
    int scissorHeight = ClampValue(pScissor->w - scissorY, 0, m_ImguiDisplayHeight - scissorY);

    nn::gfx::ScissorStateInfo scissorStateInfo;
    scissorStateInfo.SetDefault();
    scissorStateInfo.SetOriginX(scissorX);
    scissorStateInfo.SetOriginY(scissorY);
    scissorStateInfo.SetWidth(scissorWidth);
    scissorStateInfo.SetHeight(scissorHeight);

    if (m_pUpdateScissorCallback != nullptr)
    {
        m_pUpdateScissorCallback(
            &scissorStateInfo,
            m_pUpdateScissorCallbackUserData);
    }

    pCommandBuffer->SetScissors(0, 1, &scissorStateInfo);
}

void Renderer::SetTexture(nn::gfx::CommandBuffer* pCommandBuffer, ImTextureID textureId)
{
    nn::gfx::DescriptorSlot* pDescriptorSlot = nullptr;
    if (textureId == nullptr)
        pDescriptorSlot = &m_FontTextureDescriptorSlot;
    else
        pDescriptorSlot = reinterpret_cast<nn::gfx::DescriptorSlot*>(textureId);

    pCommandBuffer->SetTextureAndSampler(
        TextureShaderBindingSlot, nn::gfx::ShaderStage_Pixel,
        *pDescriptorSlot, m_FontTextureSamplerDescriptorSlot);
}

} } // namespace nns { namespace dbgui {
