﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/console/detail/console_ConsoleGraphicsFramework.h>

#include <limits>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

namespace nns {
namespace console {
namespace detail {

ConsoleBase::ConsoleBase() NN_NOEXCEPT
    : m_Style(DefaultConsoleStyle)
    , m_Initialized(false)
    , m_pDevice(nullptr)
{
}
ConsoleBase::ConsoleBase(const ConsoleStyle& style) NN_NOEXCEPT
    : m_Style(style)
    , m_Initialized(false)
    , m_pDevice(nullptr)
{
}

ConsoleBase::~ConsoleBase() NN_NOEXCEPT
{
    if (m_Initialized)
    {
        Finalize();
    }
}

void ConsoleBase::Initialize(
    nn::gfx::Device* pDevice,
    const nn::gfx::ViewportStateInfo& viewport,
    const nn::gfx::ScissorStateInfo& scissor) NN_NOEXCEPT
{
    NN_ASSERT(!m_Initialized);
    nn::gfx::ViewportScissorState::InfoType viewportScissorInfo;
    viewportScissorInfo.SetDefault();
    viewportScissorInfo.SetScissorEnabled(true);
    viewportScissorInfo.SetViewportStateInfoArray(&viewport, 1);
    viewportScissorInfo.SetScissorStateInfoArray(&scissor, 1);

    m_pDevice = pDevice;
    m_ViewportScissorState.Initialize(pDevice, viewportScissorInfo);
    m_Dimension.width = static_cast<int>(viewport.GetWidth());
    m_Dimension.height = static_cast<int>(viewport.GetHeight());
    m_Initialized = true;
}

void ConsoleBase::Finalize() NN_NOEXCEPT
{
    NN_ASSERT(m_Initialized);
    m_ViewportScissorState.Finalize(m_pDevice);
    m_Initialized = false;
}

void ConsoleBase::MakeCommand(nn::gfx::util::DebugFontTextWriter* pTextWriter, nn::gfx::CommandBuffer* pCommandBuffer) const NN_NOEXCEPT
{
    NN_ASSERT(m_Initialized);
    pCommandBuffer->SetViewportScissorState(&m_ViewportScissorState);

    pTextWriter->SetTabWidth(m_Style.tabWidth);
    pTextWriter->SetScale(m_Style.scaleH, m_Style.scaleV);
    pTextWriter->SetTextColor(m_Style.color);
    pTextWriter->SetFixedWidthEnabled(m_Style.fixedWidthEnabled);
    pTextWriter->SetFixedWidth(m_Style.fixedWidth);
    pTextWriter->SetCursor(0, 0);

    this->MakeCommandImpl(pTextWriter);

    pTextWriter->Draw(pCommandBuffer);
}

void ConsoleBase::Update() NN_NOEXCEPT
{
    this->UpdateImpl();
}

nn::image::Dimension ConsoleBase::GetDimension() const NN_NOEXCEPT
{
    return m_Dimension;
}

// ---------------------------------------------------------------------------------------------

void* ConsoleGraphicsFramework::AllocationFunction(size_t size, size_t alignment, void* pUserData) NN_NOEXCEPT
{
    return reinterpret_cast<ConsoleGraphicsFramework*>(pUserData)->Allocate(size, alignment);
}
void ConsoleGraphicsFramework::DeallocationFunction(void* ptr, void* pUserData) NN_NOEXCEPT
{
    reinterpret_cast<ConsoleGraphicsFramework*>(pUserData)->Deallocate(ptr);
}
void ConsoleGraphicsFramework::CommandMakeFunction(nns::gfx::GraphicsFramework* pFramework, int bufferIndex, void* pUserData) NN_NOEXCEPT
{
    pFramework->BeginFrame(bufferIndex);

    auto& obj = *reinterpret_cast<ConsoleGraphicsFramework*>(pUserData);

    // RenderTarget, CommandBuffer
    auto* target = pFramework->GetColorTargetView();
    auto* commandBuffer = pFramework->GetRootCommandBuffer(bufferIndex);
    commandBuffer->ClearColor(
        target,
        (obj.m_ClearColor.v[0] % 256) / 255.0f,
        (obj.m_ClearColor.v[1] % 256) / 255.0f,
        (obj.m_ClearColor.v[2] % 256) / 255.0f,
        1.0f, nullptr);
    commandBuffer->SetRenderTargets(1, &target, nullptr);

    obj.m_Callback(commandBuffer, obj.m_CallbackUserData);

    pFramework->EndFrame(bufferIndex, true /* enableCopyImage */);
}

ConsoleGraphicsFramework::ConsoleGraphicsFramework(void *buffer, size_t bufferSize) NN_NOEXCEPT
    : m_Initialized(false)
    , m_Callback(nullptr)
{
    NN_ASSERT(reinterpret_cast<uintptr_t>(buffer) % nn::os::MemoryPageSize == 0);
    NN_ASSERT(bufferSize % nn::os::MemoryPageSize == 0);
    m_Allocator.Initialize(buffer, bufferSize);
}

ConsoleGraphicsFramework::~ConsoleGraphicsFramework() NN_NOEXCEPT
{
    if (m_Initialized)
    {
        m_Framework.QueueFinish();
        m_Framework.Finalize();
    }
}

void* ConsoleGraphicsFramework::Allocate(size_t size, size_t alignment) NN_NOEXCEPT
{
    auto p = m_Allocator.Allocate(size, alignment);
    NN_ABORT_UNLESS_NOT_NULL(p);
    return p;
}
void* ConsoleGraphicsFramework::Allocate(size_t size) NN_NOEXCEPT
{
    return Allocate(size, std::alignment_of<std::max_align_t>::value);
}
void ConsoleGraphicsFramework::Deallocate(void* ptr) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(ptr);
    m_Allocator.Free(ptr);
}

void ConsoleGraphicsFramework::Initialize(
    void(*callback)(nn::gfx::CommandBuffer* pCommandBuffer, void* callbackUserData),
    void* callbackUserData,
    size_t width, size_t height,
    const nn::util::Color4u8Type& clearColor,
    size_t memorySize) NN_NOEXCEPT
{
    const auto BufferCount = 2;

    // GraphicsFramwwork
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(memorySize);

    nns::gfx::GraphicsFramework::FrameworkInfo frameWorkInfo;
    frameWorkInfo.SetDefault();
    frameWorkInfo.SetDisplayWidth(static_cast<int>(width));
    frameWorkInfo.SetDisplayHeight(static_cast<int>(height));
    frameWorkInfo.SetBufferCount(BufferCount);
    frameWorkInfo.SetDepthStencilBufferEnabled(false);
    m_Framework.Initialize(frameWorkInfo, AllocationFunction, DeallocationFunction, this);

    // CommandMakeFunction
    m_Framework.SetMakeCommandCallback(CommandMakeFunction, this);

    m_Callback = callback;
    m_CallbackUserData = callbackUserData;
    m_ClearColor = clearColor;
    m_Initialized = true;
}
void ConsoleGraphicsFramework::InitializeDebugTextWriter(
    nn::gfx::util::DebugFontTextWriter* pWriter,
    size_t width, size_t height, int charactorCountMax) NN_NOEXCEPT
{
    const auto BufferCount = 2;

    // DebugTextWriter
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(charactorCountMax);
    info.SetBufferCount(BufferCount);
    info.SetUserMemoryPoolEnabled(false);

    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(m_Framework.GetDevice(), info);
    NN_LOG("[nns::console::detail::ConsoleGraphicsFramework] nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize() -> %zu\n", debugFontHeapSize);
    auto* debugFontHeap = Allocate(debugFontHeapSize);
    NN_ASSERT_NOT_NULL(debugFontHeap);
    pWriter->Initialize(m_Framework.GetDevice(), info, debugFontHeap, debugFontHeapSize, nullptr, 0, 0);

    pWriter->SetDisplayWidth(static_cast<int>(width));
    pWriter->SetDisplayHeight(static_cast<int>(height));
    pWriter->SetTextureDescriptor(
        m_Framework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView),
        m_Framework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1));
    pWriter->SetSamplerDescriptor(
        m_Framework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler),
        m_Framework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1));

}
void ConsoleGraphicsFramework::FinalizeDebugTextWriter(nn::gfx::util::DebugFontTextWriter* pWriter) NN_NOEXCEPT
{
    Deallocate(pWriter->GetMemory());
    pWriter->Finalize();
}
nn::gfx::Device* ConsoleGraphicsFramework::GetDevice() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Initialized);
    return m_Framework.GetDevice();
}
void ConsoleGraphicsFramework::ProcessFrame() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Initialized);
    m_Framework.ProcessFrame();
}

} // ~namespace nns::console::detail
}
}
