﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "nns/atk/atk_SampleCommon.h"

#include <cstdio>
#include <nn/atk/atk_SoundSystem.h>
#include <nn/audio/audio_MemoryPool.h>
#include <nn/audio/audio_AudioRenderer.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/util/util_FormatString.h>

#include <nn/mem/mem_StandardAllocator.h>

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#endif

#include <nn/hid/hid_KeyboardKey.h>
#include <nn/settings/settings_DebugPad.h>

namespace {

    // 拡張ヒープ
    const int MemoryHeapSize = 32 * 1024 * 1024;
    static char g_HeapMemory[ MemoryHeapSize ];
    nn::mem::StandardAllocator g_Allocator;

    // メモリプール用ヒープ
    const int MemoryPoolHeapSize = 32 * 1024 * 1024;
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_PoolHeapMemory[MemoryPoolHeapSize];
    nn::mem::StandardAllocator g_PoolHeapAllocator;

    // パッド関連
    nns::atk::SamplePad g_SamplePad;

    // ファイルシステム関連
    const char MountName[] = "content";
    const int AbsolutePathMax = 128;
    char g_AbsolutePath[AbsolutePathMax];
    void* g_MountRomCacheBuffer = NULL;
}

namespace nns { namespace atk {

    // Button と DebugPad の対応関係
    const int SamplePad::DebugPadMap[Button_Count] =
    {
        nn::hid::DebugPadButton::A::Index,
        nn::hid::DebugPadButton::B::Index,
        nn::hid::DebugPadButton::X::Index,
        nn::hid::DebugPadButton::Y::Index,
        nn::hid::DebugPadButton::L::Index,
        nn::hid::DebugPadButton::R::Index,
        nn::hid::DebugPadButton::ZR::Index,
        nn::hid::DebugPadButton::Left::Index,
        nn::hid::DebugPadButton::Right::Index,
        nn::hid::DebugPadButton::Up::Index,
        nn::hid::DebugPadButton::Down::Index,
        nn::hid::DebugPadButton::Start::Index,
    };

    // Button と NPad の対応関係
    const int SamplePad::NpadMap[Button_Count] =
    {
        nn::hid::NpadButton::A::Index,
        nn::hid::NpadButton::B::Index,
        nn::hid::NpadButton::X::Index,
        nn::hid::NpadButton::Y::Index,
        nn::hid::NpadButton::L::Index,
        nn::hid::NpadButton::R::Index,
        nn::hid::NpadButton::ZR::Index,
        nn::hid::NpadButton::Left::Index,
        nn::hid::NpadButton::Right::Index,
        nn::hid::NpadButton::Up::Index,
        nn::hid::NpadButton::Down::Index,
        nn::hid::NpadButton::Plus::Index,
    };

    void SamplePad::Initialize()
    {
        nn::hid::InitializeNpad();
        nn::hid::InitializeDebugPad();

        //キーボードのキーを DebugPad のボタンに割り当てます。
        nn::settings::DebugPadKeyboardMap map;
        nn::settings::GetDebugPadKeyboardMap(&map);
        map.buttonA = nn::hid::KeyboardKey::A::Index;
        map.buttonB = nn::hid::KeyboardKey::B::Index;
        map.buttonX = nn::hid::KeyboardKey::X::Index;
        map.buttonY = nn::hid::KeyboardKey::Y::Index;
        map.buttonL = nn::hid::KeyboardKey::L::Index;
        map.buttonR = nn::hid::KeyboardKey::R::Index;
        map.buttonZR = nn::hid::KeyboardKey::Z::Index;
        map.buttonLeft = nn::hid::KeyboardKey::LeftArrow::Index;
        map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
        map.buttonUp = nn::hid::KeyboardKey::UpArrow::Index;
        map.buttonDown = nn::hid::KeyboardKey::DownArrow::Index;
        map.buttonStart = nn::hid::KeyboardKey::Space::Index;
        nn::settings::SetDebugPadKeyboardMap(map);

        const nn::hid::NpadIdType npadIds[2] =
        {
            nn::hid::NpadId::No1,
            nn::hid::NpadId::Handheld
        };
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
        nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

        ResetCurrentButtons();
        CopyStateCurrentToPrevious();
    }

    void SamplePad::Update() NN_NOEXCEPT
    {
        CopyStateCurrentToPrevious();
        ResetCurrentButtons();

        const nn::hid::NpadStyleSet styleNo1 = nn::hid::GetNpadStyleSet( nn::hid::NpadId::No1 );
        const nn::hid::NpadStyleSet styleHandheld = nn::hid::GetNpadStyleSet( nn::hid::NpadId::Handheld );

        if( styleNo1.Test<nn::hid::NpadStyleFullKey>() )
        {
            nn::hid::GetNpadState( &m_NpadFullKeyCurrentState, nn::hid::NpadId::No1 );
        }
        else if( styleHandheld.Test<nn::hid::NpadStyleHandheld>() )
        {
            nn::hid::GetNpadState( &m_NpadHandheldCurrentState, nn::hid::NpadId::Handheld );
        }

        nn::hid::GetDebugPadState( &m_DebugPadCurrentState );
    }

    bool SamplePad::IsHold(Button button) NN_NOEXCEPT
    {
        bool isHold = false;

        isHold |= m_DebugPadCurrentState.buttons.Test( DebugPadMap[button] );
        isHold |= m_NpadFullKeyCurrentState.buttons.Test( NpadMap[button] );
        isHold |= m_NpadHandheldCurrentState.buttons.Test( NpadMap[button] );

        return isHold;
    }

    bool SamplePad::IsTrigger(Button button) NN_NOEXCEPT
    {
        bool isTrigger = false;

        isTrigger |= m_DebugPadCurrentState.buttons.Test( DebugPadMap[button] ) && !m_DebugPadPreviousState.buttons.Test( DebugPadMap[button] );
        isTrigger |= m_NpadFullKeyCurrentState.buttons.Test( NpadMap[button] )  && !m_NpadFullKeyPreviousState.buttons.Test( NpadMap[button] );
        isTrigger |= m_NpadHandheldCurrentState.buttons.Test( NpadMap[button] ) && !m_NpadHandheldPreviousState.buttons.Test( NpadMap[button] );

        return isTrigger;
    }

    void SamplePad::Finalize() NN_NOEXCEPT
    {
    }

    void SamplePad::ResetCurrentButtons() NN_NOEXCEPT
    {
        m_DebugPadCurrentState.buttons.Reset();
        m_NpadFullKeyCurrentState.buttons.Reset();
        m_NpadHandheldCurrentState.buttons.Reset();
    }

    void SamplePad::CopyStateCurrentToPrevious() NN_NOEXCEPT
    {
        m_DebugPadPreviousState = m_DebugPadCurrentState;
        m_NpadFullKeyPreviousState = m_NpadFullKeyCurrentState;
        m_NpadHandheldPreviousState = m_NpadHandheldCurrentState;
    }


    void* Allocate(std::size_t size) NN_NOEXCEPT
    {
        return Allocate(size, g_Allocator);
    }

    void* Allocate(std::size_t size, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
    {
        void* pMemory = allocator.Allocate(size);
        NN_ABORT_UNLESS_NOT_NULL(pMemory);
        return pMemory;
    }

    void* Allocate(std::size_t size, int alignment) NN_NOEXCEPT
    {
        return Allocate(size, alignment, g_Allocator);
    }

    void* Allocate(std::size_t size, int alignment, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT
    {
        void* pMemory = allocator.Allocate(size, alignment);
        NN_ABORT_UNLESS_NOT_NULL(pMemory);
        return pMemory;
    }

    void Free( void* pMemory ) NN_NOEXCEPT
    {
        Free(pMemory, g_Allocator);
    }

    void Free(void* pMemory, std::size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(size);
        Free(pMemory, g_Allocator);
    }

    void Free( void* pMemory, nn::mem::StandardAllocator& allocator ) NN_NOEXCEPT
    {
        if (pMemory == nullptr)
        {
            return;
        }

        allocator.Free(pMemory);
        pMemory = nullptr;
    }

    void InitializeHeap() NN_NOEXCEPT
    {
        g_Allocator.Initialize(g_HeapMemory, sizeof(g_HeapMemory));
        g_PoolHeapAllocator.Initialize(g_PoolHeapMemory, sizeof(g_PoolHeapMemory));
    }

    void FinalizeHeap() NN_NOEXCEPT
    {
        g_PoolHeapAllocator.Finalize();
        g_Allocator.Finalize();
    }

    void InitializeFileSystem() NN_NOEXCEPT
    {
        nn::fs::SetAllocator(Allocate, Free);

        size_t cacheSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
        g_MountRomCacheBuffer = Allocate(cacheSize);
        NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::MountRom( MountName, g_MountRomCacheBuffer, cacheSize )
        );
    }

    void FinalizeFileSystem() NN_NOEXCEPT
    {
        nn::fs::Unmount(MountName);

        Free(g_MountRomCacheBuffer);
        g_MountRomCacheBuffer = NULL;
    }

    const char* GetAbsolutePath(const char* relativePath) NN_NOEXCEPT
    {
        nn::util::SNPrintf(g_AbsolutePath, AbsolutePathMax, "%s:/%s", MountName, relativePath);
        return g_AbsolutePath;
    }

    void InitializeHidDevices() NN_NOEXCEPT
    {
        g_SamplePad.Initialize();
    }

    void FinalizeHidDevices() NN_NOEXCEPT
    {
        g_SamplePad.Finalize();
    }

    bool IsHold(SamplePad::Button button) NN_NOEXCEPT
    {
        return g_SamplePad.IsHold(button);
    }

    bool IsTrigger(SamplePad::Button button) NN_NOEXCEPT
    {
        return g_SamplePad.IsTrigger(button);
    }

    void UpdateHidDevices() NN_NOEXCEPT
    {
        g_SamplePad.Update();
    }

    void* GetPoolHeapAddress() NN_NOEXCEPT
    {
        return g_PoolHeapMemory;
    }

    size_t GetPoolHeapSize() NN_NOEXCEPT
    {
        return sizeof(g_PoolHeapMemory);
    }

    void* AllocateForMemoryPool(std::size_t size) NN_NOEXCEPT
    {
        return Allocate(size, g_PoolHeapAllocator);
    }

    void* AllocateForMemoryPool(std::size_t size, int alignment) NN_NOEXCEPT
    {
        return Allocate(size, alignment, g_PoolHeapAllocator);
    }

    void FreeForMemoryPool(void* pMemory) NN_NOEXCEPT
    {
        return Free(pMemory, g_PoolHeapAllocator);
    }
}}
