﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>    // std::abs
#include <algorithm>  // std::min
#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nns/afx/afx_SimpleDelay.h>
#include "afx_Utilities.h"

namespace nns { namespace afx {

namespace {

struct SimpleDelayInternal
{
    SimpleDelayConstant c;
    SimpleDelayParameter p;

    int32_t** ppDelayBuffers;
    int32_t* pLpfStates;
    int delayBufferSampleCount;
    int delayTimeSampleCount;
    int position;
};

const SimpleDelayParameter DefaultParameter =
{
    250,
    0.4f,
    0.5f,
};

}

size_t GetRequiredMemorySizeForSimpleDelay(const SimpleDelayConstant* c)
{
    NN_ASSERT_NOT_NULL(c);
    NN_ASSERT(c->sampleRate >= 1);
    NN_ASSERT(c->channelCount >= 1);
    NN_ASSERT(c->delayTimeMax >= 1);

    const auto delaySampleMax = (c->sampleRate * c->delayTimeMax) / 1000;

    return sizeof(SimpleDelayInternal)
        + sizeof(int32_t*) * c->channelCount                  // ppDelayBuffers
        + sizeof(int32_t) * delaySampleMax * c->channelCount  // *ppDelayBuffers
        + sizeof(float) * c->channelCount;                    // pLpfState
}

void InitializeSimpleDelay(SimpleDelayType* pOutEffect, void* buffer, size_t size, const SimpleDelayConstant* c)
{
    NN_ASSERT_NOT_NULL(pOutEffect);
    NN_ASSERT_NOT_NULL(c);
    NN_ASSERT(c->sampleRate >= 1);
    NN_ASSERT(c->channelCount >= 1);
    NN_ASSERT(c->delayTimeMax >= 1);
    NN_ASSERT_NOT_NULL(buffer);
    NN_ASSERT(size >= GetRequiredMemorySizeForSimpleDelay(c));

    std::memset(buffer, 0, size);

    pOutEffect->_buffer = buffer;
    pOutEffect->_size = size;

    auto ptr = buffer;
    auto e = static_cast<SimpleDelayInternal*>(ptr);
    ptr = AddToPointer(ptr, sizeof(SimpleDelayInternal));
    std::memcpy(&e->c, c, sizeof(SimpleDelayConstant));
    std::memcpy(&e->p, &DefaultParameter, sizeof(SimpleDelayParameter));

    e->ppDelayBuffers = static_cast<int32_t**>(ptr);
    ptr = AddToPointer(ptr, sizeof(int32_t*) * c->channelCount);
    const auto delaySampleMax = (c->sampleRate * c->delayTimeMax) / 1000;
    for (auto ch = 0; ch < c->channelCount; ++ch)
    {
        e->ppDelayBuffers[ch] = static_cast<int32_t*>(ptr);
        ptr = AddToPointer(ptr, sizeof(int32_t) * delaySampleMax);
    }
    e->pLpfStates = static_cast<int32_t*>(ptr);
    ptr = AddToPointer(ptr, sizeof(int32_t) * c->channelCount);
    e->delayBufferSampleCount = delaySampleMax;
    e->delayTimeSampleCount = (c->sampleRate * e->p.delayTime) / 1000;
    e->position = 0;
}

void* FinalizeSimpleDelay(SimpleDelayType* pEffect)
{
    NN_ASSERT_NOT_NULL(pEffect);

    auto buffer = pEffect->_buffer;
    pEffect->_buffer = nullptr;
    pEffect->_size = 0;

    return buffer;
}

SimpleDelayConstant GetSimpleDelayConstant(const SimpleDelayType* pEffect)
{
    NN_ASSERT_NOT_NULL(pEffect);

    auto e = static_cast<SimpleDelayInternal*>(pEffect->_buffer);
    return e->c;
}

SimpleDelayParameter GetSimpleDelayParameter(const SimpleDelayType* pEffect)
{
    NN_ASSERT_NOT_NULL(pEffect);

    auto e = static_cast<SimpleDelayInternal*>(pEffect->_buffer);
    return e->p;
}

void SetSimpleDelayParameter(SimpleDelayType* pEffect, const SimpleDelayParameter* p)
{
    auto e = static_cast<SimpleDelayInternal*>(pEffect->_buffer);

    NN_ASSERT_NOT_NULL(pEffect);
    NN_ASSERT_NOT_NULL(p);
    NN_ASSERT(1 <= p->delayTime && p->delayTime <= e->c.delayTimeMax);
    NN_ASSERT(0.0f <= p->feedbackGain && p->feedbackGain <= 1.0f);
    NN_ASSERT(0.0f <= p->damping && p->damping <= 1.0f);

    std::memcpy(&e->p, p, sizeof(SimpleDelayParameter));
    e->p.damping = std::min(e->p.damping, 0.95f);
    e->delayTimeSampleCount = (e->c.sampleRate * p->delayTime) / 1000;
}

void ApplySimpleDelay(SimpleDelayType* pEffect, int32_t* outputs[], const int32_t* const inputs[], int channelCount, int sampleCount)
{
    NN_ASSERT_NOT_NULL(pEffect);
    NN_ASSERT_NOT_NULL(outputs);
    NN_ASSERT_NOT_NULL(inputs);

    auto e = static_cast<SimpleDelayInternal*>(pEffect->_buffer);
    NN_ASSERT(channelCount <= e->c.channelCount);

    const int Q = 15;
    const int16_t feedbackGain = static_cast<int16_t>((1 << Q) * e->p.feedbackGain);
    const int16_t lpfCoef1 = static_cast<int16_t>((1 << Q) * (1.0f - e->p.damping));
    const int16_t lpfCoef2 = static_cast<int16_t>((1 << Q) * e->p.damping);

    for (auto ch = 0; ch < channelCount; ++ch)
    {
        auto input = inputs[ch];
        auto output = outputs[ch];
        auto delay = e->ppDelayBuffers[ch];
        auto lpfState = e->pLpfStates[ch];
        auto positionRead = e->position - e->delayTimeSampleCount;
        while (positionRead < 0)
        {
            positionRead += e->delayBufferSampleCount;
        }
        auto positionWrite = e->position;

        for (auto i = 0; i < sampleCount; ++i)
        {
            auto delayOutput = delay[positionRead];

            auto feedback = static_cast<int32_t>((static_cast<int64_t>(std::abs(delayOutput)) * feedbackGain) >> Q);
            if (delayOutput < 0)
            {
                feedback = -feedback;
            }
            auto lpfInput = input[i] - feedback;
            auto lpfOutput = static_cast<int32_t>((static_cast<int64_t>(lpfCoef1) * lpfInput + static_cast<int64_t>(lpfCoef2) * lpfState) >> Q);
            lpfState = lpfOutput;

            delay[positionWrite] = lpfOutput;

            output[i] = delayOutput;

            if (++positionRead == e->delayBufferSampleCount)
            {
                positionRead = 0;
            }
            if (++positionWrite == e->delayBufferSampleCount)
            {
                positionWrite = 0;
            }
        }

        e->pLpfStates[ch] = lpfState;
    }

    e->position += sampleCount;
    while (e->position >= e->delayBufferSampleCount)
    {
        e->position -= e->delayBufferSampleCount;
    }
}

}}  // namespace nns::afx
