﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <utility>
#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nns/afx/afx_SimpleChorus.h>
#include "afx_Utilities.h"
#include "afx_SrcCoef.h"
#include "afx_LfoTable.h"

namespace nns { namespace afx {

namespace {

struct SimpleChorusLfo
{
    int32_t* table;
    int32_t lfoBuf[240]; //max sample per frame = 5ms * 48kHz
    int32_t depthSamp;
    int32_t phaseAdd;
    int32_t stepSamp;
    int32_t gradFactor;
    int32_t phase;
    int32_t sign;
    int32_t lastNum;
    int32_t lastValue;
    int32_t grad;
};

struct SimpleChorusInternal
{
    SimpleChorusConstant c;
    SimpleChorusParameter p;
    SimpleChorusLfo lfo;

    float** ppDelayBuffers;
    float** ppHistory;

    int32_t inPosition;
    int32_t outPosition;
    int32_t lastPosition;
    int32_t sizeFP;
    int32_t size;
    int32_t histIndex;
};

const SimpleChorusParameter DefaultParameter =
{
    25,
    10,
    1000,
};

}

size_t GetRequiredMemorySizeForSimpleChorus(const SimpleChorusConstant* c)
{
    NN_ASSERT_NOT_NULL(c);
    NN_ASSERT(c->sampleRate >= 1);
    NN_ASSERT(c->channelCount >= 1);

    const auto delaySampleMax = c->sampleRate / 10;

    return sizeof(SimpleChorusInternal)
        + sizeof(float*) * c->channelCount                  // *ppDelayBuffers
        + sizeof(float) * delaySampleMax * c->channelCount // **ppDelayBuffers
        + sizeof(float*) * c->channelCount                    // *ppHistory
        + sizeof(float) * 4 * c->channelCount;                // **ppHistory
}

static void ComputeInternalParameters(SimpleChorusType* pOutEffect)
{
    NN_ASSERT_NOT_NULL(pOutEffect);
    auto e = static_cast<SimpleChorusInternal*>(pOutEffect->_buffer);
    NN_ASSERT((e->p.baseDelay >= 1) && (e->p.baseDelay <= 50));
    NN_ASSERT((e->p.vibration >= 0) && (e->p.vibration <= e->p.baseDelay));
    NN_ASSERT((e->p.period >= 500) && (e->p.period <= 10000));

    SimpleChorusLfo* simpleLfo = &e->lfo;

    float baseDelayInSamples = e->p.baseDelay * e->c.sampleRate / 1000.0f;
    float depthInSamples = e->p.vibration * e->c.sampleRate / 1000.0f;
    if (depthInSamples >= baseDelayInSamples)
    {
        // NOTE : 1 sample of mergin must be needed.
        //        (to omit over-read of delay line)
        depthInSamples -= 1.f;
        if (depthInSamples < 0.f)
        {
            depthInSamples = 0.f;
        }
    }

    simpleLfo->depthSamp = FloatToFixedPoint(depthInSamples);
    simpleLfo->phaseAdd = FloatToFixedPoint(1000.0f * 256.0f / static_cast<float>(e->p.period * e->c.sampleRate));
    simpleLfo->stepSamp = FloatToFixedPoint(static_cast<float>(e->c.sampleRate * e->p.period) / (1000.0f * 256.0f));
    simpleLfo->gradFactor = FloatToFixedPoint(256.0f * static_cast<float>(e->p.vibration) / static_cast<float>(e->p.period));
    simpleLfo->phase = 0;
    simpleLfo->sign = 0;
    simpleLfo->lastNum = 0xffffffff;
    simpleLfo->lastValue = 0;
    simpleLfo->grad = 0;

    e->inPosition = 0;
    e->outPosition = e->size - static_cast<int32_t>(e->p.baseDelay * e->c.sampleRate / 1000);

    e->outPosition <<= 16;
    e->lastPosition = e->outPosition;
    e->sizeFP = e->size << 16;  //delay line length in fixed-point

    for (int ch = 0; ch < e->c.channelCount; ch++)
    {
        for (int i = 0; i < 4; i++)
        {
            e->ppHistory[ch][i] = 0.0f;
        }
    }
    e->histIndex = 0;
}

void InitializeSimpleChorus(SimpleChorusType* pOutEffect, void* buffer, size_t size, const SimpleChorusConstant* c)
{
    NN_ASSERT_NOT_NULL(pOutEffect);
    NN_ASSERT_NOT_NULL(c);
    NN_ASSERT(c->sampleRate >= 1);
    NN_ASSERT(c->channelCount >= 1);
    NN_ASSERT_NOT_NULL(buffer);
    NN_ASSERT(size >= GetRequiredMemorySizeForSimpleChorus(c));
    pOutEffect->_buffer = buffer;
    pOutEffect->_size = size;

    (void)memset(buffer, 0, size);  //clean out everything

    auto e = static_cast<SimpleChorusInternal*>(pOutEffect->_buffer);

    auto ptr = buffer;
    ptr = AddToPointer(ptr, sizeof(SimpleChorusInternal));

    std::memcpy(&e->c, c, sizeof(SimpleChorusConstant));
    std::memcpy(&e->p, &DefaultParameter, sizeof(SimpleChorusParameter));
    e->lfo.table = GetLfoSinTable();

    // Allocate
    e->ppDelayBuffers = static_cast<float**>(ptr);
    ptr = AddToPointer(ptr, sizeof(float*) * c->channelCount);
    e->size = c->sampleRate / 10;  //max delay line 100ms
    for (auto ch = 0; ch < c->channelCount; ++ch)
    {
        e->ppDelayBuffers[ch] = static_cast<float*>(ptr);
        ptr = AddToPointer(ptr, sizeof(float) * e->size);
    }

    e->ppHistory = static_cast<float**>(ptr);
    ptr = AddToPointer(ptr, sizeof(float*) * c->channelCount);

    for (auto ch = 0; ch < c->channelCount; ++ch)
    {
        e->ppHistory[ch] = static_cast<float*>(ptr);
        ptr = AddToPointer(ptr, sizeof(float) * 4);
    }

    // compute the internal variables
    ComputeInternalParameters(pOutEffect);
}

void* FinalizeSimpleChorus(SimpleChorusType* pEffect)
{
    NN_ASSERT_NOT_NULL(pEffect);

    auto buffer = pEffect->_buffer;
    pEffect->_buffer = nullptr;
    pEffect->_size = 0;

    return buffer;
}

SimpleChorusConstant GetSimpleChorusConstant(const SimpleChorusType* pEffect)
{
    NN_ASSERT_NOT_NULL(pEffect);

    auto e = static_cast<SimpleChorusInternal*>(pEffect->_buffer);
    return e->c;
}

SimpleChorusParameter GetSimpleChorusParameter(const SimpleChorusType* pEffect)
{
    NN_ASSERT_NOT_NULL(pEffect);

    auto e = static_cast<SimpleChorusInternal*>(pEffect->_buffer);
    return e->p;
}

void SetSimpleChorusParameter(SimpleChorusType* pEffect, const SimpleChorusParameter* p)
{
    NN_ASSERT_NOT_NULL(pEffect);
    NN_ASSERT_NOT_NULL(p);
    NN_ASSERT(1 <= p->baseDelay && p->baseDelay <= 50);
    NN_ASSERT(0 <= p->vibration && p->vibration <= p->baseDelay);
    NN_ASSERT(500 <= p->period && p->period <= 10000);

    auto e = static_cast<SimpleChorusInternal*>(pEffect->_buffer);
    std::memcpy(&e->p, p, sizeof(SimpleChorusParameter));

    // update the internal variables
    ComputeInternalParameters(pEffect);
}

static void CalculateLfo(int32_t* buffer, SimpleChorusLfo* lfo, int sampleCount)
{
    int32_t num;
    uint32_t nextNum;
    int32_t curValue;
    int32_t nextValue;
    int64_t level = 0;
    int64_t diff;

    for (int32_t samp = 0; samp < sampleCount; samp++)
    {
        num = lfo->phase & 0xffff0000;

        if (num != lfo->lastNum)
        {
            lfo->lastNum = num;

            num >>= 16;
            nextNum = num + 1;
            nextNum &= 0x7f;

            curValue = lfo->table[num];
            nextValue = lfo->table[nextNum];

            diff = nextValue - curValue;
            diff *= lfo->gradFactor;
            diff >>= 24;

            lfo->grad = static_cast<int32_t>(diff);

            level = static_cast<int64_t>(curValue) * static_cast<int64_t>(lfo->depthSamp);
            level >>= 24;
        }
        else
        {
            level = lfo->lastValue + lfo->grad;
        }
        lfo->lastValue = static_cast<int32_t>(level);

        if (lfo->sign >= 1)
        {
            level *= -1;
        }

        lfo->phase += lfo->phaseAdd;
        if (lfo->phase & 0xff800000)
        {
            lfo->phase &= 0x007fffff;
            lfo->sign ^= 1;
        }

        buffer[samp] = static_cast<int32_t>(level);
    }
}

void ApplySimpleChorus(SimpleChorusType* pEffect, int32_t* outputs[], const int32_t* const inputs[], int channelCount, int sampleCount)
{
    NN_ASSERT_NOT_NULL(pEffect);
    NN_ASSERT_NOT_NULL(outputs);
    NN_ASSERT_NOT_NULL(inputs);

    int32_t lfoLevel;
    int32_t readPos;
    int32_t delta;
    uint32_t deltaI;
    uint32_t deltaD;
    int32_t curReadPos;
    uint32_t index;
    uint32_t coefSetNum;

    auto e = static_cast<SimpleChorusInternal*>(pEffect->_buffer);
    NN_ASSERT(channelCount <= e->c.channelCount);

    int32_t* lfoBuf = e->lfo.lfoBuf;
    CalculateLfo(lfoBuf, &(e->lfo), sampleCount);

    for (auto samp = 0; samp < sampleCount; ++samp)
    {
        lfoLevel = lfoBuf[samp];

        readPos = e->outPosition + lfoLevel;
        if (readPos >= e->sizeFP)
        {
            readPos -= e->sizeFP;
        }
        else if (readPos < 0)
        {
            readPos += e->sizeFP;
        }

        delta = readPos - e->lastPosition;
        if (delta < 0) delta += e->sizeFP;
        deltaI = (static_cast<uint32_t>(delta) & 0xffff0000) >> 16;
        deltaD = static_cast<uint32_t>(delta) & 0x0000ffff;       // 0x0 - 0xffff

        curReadPos = e->lastPosition >> 16;
        index = e->histIndex;

        while (deltaI--)
        {
            for (auto ch = 0; ch < channelCount; ++ch)
            {
                e->ppHistory[ch][index] = e->ppDelayBuffers[ch][curReadPos];
            }
            curReadPos++;
            index++;
            index &= 0x3;

            if (curReadPos >= e->size) curReadPos = 0;
        }

        coefSetNum = (deltaD & 0x0000fe00) >> 9;  // 0 - 127
        e->lastPosition = readPos & 0xffff0000;
        auto coef = GetSrcCoef(coefSetNum);

        for (auto ch = 0; ch < channelCount; ++ch)
        {
            float out = 0.0f;
            for (uint32_t x = 0; x < 4; x++)
            {
                out += e->ppHistory[ch][index++] * (*(coef + x));
                index &= 0x3;
            }
            float data = static_cast<float>(inputs[ch][samp]);
            e->ppDelayBuffers[ch][e->inPosition] = data + out * 0; // feedback gain = 0;
            outputs[ch][samp] = static_cast<int32_t>(0.0f * data) + static_cast<int32_t>(out * 1.0f); // outGain = 1.0;
        }

        e->histIndex = index;
        if (++e->inPosition >= e->size)
        {
            e->inPosition = 0;
        }

        if ((e->outPosition += 0x00010000) >= e->sizeFP)
        {
            e->outPosition = 0;
        }
    }
}

}}  // namespace nns::afx
