﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{TutorialUtil.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines utility functions used in the tutorials.
 */

#include <nn/nn_Assert.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvntutorial/TutorialUtil.h>
#include <nvntutorial/AlignedAllocator.h>

/*
 * ConvertNVNStageToBitField
 * -------------------------
 * Takes a given NVNshaderStage value and returns the
 * equivalent NVNshaderStageBits value.
 */
NVNshaderStageBits ConvertNVNStageToBitField(NVNshaderStage stage)
{
    switch(stage)
    {
        case NVNshaderStage::NVN_SHADER_STAGE_VERTEX:
        {
            return NVNshaderStageBits::NVN_SHADER_STAGE_VERTEX_BIT;
        }
        case NVNshaderStage::NVN_SHADER_STAGE_FRAGMENT:
        {
            return NVNshaderStageBits::NVN_SHADER_STAGE_FRAGMENT_BIT;
        }
        case NVNshaderStage::NVN_SHADER_STAGE_COMPUTE:
        {
            return NVNshaderStageBits::NVN_SHADER_STAGE_COMPUTE_BIT;
        }
        case NVNshaderStage::NVN_SHADER_STAGE_GEOMETRY:
        {
            return NVNshaderStageBits::NVN_SHADER_STAGE_GEOMETRY_BIT;
        }
        case NVNshaderStage::NVN_SHADER_STAGE_TESS_CONTROL:
        {
            return NVNshaderStageBits::NVN_SHADER_STAGE_TESS_CONTROL_BIT;
        }
        case NVNshaderStage::NVN_SHADER_STAGE_TESS_EVALUATION:
        {
            return NVNshaderStageBits::NVN_SHADER_STAGE_TESS_EVALUATION_BIT;
        }
        default:
        {
            NN_ASSERT(0, "Invalid shader stage value");
            return NVNshaderStageBits::NVN_SHADER_STAGE_BITS_LARGE;
        }
    }
}

/*
 * Align
 * -----
 * Takes the given size and returns a new size at the given
 * alignment.
 */

size_t Align(size_t size, size_t alignment)
{
    size_t temp = (alignment - (size % alignment)) % alignment;

    return temp + size;
}

/*
 * AlignedAllocate
 * ---------------
 * Uses the AlignedAllocator singleton to allocate a block of
 * data with an address of the proper alignment.
 */

void* AlignedAllocate(size_t size, size_t alignment)
{
    AlignedAllocator* pAllocator = AlignedAllocator::GetInstance();

    return pAllocator->Allocate(size, alignment);
}

/*
 * AlignedDeallocate
 * -----------------
 * Frees a block of memory allocated with AlignedAllocate.
 */
void AlignedDeallocate(void* ptr)
{
    AlignedAllocator* pAllocator = AlignedAllocator::GetInstance();

    pAllocator->Deallocate(ptr);
}
