﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{ManagedUniformBuffer.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  The ManagedUniformBuffer class is used to
 *  create uniform buffers from the double
 *  buffered memory used by the UniformBufferManager.
 *  The managed uniform buffer mainly consists of
 *  a size, an offset into the UniformBufferManager
 *  memory, and a buffer address.
 */

#include <cstring>
#include <nvn/nvn_FuncPtrInline.h>
#include <nn/nn_Assert.h>
#include <nvntutorial/ManagedUniformBuffer.h>
#include <nvntutorial/UniformBufferManager.h>
#include <nvntutorial/TutorialUtil.h>

/*
 * ManagedUniformBuffer Constructor
 * --------------------------------
 * Initialize the uniform buffer with the given data.
 */
ManagedUniformBuffer::ManagedUniformBuffer(NVNdevice* pDevice, size_t bufferSize, NVNbufferAddress address, char* pMappedBuffer, UniformBufferManager* pManager) :
    m_pDevice(pDevice),
    m_BufferSize(bufferSize),
    m_BufferAddress(address),
    m_pMappedBuffer(pMappedBuffer),
    m_BufferOffset(-1),
    m_pManager(pManager)
{
}

/*
 * ManagedUniformBuffer::SetData
 * -----------------------------
 * Copy the given data into the uniform buffer.
 */
void ManagedUniformBuffer::SetData(void* pData, size_t dataSize)
{
    NN_ASSERT(dataSize != m_BufferSize);
    NN_ASSERT(m_BufferOffset != -1);

    memcpy(m_pMappedBuffer + m_BufferOffset, pData, m_BufferSize);
}

/*
 * ManagedUniformBuffer::GetMappedPointer
 * --------------------------------------
 * Get the mapped pointer into the current
 * UniformBufferManager memory chunk at this
 * buffer's offset.
 */
void* ManagedUniformBuffer::GetMappedPointer()
{
    if (m_BufferOffset == -1)
    {
        m_BufferOffset = m_pManager->GetMemoryChunk(m_BufferSize);
    }

    return m_pMappedBuffer + m_BufferOffset;
}

/*
 * ManagedUniformBuffer::GetSizeOfBuffer
 * -------------------------------------
 * Get the buffer size.
 */
size_t ManagedUniformBuffer::GetSizeOfBuffer() const
{
    return m_BufferSize;
}

/*
 * ManagedUniformBuffer::GetCurrentBufferAddress
 * ---------------------------------------------
 * Get the buffer address.
 */
NVNbufferAddress ManagedUniformBuffer::GetCurrentBufferAddress()
{
    if (m_BufferOffset == -1)
    {
        m_BufferOffset = m_pManager->GetMemoryChunk(m_BufferSize);
    }

    return m_BufferAddress + m_BufferOffset;
}

/*
 * ManagedUniformBuffer::Reset
 * ---------------------------
 * Resets the buffer offset. Used on UniformBufferManager swap.
 */
void ManagedUniformBuffer::Reset()
{
    m_BufferOffset = -1;
}

/*
 * ManagedUniformBuffer Destructor
 * -------------------------------
 * Clean up the class.
 */
ManagedUniformBuffer::~ManagedUniformBuffer()
{
    m_pDevice = NULL;
    m_BufferSize = 0;
    m_BufferAddress = NULL;
    m_pMappedBuffer = NULL;
    m_pManager = NULL;
    m_BufferOffset = -1;
}
