﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{GraphicsObjectSpecializedCube.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a class that derives from the base GraphicsObject
 *  class for drawing the graphicsObjectSpecializedCube.out asset file.
 */

#include <nn/util/util_Vector.h>
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nn/nn_Assert.h>
#include <nvntutorial/GraphicsObjectSpecializedCube.h>
#include <nvntutorial/UniformBufferManager.h>
#include <nvntutorial/ManagedUniformBuffer.h>
#include <nvntutorial/AssetFileDataHolder.h>

UniformBlockReflectionInfo GraphicsObjectSpecializedCube::vertBlockInfo;
UniformBlockReflectionInfo GraphicsObjectSpecializedCube::fragBlockInfo;

/*
 * GraphicsObjectSpecializedCube Constructor
 * ------------------------------
 * Sets up default values for various members as well as the
 * name of the asset file that this particular derivation of
 * the GraphicsObject is associated with.
 */
GraphicsObjectSpecializedCube::GraphicsObjectSpecializedCube(TextureIDManager* pTextureIDManager) :
    m_ShaderIndex(0),
    m_pUniformBlockVS(NULL),
    m_pUniformBlockFS(NULL),
    m_pTextureIDManager(pTextureIDManager)
{
    m_Name = "graphicsObjectSpecializedCube.out";

    nn::util::MatrixIdentity(&m_WorldMatrix);
    nn::util::MatrixIdentity(&m_CameraMatrix);
    nn::util::MatrixIdentity(&m_ProjectionMatrix);
}

/*
 * GraphicsObjectSpecializedCube::Init
 * ------------------------
 * Sets the data holder that the object is associated
 * with and sets up the proper render state.
 */
void GraphicsObjectSpecializedCube::Init(AssetFileDataHolder* data)
{
    SetDefaultState();

    nvnDepthStencilStateSetDepthTestEnable(&m_DepthStencilState, NVN_TRUE);
    nvnDepthStencilStateSetDepthWriteEnable(&m_DepthStencilState, NVN_TRUE);
    nvnDepthStencilStateSetDepthFunc(&m_DepthStencilState, NVN_DEPTH_FUNC_LESS);

    nvnPolygonStateSetFrontFace(&m_PolygonState, NVNfrontFace::NVN_FRONT_FACE_CCW);
    nvnPolygonStateSetCullFace(&m_PolygonState, NVNface::NVN_FACE_BACK);
    nvnPolygonStateSetPolygonMode(&m_PolygonState, NVN_POLYGON_MODE_FILL);

    nvnColorStateSetBlendEnable(&m_ColorState, 0, NVN_TRUE);

    m_pAssetData = data;
}

/*
 * GraphicsObjectSpecializedCube Destructor
 * -----------------------------
 * Empty destructor.
 */
GraphicsObjectSpecializedCube::~GraphicsObjectSpecializedCube()
{
}

/*
 * GraphicsObjectSpecializedCube::SetupUniforms
 * ---------------------------------
 * Sets up the object's uniform buffer.
 */
void GraphicsObjectSpecializedCube::SetupUniforms(UniformBufferManager* uniformBufferManager)
{
    m_pUniformBlockVS = uniformBufferManager->CreateUniformBuffer(vertBlockInfo.blockSize);
    m_pUniformBlockFS = uniformBufferManager->CreateUniformBuffer(fragBlockInfo.blockSize);
}

/*
 * GraphicsObjectSpecializedCube::UpdateUniforms
 * ----------------------------------
 * Updates the object's uniform buffer with the previously set
 * member data.
 */
void GraphicsObjectSpecializedCube::UpdateUniforms()
{
    nn::util::Matrix4x3fType scale;
    nn::util::MatrixIdentity(&scale);
    nn::util::Vector3fType scaleVector;
    nn::util::VectorSet(&scaleVector, m_ScaleX, m_ScaleY, m_ScaleZ);
    nn::util::MatrixSetScale(&scale, scaleVector);

    nn::util::Matrix4x3fType rotateX;
    nn::util::MatrixIdentity(&rotateX);
    nn::util::Vector3fType rotateXVector;
    nn::util::VectorSet(&rotateXVector, m_RotateX * 3.1459f / 180.0f, 0.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&rotateX, rotateXVector);

    nn::util::Matrix4x3fType rotateY;
    nn::util::MatrixIdentity(&rotateY);
    nn::util::Vector3fType rotateYVector;
    nn::util::VectorSet(&rotateYVector, 0.0f, m_RotateY * 3.14159f / 180.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&rotateY, rotateYVector);

    nn::util::Matrix4x3fType rotateZ;
    nn::util::MatrixIdentity(&rotateZ);
    nn::util::Vector3fType rotateZVector;
    nn::util::VectorSet(&rotateZVector, 0.0f, 0.0f, m_RotateZ * 3.14159f / 180.0f);
    nn::util::MatrixSetRotateXyz(&rotateZ, rotateZVector);

    nn::util::Matrix4x3fType translate;
    nn::util::MatrixIdentity(&translate);
    nn::util::Vector3fType translateVector;
    nn::util::VectorSet(&translateVector, m_TranslateX, m_TranslateY, m_TranslateZ);
    nn::util::MatrixSetTranslate(&translate, translateVector);

    nn::util::Matrix4x3fType tempMat1;
    nn::util::Matrix4x3fType tempMat2;

    nn::util::MatrixMultiply(&tempMat1, scale, rotateX);
    nn::util::MatrixMultiply(&tempMat2, tempMat1, rotateY);
    nn::util::MatrixMultiply(&tempMat1, tempMat2, rotateZ);
    nn::util::MatrixMultiply(&tempMat2, tempMat1, translate);

    nn::util::MatrixConvert(&m_WorldMatrix, tempMat2);

    uint8_t* uniformMapVS = reinterpret_cast<uint8_t*>(m_pUniformBlockVS->GetMappedPointer());
    uint8_t* uniformMapFS = reinterpret_cast<uint8_t*>(m_pUniformBlockFS->GetMappedPointer());

        /*
         * Uses the reflection information retrieved through ReceiveUniformBlockReflectionInformation
         * and ReceiveUniformReflectionInformation to update the uniform buffers for the object.
         *
         * Of note in this tutorial, the fragment uniform block consists of an array of 4 64-bit
         * bindless texture handles and 2 integers.  The two integers are specialized and, despite
         * basically being constant in this instance, still account for 8 bytes in the size of the
         * overall struct.  Anything written (or not written) to the uniform block's buffer at the
         * specialized variables location (i.e. the 2 ints) is ignored and not taken into account.
         */

        /* Update the Vertex uniform block */
    nn::util::Float4x4 temp;
    nn::util::MatrixStore(&temp, m_WorldMatrix);
    UniformReflectionInfo& modelUniformInfo = vertBlockInfo.uniformInfo.at("u_modelMtx");
    memcpy(uniformMapVS + modelUniformInfo.offset, &temp, sizeof(m_WorldMatrix));

    nn::util::MatrixStore(&temp, m_CameraMatrix);
    UniformReflectionInfo& viewUniformInfo = vertBlockInfo.uniformInfo.at("u_viewMtx");
    memcpy(uniformMapVS + viewUniformInfo.offset, &temp, sizeof(m_CameraMatrix));

    nn::util::MatrixStore(&temp, m_ProjectionMatrix);
    UniformReflectionInfo& projUniformInfo = vertBlockInfo.uniformInfo.at("u_projMtx");
    memcpy(uniformMapVS + projUniformInfo.offset, &temp, sizeof(m_ProjectionMatrix));

        /* Update the Fragment uniform block */
    std::vector<NVNTextureData*>& textureData = m_pAssetData->GetTextureData();
    size_t numTextures = textureData.size();
    NN_ASSERT(numTextures > 0, "No textures loaded\n");

    //UniformReflectionInfo& texUniformInfo = fragBlockInfo.uniformInfo.at("u_bindlessTex[0]");

    for(int i = 0; i < 4; ++i)
    {
        NVNtextureHandle handle = textureData[i]->m_TextureHandle;

        memcpy(uniformMapFS, &handle, sizeof(NVNtextureHandle));
        uniformMapFS += sizeof(NVNtextureHandle);
    }
}

/*
 * GraphicsObjectSpecializedCube::SetWorldMatrix
 * ----------------------------------
 * Sets the objetcts world matrix.
 */
void GraphicsObjectSpecializedCube::SetWorldMatrix(const nn::util::Matrix4x4fType& world)
{
    m_WorldMatrix = world;
}

/*
 * GraphicsObjectSpecializedCube::SetCameraMatrix
 * -----------------------------------
 * Sets the current camera matrix.
 */
void GraphicsObjectSpecializedCube::SetCameraMatrix(const nn::util::Matrix4x4fType& camera)
{
    m_CameraMatrix = camera;
}

/*
 * GraphicsObjectSpecializedCube::SetProjectionMatrix
 * ---------------------------------------
 * Sets the projection matrix for drawing the object.
 */
void GraphicsObjectSpecializedCube::SetProjectionMatrix(const nn::util::Matrix4x4fType& projection)
{
    m_ProjectionMatrix = projection;
}

/*
 * GraphicsObjectSpecializedCube::SetShaderIndex
 * ---------------------------------------
 * Sets the index for which shader to use
 */
void GraphicsObjectSpecializedCube::SetShaderIndex(int index)
{
    m_ShaderIndex = index;
}

/*
 * GraphicsObjectSpecializedCube::SetScale
 * ----------------------------
 * Sets the objects world space scale.
 */
void GraphicsObjectSpecializedCube::SetScale(float x, float y, float z)
{
    m_ScaleX = x;
    m_ScaleY = y;
    m_ScaleZ = z;
}

/*
 * GraphicsObjectSpecializedCube::SetTranslate
 * --------------------------------
 * Sets the objects world space translation.
 */
void GraphicsObjectSpecializedCube::SetTranslate(float x, float y, float z)
{
    m_TranslateX = x;
    m_TranslateY = y;
    m_TranslateZ = z;
}

/*
 * GraphicsObjectSpecializedCube::SetRotate
 * -----------------------------
 * Sets the objects world space rotation.
 */
void GraphicsObjectSpecializedCube::SetRotate(float x, float y, float z)
{
    m_RotateX = x;
    m_RotateY = y;
    m_RotateZ = z;
}

/*
 * GraphicsObjectSpecializedCube::Draw
 * ------------------------
 * Draws the object with the given command buffer, its data
 * holder, and the render states that have been set.
 */
void GraphicsObjectSpecializedCube::Draw(NVNcommandBuffer* pCommandBuffer) const
{
    std::vector<NVNModelData*>&   modelData   = m_pAssetData->GetModelData();
    std::vector<NVNProgramData*>& programData = m_pAssetData->GetProgramData();

        /* Bind the vertex buffer data. */
    Model* pModel = &modelData[0]->m_Model;
    NVNbufferAddress vboAddr = nvnBufferGetAddress(&modelData[0]->m_VertexBuffer);
    for (size_t j = 0; j < pModel->m_VertexAttributes.size(); ++j)
    {
        VertexAttribute& attr = pModel->m_VertexAttributes[j];
        nvnCommandBufferBindVertexBuffer(pCommandBuffer, attr.m_Location, vboAddr + modelData[0]->m_VertexAttributeBufferOffsets[j], attr.m_DataSize);
    }

        /* Bind the uniform buffer using reflection information */
    nvnCommandBufferBindUniformBuffer(pCommandBuffer,
                                      NVN_SHADER_STAGE_VERTEX,
                                      vertBlockInfo.bindingLocation,
                                      m_pUniformBlockVS->GetCurrentBufferAddress(),
                                      vertBlockInfo.blockSize);

    nvnCommandBufferBindUniformBuffer(pCommandBuffer,
                                      NVN_SHADER_STAGE_FRAGMENT,
                                      fragBlockInfo.bindingLocation,
                                      m_pUniformBlockFS->GetCurrentBufferAddress(),
                                      fragBlockInfo.blockSize);

        /*
         * The texture and sampler descriptor pools must be bound to any command buffer
         * that references textures or samplers registered to the pools.
         */
    m_pTextureIDManager->SetTexturePool(pCommandBuffer);
    m_pTextureIDManager->SetSamplerPool(pCommandBuffer);

        /* Bind the vertex attribute states and vertex stream states. */
    nvnCommandBufferBindVertexAttribState(pCommandBuffer, static_cast<int>(modelData[0]->m_VertexAttributeStates.size()), &modelData[0]->m_VertexAttributeStates[0]);
    nvnCommandBufferBindVertexStreamState(pCommandBuffer, static_cast<int>(modelData[0]->m_VertexStreamStates.size()), &modelData[0]->m_VertexStreamStates[0]);

        /* Bind the shader program. */
    programData[m_ShaderIndex]->BindShaderProgram(pCommandBuffer);

        /* Draw the model. */
    NVNindexType indexType = (NVNindexType)pModel->m_IndexData.m_IndexType;
    int numIndices = pModel->m_IndexData.m_DataSize / pModel->m_IndexData.m_Stride;
    nvnCommandBufferDrawElements(pCommandBuffer,
                                 (NVNdrawPrimitive)pModel->m_NvnDrawPrimitiveType,
                                 indexType,
                                 numIndices,
                                 nvnBufferGetAddress(&modelData[0]->m_IndexBuffer));
}

/*
 * GraphicsObjectSpecializedCube::ReceiveUniformBlockReflectionInformation
 * ------------------------
 * Static function that is used to receive reflection information
 * relating to the uniform blocks in the shaders.
 */
void GraphicsObjectSpecializedCube::ReceiveUniformBlockReflectionInformation(
    NVNshaderStageBits shaderStage,
    uint32_t bindingLocation,
    uint32_t blockSize,
    const std::string& blockName)
{
    if(shaderStage == NVN_SHADER_STAGE_VERTEX_BIT)
    {
        vertBlockInfo.bindingLocation = bindingLocation;
        vertBlockInfo.blockSize = blockSize;
        vertBlockInfo.blockName = blockName;
    }

    else if(shaderStage == NVN_SHADER_STAGE_FRAGMENT_BIT)
    {
        fragBlockInfo.bindingLocation = bindingLocation;
        fragBlockInfo.blockSize = blockSize;
        fragBlockInfo.blockName = blockName;
    }
}

/*
 * GraphicsObjectSpecializedCube::ReceiveUniformReflectionInformation
 * ------------------------
 * Static function that is used to receive reflection information
 * relating to the individual uniforms in the uniform blocks.
 */
void GraphicsObjectSpecializedCube::ReceiveUniformReflectionInformation(
    NVNshaderStageBits shaderStage,
    uint32_t offset,
    uint32_t dataSize,
    const std::string& uniformName)
{
    UniformReflectionInfo uniformInfo;
    uniformInfo.offset = offset;
    uniformInfo.size = dataSize;
    if(shaderStage == NVN_SHADER_STAGE_VERTEX_BIT)
    {
        vertBlockInfo.uniformInfo[uniformName] = uniformInfo;
    }

    else if(shaderStage == NVN_SHADER_STAGE_FRAGMENT_BIT)
    {
        fragBlockInfo.uniformInfo[uniformName] = uniformInfo;
    }
}
