﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneScanResult.h"

#include <cstring>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_InfraApi.h>
#include <nn/wlan/wlan_ScanResultReader.h>
#include <nn/wlan/wlan_BssDescriptionReader.h>
#include <nn/wlan/wlan_Result.h>

SceneScanResult::SceneScanResult(
        ISceneChanger* changer,
        WirelessData* pDistributor
        )
: BaseScene(changer, pDistributor)
{
    m_curY = 0;
    m_line = 0;
    m_bssCnt = 0;
    m_pBuf = NULL;
}

void SceneScanResult::Initialize() NN_NOEXCEPT
{
    m_pBuf = m_distributor->GetScanResult();
    nn::wlan::BeaconScanResultReader resultReader(m_pBuf);
    m_bssCnt = resultReader.GetCount();

    // スキャン結果を精査
    int bssCnt = 0;
    for(int i = 0; i < m_bssCnt; i++)
    {
        nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();

        // ステルスSSIDは除外する
        bool isSkip = false;
        if( beacon.GetSsid().GetLength() == 0 ||
                beacon.GetSsid().GetSsidData() == NULL )
        {
            isSkip = true;
        }
        for( int j = 0; j < beacon.GetSsid().GetLength(); ++j )
        {
            if( std::isprint(beacon.GetSsid().GetSsidData()[j]) == 0 )
            {
                // 非表示文字が含まれているものも除外する
                isSkip = true;
                break;
            }
        }

        // サポートしていない暗号タイプも除外する。ついでにセキュリティを取得しておく。
        nn::wlan::Security tmpSec;
        memset(&tmpSec, 0, sizeof(nn::wlan::Security));
        if (beacon.IsWpa2Supported())
        {
            const nn::wlan::WpaInfoElementReader<nn::wlan::InfoElementReader>* pWpa
                = beacon.GetWpa2Ie();
            int k = 0;
            for( k = 0; k < pWpa->GetPairwiseCipherSuiteCount(); k++ )
            {
                if( pWpa->GetPairwiseCipherSuite(k) == nn::wlan::WPA_CIPHER_AES_CCMP )
                {
                    // AESさえサポートしていればOK
                    tmpSec.privacyMode = nn::wlan::SecurityMode_Wpa2Aes;
                    break;
                }
            }
            if( k == pWpa->GetPairwiseCipherSuiteCount() )
            {
                isSkip = true;
            }
            if( pWpa->GetGroupCipherSuite() != nn::wlan::WPA_CIPHER_AES_CCMP )
            {
                isSkip = true;
            }
            else
            {
                tmpSec.groupPrivacyMode = nn::wlan::SecurityMode_Wpa2Aes;
            }
        }
        if( beacon.IsWpaSupported() == true )
        {
            const nn::wlan::WpaInfoElementReader<nn::wlan::WifiInfoElementReader>* pWpa
                = beacon.GetWpaIe();
            int k = 0;
            for( k = 0; k < pWpa->GetPairwiseCipherSuiteCount(); k++ )
            {
                if( pWpa->GetPairwiseCipherSuite(k) == nn::wlan::WPA_CIPHER_AES_CCMP )
                {
                    // AESさえサポートしていればOK
                    if( tmpSec.privacyMode == 0 )
                    {
                        tmpSec.privacyMode = nn::wlan::SecurityMode_WpaAes;
                    }
                    break;
                }
            }
            if( k == pWpa->GetPairwiseCipherSuiteCount() )
            {
                isSkip = true;
            }
            if( pWpa->GetGroupCipherSuite() != nn::wlan::WPA_CIPHER_AES_CCMP )
            {
                isSkip = true;
            }
            else
            {
                if( tmpSec.groupPrivacyMode == 0 )
                {
                    tmpSec.groupPrivacyMode = nn::wlan::SecurityMode_WpaAes;
                }
            }
        }
        if( beacon.GetCapabilityInformation() & 0x0010 ) // WEP
        {
            if( tmpSec.privacyMode == 0 )
            {
                tmpSec.privacyMode = nn::wlan::SecurityMode_Wep64Open;  // TORIAEZU
            }
        }
        if( tmpSec.privacyMode == 0 )
        {
            tmpSec.privacyMode = nn::wlan::SecurityMode_Open;
        }
        if( isSkip == true )
        {
            continue;
        }

        // スキャン結果の保存
        m_connectionParam[bssCnt].ssid.Set(beacon.GetSsid().GetSsidData(), beacon.GetSsid().GetLength());
        m_connectionParam[bssCnt].channel = beacon.GetChannel();
        m_connectionParam[bssCnt].rssi = beacon.GetRssi();
        m_connectionParam[bssCnt].linkLevel = beacon.GetLinkLevel();
        memcpy(&(m_connectionParam[bssCnt].security), &tmpSec, sizeof(nn::wlan::Security));
        bssCnt++;
        if( bssCnt >= SCAN_RESULT_SAVE_MAX_CNT )
        {
            // 保存最大数を超過
            break;
        }
    }
    m_bssCnt = bssCnt;
    m_dispMax = ( m_bssCnt < MAX_LINES ) ? m_bssCnt : MAX_LINES;
    NN_LOG("bssCnt:%d, m_dispMax:%d\n", m_bssCnt, m_dispMax);
} // NOLINT(impl/function_size)

void SceneScanResult::Finalize() NN_NOEXCEPT
{
}

void SceneScanResult::Update() NN_NOEXCEPT
{
    Npad npad = Npad::GetInstance();
    npad.UpdatePadState();

    if( npad.IsTrigger(Npad::A) )
    {
        m_distributor->SetConnectionParam(
                m_connectionParam[m_line].ssid,
                m_connectionParam[m_line].channel,
                m_connectionParam[m_line].security,
                NULL);
        char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
        NN_LOG("%s, %dch\n", m_connectionParam[m_line].ssid.GetHexString(ssidStr), m_connectionParam[m_line].channel);
        m_SceneChanger->ChangeScene(eScene_ConnectionParam);
    }
    else if( npad.IsTrigger(Npad::B) )
    {
        m_SceneChanger->ChangeScene(eScene_Title);
    }
    else if( npad.IsTrigger(Npad::Y) )
    {
        m_SceneChanger->ChangeScene(eScene_Scanning);
    }
    else if( npad.IsHold(Npad::UP) || npad.IsTrigger(Npad::UP) )
    {
        m_curY--;
        if( m_curY < 0 )
        {
            m_curY = 0;
        }
        m_line--;
        if( m_line < 0 )
        {
            m_line = 0;
        }
    }
    else if( npad.IsHold(Npad::DOWN) || npad.IsTrigger(Npad::DOWN) )
    {
        m_curY++;
        if( m_curY > m_dispMax - 1 )
        {
            m_curY = m_dispMax - 1;
        }
        m_line++;
        if( m_line > m_bssCnt - 1 )
        {
            m_line = m_bssCnt - 1;
        }
    }
}

void SceneScanResult::Draw(
        GraphicsSystem* pGraphicsSystem,
        FontSystem* pFontSystem
        ) NN_NOEXCEPT
{
    BaseScene::Draw(pGraphicsSystem, pFontSystem);

    nn::gfx::util::DebugFontTextWriter&
        textWriter = pFontSystem->GetDebugFontTextWriter();

    const nn::util::Unorm8x4& textColor = Color::White;
    const nn::util::Unorm8x4& textColor2 = Color::Yellow;

    textWriter.SetTextColor( textColor );
    textWriter.SetScale( FONT_SCALE, FONT_SCALE );

    // スキャン結果の表示
    // 現在のカーソル位置から表示開始位置を計算
    int dispLine = m_line - m_curY;
    if( dispLine < 0 )
    {
        dispLine = 0;
    }

    char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
    uint32_t displayCnt = (m_bssCnt > MAX_LINES) ? MAX_LINES : m_bssCnt;
    for( int i = 0; i < displayCnt; i++ )
    {
        if( i == m_curY )
        {
            textWriter.SetTextColor( textColor2 );
        }
        else
        {
            textWriter.SetTextColor( textColor );
        }
        textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE * 1), MY_INITIAL_Y + (FONT_SIZE * i) );
        textWriter.Print( "%s", m_connectionParam[dispLine].ssid.GetHexString(ssidStr) );

        // チャンネル、RSSI、セキュリティ情報を表示
        char secStr[10];
        switch( m_connectionParam[dispLine].security.privacyMode )
        {
        case nn::wlan::SecurityMode_Open:
            strcpy(secStr, "OPEN");
            break;
        case nn::wlan::SecurityMode_Wep64Open:
        case nn::wlan::SecurityMode_Wep64Shared:
        case nn::wlan::SecurityMode_Wep128Open:
        case nn::wlan::SecurityMode_Wep128Shared:
            strcpy(secStr, "WEP");
            break;
        case nn::wlan::SecurityMode_WpaAes:
            strcpy(secStr, "WPA-AES");
            break;
        case nn::wlan::SecurityMode_WpaTkip:
            strcpy(secStr, "WPA-TKIP");
            break;
        case nn::wlan::SecurityMode_Wpa2Aes:
            strcpy(secStr, "WPA2-AES");
            break;
        case nn::wlan::SecurityMode_Wpa2Tkip:
            strcpy(secStr, "WPA2-TKIP");
            break;
        default:
            strcpy(secStr, "UNKNOWN");
            break;
        }
        textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE * 24), MY_INITIAL_Y + (FONT_SIZE * i) );
        textWriter.Print( "%3dch  %3d(%d)  %s",
                m_connectionParam[dispLine].channel, m_connectionParam[dispLine].rssi,
                m_connectionParam[dispLine].linkLevel, secStr);
        dispLine++;
    }

    textWriter.SetTextColor( textColor );
    textWriter.SetCursor( MY_INITIAL_X, MY_INITIAL_Y + (FONT_SIZE * m_curY) );
    textWriter.Print( ">" );

    textWriter.SetCursor( INITIAL_X + FONT_SIZE, INITIAL_Y + (FONT_SIZE * 20) );
    textWriter.Print( "[A]:Connect [B]:Back to Title [Y]:Scan again" );

    pGraphicsSystem->BeginDraw();
    pFontSystem->Draw();
    pGraphicsSystem->EndDraw();

    pGraphicsSystem->Synchronize(
        nn::TimeSpan::FromNanoSeconds( 1000 * 1000 * 1000 / FRAME_RATE ) );
}
