﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneDirectInput.h"

#include <cstring>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/swkbd/swkbd_Api.h>
#include <nn/swkbd/swkbd_Result.h>
#include <nn/util/util_FormatString.h>

namespace {
const char* securityStrs[] = {
        "OPEN",
        "WEP64",
        "WEP128",
        "WPA-AES",
        "WPA2-AES"
};
}

SceneDirectInput::SceneDirectInput(
        ISceneChanger* changer,
        WirelessData* pDistributor
        )
: BaseScene(changer, pDistributor)
{
    m_curY = 0;
    m_curX = 0;
    m_ssid.Set("");
    m_channel = 0;
    memset(&m_security, 0, sizeof(nn::wlan::Security));
    m_pApplicationHeap = NULL;
    strcpy(m_pwStr, "");
    strcpy(m_commentStr, "");
}

void SceneDirectInput::Initialize() NN_NOEXCEPT
{
    m_distributor->GetConnectionParam(&m_ssid, &m_channel, &m_security, m_pwStr);
    switch( m_security.privacyMode )
    {
    case nn::wlan::SecurityMode_Open:
        m_curX = SecurityItem_Open;
        break;
    case nn::wlan::SecurityMode_Wep64Open:
    case nn::wlan::SecurityMode_Wep64Shared:
        m_curX = SecurityItem_Wep64;
        break;
    case nn::wlan::SecurityMode_Wep128Open:
    case nn::wlan::SecurityMode_Wep128Shared:
        m_curX = SecurityItem_Wep128;
        break;
    case nn::wlan::SecurityMode_WpaAes:
        m_curX = SecurityItem_WpaAes;
        break;
    case nn::wlan::SecurityMode_Wpa2Aes:
        m_curX = SecurityItem_Wpa2Aes;
        break;
    default:
        m_curX = SecurityItem_Open;
        break;
    }
}

void SceneDirectInput::Finalize() NN_NOEXCEPT
{
}

void SceneDirectInput::Update() NN_NOEXCEPT
{
    Npad npad = Npad::GetInstance();
    npad.UpdatePadState();

    if( npad.IsTrigger(Npad::B) )
    {
        m_SceneChanger->ChangeScene(eScene_Title);
    }
    else if( npad.IsTrigger(Npad::A) )
    {
        switch( m_curY )
        {
        case 0:
        {
            char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
            char outStr[nn::wlan::Ssid::SsidHexStringLengthMax] = "";
            InputBySwkbd(m_ssid.GetHexString(ssidStr), outStr, InputItem_Ssid);
            m_ssid.Set(outStr);
            break;
        }
        case 1:
        {
            char outStr[64] = "";
            InputBySwkbd(m_pwStr, outStr, InputItem_Pw);
            strcpy(m_pwStr, outStr);
            break;
        }
        case 3:
        {
            bool isValid = true;
            switch( m_curX )
            {
            case SecurityItem_Open:
                m_security.privacyMode = nn::wlan::SecurityMode_Open;
                break;
            case SecurityItem_Wep64:
                // 入力文字が5文字かチェック
                if( strlen(m_pwStr) != 5 )
                {
                    isValid = false;
                    strcpy(m_commentStr, "You need to input 5 characters for PW");
                    break;
                }
                else
                {
                    m_security.privacyMode = nn::wlan::SecurityMode_Wep64Open;
                }
                break;
            case SecurityItem_Wep128:
                // 入力文字が13文字かチェック
                if( strlen(m_pwStr) != 13 )
                {
                    isValid = false;
                    strcpy(m_commentStr, "You need to input 13 characters for PW");
                    break;
                }
                else
                {
                    m_security.privacyMode = nn::wlan::SecurityMode_Wep128Open;
                }
                break;
            case SecurityItem_WpaAes:
            case SecurityItem_Wpa2Aes:
                // 入力文字が8文字以上かチェック
                if( strlen(m_pwStr) < 8 )
                {
                    isValid = false;
                    strcpy(m_commentStr, "You need to input 8 or more characters for PW");
                    break;
                }
                else
                {
                    if( m_curX == SecurityItem_WpaAes )
                    {
                        m_security.privacyMode = nn::wlan::SecurityMode_WpaAes;
                        m_security.groupPrivacyMode = nn::wlan::SecurityMode_WpaAes;
                    }
                    else
                    {
                        m_security.privacyMode = nn::wlan::SecurityMode_Wpa2Aes;
                        m_security.groupPrivacyMode = nn::wlan::SecurityMode_Wpa2Aes;
                    }
                }
                break;
            default:
                break;
            }
            if( m_ssid.GetLength() == 0 )
            {
                strcpy(m_commentStr, "You need to input SSID");
                isValid = false;
            }
            if( isValid == false )
            {
                // 入力情報が不正なので接続させない
                break;
            }

            // 入力情報に不正がない場合
            if( m_security.privacyMode >= nn::wlan::SecurityMode_Wep64Open && m_security.privacyMode <= nn::wlan::SecurityMode_Wep128Shared )
            {
                // WEPの場合はASCIIコードの文字列に変換しておく必要がある
                for( int i = 0; i < strlen(m_pwStr); i++ )
                {
                    char tmpStr[10];
                    nn::util::SNPrintf(tmpStr, sizeof(tmpStr), "%d", m_pwStr[i]);
                    strcat(reinterpret_cast<char*>(m_security.key), tmpStr);
                }
                NN_LOG("WEP PW:%s\n", reinterpret_cast<char*>(m_security.key));
            }
            else if( m_security.privacyMode >= nn::wlan::SecurityMode_WpaTkip )
            {
                strcpy(reinterpret_cast<char*>(m_security.key), m_pwStr);
            }
            m_distributor->SetConnectionParam(m_ssid, m_channel, m_security, m_pwStr);
            m_SceneChanger->ChangeScene(eScene_Connecting);
            break;
        }
        default:
            break;
        }
    }
    else if( npad.IsTrigger(Npad::DOWN) )
    {
        m_curY++;
        if( m_curY > ITEM_NUM - 1 )
        {
            m_curY = 0;
        }
    }
    else if( npad.IsTrigger(Npad::UP) )
    {
        m_curY--;
        if( m_curY < 0 )
        {
            m_curY = ITEM_NUM - 1;
        }
    }
    else if( npad.IsTrigger(Npad::RIGHT) && m_curY == 2 )
    {
        m_curX++;
        if( m_curX > SecurityItem_Wpa2Aes )
        {
            m_curX = SecurityItem_Open;
        }
    }
    else if( npad.IsTrigger(Npad::LEFT) && m_curY == 2 )
    {
        m_curX--;
        if( m_curX < SecurityItem_Open )
        {
            m_curX = SecurityItem_Wpa2Aes;
        }
    }
} // NOLINT(impl/function_size)

void SceneDirectInput::Draw(
        GraphicsSystem* pGraphicsSystem,
        FontSystem* pFontSystem
        ) NN_NOEXCEPT
{
    BaseScene::Draw(pGraphicsSystem, pFontSystem);

    nn::gfx::util::DebugFontTextWriter&
        textWriter = pFontSystem->GetDebugFontTextWriter();

    const nn::util::Unorm8x4& textColor = Color::White;
    const nn::util::Unorm8x4& textColorGray = Color::Gray;
    const nn::util::Unorm8x4& textColorRed = Color::Red;

    textWriter.SetTextColor( textColor );
    textWriter.SetScale( FONT_SCALE, FONT_SCALE );

    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE * 1), MY_INITIAL_Y );
    textWriter.Print( "SSID" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE * 1), MY_INITIAL_Y + (FONT_SIZE * 1) );
    textWriter.Print( "PW" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE * 1), MY_INITIAL_Y + (FONT_SIZE * 2) );
    textWriter.Print( "SECURITY" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE * 7), MY_INITIAL_Y );
    textWriter.Print( ":" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE * 7), MY_INITIAL_Y + (FONT_SIZE * 1) );
    textWriter.Print( ":" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE * 7), MY_INITIAL_Y + (FONT_SIZE * 2) );
    textWriter.Print( ":" );

    // SSID
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE * 8), MY_INITIAL_Y );
    char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
    textWriter.Print( "[ %s ]", m_ssid.GetHexString(ssidStr) );

    // PW
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE * 8), MY_INITIAL_Y + (FONT_SIZE * 1) );
    if( m_curX == SecurityItem_Open )
    {
        textWriter.SetTextColor( textColorGray );
        textWriter.Print( "-" );
    }
    else
    {
        if( strlen(m_pwStr) > 0 )
        {
            textWriter.Print( "[ %s ]", m_pwStr );
        }
        else
        {
            textWriter.SetTextColor( textColorGray );
            textWriter.Print( "(Please input password)" );
        }
    }
    textWriter.SetTextColor( textColor );

    // SECURITY
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE * 8), MY_INITIAL_Y + (FONT_SIZE * 2) );
    textWriter.Print( "< %s >", securityStrs[m_curX] );

    textWriter.SetCursor( INITIAL_X + (FONT_SIZE * 16), INITIAL_Y + (FONT_SIZE * 16) );
    textWriter.Print( "CONNECT" );

    // カーソル
    if( m_curY == 3 )
    {
        textWriter.SetCursor( INITIAL_X + (FONT_SIZE * 15), INITIAL_Y + (FONT_SIZE * 16) );
    }
    else
    {
        textWriter.SetCursor( MY_INITIAL_X, MY_INITIAL_Y + (FONT_SIZE * m_curY) );
    }
    textWriter.Print( ">" );

    textWriter.SetCursor( INITIAL_X + FONT_SIZE, INITIAL_Y + (FONT_SIZE * 20) );
    char str[10] = "";
    if( m_curY == 2 )
    {
        strcpy(str, "Connect");
    }
    else
    {
        strcpy(str, "Edit");
    }
    textWriter.Print( "[B]:Back to Title [A]:%s", str );

    textWriter.SetScale( 1, 1 );
    textWriter.SetTextColor( textColorRed );
    textWriter.SetCursor( INITIAL_X + (FONT_SIZE * 2), INITIAL_Y + (FONT_SIZE * 17) );
    textWriter.Print( "%s", m_commentStr );

    pGraphicsSystem->BeginDraw();
    pFontSystem->Draw();
    pGraphicsSystem->EndDraw();

    pGraphicsSystem->Synchronize(
        nn::TimeSpan::FromNanoSeconds( 1000 * 1000 * 1000 / FRAME_RATE ) );
}

void SceneDirectInput::SetApplicationHeap(ApplicationHeap* pApplicationHeap) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pApplicationHeap);
    m_pApplicationHeap = pApplicationHeap;
}

void SceneDirectInput::InputBySwkbd(char* initStr, char* pOutStr, InputItem item) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(initStr);
    NN_ABORT_UNLESS_NOT_NULL(pOutStr);
    nn::swkbd::ShowKeyboardArg showKeyboardArg;
    nn::swkbd::MakePreset( &( showKeyboardArg.keyboardConfig ), nn::swkbd::Preset_Default );

    // 入力文字数制限
    if( item == InputItem_Ssid )
    {
        showKeyboardArg.keyboardConfig.textMaxLength = 32;
        showKeyboardArg.keyboardConfig.inputFormMode = nn::swkbd::InputFormMode_OneLine;
        showKeyboardArg.keyboardConfig.isUseUtf8 = true;
    }
    else if( item == InputItem_Pw )
    {
        showKeyboardArg.keyboardConfig.textMaxLength = 63;
        showKeyboardArg.keyboardConfig.inputFormMode = nn::swkbd::InputFormMode_OneLine;
        showKeyboardArg.keyboardConfig.isUseUtf8 = true;
    }

    // 共有メモリ用バッファの割り当て
    size_t in_heap_size = nn::swkbd::GetRequiredWorkBufferSize( false );
    void* swkbd_work_buffer = m_pApplicationHeap->Allocate( in_heap_size, nn::os::MemoryPageSize );

    showKeyboardArg.workBuf = swkbd_work_buffer;
    showKeyboardArg.workBufSize = in_heap_size;

    // 終了パラメータの設定
    size_t out_heap_size = nn::swkbd::GetRequiredStringBufferSize();
    nn::swkbd::String output_string;
    output_string.ptr = reinterpret_cast<char*>(
        m_pApplicationHeap->Allocate( out_heap_size, nn::os::MemoryPageSize ) );
    output_string.bufSize = out_heap_size;

    // 初期入力文字列の設定
    if( strlen(initStr) > 0 )
    {
        nn::swkbd::SetInitialTextUtf8(&showKeyboardArg, initStr);
    }
    else
    {
        // ガイド文字列の設定
        char guide_string[32];
        if( item == InputItem_Ssid )
        {
            strcpy(guide_string, "Please input SSID");
        }
        else if( item == InputItem_Pw )
        {
            strcpy(guide_string, "Please input password");
        }
        else
        {
            strcpy(guide_string, "Please input word");
        }
        nn::swkbd::SetGuideTextUtf8( &showKeyboardArg.keyboardConfig, guide_string );
    }

    nn::Result result = nn::swkbd::ShowKeyboard( &output_string, showKeyboardArg );
    if( nn::swkbd::ResultCanceled::Includes( result ) )
    {
        NN_LOG( " -- cancel --\n" );
    }
    else if( result.IsSuccess() )
    {
        NN_LOG( " -- ok --\n" );
    }

    // 結果文字列を受け取る
    char* str_ptr = reinterpret_cast< char* >( output_string.ptr );
    strcpy(pOutStr, str_ptr);

    // 共有メモリ用バッファの解放
    m_pApplicationHeap->Deallocate( output_string.ptr );
    m_pApplicationHeap->Deallocate( swkbd_work_buffer );
}
