﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <array>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_InfraApi.h>
#include <nn/wlan/wlan_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/util/util_FormatString.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/bsdsocket/cfg/cfg.h>
#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#ifdef __INTELLISENSE__
#include <sys/poll.h>
#endif

#include "SceneConnected.h"
#include "Async.h"

namespace {
// スレッドスタック
const size_t ThreadStackSize = 8192;
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack[ ThreadStackSize ];
}

// nibble を ascii に
inline char NibbleToAscii(uint8_t nibble)
{
    nibble &= 0xf;
    return nibble < 0xa ? nibble + '0' : nibble - 0xa + 'A';
}

int32_t SceneConnected::m_rssi;
nn::os::EventType SceneConnected::m_event;
nn::os::SystemEventType SceneConnected::m_connectionEvent;
nn::wlan::ConnectionStatus SceneConnected::m_connectionStatus;
nn::os::MutexType SceneConnected::m_mutex;
bool SceneConnected::m_isTcpConnected;
int SceneConnected::m_recvDataCnt;
int SceneConnected::m_recvDataSize;
SceneConnected::SceneConnected(
        ISceneChanger* changer,
        WirelessData* pDistributor
        )
: BaseScene(changer, pDistributor)
{
    m_rssi = -128;
    memset(&m_ownIp, 0, sizeof(nn::wlan::WlanIpv4Address));
    m_pageId = 0;
    m_page2CurX = 0;
    m_page2CurY = 0;
    for( int i = 0; i < NUM_OF_IP_CHAR; i++ )
    {
        m_inputIp[i] = 0;
}
    for( int i = 0; i < NUM_OF_PORT_CHAR; i++ )
    {
        m_inputPort[i] = 4 + i;  // デフォルトポートは45678
    }
    m_sockFd = 0;
    m_isTcpConnected = false;
    m_recvDataCnt = 0;
    m_recvDataSize = 0;
}

void SceneConnected::Initialize() NN_NOEXCEPT
{
    strcpy(m_resultStr, "");
    m_resultTick = nn::os::GetSystemTick();
    m_prevTick = nn::os::GetSystemTick();
    SetDummyTcpSessionInfo();
    memset(&m_wakeCounts, 0, sizeof(nn::wlan::WowlWakeCount));
    memset(&m_sleepStats, 0, sizeof(nn::wlan::WowlSleepStats));

    m_distributor->GetOwnIp(&m_ownIp);

    nn::os::InitializeMutex(&m_mutex, false, 0);
    nn::os::InitializeEvent(&m_event, false, nn::os::EventClearMode_AutoClear);

    nn::os::LockMutex(&m_mutex);
    nn::wlan::Infra::GetConnectionStatus(&m_connectionStatus);
    nn::os::UnlockMutex(&m_mutex);

    nn::Result result;
    result = nn::os::CreateThread(
            &m_monitorThread, MonitorThreadFunc, NULL, g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ABORT_UNLESS( result.IsSuccess(), "Cannot create monitor thread." );

    nn::os::StartThread( &m_monitorThread );
}

void SceneConnected::Finalize() NN_NOEXCEPT
{
    nn::os::FinalizeEvent(&m_event);
    nn::os::FinalizeMutex(&m_mutex);
}

void SceneConnected::Update() NN_NOEXCEPT
{
    switch( m_pageId )
    {
    case 0:
        UpdatePage1();
        break;
    case 1:
        if( m_isTcpConnected == false )
        {
            UpdatePage2();
        }
        else
        {
            UpdatePage3();
        }
        break;
    default:
        UpdatePage1();
        break;
    }
}

void SceneConnected::Draw(
        GraphicsSystem* pGraphicsSystem,
        FontSystem* pFontSystem
        ) NN_NOEXCEPT
{
    BaseScene::Draw(pGraphicsSystem, pFontSystem);

    nn::gfx::util::DebugFontTextWriter&
        textWriter = pFontSystem->GetDebugFontTextWriter();

    switch( m_pageId )
    {
    case 0:
        DrawPage1(textWriter);
        break;
    case 1:
        if( m_isTcpConnected == false )
        {
            DrawPage2(textWriter);
        }
        else
        {
            DrawPage3(textWriter);
        }
        break;
    default:
        DrawPage1(textWriter);
        break;
    }

    pGraphicsSystem->BeginDraw();
    pFontSystem->Draw();
    pGraphicsSystem->EndDraw();

    pGraphicsSystem->Synchronize(
        nn::TimeSpan::FromNanoSeconds( 1000 * 1000 * 1000 / FRAME_RATE ) );
} //NOLINT(impl/function_size)

void SceneConnected::SetDummyTcpSessionInfo() NN_NOEXCEPT
{
    // WoWLスリープに入れる設定をしておく
    // ダミーの情報を渡す。基本的にデータ受信起床はさせない。
    nn::wlan::MacAddress dstMac(0x12, 0x34, 0x56, 0x78, 0x90, 0xab);
    nn::wlan::WlanIpv4Address dstIp = {
            { 10, 20, 30, 40 }
    };
    nn::Result result = nn::wlan::Infra::SetTcpSessionInformation(dstMac,
            m_ownIp, dstIp, 50000, 50001, 0xC18D8B0, 64240);
    if( result.IsFailure() )
    {
        NN_LOG("SetDummyTcpSessionInformation failed.\n");
    }
    else
    {
        NN_LOG("SetDummyTcpSessionInformation success.\n");
    }
}

void SceneConnected::UpdateWowlStats() NN_NOEXCEPT
{
    nn::wlan::Infra::GetWowlStats(&m_wakeCounts, &m_sleepStats);
}

void SceneConnected::ClearWowlStats() NN_NOEXCEPT
{
    memset(&m_wakeCounts, 0, sizeof(nn::wlan::WowlWakeCount));
    memset(&m_sleepStats, 0, sizeof(nn::wlan::WowlSleepStats));
    nn::wlan::Infra::ClearWowlStats(true, true);
}

bool SceneConnected::ExportStatsToSd() NN_NOEXCEPT
{
    bool ret = true;
    nn::Result result;

    NN_LOG("Mount SD card as \"sd:/\"\n");

    // SD カードをマウント名 "sd" でマウント
    result = nn::fs::MountSdCardForDebug("sd");
    if( nn::fs::ResultSdCardAccessFailed::Includes(result) )
    {
        NN_LOG("SD card not found.");
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    do {
        // ディレクトリを作成
        NN_LOG("Create\tsd:/WowlFieldTestTool\n");
        result = nn::fs::CreateDirectory("sd:/WowlFieldTestTool");
        if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
        {
            // ストレージに空き領域が不足
            NN_LOG("Usable space not enough.\n");
            ret = false;
            break;
        }

        // ファイルが存在しない場合は作成
        nn::fs::DirectoryEntryType directoryEntryType;
        result = nn::fs::GetEntryType(&directoryEntryType, "sd:/WowlFieldTestTool/test_stats.txt");
        if( nn::fs::ResultPathNotFound().Includes(result) )
        {
            // 対象ファイルが存在しないので作成
            NN_LOG("Create \"sd:/WowlFieldTestTool/test_stats.txt\"\n");
            result = nn::fs::CreateFile("sd:/WowlFieldTestTool/test_stats.txt", 0);
            if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
            {
                // 空き領域が不足
                NN_LOG("Usable space not enough.\n");
                ret = false;
                break;
            }
        }

        // ファイル末尾に追記
        nn::fs::FileHandle fileHandle;
        NN_LOG("Open \"sd:/WowlFieldTestTool/test_stats.txt\"\n");
        result = nn::fs::OpenFile(&fileHandle, "sd:/WowlFieldTestTool/test_stats.txt", nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
        if( nn::fs::ResultTargetLocked::Includes(result) )
        {
            // 対象ファイルが既にオープンされている
            // ファイルが既にオープンされている可能性は無いはずだが、とりあえず失敗としておく
            ret = false;
            break;
        }

        int64_t fileSize = 0;
        (void)nn::fs::GetFileSize(&fileSize, fileHandle);

        char writeBuffer[500];
        nn::util::SNPrintf(writeBuffer, sizeof(writeBuffer), "------- Exported stats -------\n");

        // 現在時刻を書き出しておく
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Initialize() );
        // 協定世界時 (UTC) の 西暦1970年1月1日午前0時0分0秒 からの経過秒数で現在時刻を取得
        nn::time::PosixTime posixTime;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime) );

        // PosixTime を CalendarTime へ変換
        // 計算に利用されるタイムゾーンはデバイスに設定されたものを利用
        nn::time::CalendarTime calendarTime;
        nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime) );
        char timeStr[30];
        nn::util::SNPrintf(timeStr, sizeof(timeStr), "%04d%02d%02d-%02d:%02d:%02d\n\n",
                 calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second);
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Finalize() );

        strcat(writeBuffer, timeStr);

        // 統計情報を書き足していく
        char statStr[50];
        uint64_t cpuUsage = 0;
        uint64_t cpuUsageDec = 0;
        if( m_sleepStats.totalMeasuredTime > 0 )
        {
            cpuUsage = (static_cast<uint64_t>(m_sleepStats.cpuActiveTime) * 100) /
                (static_cast<uint64_t>(m_sleepStats.totalMeasuredTime) * 1000);
            cpuUsageDec = (((static_cast<uint64_t>(m_sleepStats.cpuActiveTime) * 100) -
                (cpuUsage * (static_cast<uint64_t>(m_sleepStats.totalMeasuredTime) * 1000))) * 100) /
                    (static_cast<uint64_t>(m_sleepStats.totalMeasuredTime) * 1000);
        }

        nn::util::SNPrintf(statStr, sizeof(statStr), "Total measured time: %u000[us] = %u[sec]\n", m_sleepStats.totalMeasuredTime,
                m_sleepStats.totalMeasuredTime / 1000);
        strcat(writeBuffer, statStr);
        nn::util::SNPrintf(statStr, sizeof(statStr), "CPU active time: %u[us] (%02lu.%02lu%%)\n", m_sleepStats.cpuActiveTime,
                cpuUsage, cpuUsageDec );
        strcat(writeBuffer, statStr);
        nn::util::SNPrintf(statStr, sizeof(statStr), "Idle time: %u[us]\n", m_sleepStats.idleTime);
        strcat(writeBuffer, statStr);
        nn::util::SNPrintf(statStr, sizeof(statStr), "CPU wakes: %u\n", m_sleepStats.cpuWakes);
        strcat(writeBuffer, statStr);
        nn::util::SNPrintf(statStr, sizeof(statStr), "DTIM: %u\n", m_sleepStats.dtim);
        strcat(writeBuffer, statStr);
        nn::util::SNPrintf(statStr, sizeof(statStr), "ARP reply: %u\n", m_sleepStats.arpReply);
        strcat(writeBuffer, statStr);
        nn::util::SNPrintf(statStr, sizeof(statStr), "TCP KA ack: %u\n", m_sleepStats.tcpKaAck);
        strcat(writeBuffer, statStr);
        nn::util::SNPrintf(statStr, sizeof(statStr), "GTK renewal: %u\n", m_sleepStats.gtkRenewal);
        strcat(writeBuffer, statStr);
        nn::util::SNPrintf(statStr, sizeof(statStr), "Tx packets: %u\n", m_sleepStats.txPackets);
        strcat(writeBuffer, statStr);
        nn::util::SNPrintf(statStr, sizeof(statStr), "Rx packets: %u\n", m_sleepStats.rxPackets);
        strcat(writeBuffer, statStr);
        nn::util::SNPrintf(statStr, sizeof(statStr), "Tx bytes: %u[byte]\n", m_sleepStats.txBytes);
        strcat(writeBuffer, statStr);
        nn::util::SNPrintf(statStr, sizeof(statStr), "Rx bytes: %u[byte]\n", m_sleepStats.rxBytes);
        strcat(writeBuffer, statStr);
        nn::util::SNPrintf(statStr, sizeof(statStr), "Tx time: %u[us]\n", m_sleepStats.txActiveTime);
        strcat(writeBuffer, statStr);
        nn::util::SNPrintf(statStr, sizeof(statStr), "Rx time: %u[us]\n\n\n", m_sleepStats.rxActiveTime);
        strcat(writeBuffer, statStr);

        NN_LOG("Write to \"sd:/WowlFieldTestTool/test_stats.txt\"\n");
        size_t length = strlen(writeBuffer);
        result = nn::fs::WriteFile(fileHandle, fileSize, writeBuffer, length, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
        {
            // 空き領域が不足
            NN_LOG("Usable space not enough.\n");
            ret = false;
        }

        NN_LOG("Close \"sd:/WowlFieldTestTool/test_stats.txt\"\n");
        nn::fs::CloseFile(fileHandle);

    } while( 0 );

    // アンマウントします。
    NN_LOG("Unmount \"sd:/\"\n");
    nn::fs::Unmount("sd");

    return ret;
} //NOLINT(impl/function_size)

void SceneConnected::MonitorThreadFunc(void* arg) NN_NOEXCEPT
{
    while( 1 )
    {
        nn::os::LockMutex(&m_mutex);
        nn::wlan::Infra::GetConnectionStatus(&m_connectionStatus);
        nn::os::UnlockMutex(&m_mutex);
        if( nn::os::TryWaitEvent(&m_event) == true )
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));
    }
}

void SceneConnected::UpdatePage1() NN_NOEXCEPT
{
    Npad npad = Npad::GetInstance();
    npad.UpdatePadState();

    if( npad.IsTrigger(Npad::PLUS) )
    {
        // 切断
        nn::wlan::Infra::OpenMode();  // システム半起床でwlanが既にcloseしている場合があるので念のため呼んでおく。
        nn::wlan::Infra::Disconnect();
        // bsdsocketを閉じておく
        nn::bsdsocket::cfg::SetIfDown("wl0");
        nn::os::SignalEvent(&m_event);
        nn::os::WaitThread(&m_monitorThread);
        nn::os::DestroyThread(&m_monitorThread);

        m_SceneChanger->ChangeScene(eScene_Title);
    }
    else if( npad.IsTrigger(Npad::A) )
    {
        nn::wlan::ConnectionState state;

        // 現在接続ステートを取得
        nn::os::LockMutex(&m_mutex);
        state = m_connectionStatus.state;
        nn::os::UnlockMutex(&m_mutex);

        if( state == nn::wlan::ConnectionState_Connected )
        {
            SetDummyTcpSessionInfo();
            strcpy(m_resultStr, "A : DONE");
            m_resultTick = nn::os::GetSystemTick();
        }
        else
        {
            // 切断
            nn::wlan::Infra::OpenMode();  // システム半起床でwlanが既にcloseしている場合があるので念のため呼んでおく。
            nn::wlan::Infra::Disconnect();
            // bsdsocketを閉じておく
            nn::bsdsocket::cfg::SetIfDown("wl0");
            nn::os::SignalEvent(&m_event);
            nn::os::WaitThread(&m_monitorThread);
            nn::os::DestroyThread(&m_monitorThread);

            m_SceneChanger->ChangeScene(eScene_Connecting);
        }
    }
    else if( npad.IsTrigger(Npad::X) )
    {
        UpdateWowlStats();
        strcpy(m_resultStr, "X : DONE");
        m_resultTick = nn::os::GetSystemTick();
    }
    else if( npad.IsTrigger(Npad::Y) )
    {
        if( ExportStatsToSd() == true )
        {
            strcpy(m_resultStr, "Y : SUCCESS");
            m_resultTick = nn::os::GetSystemTick();
        }
        else
        {
            strcpy(m_resultStr, "Y : FAILURE");
            m_resultTick = nn::os::GetSystemTick();
        }
    }
    else if( npad.IsTrigger(Npad::B) )
    {
        ClearWowlStats();
        strcpy(m_resultStr, "B : DONE");
        m_resultTick = nn::os::GetSystemTick();
    }
    else if( npad.IsTrigger(Npad::L))
    {
        m_pageId--;
        if( m_pageId < 0)
        {
            m_pageId = 0;
}
    }
    else if( npad.IsTrigger(Npad::R))
    {
        m_pageId++;
        if( m_pageId >= MAX_PAGE_NUM)
        {
            m_pageId = MAX_PAGE_NUM - 1;
        }
    }
} //NOLINT(impl/function_size)

void SceneConnected::UpdatePage2() NN_NOEXCEPT
{
    Npad npad = Npad::GetInstance();
    npad.UpdatePadState();

    if( npad.IsTrigger(Npad::PLUS) )
    {
        // 切断
        nn::wlan::Infra::OpenMode();  // システム半起床でwlanが既にcloseしている場合があるので念のため呼んでおく。
        nn::wlan::Infra::Disconnect();
        // bsdsocketを閉じておく
        nn::bsdsocket::cfg::SetIfDown("wl0");
        nn::os::SignalEvent(&m_event);
        nn::os::WaitThread(&m_monitorThread);
        nn::os::DestroyThread(&m_monitorThread);

        m_SceneChanger->ChangeScene(eScene_Title);
    }
    else if( npad.IsTrigger(Npad::A) )
    {
        // TCP CONNECT
        if( m_page2CurY == 2 )
        {
            // TCP CONNECT
            char ipStr[16];
            nn::util::SNPrintf(ipStr, sizeof(ipStr), "%d.%d.%d.%d",
                    (m_inputIp[0] * 100) +  (m_inputIp[1] * 10) +  m_inputIp[2],
                    (m_inputIp[3] * 100) +  (m_inputIp[4] * 10) +  m_inputIp[5],
                    (m_inputIp[6] * 100) +  (m_inputIp[7] * 10) +  m_inputIp[8],
                    (m_inputIp[9] * 100) + (m_inputIp[10] * 10) + m_inputIp[11]);
            NN_LOG("dest ip:%s\n", ipStr);
            uint16_t port = (m_inputPort[0] * 10000) + (m_inputPort[1] * 1000) +
                    (m_inputPort[2] * 100) + (m_inputPort[3] * 10) + m_inputPort[4];
            NN_LOG("dest port:%d\n", port);
            m_recvDataCnt = 0;
            m_recvDataSize = 0;

            bool ret = ConnectTcp(ipStr, port);
            if( ret == true )
            {
                strcpy(m_resultStr, "SUCCESS");
                m_resultTick = nn::os::GetSystemTick();
                m_isTcpConnected = true;
            }
            else
            {
                strcpy(m_resultStr, "FAILURE");
                m_resultTick = nn::os::GetSystemTick();
                m_isTcpConnected = false;
            }
        }
    }
    else if( npad.IsTrigger(Npad::UP))
    {
        if( m_page2CurX == 0 )
        {
            m_page2CurY--;
            if( m_page2CurY < 0)
            {
                m_page2CurY = 0;
            }
        }
        else
        {
            switch( m_page2CurY )
            {
            case 0:
                m_inputIp[m_page2CurX - 1]++;
                if( m_inputIp[m_page2CurX - 1] > 9 )
                {
                    m_inputIp[m_page2CurX - 1] = 0;
                }
                break;
            case 1:
                m_inputPort[m_page2CurX - 1]++;
                if( m_inputPort[m_page2CurX - 1] > 9 )
                {
                    m_inputPort[m_page2CurX - 1] = 0;
                }
                break;
            default:
                break;
            }
        }
    }
    else if( npad.IsTrigger(Npad::DOWN))
    {
        if( m_page2CurX == 0 )
        {
            m_page2CurY++;
            if( m_page2CurY >= MAX_ITEM_PAGE2)
            {
                m_page2CurY = MAX_ITEM_PAGE2 - 1;
            }
        }
        else
        {
            switch( m_page2CurY )
            {
            case 0:
                m_inputIp[m_page2CurX - 1]--;
                if( m_inputIp[m_page2CurX - 1] < 0 )
                {
                    m_inputIp[m_page2CurX - 1] = 9;
                }
                break;
            case 1:
                m_inputPort[m_page2CurX - 1]--;
                if( m_inputPort[m_page2CurX - 1] < 0 )
                {
                    m_inputPort[m_page2CurX - 1] = 9;
                }
                break;
            default:
                break;
            }
        }
    }
    else if( npad.IsHold(Npad::LEFT) || npad.IsTrigger(Npad::LEFT) )
    {
        switch( m_page2CurY )
        {
        case 0:
        case 1:
            m_page2CurX--;
            if( m_page2CurX < 0 )
            {
                m_page2CurX = 0;
            }
            break;
        case 2:
            break;
        default:
            break;
        }
    }
    else if( npad.IsHold(Npad::RIGHT) || npad.IsTrigger(Npad::RIGHT) )
    {
        switch( m_page2CurY )
        {
        case 0:
            // IP指定行
            m_page2CurX++;
            if( m_page2CurX > NUM_OF_IP_CHAR )
            {
                m_page2CurX = NUM_OF_IP_CHAR;
            }
            break;
        case 1:
            // Port指定行
            m_page2CurX++;
            if( m_page2CurX > NUM_OF_PORT_CHAR )
            {
                m_page2CurX = NUM_OF_PORT_CHAR;
            }
            break;
        case 2:
            break;
        default:
            break;
        }
    }
    else if( npad.IsTrigger(Npad::L))
    {
        m_pageId--;
        if( m_pageId < 0)
        {
            m_pageId = 0;
        }
    }
    else if( npad.IsTrigger(Npad::R))
    {
        m_pageId++;
        if( m_pageId >= MAX_PAGE_NUM)
        {
            m_pageId = MAX_PAGE_NUM - 1;
        }
    }
} //NOLINT(impl/function_size)

void SceneConnected::UpdatePage3() NN_NOEXCEPT
{
    Npad npad = Npad::GetInstance();
    npad.UpdatePadState();

    if( npad.IsTrigger(Npad::PLUS) )
    {
        // Socket close
        CloseSocket(m_sockFd);
        // 切断
        nn::wlan::Infra::OpenMode();  // システム半起床でwlanが既にcloseしている場合があるので念のため呼んでおく。
        nn::wlan::Infra::Disconnect();
        // bsdsocketを閉じておく
        nn::bsdsocket::cfg::SetIfDown("wl0");
        nn::os::SignalEvent(&m_event);
        nn::os::WaitThread(&m_monitorThread);
        nn::os::DestroyThread(&m_monitorThread);

        m_SceneChanger->ChangeScene(eScene_Title);
    }
    else if( npad.IsTrigger(Npad::A) )
    {
        // SET TCP SESSION INFO
        if( SetTcpSessionInfo(m_sockFd) == true )
        {
            strcpy(m_resultStr, "A:SUCCESS");
        }
        else
        {
            strcpy(m_resultStr, "A:FAILURE");
        }
        m_resultTick = nn::os::GetSystemTick();
    }
    else if( npad.IsTrigger(Npad::B) )
    {
        CloseSocket(m_sockFd);
    }
    else if( npad.IsTrigger(Npad::L))
    {
        m_pageId--;
        if( m_pageId < 0)
        {
            m_pageId = 0;
        }
    }
    else if( npad.IsTrigger(Npad::R))
    {
        m_pageId++;
        if( m_pageId >= MAX_PAGE_NUM)
        {
            m_pageId = MAX_PAGE_NUM - 1;
        }
    }
} //NOLINT(impl/function_size)

void SceneConnected::DrawConnectState(nn::gfx::util::DebugFontTextWriter& textWriter) NN_NOEXCEPT
{
    const nn::util::Unorm8x4& textColor = Color::White;
    const nn::util::Unorm8x4& textColorGreen = Color::Green;
    const nn::util::Unorm8x4& textColorRed = Color::Red;

    textWriter.SetTextColor( textColor );
    textWriter.SetScale( FONT_SCALE, FONT_SCALE );
    textWriter.SetCursor( 1280 / 2 - 64, INITIAL_Y + 78 );
    textWriter.Print("STATE");

    char stateStr[15];
    char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
    int16_t channel;
    nn::wlan::ConnectionState state;

    nn::os::LockMutex(&m_mutex);
    // 現在接続ステートを取得
    state = m_connectionStatus.state;
    // SSIDを取得
    m_connectionStatus.ssid.GetHexString(ssidStr);
    // チャンネルを取得
    channel = m_connectionStatus.channel;
    nn::os::UnlockMutex(&m_mutex);

    if( state == nn::wlan::ConnectionState_Connected )
    {
        textWriter.SetTextColor( textColorGreen );
        strcpy(stateStr, "CONNECTED");
    }
    else
    {
        textWriter.SetTextColor( textColorRed );
        strcpy(stateStr, "DISCONNECTED");
    }

    // ステートを表示
    textWriter.SetScale( 1.5, 1.5 );
    textWriter.SetCursor( INITIAL_X + ( FONT_SIZE * 1 ), INITIAL_Y + (FONT_SIZE * 4) );
    textWriter.Print( "%s", stateStr );
    // IPを表示
    textWriter.SetTextColor( textColor );
    if( state == nn::wlan::ConnectionState_Connected )
    {
        textWriter.SetCursor( INITIAL_X + ( FONT_SIZE * 8 ), INITIAL_Y + (FONT_SIZE * 4) );
        textWriter.Print( "%d.%d.%d.%d", m_ownIp.addr[0], m_ownIp.addr[1], m_ownIp.addr[2], m_ownIp.addr[3] );
    }
    // SSIDを表示
    textWriter.SetCursor( INITIAL_X + ( FONT_SIZE * 16 ), INITIAL_Y + (FONT_SIZE * 4) );
    textWriter.Print( "%s", ssidStr );
    // チャンネルを表示
    textWriter.SetCursor( 1280 - 300, INITIAL_Y + (FONT_SIZE * 4) );
    textWriter.Print( "%dch", channel );
    // RSSIを表示
    if( (nn::os::GetSystemTick() - m_prevTick).ToTimeSpan().GetMilliSeconds() > 1000 )
    {
        nn::wlan::Infra::GetRssi(&m_rssi);
        m_prevTick = nn::os::GetSystemTick();
    }
    textWriter.SetCursor( 1280 - 200, INITIAL_Y + (FONT_SIZE * 4) );
    textWriter.Print( "%d[dBm](%d)", m_rssi, nn::wlan::Infra::ConvertRssiToLinkLevel(m_rssi) );
} //NOLINT(impl/function_size)

void SceneConnected::DrawPage1(nn::gfx::util::DebugFontTextWriter& textWriter) NN_NOEXCEPT
{
    DrawConnectState(textWriter);

    const nn::util::Unorm8x4& textColor = Color::White;
    const nn::util::Unorm8x4& textColorOrange = Color::Orange;

    nn::wlan::ConnectionState state;
    nn::os::LockMutex(&m_mutex);
    // 現在接続ステートを取得
    state = m_connectionStatus.state;
    nn::os::UnlockMutex(&m_mutex);

    // STATSの表示
    int statsY = INITIAL_Y + 78 + 92;
    textWriter.SetTextColor( textColor );
    textWriter.SetScale( FONT_SCALE, FONT_SCALE );
    textWriter.SetCursor( 1280 / 2 - 64, statsY );
    textWriter.Print( "STATS");

    // WoWL stats
    uint64_t cpuUsage = 0;
    uint64_t cpuUsageDec = 0;
    if( m_sleepStats.totalMeasuredTime > 0 )
    {
        cpuUsage = (static_cast<uint64_t>(m_sleepStats.cpuActiveTime) * 100) /
            (static_cast<uint64_t>(m_sleepStats.totalMeasuredTime) * 1000);
        cpuUsageDec = (((static_cast<uint64_t>(m_sleepStats.cpuActiveTime) * 100) -
            (cpuUsage * (static_cast<uint64_t>(m_sleepStats.totalMeasuredTime) * 1000))) * 100) /
                (static_cast<uint64_t>(m_sleepStats.totalMeasuredTime) * 1000);
    }

    textWriter.SetScale( 1.5, 1.5 );
    int i = 1;
    textWriter.SetCursor( INITIAL_X + ( FONT_SIZE * 1 ), statsY + (FONT_SIZE * i++) );
    textWriter.Print( "Total measured time: %u000[us] = %u[sec]", m_sleepStats.totalMeasuredTime,
            m_sleepStats.totalMeasuredTime / 1000);
    textWriter.SetCursor( INITIAL_X + ( FONT_SIZE * 7 ), statsY + (FONT_SIZE * i) );
    textWriter.Print( "CPU active time: %u[us] (%02lu.%02lu%%)", m_sleepStats.cpuActiveTime,
            cpuUsage, cpuUsageDec );
    textWriter.SetCursor( INITIAL_X + ( FONT_SIZE * 25 ), statsY + (FONT_SIZE * i++) );
    textWriter.Print( "Idle time: %u[us]", m_sleepStats.idleTime);
    textWriter.SetCursor( INITIAL_X + ( FONT_SIZE * 1 ), statsY + (FONT_SIZE * i++) );
    textWriter.Print( "CPU wakes: %u", m_sleepStats.cpuWakes);
    textWriter.SetCursor( INITIAL_X + ( FONT_SIZE * 1 ), statsY + (FONT_SIZE * i++) );
    textWriter.Print( "DTIM: %u", m_sleepStats.dtim);
    textWriter.SetCursor( INITIAL_X + ( FONT_SIZE * 1 ), statsY + (FONT_SIZE * i++) );
    textWriter.Print( "ARP reply: %u", m_sleepStats.arpReply);
    textWriter.SetCursor( INITIAL_X + ( FONT_SIZE * 1 ), statsY + (FONT_SIZE * i++) );
    textWriter.Print( "TCP KA ack: %u", m_sleepStats.tcpKaAck);
    textWriter.SetCursor( INITIAL_X + ( FONT_SIZE * 1 ), statsY + (FONT_SIZE * i++) );
    textWriter.Print( "GTK renewal: %u", m_sleepStats.gtkRenewal);

    textWriter.SetCursor( INITIAL_X + ( FONT_SIZE * 1 ), statsY + (FONT_SIZE * i) );
    textWriter.Print( "Tx packets: %u", m_sleepStats.txPackets);
    textWriter.SetCursor( 1280 / 2 + INITIAL_X, statsY + (FONT_SIZE * i++) );
    textWriter.Print( "Rx packets: %u", m_sleepStats.rxPackets);

    textWriter.SetCursor( INITIAL_X + ( FONT_SIZE * 1 ), statsY + (FONT_SIZE * i) );
    textWriter.Print( "Tx bytes: %u[byte]", m_sleepStats.txBytes);
    textWriter.SetCursor( 1280 / 2 + INITIAL_X, statsY + (FONT_SIZE * i++) );
    textWriter.Print( "Rx bytes: %u[byte]", m_sleepStats.rxBytes);

    textWriter.SetCursor( INITIAL_X + ( FONT_SIZE * 1 ), statsY + (FONT_SIZE * i) );
    textWriter.Print( "Tx time: %u[us]", m_sleepStats.txActiveTime);
    textWriter.SetCursor( 1280 / 2 + INITIAL_X, statsY + (FONT_SIZE * i++) );
    textWriter.Print( "Rx time: %u[us]", m_sleepStats.rxActiveTime);

    textWriter.SetScale( 1.5, 1.5 );
    textWriter.SetCursor( INITIAL_X + FONT_SIZE, INITIAL_Y + (FONT_SIZE * 20) );
    char aStr[15];
    if( state == nn::wlan::ConnectionState_Connected )
    {
        strcpy(aStr, "Set TCP info");
    }
    else
    {
        strcpy(aStr, "Reconnect");
    }
    textWriter.Print( "[+]:Go to Title  [X]:Update stats  [B]:Clear stats  [Y]:Export to SD  [A]:%s", aStr );

    // ボタン入力結果表示
    if( (nn::os::GetSystemTick() - m_resultTick).ToTimeSpan().GetMilliSeconds() < 2000 )
    {
        textWriter.SetScale( 1, 1 );
        textWriter.SetCursor( 1280 - 100, 720 - 20 );
        textWriter.Print( "%S", m_resultStr );
    }


    textWriter.SetTextColor( textColorOrange );
    textWriter.SetScale( 1, 1 );
    textWriter.SetCursor( INITIAL_X + FONT_SIZE, INITIAL_Y + (FONT_SIZE * 19) );
    textWriter.Print( "Please detach JOY-CONs before putting the system in sleep mode." );
} //NOLINT(impl/function_size)

void SceneConnected::DrawPage2(nn::gfx::util::DebugFontTextWriter& textWriter) NN_NOEXCEPT
{
    DrawConnectState(textWriter);

    const nn::util::Unorm8x4& textColor = Color::White;

    // TCP接続状態
    int tcpY = INITIAL_Y + 210;
    int ipY = tcpY + 100;
    int portY = tcpY + 240;
    int connectY = tcpY + 350;
    textWriter.SetTextColor( textColor );
    textWriter.SetScale( FONT_SCALE, FONT_SCALE );
    textWriter.SetCursor( 1280 / 2 - (32 * 5), tcpY );
    textWriter.Print( "TCP CONNECTION" );

    // 宛先IP
    textWriter.SetCursor( INITIAL_X + 64, ipY );
    textWriter.Print( "Destination IP" );
    textWriter.SetCursor( INITIAL_X + 200, ipY + 50 );
    textWriter.Print( "%d%d%d.%d%d%d.%d%d%d.%d%d%d",
            m_inputIp[0],  m_inputIp[1],  m_inputIp[2],
            m_inputIp[3],  m_inputIp[4],  m_inputIp[5],
            m_inputIp[6],  m_inputIp[7],  m_inputIp[8],
            m_inputIp[9], m_inputIp[10], m_inputIp[11]);

    // 宛先Port
    textWriter.SetCursor( INITIAL_X + 64, portY );
    textWriter.Print( "Destination Port" );
    textWriter.SetCursor( INITIAL_X + 200, portY + 50 );
    textWriter.Print( "%d%d%d%d%d",
            m_inputPort[0],  m_inputPort[1],  m_inputPort[2], m_inputPort[3],  m_inputPort[4]);

    // TCP CONNECT
    textWriter.SetCursor( 1280 / 2 - (32 * 3), connectY );
    textWriter.Print( "CONNECT" );

    // カーソル
    int dispCurY = 0;
    int dispCurX = 0;
    if( m_page2CurX == 0 )
{
        if( m_page2CurY == 0 )
    {
            dispCurX = INITIAL_X + 32;
            dispCurY = ipY;
    }
        else if( m_page2CurY == 1 )
        {
            dispCurX = INITIAL_X + 32;
            dispCurY = portY;
        }
        else if( m_page2CurY == 2 )
        {
            dispCurX = 1280 / 2 - (32 * 4);
            dispCurY = connectY;
        }
        textWriter.SetCursor( dispCurX, dispCurY );
        textWriter.Print( ">" );
    }
    else
    {
        if( m_page2CurY == 0 )
        {
            dispCurX = INITIAL_X + 200 + (22 * (m_page2CurX - 1) + 2 );
            // 表示上はドット「.」があるので、その分カーソル位置をズラしておく必要がある
            if( m_page2CurX > 3 )
            {
                dispCurX += 12;
    }
            if( m_page2CurX > 6 )
            {
                dispCurX += 12;
}
            if( m_page2CurX > 9 )
            {
                dispCurX += 12;
            }
            dispCurY = ipY + 50 + 32;
        }
        else if( m_page2CurY == 1 )
        {
            dispCurX = INITIAL_X + 200 + (22 * (m_page2CurX - 1) + 2);
            dispCurY = portY + 50 + 32;
        }
        textWriter.SetCursor( dispCurX, dispCurY );
        textWriter.Print( "^" );
    }

    textWriter.SetScale( 1.5, 1.5 );
    textWriter.SetCursor( INITIAL_X + FONT_SIZE, INITIAL_Y + (FONT_SIZE * 20) );
    textWriter.Print( "[+]:Go to Title  [L/R]:Move pages" );

    // ボタン入力結果表示
    if( (nn::os::GetSystemTick() - m_resultTick).ToTimeSpan().GetMilliSeconds() < 2000 )
{
        textWriter.SetScale( 1, 1 );
        textWriter.SetCursor( 1280 - 100, 720 - 20 );
        textWriter.Print( "%S", m_resultStr );
}
} //NOLINT(impl/function_size)

void SceneConnected::DrawPage3(nn::gfx::util::DebugFontTextWriter& textWriter) NN_NOEXCEPT
{
    DrawConnectState(textWriter);

    const nn::util::Unorm8x4& textColor = Color::White;
    const nn::util::Unorm8x4& textColorG = Color::Green;

    // TCP接続状態
    int tcpY = INITIAL_Y + 210;
    textWriter.SetTextColor( textColor );
    textWriter.SetScale( FONT_SCALE, FONT_SCALE );
    textWriter.SetCursor( 1280 / 2 - (32 * 5), tcpY );
    textWriter.Print( "TCP CONNECTION" );

    textWriter.SetTextColor( textColorG );
    textWriter.SetCursor( 1280 / 2 - (32 * 4), tcpY + 100 );
    textWriter.Print( "CONNECTED" );

    // TCPデータ受信数
    textWriter.SetTextColor( textColor );
    textWriter.SetScale( 1.5, 1.5 );
    textWriter.SetCursor( INITIAL_X + 64, tcpY + 150 );
    textWriter.Print( "Received Data Count:%d", m_recvDataCnt );
    // TCPデータ受信サイズ(累計)
    textWriter.SetCursor( INITIAL_X + 64, tcpY + 150 + 25 );
    textWriter.Print( "Total Received Data Size:%d", m_recvDataSize );

    textWriter.SetTextColor( textColor );
    textWriter.SetScale( 1.5, 1.5 );
    textWriter.SetCursor( INITIAL_X + FONT_SIZE, INITIAL_Y + (FONT_SIZE * 20) );
    textWriter.Print( "[+]:Go to Title  [A]:Set Tcp info [B]:Close socket [L/R]:Move pages" );

    // ボタン入力結果表示
    if( (nn::os::GetSystemTick() - m_resultTick).ToTimeSpan().GetMilliSeconds() < 2000 )
    {
        textWriter.SetScale( 1, 1 );
        textWriter.SetCursor( 1280 - 100, 720 - 20 );
        textWriter.Print( "%S", m_resultStr );
}
} //NOLINT(impl/function_size)

bool SceneConnected::ConnectTcp(const char* cp, uint16_t port) NN_NOEXCEPT
{
    auto socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);

    {
        int opt = 1;
        auto ret = nn::socket::SetSockOpt(socket, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, &opt, sizeof(opt));
        if (ret < 0)
        {
            NN_LOG(">warn:failed to setting nn::socket::Option::Tcp_NoDelay flag\n");
        }

        // disable SACK/WindowScaling/TimeStamp
        opt = 1;
        ret = nn::socket::SetSockOpt(socket, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoOpt, &opt, sizeof(opt));
        if (ret < 0)
    {
            NN_LOG(">warn:failed to setting nn::socket::Option::Tcp_NoDelay flag\n");
        }

        opt = 4096;
        ret = nn::socket::SetSockOpt(socket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &opt, sizeof(opt));
        if (ret  < 0)
        {
            NN_LOG(">warn:failed to setting So_RcvBuf value\n");
        }
    }

    nn::socket::InAddr addr;
    nn::socket::InetAton(cp, &addr);

    nn::socket::SockAddrIn destAddr = {};
    destAddr.sin_addr = addr;
    destAddr.sin_port = nn::socket::InetHtons(port);
    destAddr.sin_family = nn::socket::Family::Af_Inet;

    int result = nn::socket::Connect(socket, reinterpret_cast<nn::socket::SockAddr*>(&destAddr), sizeof(destAddr));
    if (result < 0)
    {
        return false;
    }

    // データ受信コールバック
    static const auto ReceiveCallback = [](int socket, nn::socket::PollEvent pollResult, void* userptr)
        {
        const size_t AddressByteWidth = 2;
        const size_t BytesInLine = 16;
        const size_t LineWidth = AddressByteWidth * 2 + 1 + 3 * BytesInLine + 1 + BytesInLine + 1;

        if ((pollResult & nn::socket::PollEvent::PollHup) != nn::socket::PollEvent::PollNone)
        {
            // 切断された
            NN_LOG(">socket_disconnected\n");
            nn::socket::Close(socket);
            m_isTcpConnected = false;
            return;
        }
        else if( (pollResult & nn::socket::PollEvent::PollNVal) != nn::socket::PollEvent::PollNone)
        {
            // ソケット閉じた
            NN_LOG(">socket_closed\n");
            nn::socket::Close(socket);
            m_isTcpConnected = false;
            return;
        }

        NN_LOG(">recv\n");
        m_recvDataCnt++;

        {
            char format[1 + 4 + 1 + BytesInLine * 3 + 1 + BytesInLine + 1 + 1];
            nn::util::SNPrintf(format, sizeof(format), ">%%-%ds +0 +1 +2 +3 +4 +5 +6 +7 +8 +9 +A +B +C +D +E +F 0123456789ABCDEF\n", AddressByteWidth * 2);

            NN_LOG(format, "addr");
        }

        std::array<char, LineWidth> lineBuffer;
        lineBuffer.back() = '\0';

        int totalReceivedCount = 0;
        char* lineHexWritePtr;
        char* lineAsciiWritePtr;

        uint8_t recvBuf[128];
        for (;;)
            {
            auto readPtr = recvBuf;
            auto recvSize = nn::socket::Recv(socket, recvBuf, sizeof(recvBuf), nn::socket::MsgFlag::Msg_None);
            m_recvDataSize += static_cast<int>(recvSize);
            auto readEnd = readPtr + recvSize;

            for (; readPtr < readEnd; ++readPtr)
            {
                if (!(totalReceivedCount & 0xf))
                {
                    auto lineWritePtr = lineBuffer.begin();
                    lineHexWritePtr = lineWritePtr + AddressByteWidth * 2 + 1;
                    lineAsciiWritePtr = lineHexWritePtr + 3 * BytesInLine;

                    for (int i = AddressByteWidth * 2; i--;)
                    {
                        *lineWritePtr++ = NibbleToAscii(totalReceivedCount >> (i * 4));
            }

                    while (lineWritePtr < lineBuffer.end())
                    {
                        *lineWritePtr++ = ' ';
        }
                }

                *(lineHexWritePtr) = NibbleToAscii(*readPtr >> 4);
                *(lineHexWritePtr + 1) = NibbleToAscii(*readPtr);
                lineHexWritePtr += 3;

                *lineAsciiWritePtr++ = *readPtr >= 0x20 && *readPtr < 0x80 ? *readPtr : '.';

                if (!(~totalReceivedCount & 0xf))
        {
                    NN_LOG(">%s\n", lineBuffer.data());
                }

                ++totalReceivedCount;
            }

            int bufferedDataLength;
            nn::socket::Ioctl(socket, nn::socket::IoctlCommand::FionRead, &bufferedDataLength, sizeof(bufferedDataLength));
            if (bufferedDataLength == 0)
            {
            break;
        }
        }

        if (totalReceivedCount & 0xf)
        {
            NN_LOG(">%s\n", lineBuffer.data());
        }

        if (totalReceivedCount > 0)
        {
            Async::WaitRecv(socket, reinterpret_cast<void(*)(int, nn::socket::PollEvent, void*)>(userptr), userptr);
        }
        else
        {
            // RST 投げられたらここに来るようなので、たぶん RST
            NN_LOG(">socket_rst\n");
            nn::socket::Close(socket);
            m_isTcpConnected = false;
        }
    };
    Async::WaitRecv(socket, ReceiveCallback, reinterpret_cast<void*>(static_cast<void(*)(int, nn::socket::PollEvent, void*)>(ReceiveCallback)));

    NN_LOG(">socket_fd:%d\n", socket);
    m_sockFd = socket;
    return true;
} //NOLINT(impl/function_size)

void SceneConnected::CloseSocket(int fd) NN_NOEXCEPT
{
    auto result = nn::socket::Close(fd);
    if (result < 0)
    {
        NN_LOG("Failed to close socket %d.\n", fd);
    }
}

bool SceneConnected::SetTcpSessionInfo(int fd) NN_NOEXCEPT
{
    nn::bsdsocket::cfg::IfState state;
    nn::bsdsocket::cfg::GetIfState("wl0", &state);

    nn::socket::TcpInfo ti;
    nn::socket::SockLenT len = sizeof(ti);

    if (nn::socket::GetSockOpt(fd, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_Info, reinterpret_cast<void*>(&ti), &len) < 0)
        {
        NN_LOG(">GetSockOpt failed (error %d)\n", nn::socket::GetLastError());
        return false;
        }

    nn::socket::SockAddrIn saiClient;
    len = sizeof(saiClient);
    if (nn::socket::GetSockName(fd, reinterpret_cast<nn::socket::SockAddr*>(&saiClient), &len) < 0)
    {
        NN_LOG(">GetSockName failed (error %d)\n", nn::socket::GetLastError());
        return false;
    }
    if (saiClient.sin_addr.S_addr == nn::socket::InAddr_Any)
    {
        saiClient.sin_addr.S_addr = state.addr.S_addr;
    }

    nn::socket::SockAddrIn saiServer;
    len = sizeof(saiServer);
    if (nn::socket::GetPeerName(fd, reinterpret_cast<nn::socket::SockAddr*>(&saiServer), &len) < 0)
        {
        NN_LOG(">GetPeerName failed (error %d)\n", nn::socket::GetLastError());
        return false;
        }

    uint8_t remoteHardwareAddress[nn::socket::Ether_Addr_Len];

    nn::socket::SockAddrIn saiNext;
    if ((saiServer.sin_addr.S_addr & state.subnetMask.S_addr) == (state.gatewayAddr.S_addr & state.subnetMask.S_addr))
    {
        saiNext = saiServer;
    }
    else
    {
        saiNext.sin_addr = state.gatewayAddr;
    }

    nn::Result result = nn::bsdsocket::cfg::LookupArpEntry(remoteHardwareAddress, nn::socket::Ether_Addr_Len, saiNext.sin_addr);
    if (result.IsFailure())
    {
        NN_LOG(">LookupArpEntry failed (result 0x%08x)\n", result.GetInnerValueForDebug());
        return false;
    }

    NN_LOG(">       seq: % 16u[0x%08x]\n", ti.tcpi_snd_nxt, ti.tcpi_snd_nxt);
    NN_LOG(">       ack: % 16u[0x%08x]\n", ti.tcpi_rcv_nxt, ti.tcpi_rcv_nxt);
    NN_LOG(">      rwin: % 16u\n", ti.tcpi_rcv_space);
    NN_LOG(">     local: % 16s:%u\n", nn::socket::InetNtoa(saiClient.sin_addr), nn::socket::InetNtohs(saiClient.sin_port));
    NN_LOG(">    remote: % 16s:%u\n", nn::socket::InetNtoa(saiServer.sin_addr), nn::socket::InetNtohs(saiServer.sin_port));
    NN_LOG("> next addr: % 16s[%02x:%02x:%02x:%02x:%02x:%02x]\n",
        nn::socket::InetNtoa(saiNext.sin_addr),
        remoteHardwareAddress[0], remoteHardwareAddress[1], remoteHardwareAddress[2], remoteHardwareAddress[3], remoteHardwareAddress[4], remoteHardwareAddress[5]);

    nn::wlan::WlanIpv4Address src;
    nn::wlan::WlanIpv4Address dst;
    char srcIpStr[20];
    char dstIpStr[20];
    char* tok;

    std::strcpy(srcIpStr, nn::socket::InetNtoa(saiClient.sin_addr));
    std::strcpy(dstIpStr, nn::socket::InetNtoa(saiServer.sin_addr));
    tok = std::strtok(srcIpStr, ".");
    src.addr[0] = std::atoi(tok);
    for( int i = 1; i < 4; i++ )
{
        tok = std::strtok(NULL, ".");
        src.addr[i] = std::atoi(tok);
    }

    tok = std::strtok(dstIpStr, ".");
    dst.addr[0] = std::atoi(tok);
    for( int i = 1; i < 4; i++ )
    {
        tok = std::strtok(NULL, ".");
        dst.addr[i] = std::atoi(tok);
    }

    nn::wlan::MacAddress dstMac(remoteHardwareAddress);
    result = nn::wlan::Infra::SetTcpSessionInformation(dstMac,
            src, dst, nn::socket::InetNtohs(saiClient.sin_port), nn::socket::InetNtohs(saiServer.sin_port),
            ti.tcpi_rcv_nxt, static_cast<uint16_t>(ti.tcpi_rcv_space));
    if( result.IsFailure() )
        {
        NN_LOG("SetTcpSessionInformation failed.\n");
        return false;
        }

    return true;
    }
