﻿using System;
using System.Drawing;
using System.Net;
using System.Text;
using System.Windows.Forms;
using System.Net.NetworkInformation;
using System.ComponentModel;

namespace WowFunctionTestServer
{
    public partial class MainForm : Form
    {
        Server m_Server;

        class ClientListViewItem : ListViewItem
        {
            public readonly Client Client;

            public ClientListViewItem(Client client) : base(((IPEndPoint) client.Socket.RemoteEndPoint).Address.ToString())
            {
                Client = client;

                SubItems.Add(((IPEndPoint) Client.Socket.RemoteEndPoint).Port.ToString());
                SubItems.Add(((IPEndPoint) Client.Socket.LocalEndPoint).Address.ToString());
                SubItems.Add(((IPEndPoint) Client.Socket.LocalEndPoint).Port.ToString());
            }
        }

        public MainForm()
        {
            InitializeComponent();
        }

        private void MainForm_Load(object sender, EventArgs e)
        {
            // 停止ボタンを無効化しておく
            StopButton.Enabled = false;

            m_Server = new Server(45678);
            m_Server.Connected += ClientConnected;

            m_Server.Run();

            // TCP KA パラメータのセット
            int idle;
            int interval;
            int cnt;
            bool parsed = Int32.TryParse(textBox_tcpka_idle.Text, out idle);
            if (!parsed)
            {
                idle = 10;  // デフォルトは10secにしておく
            }
            parsed = Int32.TryParse(textBox_tcpka_interval.Text, out interval);
            if (!parsed)
            {
                interval = 10;  // デフォルトは10secにしておく
            }
            parsed = Int32.TryParse(textBox_tcpka_cnt.Text, out cnt);
            if (!parsed)
            {
                cnt = 10;  // デフォルトは10にしておく
            }
            m_Server.SetTcpKaParam(idle, interval, cnt);
        }

        private void ClientConnected(object sender, ConnectedEventArgs e)
        {
            e.Client.Closed += ClientDisconnected;
            e.Client.Received += ClientReceived;

            Invoke(new Action(() => {
                var item = new ClientListViewItem(e.Client);
                item.Checked = true;

                lock (ClientsListView.Items)
                {
                    ClientsListView.Items.Add(item);
                }
            }));
        }

        private void ClientDisconnected(object sender, EventArgs e)
        {
            Invoke(new Action(() => {
                lock (ClientsListView.Items)
                {
                    foreach (ClientListViewItem listviewItem in ClientsListView.Items)
                    {
                        if (listviewItem.Client == sender)
                        {
                            listviewItem.BackColor = Color.Gray;
                            listviewItem.Checked = false;
                            //ClientsListView.Items.Remove(listviewItem);
                            break;
                        }
                    }
                }
            }));
        }

        private void ClientReceived(object sender, ReceivedEventArgs e)
        {
            Invoke(new Action(() => {
                ClientListViewItem item = null;
                lock (ClientsListView.Items)
                {
                    foreach (ClientListViewItem listviewItem in ClientsListView.Items)
                    {
                        if (listviewItem.Client == sender)
                        {
                            item = listviewItem;
                            break;
                        }
                    }
                }
                if (item == null)
                {
                    return;
                }
            }));
        }

        private void FinButton_Click(object sender, EventArgs e)
        {
            lock (ClientsListView.Items)
            {
                foreach (ClientListViewItem listviewItem in ClientsListView.Items)
                {
                    if (listviewItem.Checked)
                    {
                        listviewItem.Client.Fin();
                    }
                }
            }
        }

        private void RstButton_Click(object sender, EventArgs e)
        {
            lock (ClientsListView.Items)
            {
                foreach (ClientListViewItem listviewItem in ClientsListView.Items)
                {
                    if (listviewItem.Checked)
                    {
                        listviewItem.Client.Rst();
                    }
                }
            }
        }

        private void SendButton_Click(object sender, EventArgs e)
        {
            lock (ClientsListView.Items)
            {
                foreach (ClientListViewItem listviewItem in ClientsListView.Items)
                {
                    if (listviewItem.Checked)
                    {
                        listviewItem.Client.Send(Encoding.UTF8.GetBytes(DataTextBox.Text));
                    }
                }
            }
        }

        private void PingButton_Click(object sender, EventArgs e)
        {
            lock (ClientsListView.Items)
            {
                foreach (ClientListViewItem listviewItem in ClientsListView.Items)
                {
                    if (listviewItem.Checked)
                    {
                        var ping = new Ping();

                        ping.PingCompleted += (object obj, PingCompletedEventArgs args) => {
                            if (args.Reply.Status == IPStatus.Success)
                            {
                                LogTextBox.Text += string.Format("response from {0}: {1} byte {2}ms\r\n",
                                        args.Reply.Address.ToString(),
                                        args.Reply.Buffer.Length,
                                        args.Reply.RoundtripTime);
                            }
                            else
                            {
                                LogTextBox.Text += string.Format("{0}\r\n",
                                        args.Reply.Status.ToString());
                            }
                        };

                        ping.SendAsync(((IPEndPoint) listviewItem.Client.Socket.RemoteEndPoint).Address, null);
                    }
                }
            }
        }

        private void ClientsListView_ItemCheck(object sender, ItemCheckEventArgs e)
        {
            if (ClientsListView.Items[e.Index].BackColor == Color.Gray)
            {
                e.NewValue = CheckState.Unchecked;
            }
        }

        private void SendDataBgWorker_RunWorkerCompleted(object sender, System.ComponentModel.RunWorkerCompletedEventArgs e)
        {
            if (e.Error != null)
            {
                //エラーが発生したとき
                String str = "エラー:" + e.Error.Message;
                LogTextBox.AppendText(str);
            }
            else if (e.Cancelled)
            {
                //キャンセルされたとき
                LogTextBox.AppendText("キャンセルされました。");
            }
            else
            {
                //正常に終了したとき
                //結果を取得する
                int result = (int)e.Result;
                String str = result.ToString() + "回で完了しました。";
                LogTextBox.AppendText(str);
            }

            //Buttonを有効に戻す
            FinButton.Enabled = true;
            RstButton.Enabled = true;
            PingButton.Enabled = true;
            SendButton.Enabled = true;
            SendDataPeriodicButton.Enabled = true;
            StopButton.Enabled = false;
        }

        private void SendDataBgWorker_ProgressChanged(object sender, System.ComponentModel.ProgressChangedEventArgs e)
        {
            String str = "Send packet : " + (string)e.UserState + "\n";
            LogTextBox.AppendText(str);
        }

        private void SendDataBgWorker_DoWork(object sender, System.ComponentModel.DoWorkEventArgs e)
        {
            BackgroundWorker bgWorker = (BackgroundWorker)sender;
            e.Cancel = false;
            int i = 0;
            for (i = 0; ; i++)
            {
                lock (ClientsListView.Items)
                {
                    foreach (ClientListViewItem listviewItem in ClientsListView.Items)
                    {
                        if (listviewItem.Checked)
                        {
                            String dataText = DataTextBox.Text + i.ToString();
                            listviewItem.Client.Send(Encoding.UTF8.GetBytes(dataText));
                            bgWorker.ReportProgress(i, dataText);
                        }
                    }
                }

                int sleepTime;
                bool parsed = Int32.TryParse(SendDataPeriodicIntervalText.Text, out sleepTime);
                if (!parsed)
                {
                    sleepTime = 3;  // デフォルトは3secにしておく
                }

                for (int j = 0; j < sleepTime * 1000; j++)
                {
                    System.Threading.Thread.Sleep(1);
                    // キャンセルされたか調べる
                    if (bgWorker.CancellationPending)
                    {
                        e.Cancel = true;
                        break;
                    }
                }
                if (e.Cancel == true )
                {
                    break;
                }
            }

            e.Result = i;
        }

        private void SendDataPeriodicButton_Click(object sender, EventArgs e)
        {
            //処理が行われているときは、何もしない
            if (SendDataBgWorker.IsBusy)
                return;

            //キャンセルボタン以外を無効にする
            FinButton.Enabled = false;
            RstButton.Enabled = false;
            PingButton.Enabled = false;
            SendButton.Enabled = false;
            SendDataPeriodicButton.Enabled = false;
            StopButton.Enabled = true;

            //処理を開始する
            SendDataBgWorker.RunWorkerAsync();
        }

        private void StopButton_Click(object sender, EventArgs e)
        {
            StopButton.Enabled = false;

            // リピート送信を停止する
            SendDataBgWorker.CancelAsync();
        }

        private void button_tcpka_param_Click(object sender, EventArgs e)
        {
            int idle;
            int interval;
            int cnt;
            bool parsed = Int32.TryParse(textBox_tcpka_idle.Text, out idle);
            if (!parsed)
            {
                idle = 10;  // デフォルトは10secにしておく
            }
            parsed = Int32.TryParse(textBox_tcpka_interval.Text, out interval);
            if (!parsed)
            {
                interval = 10;  // デフォルトは10secにしておく
            }
            parsed = Int32.TryParse(textBox_tcpka_cnt.Text, out cnt);
            if (!parsed)
            {
                cnt = 10;  // デフォルトは10にしておく
            }
            m_Server.SetTcpKaParam(idle, interval, cnt);
        }
    }
}
