﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "InteractiveShell.h"
#include <nn/nn_Log.h>
#include <nn/diag/diag_LogObserver.h>
#include <nn/htcs/htcs_Library.h>
#include <nn/htcs/htcs_Socket.h>


namespace
{

uint8_t* g_pHtcsBuffer = nullptr;
bool g_ExitFlag = false;

}


namespace InteractiveShell
{

// 初期化
void Initialize()
{
    auto htcsBufferSize = nn::htcs::GetWorkingMemorySize(nn::htcs::SocketCountMax);
    g_pHtcsBuffer = new uint8_t[htcsBufferSize];

    nn::htcs::Initialize(g_pHtcsBuffer, htcsBufferSize);
}


// 終了
void Finalize()
{
    nn::htcs::Finalize();

    delete[] g_pHtcsBuffer;
    g_pHtcsBuffer = nullptr;
}

// コマンドライン解析
// FIXME: バッファオーバーフロー/argc で制限
void ParseCommandline(char** pArgV, int* pArgC, char* line)
{
    bool isQuoting = false;

    *pArgC = 0;
    pArgV[0] = line;

    for (auto pRead = line, pWrite = line;; ++pRead)
    {
        if (*pRead == '\0' || *pRead == '\r' || *pRead == '\n' || (!isQuoting && *pRead == ' '))
        {
            *(pWrite++) = '\0';

            for (; *(pRead + 1) && (*(pRead + 1) == '\r' || *(pRead + 1) == '\n' || *(pRead + 1) == ' '); ++pRead)
            {
            }

            pArgV[++(*pArgC)] = pRead + 1;

            if (!*(pRead + 1))
            {
                break;
            }
        }
        else if (*pRead == '"')
        {
            isQuoting = !isQuoting;
        }
        else if (*pRead == '\\')
        {
            if (*(pRead + 1))
            {
                switch (*++pRead)
                {
                case ' ':
                    *(pWrite++) = ' ';
                    break;
                case '"':
                    *(pWrite++) = '"';
                    break;
                case 'n':
                    *(pWrite++) = '\n';
                    break;
                case 'r':
                    *(pWrite++) = '\r';
                    break;
                case 't':
                    *(pWrite++) = '\t';
                    break;
                case 'x':
                    if (
                        ((*(pRead + 1) >= '0' && *(pRead + 1) <= '9') || (*(pRead + 1) >= 'a' && *(pRead + 1) <= 'f') || (*(pRead + 1) >= 'A' && *(pRead + 1) <= 'F')) &&
                        ((*(pRead + 2) >= '0' && *(pRead + 2) <= '9') || (*(pRead + 2) >= 'a' && *(pRead + 2) <= 'f') || (*(pRead + 2) >= 'A' && *(pRead + 2) <= 'F')))
                    {
                        *pWrite = (*++pRead < 'A' ? *pRead - '0' : (*pRead < 'a' ? *pRead - 'A' + 0x10 : *pRead - 'a' + 0x10)) << 4;
                        *(pWrite++) |= *++pRead < 'A' ? *pRead - '0' : (*pRead < 'a' ? *pRead - 'A' + 0x10 : *pRead - 'a' + 0x10);
                    }
                    break;
                default:
                    break;
                }
            }
        }
        else
        {
            *(pWrite++) = *pRead;
        }
    }
}

// コマンドを待つ
void Run(void(*pCommandHandler)(char**, int))
{
    char recvBuffer[1025];
    recvBuffer[sizeof(recvBuffer) - 1] = '\0';

    g_ExitFlag = false;
    while (!g_ExitFlag)
    {
        // シェルクライアント待ち受け
        int htcsPeerSocket = -1;
        {
            auto htcsSocket = nn::htcs::Socket();
            if (htcsSocket < 0)
            {
                continue;
            }

            nn::htcs::SockAddrHtcs htcsAddr = {
                nn::htcs::HTCS_AF_HTCS,
                nn::htcs::GetPeerNameAny(),
                { "wowtest" }
            };
            if (nn::htcs::Bind(htcsSocket, &htcsAddr) < 0)
            {
                nn::htcs::Close(htcsSocket);
                continue;
            }

            if (nn::htcs::Listen(htcsSocket, 1) < 0)
            {
                nn::htcs::Close(htcsSocket);
                continue;
            }

            nn::htcs::SockAddrHtcs htcsPeerAddr;
            htcsPeerSocket = nn::htcs::Accept(htcsSocket, &htcsPeerAddr);
            if (htcsPeerSocket < 0)
            {
                nn::htcs::Close(htcsSocket);
                continue;
            }

            nn::htcs::Close(htcsSocket);
        }

        // シェル初期化
        {
            char buf[] = "Hi!\n";
            nn::htcs::Send(htcsPeerSocket, buf, sizeof(buf) - 1, 0);
        }

        nn::diag::LogObserverHolder logObserverHolder;
        {
            static const auto LogObserver = [](const nn::diag::LogMetaData& metaInfo, const nn::diag::LogBody& logBody, void* userptr)
            {
                nn::htcs::Send(*reinterpret_cast<int*>(userptr), logBody.message, logBody.messageBytes, 0);
            };
            nn::diag::InitializeLogObserverHolder(&logObserverHolder, LogObserver, &htcsPeerSocket);
        }
        nn::diag::RegisterLogObserver(&logObserverHolder);

        // シェルループ
        while (!g_ExitFlag)
        {
            // TODO: コマンドライン最大長が 1024 バイト固定なのをなんとかする
            // 受信
            auto receivedBytes = nn::htcs::Recv(htcsPeerSocket, &recvBuffer, sizeof(recvBuffer) - 1, 0);
            if (receivedBytes <= 0)
            {
                // おそらく接続が切れたのでシェルループ終了
                break;
            }

            int lineLength = 0;
            for (; lineLength < receivedBytes && recvBuffer[lineLength] && recvBuffer[lineLength] != '\r' && recvBuffer[lineLength] != '\n'; ++lineLength)
            {
            }
            recvBuffer[lineLength] = '\0';

            // コマンドライン解析
            char* argv[16] = {};
            int argc = 0;

            ParseCommandline(argv, &argc, recvBuffer);

            // コマンド実行
            pCommandHandler(argv, argc);
        }
        // シェル終了
        nn::diag::UnregisterLogObserver(&logObserverHolder);

        if (htcsPeerSocket >= 0)
        {
            nn::htcs::Close(htcsPeerSocket);
        }
    }
}


// シェルを終了する
void Quit()
{
    g_ExitFlag = true;
}

}

