﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../common.h"

//#define PRINT_TSF_VALUE
//#define SCAN_ON_CONNECTION

namespace {

const size_t  buffSize = 100 * 1024; // 100KB
NN_ALIGNAS(4096) char  g_scanBuffer[ buffSize ];
nn::wlan::WlanState g_wlanState;

#ifdef ENABLE_RX_THREAD
uint8_t  g_rxBuffer[2048]; // 受信用バッファ
const size_t ThreadStackSize = 4096;              // スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStackRx[ ThreadStackSize ];   // 受信用スレッドスタック
nn::os::ThreadType  g_ThreadRx;
static bool g_ExitFlagRx = false;
#endif
}

#ifdef ENABLE_RX_THREAD
void RxThreadFunc(void* arg)
{
    nn::Result result;
    g_ExitFlagRx = true;

    uint32_t* rxId = reinterpret_cast<uint32_t*>(arg);
    uint32_t totalPacketSize = 0;

    while( g_ExitFlagRx )
    {
        size_t rxSize = 0;
        result = nn::wlan::Socket::GetFrameRaw(g_rxBuffer, sizeof(g_rxBuffer), &rxSize, *rxId); // ブロックされる
        if( result.IsSuccess() )
        {
            totalPacketSize += rxSize;
            NN_LOG("%d[KBytes]\n", totalPacketSize / 1024);
            //WlanTest::DumpBuffer(g_rxBuffer, rxSize);
#if 0
            // Assuming ping
            // To send back the received data to sender
            // swap dst for src
            uint8_t tmp[nn::wlan::MacAddress::MacAddressSize];
            std::memcpy(&tmp[0], &g_rxBuffer[0], nn::wlan::MacAddress::MacAddressSize);
            //std::memcpy(&g_rxBuffer[0], &g_rxBuffer[6], nn::wlan::MacAddress::MacAddressSize);
            std::memcpy(&g_rxBuffer[0], nn::wlan::MacAddress::CreateBroadcastMacAddress().GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
            std::memcpy(&g_rxBuffer[6], &tmp[0], nn::wlan::MacAddress::MacAddressSize);

            // swap ip address
            uint8_t ip[4];
            std::memcpy(&ip[0], &g_rxBuffer[26], 4);
            std::memcpy(&g_rxBuffer[26], &g_rxBuffer[30], 4);
            std::memcpy(&g_rxBuffer[30], &ip[0], 4);
            g_rxBuffer[34] = 0x00;
            WlanTest::DumpBuffer(g_rxBuffer, rxSize);
            nn::wlan::Socket::PutFrameRaw(g_rxBuffer, rxSize);
#endif
        }
    }
}
#endif

void TestCaseInfra()
{
    nn::Result result;

    // 接続に変化があった際に通知を受けるイベントオブジェクト
    nn::os::SystemEventType connectionEvent;

    // 接続状態を格納するためのオブジェクト
    nn::wlan::ConnectionStatus connectionStatus;

    // Infra用APIを使用するために、InfraManagerを初期化する
    nn::wlan::InitializeInfraManager();

#ifdef ENABLE_RX_THREAD
    // Socket用APIを使用するために、SocketManagerを初期化する
    nn::wlan::InitializeSocketManager();
#endif

    nn::wlan::Infra::GetState(&g_wlanState);
    NN_LOG("WLAN STATE : %s\n", WlanTest::wlanStateStr[g_wlanState]);

    nn::wlan::Infra::OpenMode();
    NN_LOG("WlanTest: OpenMode done\n");

    nn::wlan::MacAddress macAddr;
    nn::wlan::Infra::GetMacAddress(&macAddr);
    char macStr[nn::wlan::MacAddress::MacStringSize];
    NN_LOG("mac addr : %s\n", macAddr.GetString(macStr));

    // ConnectionEventをWLANプロセスのものにアタッチ
    nn::wlan::Infra::GetConnectionEvent(&connectionEvent);
    NN_LOG("WLANClient: GetConnectionEvent\n");

    nn::wlan::ScanParameters scanParam = {
            nn::wlan::ScanType_Passive,
            {1, 6, 11},
            0,  // 全チャンネルスキャン
            120,
            0,
            NULL,
            0,
            nn::wlan::MacAddress::CreateBroadcastMacAddress()
    };

    nn::wlan::Infra::StartScan(g_scanBuffer, buffSize, scanParam);
    NN_LOG("WlanTest: Scan finished.\n");

    // Dump scan buffer
    WlanTest::DumpScanResultWithReader(g_scanBuffer);

    nn::wlan::Infra::GetState(&g_wlanState);
    NN_LOG("WLAN STATE : %s\n", WlanTest::wlanStateStr[g_wlanState]);

    // 接続試行
    nn::wlan::BeaconScanResultReader resultReader(g_scanBuffer);
    uint32_t bssCount = resultReader.GetCount();

    nn::wlan::Ssid mySsid("WLAN_TEST_WPA2_AES");  // 接続先のSSID
    nn::wlan::MacAddress myBssid;               // 接続先のBSSID格納先 (SSIDが分かっていれば不要)
    int16_t channel;                            // 接続先のChannel格納先
    bool Exist = false;                         // 接続先が見つかっているかどうかのフラグ
    for(uint32_t i = 0; i < bssCount; i++)
    {
        nn::wlan::BeaconDescriptionReader beacon = resultReader.GetNextDescription();
        if( mySsid == beacon.GetSsid() )
        {
            NN_LOG("WlanInfra: Found access point! ch:%d\n", beacon.GetChannel());
            myBssid = beacon.GetBssid();
            channel = beacon.GetChannel();
            Exist = true;
        }
    }

    // 接続対象のAPが見つかっていなかったら終了
    if( Exist == false )
    {
        NN_LOG("Cannot find the target AP\n");
        return;
    }

    // 相手先のセキュリティ情報入力
    nn::wlan::Security security = {
            nn::wlan::SecurityMode_Wpa2Aes,
            nn::wlan::SecurityMode_Wpa2Aes,
            0,
            "012345678",  // Key
    };

    // 接続成功するまで接続試行
    while( 1 )
    {
        nn::wlan::Infra::Connect(mySsid, myBssid, channel, security, true);

        // Connect関数を抜けた時点で接続行為は終了しているので、実際に接続に成功したかはGetConnectionStatusで確認する
        nn::os::WaitSystemEvent(&connectionEvent);
        NN_LOG("WlanTest: Connection event signaled\n");

        nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        WlanTest::PrintConnectionStatus(&connectionStatus);

        if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
        {
            NN_LOG("WlanTest: CONNECTED!!!\n");
            break;
        }

        // 3秒待って再試行
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(3000));
    }

#ifdef ENABLE_RX_THREAD
    // To create rx entry
    uint32_t rxId;  // output variable for rxid
    uint16_t ethertypes[] = { 0x0800 };  // Ethertype (IPv4)

    result = nn::wlan::Socket::CreateRxEntry(&rxId, ethertypes, sizeof(ethertypes) / sizeof(uint16_t), 50);
    NN_ASSERT( result.IsSuccess() );
    NN_LOG("WlanTest: Created Rx Entry [%d]\n", rxId);

    // To create rx thread
    // waiting for ipv4 packet
    result = nn::os::CreateThread( &g_ThreadRx, RxThreadFunc, &rxId, g_ThreadStackRx, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_ThreadTx." );
    nn::os::StartThread( &g_ThreadRx );
#endif

    // リンクレベルを取得および表示し続ける
    int32_t rssi;
    nn::os::Tick sysTick = nn::os::GetSystemTick();
#if defined(PRINT_TSF_VALUE)
    nn::wlan::Socket::EnableTsfTimerFunction();
    nn::os::Tick tsfTick = nn::os::GetSystemTick();
    int64_t prevDelta = 0;
#endif
    while( 1 )
    {
        if( nn::os::TimedWaitSystemEvent(&connectionEvent, nn::TimeSpan::FromMilliSeconds(16)) == true )
        {
            nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
            WlanTest::PrintConnectionStatus(&connectionStatus);

            if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
            {
                NN_LOG("Disconnect\n");
                break;
            }
        }

        // 5秒毎にRSSI値とリンクレベルを表示する
        if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMilliSeconds() >= 5000 )
        {
            nn::wlan::Infra::GetRssi(&rssi);
            NN_LOG("+++++++++++++++++\n");
            NN_LOG("RSSI : %d[dbm]\n", rssi);
            NN_LOG("LINK LEVEL : %d\n", nn::wlan::Infra::ConvertRssiToLinkLevel(rssi));
            sysTick = nn::os::GetSystemTick();
#if defined(SCAN_ON_CONNECTION)
            nn::wlan::Infra::StartScan(g_scanBuffer, buffSize, scanParam);
            NN_LOG("WlanTest: Scan finished.\n");
            break;
#endif
        }

#if defined(PRINT_TSF_VALUE)
        // 1秒ごとにTSF値を出力する
        if( (nn::os::GetSystemTick() - tsfTick).ToTimeSpan().GetMilliSeconds() >= 1000 )
        {
            int64_t deltaTime;
            nn::wlan::Socket::GetDeltaTimeBetweenSystemAndTsf(&deltaTime);
            NN_LOG("[TSF GAP]%lld[usec] (%lld from prev)\n", deltaTime, deltaTime - prevDelta);
            prevDelta = deltaTime;
            tsfTick = nn::os::GetSystemTick();
        }
#endif
    }

#ifdef ENABLE_RX_THREAD
    // 受信用スレッドを終了する
    g_ExitFlagRx = false;
    nn::wlan::Socket::CancelGetFrame(rxId);

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &g_ThreadRx );
    // スレッドを破棄する
    nn::os::DestroyThread( &g_ThreadRx );
    nn::wlan::Socket::DeleteRxEntry(rxId);
#endif

    nn::wlan::Infra::Disconnect();

    nn::wlan::Infra::CloseMode();
    NN_LOG("WlanTest: CloseMode done\n");

#if defined(PRINT_TSF_VALUE)
    nn::wlan::Socket::DisableTsfTimerFunction();
#endif

    nn::wlan::FinalizeInfraManager();
#ifdef ENABLE_RX_THREAD
    nn::wlan::FinalizeSocketManager();
#endif
    NN_LOG("\n\n End WlanTest \n\n");
} // NOLINT(impl/function_size)

extern "C" void nnMain()
{
    NN_LOG("\n\n Start WlanTest Infra \n\n");
    WlanTest::SystemInitialize();
    TestCaseInfra();
    WlanTest::SystemFinalize();
}

