﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// --auto MC2 --role MASTER --input D:/home/WIT/input-v8_0/MC2_AgingTest.txt
// --auto MC2 --role CLIENT --input D:/home/WIT/input-v8_0/MC2_AgingTest.txt
// --auto MC1 --input D:/home/WIT/input-v8_0/MC1_AgingTest.txt

#include <cstdlib>
#include <nn/nn_Macro.h>
#include <nn/oe.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include "Arg.h"
#include "Program.h"

#ifdef BT_STAT_SD_OUT
#include "BtStatSdOut.h"
#endif

using namespace WlanTest;

namespace
{

bool g_InitialFlightMode = false;
nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

} // namespace

void Initialize()
{
    srand(nn::os::GetSystemTick().ToTimeSpan().GetNanoSeconds());

    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), "WitMainThread");

    // デバッグ設定書き込み
    const bool isEnabled = true;
    nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &isEnabled, sizeof(isEnabled));

    // nifm を停止する
    nn::nifm::Initialize();
    g_InitialFlightMode = Application::IsWlanFlightModeEnabled();
    Application::SetWlanFlightMode(true);

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }
#endif

    nn::Result result;

    nn::bluetooth::InitializeBluetoothDriver();

    nn::oe::Initialize();

    // 終了リスクエストをハンドリングする
    nn::oe::EnterExitRequestHandlingSection();

    // アプリへのフォーカスが外れてもアプリの進行を止めないようにする
    nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_Notify);

    result = nn::wlan::InitializeLocalManager();
    NN_ASSERT(result.IsSuccess());

    result = nn::wlan::InitializeDetectManager();
    NN_ASSERT(result.IsSuccess());

    result = nn::wlan::InitializeSocketManager();
    NN_ASSERT(result.IsSuccess());

    result = nn::wlan::InitializeInfraManager();
    NN_ASSERT(result.IsSuccess());

    result = nn::socket::Initialize(g_SocketConfigWithMemory);
    NN_ASSERT(result.IsSuccess());

    nn::btm::InitializeBtmInterface();

#ifdef BT_STAT_SD_OUT
    startBtRfMonitor();
#endif
}

void Finalize()
{
#ifdef BT_STAT_SD_OUT
    stopBtRfMonitor();
#endif

    NPad::Finalize();
    nn::btm::FinalizeBtmInterface();
    nn::socket::Finalize();
    nn::wlan::FinalizeDetectManager();
    nn::wlan::FinalizeInfraManager();
    nn::wlan::FinalizeSocketManager();
    nn::wlan::FinalizeLocalManager();
    nn::bluetooth::FinalizeBluetoothDriver();

    // プログラム起動時のフライトモードに戻す
    Application::SetWlanFlightMode(g_InitialFlightMode);

    // 終了リクエストのハンドリングを終了
    nn::oe::LeaveExitRequestHandlingSection();
}

extern "C" void nnMain()
{
    NN_LOG("\n\n Start WirelessIntegratedTest v%u.%u.%u %s\n\n",
           WIT_VERSION_MAJOR, WIT_VERSION_MINOR, WIT_VERSION_MICRO, WIT_VERSION_SUFFIX);

    Initialize();

    int argc = nn::os::GetHostArgc();
    if( argc > 1 )
    {
        // ##? bluetooth と hid が同時に使えないため Pad を無効化しておく
        Pad::DisablePad();

        Display::EnableConsoleOutput();
        Display::DisableDisplayOutput();

        WitArg witArg;
        witArg.Initialize();

        if( witArg.Parse() )
        {
            int testId = witArg.GetTestId();
            int roleId = witArg.GetRoleId();
            char inputFilePath[WitArg::INPUT_FILE_NAME_SIZE];
            nn::util::Strlcpy(inputFilePath, witArg.GetInputFilePath(), sizeof(inputFilePath));

            Test* pTest = GenerateAutoTest(testId, roleId, inputFilePath);

            if( pTest )
            {
                TestConductor conductor(pTest);
                conductor.Execute();

                delete pTest;
            }
            else
            {
                NN_LOG("  - failed : Parametes cannot be set. Please check arguments and parameters in your input file.\n");
                NN_LOG("           : usage '--auto <Test name> [--role <Role name>] [--input <file path>]'\n");
                NN_LOG("           : e.g. '--auto MC2 --role MASTER --input C:/wit/input.txt'\n");
            }
        }
        else
        {
            NN_LOG("Exit for error...\n");
        }
    }
    else
    {
        Display::EnableConsoleOutput();
        Display::EnableDisplayOutput();

        TestSelector selector;
        selector.Execute();
    }

    Finalize();

}
