﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "TesterViewer.h"

namespace WlanTest {

using namespace std;

// title にはコンソール表示のために改行する
const char* TEST_SUMMARY_TITLE_LABEL          = "Test Summary\n";
const char* ELAPSED_TIME_LABEL                = "Elapsed Time    : ";
const char* TEST_TIME_LABEL                   = "Test Time       : ";
const char* TARGET_TX_RATE_LABEL              = "Target Tx Rate  : ";
const char* TARGET_RX_RATE_LABEL              = "Target Rx Rate  : ";
const char* TARGET_PLR_LABEL                  = "Target PLR      : ";
const char* TARGET_MAX_LATENCY_LABEL          = "Target Max Lat. : ";
const char* TARGET_AVG_LATENCY_LABEL          = "Target Avg Lat. : ";
const char* RESULT_LABEL                      = "Total Result    : ";
const char* TARGET_BT_PLR_LABEL               = "Target Bt PLR   : ";


void SetResultFontColor(Label* pLabel, const TestResult result)
{
    NN_ASSERT(pLabel);
    if( result == RESULT_SUCCESS )
    {
        pLabel->TextColor = ToColor(GREEN);
    }
    else if( result == RESULT_FAILURE )
    {
        pLabel->TextColor = ToColor(RED);
    }
    else
    {
        pLabel->TextColor = ToColor(WHITE);
    }
}

void CreateTitleLabel(Label* pLabel, const char* title, const uint32_t x, const uint32_t y)
{
    NN_ASSERT(pLabel);
    pLabel->Text = title;
    pLabel->X = x;
    pLabel->Y = y;
    pLabel->FitSize();
}

void CreateContentLabel(Label* pLabel, const char* text, const uint32_t x, const uint32_t y, const uint32_t width)
{
    NN_ASSERT(pLabel);
    pLabel->Text = text;
    pLabel->X = x;
    pLabel->Y = y;
    pLabel->FitSize();
    pLabel->Width += width;
}





void UpdateElapsedTimeLabel(Label* pLabel, const nn::TimeSpan& ts)
{
    NN_ASSERT(pLabel);
    string str;
    if(ts == nn::TimeSpan(0))
    {
        str = "-";
    }
    else
    {
        str = ToString(ts);
    }
    pLabel->Text = ELAPSED_TIME_LABEL + str;
}

void UpdateTestTimeLabel(Label* pLabel, const nn::TimeSpan& ts)
{
    NN_ASSERT(pLabel);
    pLabel->Text = TEST_TIME_LABEL + ToString(ts);
}

void UpdateTargetThroughput(Label* pLabel, const char* label, const uint32_t tps, const TestResult result)
{
    NN_ASSERT(pLabel);
    string str = "-";
    if( tps != 0 )
    {
        SetResultFontColor(pLabel, result);

        ostringstream oss;
        oss << ToStringForBinaryPrefix(tps) << "bps";
        str = oss.str();
    }
    else
    {
        SetResultFontColor(pLabel, RESULT_NOT_COMPLETED);
    }

    pLabel->Text = label + str;
}

void UpdateTargetPlr(Label* pLabel, const char* label, const double plr, const TestResult result)
{
    NN_ASSERT(pLabel);
    string str = "-";
    if( plr >= 0.0f )
    {
        SetResultFontColor(pLabel, result);

        ostringstream oss;
        oss << plr << " (" << plr * 100.0f << "%%)";
        str = oss.str();
    }
    else
    {
        SetResultFontColor(pLabel, RESULT_NOT_COMPLETED);
    }

    pLabel->Text = label + str;
}

void UpdateTargetLatency(Label* pLabel, const char* label, const uint32_t latency, const TestResult result)
{
    NN_ASSERT(pLabel);
    string str = "-";
    if( latency != 0 )
    {
        SetResultFontColor(pLabel, result);

        ostringstream oss;
        oss << latency << " ms";
        str = oss.str();
    }
    else
    {
        SetResultFontColor(pLabel, RESULT_NOT_COMPLETED);
    }

    pLabel->Text = label + str;
}

void UpdateTotalResult(Label* pLabel, const char* label, const TestResult result, const bool isTesting)
{
    NN_ASSERT(pLabel);
    string str;

    if( isTesting )
    {
        SetResultFontColor(pLabel, RESULT_NOT_COMPLETED);
        str = "TESTING";
    }
    else
    {
        SetResultFontColor(pLabel, result);

        switch( result )
        {
        case RESULT_SUCCESS : str = "SUCCESS"; break;
        case RESULT_FAILURE : str = "FAILURE"; break;
        case RESULT_NOT_COMPLETED : str = "CANCELED"; break;
        default: break;
        }
    }

    pLabel->Text = label + str;
}


}
