﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/os.h>

#include <cstring>
#include <map>

#include "Test.h"
#include "TestUtil.h"


namespace WlanTest {

using namespace std;

struct WitParam
{
    TestParam testParam;
    LogParam logParam;
    MasterParam masterParam;
    ClientParam clientParam;
    SocketParam socketParam;
    DataGeneratorParam txParam;
};

const char CLASS_NAME_TEST[]       = "TEST";
const char CLASS_NAME_MASTER[]     = "MASTER";
const char CLASS_NAME_CLIENT[]     = "CLIENT";
const char CLASS_NAME_TX[]         = "TX";
const char CLASS_NAME_LOG[]        = "LOG";
const char CLASS_NAME_INFRA[]      = "INFRA";

const nn::TimeSpan RcViewer::LoggingInterval = nn::TimeSpan::FromSeconds(1);

const char* LogParam::ITEM_STR_LOGGING_INTERVAL  = "LOGGING_INTERVAL";

void SetDefaultParam(int testId, int roleId, WitParam& param)
{
    param.testParam.isAutoTest       = true;

    switch(testId)
    {
    // Local test
    case TEST_ID_H3 :
        {
            param.masterParam.numOfClients     = 7;
            param.testParam.minRxRate        = 250 * 1024;
            param.testParam.maxPlr           = 0.01f;
            param.testParam.maxLatencyMax    = 5;
            param.testParam.maxLatencyAvg    = 2;
            param.testParam.wlanMode         = nn::btm::WlanMode_Local4;
        }
        break;

    case TEST_ID_H4 :
        {
            param.masterParam.numOfClients     = 7;
            param.testParam.minRxRate        = 200 * 1024 * 1024;
            param.testParam.maxPlr           = 5.0f * 0.01f;
            param.testParam.maxLatencyMax    = 0;
            param.testParam.maxLatencyAvg    = 0;
            param.testParam.wlanMode         = nn::btm::WlanMode_Local4;
        }
        break;

    case TEST_ID_H5 :
        {
            param.masterParam.numOfClients     = 1;
            param.testParam.minRxRate        = 200 * 1024 * 1024;
            param.testParam.maxPlr           = 0;
            param.testParam.maxLatencyMax    = 0;
            param.testParam.maxLatencyAvg    = 0;
            param.testParam.wlanMode         = nn::btm::WlanMode_Local4;
        }
        break;

    case TEST_ID_H6 :
        {
            double minRxRate = 0.0f;
            double maxPlr = 0.0f;
            if( roleId == ROLE_ID_MASTER )
            {
                minRxRate = 100 * 1024;
                maxPlr = 0.01f;
            }
            else if( roleId == ROLE_ID_CLIENT )
            {
                minRxRate = 10 * 1024 * 1024;
                maxPlr = 0.0001f;
            }

            param.masterParam.numOfClients   = 7;

            param.testParam.minRxRate        = minRxRate;
            param.testParam.maxPlr           = maxPlr;
            param.testParam.maxLatencyMax    = 0;
            param.testParam.maxLatencyAvg    = 0;
            param.testParam.wlanMode         = nn::btm::WlanMode_Local4;
        }
        break;

    case TEST_ID_MC2 :
        {
            param.masterParam.numOfClients   = 3;

            param.testParam.minRxRate        = 250 * 1024;
            param.testParam.maxPlr           = 0.01f;
            param.testParam.maxLatencyMax    = 5;
            param.testParam.maxLatencyAvg    = 2;
            param.testParam.wlanMode         = nn::btm::WlanMode_Local4;
        }
        break;

    // Infra test
    case TEST_ID_H1 :
        {
            param.masterParam.numOfClients   = 0;

            param.testParam.minRxRate        = 500 * 1024;
            param.testParam.maxPlr           = 0.0001;
            param.testParam.maxLatencyMax    = 5;
            param.testParam.maxLatencyAvg    = 2;
            param.testParam.wlanMode         = nn::btm::WlanMode_None;
        }
        break;

    case TEST_ID_H2 :
        {
            param.masterParam.numOfClients   = 0;

            param.testParam.minRxRate        = 500 * 1024 * 1024;
            param.testParam.maxPlr           = 0;
            param.testParam.maxLatencyMax    = 0;
            param.testParam.maxLatencyAvg    = 0;
            param.testParam.wlanMode         = nn::btm::WlanMode_None;
        }
        break;

    case TEST_ID_C1 :
        {
            param.masterParam.numOfClients   = 0;

            param.testParam.minRxRate        = 500 * 1024;
            param.testParam.maxPlr           = 0.01;
            param.testParam.maxLatencyMax    = 0;
            param.testParam.maxLatencyAvg    = 0;
            param.testParam.wlanMode         = nn::btm::WlanMode_None;
        }
        break;

    case TEST_ID_C2 :
        {
            param.masterParam.numOfClients   = 0;

            param.testParam.minRxRate        = 500 * 1024;
            param.testParam.maxPlr           = 0.01;
            param.testParam.maxLatencyMax    = 0;
            param.testParam.maxLatencyAvg    = 0;
            param.testParam.wlanMode         = nn::btm::WlanMode_None;
        }
        break;

    case TEST_ID_C3 :
        {
            param.masterParam.numOfClients   = 0;

            param.testParam.minRxRate        = 500 * 1024;
            param.testParam.maxPlr           = 0.01;
            param.testParam.maxLatencyMax    = 0;
            param.testParam.maxLatencyAvg    = 0;
            param.testParam.wlanMode         = nn::btm::WlanMode_None;
        }
        break;

    case TEST_ID_MC1 :
        {
            param.masterParam.numOfClients   = 0;

            param.testParam.minRxRate        = 500 * 1024;
            param.testParam.maxPlr           = 0.0001;
            param.testParam.maxLatencyMax    = 0;
            param.testParam.maxLatencyAvg    = 0;
            param.testParam.wlanMode         = nn::btm::WlanMode_None;
        }
        break;

    default :
        break;
    }
} //NOLINT(impl/function_size)

bool ParseLocalTestFile(const char* filePath, WitParam* witParam)
{
    NN_ASSERT(witParam);

    const size_t BufferSize = 4096;
    static uint8_t buf[BufferSize];

    if( !ReadFile(buf, 4096, filePath) )
    {
        return false;
    }

    // ファイル読込
    stringstream ss(reinterpret_cast<char*>(buf));
    string str;
    while(std::getline(ss, str, '\n'))
    {
         // "#" 以降のコメントを削除
        string::size_type index = str.find('#');
        if( index != string::npos )
        {
            str = str.substr(0, index);
        }

         // 文字列を分解
        string classStr;
        string::size_type classIndex = str.find("::");
        if( classIndex != string::npos )
        {
            classStr = str.substr(0, classIndex);
            classStr = Trim(classStr);
            classStr = ToUpper(classStr);
        }

        string itemStr;
        string::size_type itemIndex = str.find(":", classIndex + 2);
        if( itemIndex != string::npos )
        {
            itemStr = str.substr(classIndex + 2, itemIndex - (classIndex + 2));
            itemStr = Trim(itemStr);
            itemStr = ToUpper(itemStr);
        }

        string valueStr;
        valueStr = str.substr(itemIndex + 1, str.size());
        valueStr = Trim(valueStr);
        valueStr = Trim(valueStr, "\n\r");

        if( classStr.size() == 0 || itemStr.size() == 0 || valueStr.size() == 0 )
        {
            continue;
        }

        if( classStr == string(CLASS_NAME_TEST) )
        {
            if( !witParam->testParam.SetParam(itemStr, valueStr) )
            {
                NN_LOG("  -failed : Inappropriate parameter\n");
                NN_LOG("           -> %s(%s, %s)\n", str.c_str(), itemStr.c_str(), valueStr.c_str());
                return false;
            }
        }
        else if( classStr == string(CLASS_NAME_LOG) )
        {
            if( !witParam->logParam.SetParam(itemStr, valueStr) )
            {
                NN_LOG("  - failed : Inappropriate parameter\n");
                NN_LOG("            -> %s(%s, %s)\n", str.c_str(), itemStr.c_str(), valueStr.c_str());
                return false;
            }
        }
        else if( classStr == string(CLASS_NAME_MASTER) )
        {
            if( !witParam->masterParam.SetParam(itemStr, valueStr) )
            {
                NN_LOG("  - failed : Inappropriate parameter\n");
                NN_LOG("            -> %s(%s, %s)\n", str.c_str(), itemStr.c_str(), valueStr.c_str());
                return false;
            }
        }
        else if( classStr == string(CLASS_NAME_CLIENT) )
        {
            if( !witParam->clientParam.SetParam(itemStr, valueStr) )
            {
                NN_LOG("  - failed : Inappropriate parameter\n");
                NN_LOG("            -> %s(%s, %s)\n", str.c_str(), itemStr.c_str(), valueStr.c_str());
                return false;
            }
        }
        else if( classStr == string(CLASS_NAME_INFRA) )
        {
            if( !witParam->socketParam.SetParam(itemStr, valueStr) )
            {
                NN_LOG("  - failed : Inappropriate parameter\n");
                NN_LOG("            -> %s(%s, %s)\n", str.c_str(), itemStr.c_str(), valueStr.c_str());
                return false;
            }
        }
        else if( classStr == string(CLASS_NAME_TX) )
        {
            if( !witParam->txParam.SetParam(itemStr, valueStr) )
            {
                NN_LOG("  - failed : Inappropriate parameter\n");
                NN_LOG("            -> %s(%s, %s)\n", str.c_str(), itemStr.c_str(), valueStr.c_str());
                return false;
            }
        }
        else
        {
            NN_LOG("Ignore line : '%s'\n", str.c_str());
        }
    }

    return true;
} //NOLINT(impl/function_size)

LocalTest* GenerateLocalTest(int testId, int roleId, const char* filePath)
{
    WitParam witParam;
    LocalTest *pTest;

    // 各テストの初期パラメータ設定
    SetDefaultParam(testId, roleId, witParam);

    // ファイルからのパラメータ設定
    if( strlen(filePath) > 0 )
    {
        if( !ParseLocalTestFile(filePath, &witParam) )
        {
            return nullptr;
        }
    }

    if( roleId == ROLE_ID_MASTER )
    {
        pTest = new LocalMasterTest(witParam.testParam, witParam.masterParam, witParam.txParam);
    }
    else if( roleId == ROLE_ID_CLIENT )
    {
        pTest = new LocalClientTest(witParam.testParam, witParam.clientParam, witParam.txParam);
    }
    else
    {
        NN_LOG("  - failed : Role is not specified\n");
        pTest = nullptr;
    }

    if( pTest )
    {
        NN_LOG("\nStart %s Test\n", TestId2TestName(testId));

        NN_LOG(" << Test param >>\n");
        NN_LOG("  Role                     : %s\n", RoleId2RoleName(roleId));
        witParam.testParam.Print();
        NN_LOG("\n");

        NN_LOG(" << Log param >>\n");
        witParam.logParam.Print();
        NN_LOG("\n");

        NN_LOG(" << Master param >>\n");
        witParam.masterParam.Print();
        NN_LOG("\n");

        NN_LOG(" << Client param >>\n");
        witParam.clientParam.Print();
        NN_LOG("\n");

        NN_LOG(" << Socket param >>\n");
        witParam.socketParam.Print();
        NN_LOG("\n");

        NN_LOG(" << Tx param >>\n");
        witParam.txParam.Print();
        NN_LOG("\n");
    }

    return pTest;
}

InfraTest* GenerateInfraTest(int testId, int roleId, const char* filePath)
{
    WitParam witParam;
    InfraTest *pTest;

    // 各テストの初期パラメータ設定
    SetDefaultParam(testId, roleId, witParam);

    // ファイルからのパラメータ設定
    if( strlen(filePath) > 0 )
    {
        if( !ParseLocalTestFile(filePath, &witParam) )
        {
            return nullptr;
        }
    }

    pTest = new InfraTest(witParam.testParam, witParam.socketParam, witParam.txParam);

    if( pTest )
    {
        NN_LOG("\nStart %s Test\n", TestId2TestName(testId));

        NN_LOG(" << Test param >>\n");
        NN_LOG("  Role                     : %s\n", RoleId2RoleName(roleId));
        witParam.testParam.Print();
        NN_LOG("\n");

        NN_LOG(" << Log param >>\n");
        witParam.logParam.Print();
        NN_LOG("\n");

        NN_LOG(" << Master param >>\n");
        witParam.masterParam.Print();
        NN_LOG("\n");

        NN_LOG(" << Client param >>\n");
        witParam.clientParam.Print();
        NN_LOG("\n");

        NN_LOG(" << Socket param >>\n");
        witParam.socketParam.Print();
        NN_LOG("\n");

        NN_LOG(" << Tx param >>\n");
        witParam.txParam.Print();
        NN_LOG("\n");
    }

    return pTest;
}

Test* GenerateAutoTest(int testId, int roleId, const char* filePath)
{
    NN_ASSERT(filePath);

    switch(testId)
    {
    // Infra test
    case TEST_ID_H1 :
    case TEST_ID_H2 :
    case TEST_ID_C1 :
    case TEST_ID_C2 :
    case TEST_ID_C3 :
    case TEST_ID_MC1 :
        {
            // Infra mode では roleId を使用しない。Client として代入しておく。
            roleId = ROLE_ID_CLIENT;

            InfraTest* pTest = GenerateInfraTest(testId, roleId, filePath);
            return pTest;
        }
        break;

    // Local test
    case TEST_ID_H3 :
    case TEST_ID_H4 :
    case TEST_ID_H5 :
    case TEST_ID_H6 :
    case TEST_ID_MC2 :
        {
            LocalTest* pTest = GenerateLocalTest(testId, roleId, filePath);
            return pTest;
        }
        break;

    default :
        {
            return nullptr;
        }
        break;
    }

    return nullptr;
}

void TestController::Set()
{
    target->SetLoggingInterval(loggingInterval);
    if( packetDump )
    {
        PacketLogger::EnableLog();
    }
    else
    {
        PacketLogger::DisableLog();
    }
}

}
