﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SocketNode.h"
#include "Log.h"

#define INITIAL_IP_ADDR     "0.0.0.0"
#define STATIC_GW_ADDR      "192.168.11.1"
#define STATIC_SUBNET_MASK  "255.255.255.0"
#define STATIC_DNS_1        "192.168.11.1"
#define STATIC_DNS_2        "192.168.11.1"

const uint16_t INITIAL_PORT = 50285;

namespace
{
// OUI + PI (不要であるため適当な値)
const nn::wlan::ReceivedDataMatchInfo DammyMatchInfo = {
    { 0x00, 0x00, 0x00, 0x00, 0x00 }, 5
};
}

namespace WlanTest {

const char* SocketParam::ITEM_STR_CHANNEL               = "CHANNEL";
const char* SocketParam::ITEM_STR_SSID                  = "SSID";
const char* SocketParam::ITEM_STR_SECURITY_MODE         = "SECURITY_MODE";
const char* SocketParam::ITEM_STR_SECURITY_KEY          = "SECURITY_KEY";
const char* SocketParam::ITEM_STR_TYPE                  = "TYPE";
const char* SocketParam::ITEM_STR_PACKET_FORMAT         = "PACKET_FORMAT";
const char* SocketParam::ITEM_STR_IP_ADDRESS            = "IP_ADDRESS";
const char* SocketParam::ITEM_STR_GW_IP_ADDRESS         = "GW_IP_ADDRESS";
const char* SocketParam::ITEM_STR_REMOTE_IP_ADDRESS     = "REMOTE_IP_ADDRESS";
const char* SocketParam::ITEM_STR_PORT                  = "PORT";

/*---------------------------------------------------------------------------
　　　　　SocketReceiver
---------------------------------------------------------------------------*/

SocketReceiver::SocketReceiver()
{
    m_ProtocolType = SOCKET_TYPE_UDP;
    m_RequestStop = false;
    m_IsStop = true;
}

void SocketReceiver::ReceiveThreadFunction(void* receiver)
{
    DBG_VLOG("Thread start - SocketReceiver::ReceiveThread");
    reinterpret_cast<SocketReceiver*>(receiver)->Receive();
    DBG_VLOG("Thread stop - SocketReceiver::ReceiveThread");
}

void SocketReceiver::Receive()
{
    if(m_Target == NULL || m_ProtocolType == SOCKET_TYPE_UNKNOWN)
    {
        NN_LOG("Target: %d\n ProtocolType : %d\n", m_Target, m_ProtocolType);
        NN_LOG("Can't receive because Target or ProtocolType is not not set.");
        m_RequestStop = true;
        return;
    }

    nn::Result result = nn::ResultSuccess();
    ReceiveEventArgs args;
    uint8_t buffer[2334];
    args.Data = buffer;

    PacketLogger& logger = PacketLogger::GetInstance();
    if( logger.IsEnabled() )
    {
        logger.Write("SourceAddress, Port, Size(byte), Seqno\n");
    }

    // IDLE から状態遷移するまで一定時間待つ
    nn::os::Tick tick = nn::os::GetSystemTick();
    while( !m_Target->IsConnected() )
    {
        if( m_RequestStop )
        {
            break;
        }

        Sleep(nn::TimeSpan::FromMilliSeconds(100));
    }

    while( !m_RequestStop )
    {
        if( m_Target->WaitReceive(nn::TimeSpan::FromMilliSeconds(1000)) )
        {
            // Receive を WlanGetFrame を介して呼ぶ
            //result = m_Target->Receive(buffer, reinterpret_cast<size_t*>(&(args.Size)), sizeof(buffer));
            result = m_Target->WlanGetFrame(0, buffer, &args.Size, sizeof(buffer), &args.Rssi);
        }
        else
        {
            continue;
        }

        if( result.IsFailure() )
        {
            m_Statistics.ReceiveErrorCount++;
            Sleep(nn::TimeSpan::FromMilliSeconds(1));    // 切断時の連続受信失敗を避けるため、スリープする
        }
        else
        {
            if( m_Statistics.ReceiveCount == 0 )
            {
                m_Statistics.FirstReceiveTime = nn::os::GetSystemTick();
            }
            m_Statistics.LastReceiveTime = nn::os::GetSystemTick();
            m_Statistics.ReceiveCount++;
            m_Statistics.ReceiveSize += args.Size;

            // パケットの中身を表示
            // for(int i=0; i<40; i++)
            // {
            //     NN_LOG("%02hx ", buffer[i]);
            //     if((i + 1)%8==0)
            //     {
            //         NN_LOG("\n");
            //     }
            // }

            args.Receiver = PROTOCOL_RECEIVER;
            // 受信イベントを発生させる
            if( m_Handler != NULL )
            {
                (*m_Handler)(&args);
            }
            else
            {
                NN_LOG("ReceiveEvent does not exist.\n");
            }
        }
    }

    if( logger.IsEnabled() )
    {
        logger.Write("end\n");
    }

    if( result.IsFailure() )
    {
        NN_LOG(" - failed : WlanGetFrame\n");
        WlanTest::TPrintResult(result);
    }

    m_IsStop = true;
}

nn::Result SocketReceiver::Start(int32_t priority)
{
    static NN_ALIGNAS(4096) char s_SocketReceiveThreadStack[4096 * 5];
    nn::Result result;

    if(!m_IsStop)
    {
        return WlanTest::ResultBusy();
    }
    m_IsStop = false;
    m_RequestStop = false;
    result = nn::os::CreateThread(&m_ReceiveThread, SocketReceiver::ReceiveThreadFunction, this, s_SocketReceiveThreadStack, sizeof(s_SocketReceiveThreadStack), priority);
    if( result.IsFailure() )
    {
        NN_LOG("ReceiveThread can not run.\n");
        return result;
    }

    nn::os::SetThreadNamePointer(&m_ReceiveThread, "WitSocketRxThread");
    nn::os::StartThread(&m_ReceiveThread);

    return result;
}

nn::Result SocketReceiver::Stop()
{
    nn::Result result = nn::ResultSuccess();
    if(!m_IsStop)
    {
        m_RequestStop = true;
        nn::os::DestroyThread(&m_ReceiveThread);
    }
    return result;
}

/*---------------------------------------------------------------------------
　　　　　InfraNode
---------------------------------------------------------------------------*/

InfraNode::InfraNode() : m_Cs(false, 0)
{
    m_Ssid = nn::wlan::Ssid("wireless_test_1");
    m_Bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();
    m_Channel = -1;

    m_IpAddress = INITIAL_IP_ADDR;
    m_RemoteIpAddress = INITIAL_IP_ADDR;
    m_GwIpAddress = INITIAL_IP_ADDR;

    m_Security.privacyMode = nn::wlan::SecurityMode_Open;
    m_Security.groupPrivacyMode = nn::wlan::SecurityMode_Open;
    std::memset(m_Security.key, 0x00, 64);

    m_IsWaiting = false;
}

InfraNode::~InfraNode()
{
    StopReceiveCommand();
    StopReceiveData();
}

nn::Result InfraNode::Initialize()
{
    nn::Result result = WlanInitialize();
    if(result.IsSuccess())
    {
        m_IsInitialized = true;
    }

    //CreateRxEntry();

    return result;
}

nn::Result InfraNode::Finalize()
{
    nn::Result result;

    // result = DeleteRxEntry();
    // if( result.IsFailure() )
    // {
    //     NN_LOG(" - failed : DeleteRxEntry()\n");
    // }

    result = WlanFinalize();
    if(result.IsSuccess())
    {
        m_IsInitialized = false;
    }

    return result;
}

nn::Result InfraNode::WlanInitialize()
{
    nn::Result result = nn::ResultSuccess();
    // result = nn::wlan::InitializeInfraManager();
    // if(result.IsFailure())
    // {
    //     NN_LOG("  - failed : InitializeInfraManager()\n");
    //     return result;
    // }

    // 接続前に socket::Initialize() をするとハングする
    // result = nn::socket::Initialize();
    // if(result.IsFailure())
    // {
    //     NN_LOG("  - failed : socket::Initialize()\n");
    //     return result;
    // }

    return result;
}
nn::Result InfraNode::WlanFinalize()
{
    return nn::ResultSuccess();
}

nn::Result InfraNode::WlanOpenMode()
{
    return nn::wlan::Infra::OpenMode();
}

nn::Result InfraNode::WlanCloseMode()
{
    return nn::wlan::Infra::CloseMode();
}

nn::Result InfraNode::WlanLocalCreateRxEntry(uint32_t* pRxId, const uint16_t pProtocols[], const int32_t& count, const int32_t& capacity)
{
    return nn::ResultSuccess();
}

nn::Result InfraNode::WlanSocketCreateRxEntry(uint32_t* pRxid, const uint16_t pProtocols[], const int32_t& count, const int32_t& capacity)
{
    return nn::ResultSuccess();
}

nn::Result InfraNode::WlanLocalCreateRxEntryForAf(uint32_t* pRxid, const uint16_t pProtocols[], const int32_t& count, const int32_t& capacity)
{
    return nn::ResultSuccess();
}

nn::Result InfraNode::WlanDetectCreateRxEntryForAf(uint32_t* pRxid, const uint16_t pProtocols[], const int32_t& count, const int32_t& capacity)
{
    return nn::ResultSuccess();
}

nn::Result InfraNode::WlanLocalDeleteRxEntry(uint32_t* pRxId)
{
    return nn::ResultSuccess();
}

nn::Result InfraNode::WlanSocketDeleteRxEntry(uint32_t* pRxId)
{
    return nn::ResultSuccess();
}

nn::Result InfraNode::WlanLocalDeleteRxEntryForAf(uint32_t* pRxId)
{
    return nn::ResultSuccess();
}

nn::Result InfraNode::WlanDetectDeleteRxEntryForAf(uint32_t* pRxId)
{
    return nn::ResultSuccess();
}

nn::Result InfraNode::WlanLocalAddMatchingData(const uint32_t& rxId, const nn::wlan::ReceivedDataMatchInfo& pMatchInfo)
{
    return nn::ResultSuccess();
}

nn::Result InfraNode::WlanLocalRemoveMatchingData(const uint32_t& rxId, const nn::wlan::ReceivedDataMatchInfo& pMatchInfo)
{
    return nn::ResultSuccess();
}

nn::Result InfraNode::WlanGetFrame( uint32_t rxId, uint8_t pOutput[], size_t* pSize, size_t maxSize, int8_t* pRssi )
{
    return nn::wlan::Socket::GetFrameRaw( pOutput, maxSize, pSize, rxId );
}

nn::Result InfraNode::WlanPutFrame( const uint8_t pInput[], size_t size, bool selfCts)
{
    return nn::wlan::Socket::PutFrameRaw(pInput, size);
}

nn::Result InfraNode::WlanGetActionFrame(nn::wlan::MacAddress* pOutSrcMac, uint8_t pOutBuf[], size_t size, size_t* pOutSize, uint32_t rxId, uint16_t* pChannel, int16_t* pRssi)
{
    NN_ASSERT(false);
    return nn::ResultSuccess();
}

nn::Result InfraNode::WlanPutActionFrame(const nn::wlan::MacAddress& dstMac, const uint8_t* pData, size_t size, int16_t channel, uint32_t dwellTime)
{
    NN_ASSERT(false);
    return nn::ResultSuccess();
}

nn::Result InfraNode::WlanCancelGetFrame(uint32_t rxId)
{
    return nn::wlan::Socket::CancelGetFrame(rxId);
}

nn::Result InfraNode::WlanCancelGetActionFrame(uint32_t rxId)
{
    return nn::ResultSuccess();
}


nn::Result InfraNode::WlanGetState(nn::wlan::WlanState *pState)
{
    return nn::wlan::Infra::GetState(pState);
}

nn::Result InfraNode::Open()
{
    nn::Result result;

    ModeChanger::ToInfraMode();

    // チャネル選択
    // int channel = m_Channel;
    // if(channel == 0)
    // {
    //     InfraScanner scanner;
    //     scanner.SetSsid(m_Ssid);
    //     channel = scanner.SearchChannel();
    //     if(channel == 0)
    //     {
    //         NN_LOG(" - failed : Master Not Found\n");
    //         result = WlanTest::ResultNotFound();
    //         return result;
    //     }
    //     NN_LOG(" - Ch : %d\n", channel);
    //     m_Channel = channel;
    // }

    // AP への接続
    if( m_ConnectionStatus.state != nn::wlan::ConnectionState_Connected )
    {
        char buf[nn::wlan::Ssid::SsidHexStringLengthMax] = {0};
        NN_LOG(" === Connect Params ===============================\n");
        NN_LOG("   Ssid              : %s\n", m_Ssid.GetHexString(buf));
        NN_LOG("   Bssid             : %s\n", m_Bssid.GetString(buf));
        NN_LOG("   Channel           : %d\n", m_Channel);
        NN_LOG("   Security mode     : %d\n", m_Security.groupPrivacyMode);
        NN_LOG("   Security key      : %s\n", reinterpret_cast<char*>(m_Security.key));
        NN_LOG("   Auto keep alive   : %d\n", true);
        NN_LOG("   Bmiss timeout     : %d\n", 10);
        NN_LOG(" ===============================================\n");

        result = nn::wlan::Infra::Connect(m_Ssid, m_Bssid, m_Channel, m_Security, true);
        if( result.IsSuccess() )
        {
            nn::wlan::ConnectionStatus connectionStatus;
            nn::Result conResult = nn::wlan::Infra::GetConnectionStatus(&connectionStatus);

            if( conResult.IsSuccess() && connectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                // Auto の場合もあるので改めてチャネルを取得する
                m_Channel = connectionStatus.channel;
            }
        }
        else
        {
            NN_LOG(" - failed : Infra::Connect\n");
            WlanTest::TPrintResult(result);
            return result;
        }
    }

    return result;
}

nn::Result InfraNode::Close()
{
    nn::Result result;
    result = nn::wlan::Infra::Disconnect();
    if( result.IsFailure() )
    {
        NN_LOG(" - failed : Disconnect\n");
        WlanTest::TPrintResult(result);
    }

    ModeChanger::ToReadyMode();

    return result;
}

nn::wlan::MacAddress InfraNode::GetApAddress()
{
    nn::wlan::ConnectionStatus sta;
    nn::wlan::Infra::GetConnectionStatus( &sta );
    nn::wlan::MacAddress m(sta.bssid);

    return m;
}

bool InfraNode::IsConnected()
{
    bool isConnected = false;

    m_Cs.Lock();

    isConnected = (m_ConnectionStatus.state == nn::wlan::ConnectionState_Connected);

    m_Cs.Unlock();

    return isConnected;
}

bool InfraNode::IsAfRxReady()
{
    return true;
}

nn::Result InfraNode::CancelConnect()
{
    return nn::wlan::Infra::CancelConnect();
}

nn::Result InfraNode::CreateRxEntry()
{
    nn::Result result;

    m_Cs.Lock();

    // RxId == 0 の場合に Rx エントリが未作成とみなし、新たに作成する
    if( m_SocketDataRxId == 0 )
    {
        uint16_t etherTypes[1] = { PID_OUI_EXT };
        result = WlanSocketCreateRxEntry(&m_SocketDataRxId, etherTypes, sizeof(etherTypes) / sizeof(etherTypes[0]), 80);
        NN_ASSERT(result.IsSuccess());
    }

    if( m_SocketControlRxId == 0 )
    {
        uint16_t etherTypes[1] = { PID_LOCAL_EXP1 };
        result = WlanSocketCreateRxEntry(&m_SocketControlRxId, etherTypes, sizeof(etherTypes) / sizeof(etherTypes[0]), 30);
        NN_ASSERT(result.IsSuccess());
    }

    if( m_LocalRxId == 0 )
    {
        // 機能的には不要だが、下記を送信可能であるかのフラグ管理として wlan が使用しているため呼ぶ
        // 呼ばないと wlan::Socket の送信もできない。 wlan::Local の送受信は使用しない。

        // 不要であるため他と重複しない適当な値
        uint16_t etherTypes[1] = { PID_LOCAL_EXP2 };
        result = WlanLocalCreateRxEntry(&m_LocalRxId, etherTypes, sizeof(etherTypes) / sizeof(etherTypes[0]), 30);
        NN_ASSERT(result.IsSuccess());

        result = WlanLocalAddMatchingData(m_LocalRxId, DammyMatchInfo);
        NN_ASSERT(result.IsSuccess());
    }


    m_Cs.Unlock();

    return nn::ResultSuccess();
}

nn::Result InfraNode::DeleteRxEntry()
{
    if( !m_IsInitialized )
    {
        return WlanTest::ResultNotInitialized();
    }

    m_Cs.Lock();
    NN_ABORT_UNLESS_RESULT_SUCCESS( WlanSocketDeleteRxEntry(&m_SocketDataRxId) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( WlanSocketDeleteRxEntry(&m_SocketControlRxId) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( WlanLocalRemoveMatchingData(m_LocalRxId, DammyMatchInfo) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( WlanLocalDeleteRxEntry(&m_LocalRxId) );
    m_Cs.Unlock();

    return nn::ResultSuccess();
}

/*---------------------------------------------------------------------------
          SocketNode
---------------------------------------------------------------------------*/
SocketNode::SocketNode()
{
    if( m_Receiver )
    {
        delete m_Receiver;
    }
    m_Receiver = new SocketReceiver();

    m_Type = SOCKET_TYPE_UNKNOWN;
    m_Socket = 0;
    FD_ZERO(&m_ReadFds);
    m_Port = INITIAL_PORT;
    m_InterfaceName = "wl0";

    m_Errors.Clear();
    SetRxPacketFormat(PACKET_FORMAT_WIT);

    m_IsIpAddrAssigned = false;
    m_EnableDhcp = false;
    m_SocketInitialized = false;
}

SocketNode::~SocketNode()
{
}

nn::Result SocketNode::WlanGetFrame( uint32_t rxId, uint8_t pOutput[], size_t* pSize, size_t maxSize, int8_t* pRssi)
{
    NN_UNUSED(rxId);
    NN_UNUSED(pRssi);

    return Receive(pOutput, pSize, maxSize);
}

nn::Result SocketNode::Initialize()
{
    InfraNode::Initialize();

    m_Errors.Clear();
    m_IsIpAddrAssigned = false;
    m_EnableDhcp = false;
    m_SocketInitialized = false;

    return nn::ResultSuccess();
}

nn::Result SocketNode::Finalize()
{
    nn::Result result = InfraNode::Finalize();
    if(result.IsSuccess())
    {
        m_IsInitialized = false;
    }

    return result;
}

nn::Result SocketNode::Open()
{
    nn::Result result;

    // AP への接続
    result = InfraNode::Open();
    if( result.IsFailure() )
    {
        return result;
    }

    if( !InfraNode::IsConnected() )
    {
        return WlanTest::ResultFailure();
    }

    if( !m_SocketInitialized )
    {
        m_SocketInitialized = true;

        // ソケットの生成
        CreateSocket(m_Socket);

        // notify interface linkup to BSD socket process
        struct ifreq ifr;
        nn::util::Strlcpy(ifr.ifr_name, m_InterfaceName.c_str(), sizeof(ifr.ifr_name));
        ifr.ifr_flags = IFF_UP;
        nn::socket::Ioctl(m_Socket, SIOCSIFFLAGS, &ifr, sizeof(struct ifreq));

        ConfigureInterface();
    }

    return result;
}

nn::Result SocketNode::Close()
{
    if(m_SocketInitialized)
    {
        nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(m_InterfaceName.c_str()));
        nn::socket::Close(m_Socket);
        m_SocketInitialized = false;
        m_IsIpAddrAssigned = false;
    }
    InfraNode::Close();

    return nn::ResultSuccess();
}

void SocketNode::MaintainConnection()
{
    nn::Result result;
    nn::wlan::ConnectionState lastState = nn::wlan::ConnectionState_Idle;
    m_ConnectionStatus.state = nn::wlan::ConnectionState_Idle;

    result = nn::wlan::Infra::GetConnectionEvent(&m_ConnectionEvent);
    if( result.IsFailure() )
    {
        NN_LOG("GetConnectionEvent failed\n");
        WlanTest::TPrintResult(result);
        goto End;
    }

    m_RequestStop = false;
    m_IsStop = false;
    while( !m_RequestStop )
    {
        nn::wlan::WlanState state = nn::wlan::WlanState_Stop;
        result = nn::wlan::Infra::GetState( &state );
        if( result.IsFailure() )
        {
            NN_LOG("GetState failed\n");
            WlanTest::TPrintResult(result);
            goto End;
        }

        // WiFi OFFだったならば
        // if(state == STATE_FLIGHT_MODE)
        // {
        //     StopReceive();
        //     StopMaintainConnection();
        //     break;
        // }

        // 接続していなければ接続する
        if( !IsConnected() )
        {
            Open();

            // 接続できた場合は Connect() を抜けた時点で接続が完了していることが保障されている
            result = nn::wlan::Infra::GetConnectionStatus(&m_ConnectionStatus);
            if( result.IsSuccess() && m_ConnectionStatus.state != nn::wlan::ConnectionState_Connected )
            {
                m_Statistics.ConnectErrorCount++;
            }
        }

        m_Cs.Lock();

        // 接続しているか調べる
        result = nn::wlan::Infra::GetConnectionStatus(&m_ConnectionStatus);
        if( result.IsFailure() )
        {
            NN_LOG("GetConnectionStatus failed\n");
            WlanTest::TPrintResult(result);
        }

        if( m_ConnectionStatus.state != lastState )
        {
            if( m_ConnectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                m_Statistics.ConnectCount++;

                // char buf[nn::wlan::Ssid::SsidHexStringLengthMax];
                // NN_LOG("  Connected to AP (%s on ch%d)\n", m_ConnectionStatus.ssid.GetHexString(buf), m_ConnectionStatus.channel);
                 NN_LOG("  Connected to AP\n");
            }
            if( m_ConnectionStatus.state == nn::wlan::ConnectionState_Idle && lastState == nn::wlan::ConnectionState_Connected )
            {
                m_Statistics.Disconnected++;
                m_IsIpAddrAssigned = false;
            }
        }
        lastState = m_ConnectionStatus.state;

        m_Cs.Unlock();
        nn::os::TimedWaitSystemEvent(&m_ConnectionEvent, nn::TimeSpan::FromMilliSeconds(100));
    }
End:
    // 保持しているコネクションステータスを切断にする
    // ここで取得しなければ、切断後も接続中のものが残ってしまう
    result = nn::wlan::Infra::GetConnectionStatus(&m_ConnectionStatus);

    if( result.IsFailure() )
    {
        NN_LOG("GetConnectionStatus failed\n");
        WlanTest::TPrintResult(result);
    }

    m_IsStop = true;
    nn::os::DestroySystemEvent(&m_ConnectionEvent);
}

void SocketNode::StopMaintainConnection()
{
    if( !m_IsStop )
    {
        m_RequestStop = true;

        CancelConnect();

        //#?? Detach できないから明示的に破棄する必要がある
        nn::os::DestroyThread(&m_MaintainConnectionThread);

        // Connect() 中に Disconnect() をしないように、MaintainConnectionThread が終了後に Close() する
        Close();
    }
}

bool SocketNode::WaitReceive(nn::TimeSpan timeSpan)
{
    int64_t sec = timeSpan.GetSeconds();
    int64_t us  = timeSpan.GetMicroSeconds();

    if( us != 0 )
    {
        us -= nn::TimeSpan::FromSeconds(sec).GetMicroSeconds();
    }

    timeval tv;
    tv.tv_sec  = sec;
    tv.tv_usec = us;

    fd_set fds;
    // 読み込み用 fd_set の初期化
    memcpy(&fds, &m_ReadFds, sizeof(fd_set));

    // fdsに設定されたソケットが読み込み可能になるまでタイムアウトつきで待機する
    int n = nn::socket::Select(m_Socket + 1, &fds, NULL, NULL, &tv);

    // タイムアウトの場合は 0 となる
    if( n == 0 )
    {
        return false;
    }

    return FD_ISSET(m_Socket, &fds);
}

void SocketNode::ConfigureInterface()
{
    nn::bsdsocket::cfg::IfSettings ifcfg;
    memset(&ifcfg, 0, sizeof(ifcfg));

    if(m_EnableDhcp)
    {
        // dhcp
        ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
        ifcfg.mtu  = 1500;
    }
    else
    {
        // static ip
        ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
        ifcfg.mtu  = 1500;

        nn::socket::InetAton(m_IpAddress.c_str(), &ifcfg.u.modeStatic.addr);
        nn::socket::InetAton(m_GwIpAddress.c_str(), &ifcfg.u.modeStatic.gatewayAddr);
        nn::socket::InetAton(STATIC_SUBNET_MASK, &ifcfg.u.modeStatic.subnetMask);
        ifcfg.u.modeStatic.broadcastAddr.S_addr =
            (ifcfg.u.modeStatic.addr.S_addr & ifcfg.u.modeStatic.subnetMask.S_addr) |
            ~ifcfg.u.modeStatic.subnetMask.S_addr;
        // nn::socket::InetAton(STATIC_DNS_1, &ifcfg.dnsAddrs[0]);
        // nn::socket::InetAton(STATIC_DNS_2, &ifcfg.dnsAddrs[1]);
        nn::socket::InetAton(m_GwIpAddress.c_str(), &ifcfg.dnsAddrs[0]);
        nn::socket::InetAton(STATIC_DNS_1, &ifcfg.dnsAddrs[1]);
    }

    nn::Result result = nn::bsdsocket::cfg::SetIfUp(const_cast<char*>(m_InterfaceName.c_str()), &ifcfg);
    if( result.IsFailure() )
    {
        NN_LOG("failed to configure interface %s - %d:%d\n",
               m_InterfaceName.c_str(),
               result.GetModule(),
               result.GetDescription());
        NN_ASSERT(false);
    }

    nn::bsdsocket::cfg::IfState ifState;
    result = nn::bsdsocket::cfg::GetIfState(m_InterfaceName.c_str(), &ifState);
    if( result.IsFailure() )
    {
        NN_LOG("failed to get interface state %s - %d:%d\n",
               m_InterfaceName.c_str(),
               result.GetModule(),
               result.GetDescription());
        NN_ASSERT(false);
    }

    m_IpAddress = nn::socket::InetNtoa(ifState.addr);
    m_IsIpAddrAssigned = true;
}

nn::Result SocketNode::Send(uint8_t data[], size_t size, uint8_t ieInd, size_t* pSentSize)
{
    NN_UNUSED(ieInd);

    nn::Result result;
    result = SocketSend(data, size, 0, pSentSize);

    if(result.IsSuccess())
    {
        if(m_Statistics.SendCount == 0)
        {
            m_Statistics.FirstSendTime = nn::os::GetSystemTick();
        }
        m_Statistics.LastSendTime = nn::os::GetSystemTick();
        m_Statistics.SendCount++;
        m_Statistics.SendSize += size;
    }
    else
    {
        m_Statistics.SendErrorCount++;
    }

    return result;
}

nn::Result SocketNode::Receive(uint8_t pOutput[], size_t* pSize, const size_t maxSize)
{
    nn::Result result;
    sockaddr_in fromAddr = { 0 };
    result = SocketRecv(pOutput, pSize, &fromAddr, maxSize, 0);

    // ##? パケットの中身の処理は Sink でやりたい
    if( result.IsSuccess() )
    {
        string remote = nn::socket::InetNtoa(fromAddr.sin_addr);
        uint16_t port = nn::socket::InetNtohs(fromAddr.sin_port);
        uint64_t seqNo = 0;
        bool isSuccess = false;
        if( m_RemoteIpAddress == remote )
        {
            if( m_RxPacketFormat == PACKET_FORMAT_WIT )
            {
                if( *pSize >= sizeof(uint64_t) )
                {
                    seqNo = *reinterpret_cast<uint64_t*>(pOutput);
                    isSuccess = true;
                }
            }
            else if( m_RxPacketFormat == PACKET_FORMAT_RTP )
            {
                if( *pSize >= 4 )
                {
                    seqNo = *reinterpret_cast<uint16_t*>(pOutput + 2);
                    seqNo = ((seqNo & 0xFF00) >> 8) | ((seqNo & 0x00FF) << 8);
                    isSuccess = true;
                }
            }
            else
            {
                NN_LOG(" Unknown rx packet format\n");
            }

            if( isSuccess )
            {
                m_Errors.Count(seqNo);
                //NN_LOG(" from %s:%u : seqno %llu\n", remote.c_str(), port, seqNo);
            }
            else
            {
                NN_LOG(" ** Recv size is too small to get WIT sequence no (size = %d)\n", *pSize);
            }
        }
        else
        {
            NN_LOG(" from %s:%u (?)\n", remote.c_str(), port);
        }

        PacketLogger& logger = PacketLogger::GetInstance();
        if( logger.IsEnabled() )
        {
            static char buf[100];
            if( isSuccess )
            {
                // source IP, target port, size(byte), seqno
                // (15+1)   + (5+1)      + (8+1)     + (8)   = 39
                snprintf(buf, sizeof(buf), "%s,%u,%llu,%llu\n",
                         remote.c_str(), port, static_cast<uint64_t>(*pSize), seqNo);
            }
            else
            {
                // source IP, target port, size(byte)
                // (15+1)   + (5+1)      + (8+1)      = 31
                snprintf(buf, sizeof(buf), "%s,%u,%llu\n",
                         remote.c_str(), port, static_cast<uint64_t>(*pSize));
            }
            logger.Write(buf);
        }
    }

    return result;
}

void SocketNode::ClearStatistics()
{
    InfraNode::ClearStatistics();
    m_Errors.Clear();
}

bool SocketNode::IsConnected()
{
    return InfraNode::IsConnected() && m_IsIpAddrAssigned;
}

/*---------------------------------------------------------------------------
          UdpNode
---------------------------------------------------------------------------*/
UdpNode::UdpNode()
{
    m_Type = SOCKET_TYPE_UDP;
    std::memset(&m_TxSockAddr, 0, sizeof(m_TxSockAddr));
    std::memset(&m_RxSockAddr, 0, sizeof(m_RxSockAddr));
}

UdpNode::~UdpNode()
{
}

nn::Result UdpNode::Initialize()
{
    return SocketNode::Initialize();
}

nn::Result UdpNode::Finalize()
{
    return SocketNode::Finalize();
}

nn::Result UdpNode::Open()
{
    nn::Result result;

    // AP への接続、ソケットの準備
    result = SocketNode::Open();
    if(result.IsFailure())
    {
        return result;
    }

    // ソケットへの接続
    result = OpenUdpServer(m_RxSockAddr);
    if(result.IsFailure())
    {
        return result;
    }

    result = OpenUdpClient(m_TxSockAddr);
    if(result.IsFailure())
    {
        return result;
    }

    return result;
}

nn::Result UdpNode::Close()
{
    return SocketNode::Close();
}

void UdpNode::CreateSocket(int& socket)
{
    socket = nn::socket::Socket(PF_INET, SOCK_DGRAM, 0);
    NN_ABORT_UNLESS(socket >= 0);
}

nn::Result UdpNode::OpenUdpServer(sockaddr_in& sockAddr)
{
    nn::Result result;

    // 自身の IP アドレスを設定する
    // ConfigureInterface() で Static IP を有効にしていれば IN_ANYADDR を自身の IP に設定しても
    // Static IP アドレスが有効になるはず
    in_addr outAddr = { 0 };
    nn::socket::InetAton(m_IpAddress.c_str(), &outAddr);
    sockAddr.sin_addr = outAddr;
    sockAddr.sin_port = nn::socket::InetHtons(m_Port);
    sockAddr.sin_family = AF_INET;

    // ソケットへの接続
    if( nn::socket::Bind(m_Socket, reinterpret_cast<sockaddr *>(&sockAddr), sizeof(sockAddr)) < 0 )
    {
        NN_LOG("server:  Bind failed (error %d, %s:%u)\n", nn::socket::GetLastErrno(), m_IpAddress.c_str(), m_Port);
        return WlanTest::ResultFailure();
    }

    // パケットが届いてるか確認するために fd に結び付ける
    // fd_set の初期化
    FD_ZERO(&m_ReadFds);

    // 読み込むソケットの登録
    FD_SET(m_Socket, &m_ReadFds);

    sockaddr_in tmpSockAddr = { 0 };
    socklen_t   saLen = sizeof(tmpSockAddr);
    if( nn::socket::GetSockName(m_Socket, reinterpret_cast<sockaddr *>(&tmpSockAddr), &saLen) < 0 )
    {
        NN_LOG("server:  GetSockName failed (error %d)\n", nn::socket::GetLastErrno());
        return WlanTest::ResultFailure();
    }

    NN_LOG("  [IP Addr: %s:%d]\n", m_IpAddress.c_str(), static_cast<int>(nn::socket::InetNtohs(tmpSockAddr.sin_port)));

    return nn::ResultSuccess();
}

nn::Result UdpNode::OpenUdpClient(sockaddr_in& sockAddr)
{
    // 送信先のソケットを設定
    in_addr outAddr = { 0 };
    nn::socket::InetAton(m_RemoteIpAddress.c_str(), &outAddr);
    sockAddr.sin_addr = outAddr;
    sockAddr.sin_port = nn::socket::InetHtons(m_Port);
    sockAddr.sin_family = AF_INET;

    NN_LOG("  [Remote : %s:%u]\n", m_RemoteIpAddress.c_str(), m_Port);

    return nn::ResultSuccess();
}

nn::Result UdpNode::SocketSend( const uint8_t pInput[], size_t size, int flags, size_t* pSentSize)
{
    ssize_t sentSize = -1;
    sentSize = nn::socket::SendTo(m_Socket, pInput, size, flags, reinterpret_cast<sockaddr*>(&m_TxSockAddr), sizeof(m_TxSockAddr));

    if( sentSize >= 0 )
    {
        // NN_LOG("[send - %s:? -> ", nn::socket::InetNtoa(m_RxSockAddr.sin_addr));
        // NN_LOG("%s:%u]", nn::socket::InetNtoa(m_TxSockAddr.sin_addr), nn::socket::InetNtohs(m_TxSockAddr.sin_port));

        // uint32_t len = (size < 200) ? size : 200;
        // char buf[200];
        // nn::util::Strlcpy(buf, reinterpret_cast<const char*>(pInput), len);
        // buf[199] = '\0';
        // NN_LOG(" socket : %d, sent size : %d, buf size : %d, data : %s\n", m_Socket, sentSize, size, buf);
        *pSentSize = sentSize;
        return nn::ResultSuccess();
    }
    else
    {
        NN_LOG("  - failed : Send(), error : %d\n", sentSize);
        return WlanTest::ResultFailure();
    }
}

nn::Result UdpNode::SocketRecv( uint8_t pOutput[], size_t* pSize, sockaddr_in *pFromAddr, const size_t maxSize, int flags)
{
    ssize_t recvSize = -1;
    sockaddr_in sockAddr = { 0 };
    socklen_t saLen = sizeof(sockAddr);

    recvSize = nn::socket::RecvFrom(m_Socket, pOutput, maxSize, flags, reinterpret_cast<sockaddr*>(&sockAddr), &saLen);

    if(recvSize >= 0)
    {
        *pSize = recvSize;
        *pFromAddr = sockAddr;

        //NN_LOG("[recv %s:%u -> ", nn::socket::InetNtoa(sockAddr.sin_addr), nn::socket::InetNtohs(sockAddr.sin_port));
        // NN_LOG("%s:%u]", nn::socket::InetNtoa(m_RxSockAddr.sin_addr), nn::socket::InetNtohs(m_RxSockAddr.sin_port));

        // NN_LOG(" socket : %d, recv size : %d, buf size : %d, data :", m_Socket, recvSize, maxSize);
        // for(int i=0; i<recvSize; ++i)
        // {
        //     NN_LOG(" %02x", pOutput[i]);
        // }
        // NN_LOG("\n");

        return nn::ResultSuccess();
    }
    else
    {
        NN_LOG("  - failed : Recv(), error : %d\n", recvSize);
        return WlanTest::ResultFailure();
    }
}



}

