﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Rssi.h"
#include "Node.h"

namespace WlanTest {

RssiStatistics::RssiStatistics() :
    m_IsStop(true),
    m_RecordInterval(500),
    m_RecordCount(360),
    m_pNode(nullptr),
    m_Cs(false, 0)
{
}

RssiStatistics::~RssiStatistics()
{
    StopRecordRssi();
}

void RssiStatistics::Clear()
{
    m_Cs.Lock();
    m_Statistics.clear();
    m_Cs.Unlock();
}

nn::Result RssiStatistics::StartRecordRssi(int32_t priority)
{
    static NN_ALIGNAS(4096) char s_RecordRssiThreadStack[4096 * 5];
    nn::Result result;

    if( m_pNode == nullptr )
    {
        return WlanTest::ResultNotInitialized();
    }

    if( !m_IsStop )
    {
        return WlanTest::ResultBusy();
    }

    m_IsStop = false;

    result = nn::os::CreateThread(&m_RecordRssiThread, RecordRssiThreadFunc, this, s_RecordRssiThreadStack, sizeof(s_RecordRssiThreadStack), priority);

    nn::os::SetThreadNamePointer(&m_RecordRssiThread, "WitRecordRssiThread");
    nn::os::StartThread(&m_RecordRssiThread);
    if( result.IsFailure() )
    {
        NN_LOG("Can not run RecordRssiThread.\n");
        return result;
    }

    return result;
}

void RssiStatistics::StopRecordRssi()
{
    if( !m_IsStop )
    {
        m_IsStop = true;
        nn::os::DestroyThread(&m_RecordRssiThread);
    }
}

bool RssiStatistics::IsStop()
{
    return m_IsStop;
}

int32_t RssiStatistics::FindOldestIndex()
{
    int32_t index = -1;
    if( m_Statistics.empty() )
    {
        return index;
    }

    nn::os::Tick oldestRecordTick = m_Statistics[0].back().recordTime;
    index = 0;
    for(int i=0; i<m_Statistics.size(); ++i)
    {
        if( oldestRecordTick > m_Statistics[i].back().recordTime )
        {
            index = i;
            oldestRecordTick = m_Statistics[i].back().recordTime;
        }
    }

    return index;
}

void RssiStatistics::RecordRssiThreadFunc(void* pRssiStatistics)
{
    DBG_VLOG("Thread start - Node::RecordRssiThread");

    reinterpret_cast<RssiStatistics*>(pRssiStatistics)->RecordRssi();

    DBG_VLOG("Thread stop - Node::RecordRssiThread");
}


void RssiStatistics::RecordRssi()
{
    NN_ASSERT_NOT_NULL(m_pNode);

    uint64_t count = 0;
    nn::os::Tick lastTick = nn::os::GetSystemTick();
    nn::os::Tick tick;
    while( !m_IsStop )
    {
        tick = nn::os::GetSystemTick();
        if( (tick - lastTick).ToTimeSpan().GetMilliSeconds() >= m_RecordInterval )
        {
            vector<Rssi> rssiList;
            bool isSuccess = m_pNode->GetRssi(&rssiList);

            if( !rssiList.empty() && isSuccess )
            {
                m_Cs.Lock();

                for(const auto& rssi : rssiList)
                {
                    // 同じ MacAddress を持つ統計情報に追加
                    RssiHistory* pRssiHistory = nullptr;
                    for(auto& stat : m_Statistics)
                    {
                        if( stat.back().address == rssi.address )
                        {
                            pRssiHistory = &stat;
                            break;
                        }
                    }

                    // 新たな MacAddress であれば新規追加 or 古い情報に上書き
                    if( pRssiHistory == nullptr )
                    {
                        if( m_Statistics.size() < ConnectableNodeCountMax )
                        {
                            RssiHistory rssiHistory;
                            m_Statistics.push_back(rssiHistory);
                            pRssiHistory = &m_Statistics.back();
                        }
                        else
                        {
                            // 接続ノードが多い場合は古い情報を削除する
                            int32_t index = FindOldestIndex();
                            NN_ASSERT(index != -1);
                            m_Statistics[index].clear();
                            pRssiHistory = &m_Statistics[index];
                        }
                    }

                    pRssiHistory->push_back(rssi);
                    while( pRssiHistory->size() > m_RecordCount )
                    {
                        pRssiHistory->pop_front();
                    }
                }

                m_Cs.Unlock();
            }
            lastTick = tick;
        }

        Sleep(nn::TimeSpan::FromMilliSeconds(10));
    }
}

const vector<RssiHistory>& RssiStatistics::GetStatistics()
{
    return m_Statistics;
}

void RssiStatistics::SetNode(Node* pNode)
{
    m_pNode = pNode;
}

void RssiStatistics::SetRecordInterval(uint32_t msec)
{
    m_RecordInterval = msec;
}

uint32_t RssiStatistics::GetRecordInterval()
{
    return m_RecordInterval;
}

void RssiStatistics::SetRecordCount(uint32_t count)
{
    m_RecordCount = count;
}

uint32_t RssiStatistics::GetRecordCount()
{
    return m_RecordCount;
}


}
