﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Detector.h"

namespace {

uint8_t  g_txBuffer[2000];

}
namespace WlanTest {

/*---------------------------------------------------------------------------
　　　　　DetectorHash
---------------------------------------------------------------------------*/

DetectorHash::DetectorHash() :
    m_HashMode(HashMode_Fixed),
    m_Hash({{0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}})
{
}

DetectorHash::DetectorHash(HashMode mode, nn::wlan::DetectHash hash) :
    m_HashMode(mode),
    m_Hash(hash)
{
}

void DetectorHash::Update()
{
    switch(m_HashMode)
    {
    case HashMode_Fixed :
        {
        }
        break;

    case HashMode_Incremental :
        {
            for(int i=HashSize - 1; i>=0; --i)
            {
                if( m_Hash.hash[i] == 0xff )
                {
                    m_Hash.hash[i] = 0;
                }
                else
                {
                    ++m_Hash.hash[i];
                    break;
                }
            }
        }
        break;

    case HashMode_Random :
        {
            for(int i=0; i<HashSize; ++i)
            {
                m_Hash.hash[i] = rand() % 0xFF;
            }
        }
        break;

    default :
        {
        }
        break;
    }
}

nn::wlan::DetectHash DetectorHash::GetHash() const
{
    return m_Hash;
}

void DetectorHash::SetHash(const nn::wlan::DetectHash& hash)
{
    m_Hash = hash;
}

DetectorHash::HashMode DetectorHash::GetMode() const
{
    return m_HashMode;
}

void DetectorHash::SetMode(const HashMode& mode)
{
    m_HashMode = mode;
}

bool DetectorHash::operator==(const DetectorHash& hash) const
{
    return m_HashMode == hash.GetMode() && m_Hash == hash.GetHash();
}

bool DetectorHash::operator!=(const DetectorHash& hash) const
{
    return !(*this == hash);
}


/*---------------------------------------------------------------------------
　　　　　Detector
---------------------------------------------------------------------------*/

Detector::Detector() :
    m_TxMode(TxMode_Disabled),
    m_Channel(nn::wlan::WirelessChannel_1ch),
    m_TxInterval(100),
    m_PayloadSize(1300),
    m_Hash(DetectorHash::HashMode_Fixed, nn::wlan::DetectHash({{}}))
{
}

Detector::~Detector()
{
}

nn::Result Detector::Initialize()
{
    nn::Result result = WlanInitialize();
    if(result.IsFailure())
    {
        NN_LOG(" - failed : LocalNode::WlanInitialize()\n");
        return result;
    }

    ModeChanger::ToDetectorMode(m_Channel);
    result = nn::wlan::Socket::EnableTsfTimerFunction();
    if(result.IsFailure())
    {
        NN_LOG(" - failed : EnableTsfTimerFunction()\n");
        return result;
    }

    // 送受信する際に Rx エントリが必須なので初期化時に作成する
    CreateRxEntry();

    m_IsInitialized = true;

    return result;
}

nn::Result Detector::Finalize()
{
    nn::Result result = DeleteRxEntry();
    if( result.IsFailure() )
    {
        NN_LOG(" - failed : DeleteRxEntry()\n");
    }

    result = nn::wlan::Socket::DisableTsfTimerFunction();
    if( result.IsFailure() )
    {
        NN_LOG(" - failed : DisableTsfTimerFunction()\n");
    }

    result = WlanFinalize();
    if(result.IsFailure())
    {
        NN_LOG(" - failed : LocalNode::WlanFinalize()\n");
    }

    ModeChanger::ToReadyMode();
    m_IsInitialized = false;

    return result;
}

nn::Result Detector::Open()
{
    nn::Result result;
    result = nn::wlan::Detect::StartCommunication();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

nn::Result Detector::Close()
{
    nn::Result result;

    nn::wlan::WlanState state;
    result = nn::wlan::Detect::GetState(&state);
    NN_ASSERT(result.IsSuccess(), "  - failed : GetState()");

    if( state == nn::wlan::WlanState_Detect )
    {
        result = nn::wlan::Detect::StopCommunication();
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    return result;
}

nn::Result Detector::WlanGetActionFrame(nn::wlan::MacAddress* pOutSrcMac, uint8_t pOutBuf[], size_t size, size_t* pOutSize, uint32_t rxId, uint16_t* pChannel, int16_t* pRssi)
{
    nn::os::Tick tick;
    return nn::wlan::Detect::GetActionFrame(pOutSrcMac, pOutBuf, size, pOutSize, rxId, pChannel, pRssi, &tick);
}

nn::Result Detector::WlanCancelGetFrame(uint32_t rxId)
{
    NN_UNUSED(rxId);
    return nn::ResultSuccess();
}

nn::Result Detector::WlanCancelGetActionFrame(uint32_t rxId)
{
    return nn::wlan::Detect::CancelGetActionFrame(rxId);
}

void GeneratePacket(uint8_t* buf, const size_t bufSize, const uint64_t sequenceNumber, const int16_t channel, const uint32_t length)
{
    NN_ASSERT(bufSize >= length, "bufSize : %u, length : %u", bufSize, length);
    NN_ASSERT(length >= sizeof(WitHeader) + sizeof(ActionFrameSearchInfoCommand));

    WitHeader header;
    GenerateWitHeader(&header, WitCommand_ActionFrameSearchInfo, sizeof(ActionFrameSearchInfoCommand));
    std::memcpy(buf, reinterpret_cast<uint8_t*>(&header), sizeof(header));

    ActionFrameSearchInfoCommand command;
    command.sequenceNumber = sequenceNumber;
    command.channel = channel;

    int64_t tsfTime = 0;
    nn::Result result = GetTsfTime(&tsfTime);
    NN_ASSERT(result.IsSuccess());
    command.tsfTime = tsfTime;

    std::memcpy(buf + sizeof(header), reinterpret_cast<uint8_t*>(&command), sizeof(command));
}

nn::Result Detector::WlanPutActionFrame(const nn::wlan::MacAddress& dstMac, const uint8_t* pData, size_t size, int16_t channel, uint32_t dwellTime)
{
    NN_UNUSED(channel);

    // Standalone でも受信できるように DhpCmdPeriodic を設定して送信する
    nn::wlan::DetectHeader header = {
        DhpMajor,
        DhpMinor,
        DhpCmdPeriodic,
        0, // reserved
        m_Hash.GetHash()
    };

    return nn::wlan::Detect::PutActionFrameOneShotEx(dstMac, GetMacAddress(), nn::wlan::ActionFrameType_Detect, header, pData, size, dwellTime);
}

nn::Result Detector::Send(uint8_t data[], size_t size, uint8_t ieInd, size_t* pSentSize)
{
    static nn::wlan::MacAddress BroadcastMacaddress = nn::wlan::MacAddress::CreateBroadcastMacAddress();

    //    パケットの中身を表示する
    // for(int i=0; i<30; i++)
    // {
    //    NN_LOG("%02x ", data[i]);
    //       if( i % 8 == 7)
    //           NN_LOG("\n");
    // }
    // NN_LOG("\n");

    nn::Result result;
    static uint64_t seq = 0;
    // ##fixme DataGenerator でペイロードを生成する
    GeneratePacket(data, size, seq++, m_Channel, size);
    m_Hash.Update();
    result = WlanPutActionFrame(BroadcastMacaddress, data, size, m_Channel, 0);
    if( result.IsFailure() )
    {
        m_Statistics.SendErrorCount++;
        Sleep(nn::TimeSpan::FromMilliSeconds(1));    // 切断時の連続送信失敗を避けるため、スリープする
    }
    else
    {
        if(m_Statistics.SendCount == 0)
        {
            m_Statistics.FirstSendTime = nn::os::GetSystemTick();
        }
        m_Statistics.LastSendTime = nn::os::GetSystemTick();
        m_Statistics.SendCount++;
        m_Statistics.SendSize += size;
        *pSentSize = size;
    }

    return result;
}

nn::wlan::MacAddress Detector::GetMacAddress()
{
    nn::Result result;

    if( m_Address == nn::wlan::MacAddress::CreateZeroMacAddress() )
    {
        result = nn::wlan::Detect::GetMacAddress(&m_Address);
        if( result.IsFailure() )
        {
            NN_LOG("GetMacAddress failed\n");
            WlanTest::TPrintResult(result);
        }
    }

    return m_Address;
}

void Detector::StartPeriodicActionFrame()
{
    NN_LOG(" === StartPeriodicActionFrame Params ===============================\n");
    NN_LOG("   Channel           : %d\n", m_Channel);
    NN_LOG("   Interval          : %u\n", m_TxInterval);
    NN_LOG("   Payload Size      : %u\n", m_PayloadSize);
    NN_LOG("   Hash Mode         : 0x%x (0:Fixed, 1:Increment, 2:Random\n", m_Hash.GetMode());
    NN_LOG("   Hash              : 0x%016x\n", ToString(m_Hash.GetHash()).c_str());
    NN_LOG(" ====================================================\n");

    NN_ASSERT(m_PayloadSize < sizeof(g_txBuffer), "  - failed : size %u, buf %u\n", m_PayloadSize, sizeof(g_txBuffer));
    std::memset(g_txBuffer, 0x00, sizeof(g_txBuffer));
    g_txBuffer[0] = 0x7F; // Category
    g_txBuffer[1] = 0x00; // OUI1
    g_txBuffer[2] = 0x22; // OUI2
    g_txBuffer[3] = 0xAA; // OUI3
    g_txBuffer[4] = nn::wlan::ActionFrameType_Beacon;  // Subtype

    nn::Result result = nn::wlan::Detect::StartPeriodicActionFrame(
        nn::wlan::ActionFrameType_Detect,
        m_Hash.GetHash(), reinterpret_cast<uint8_t*>(g_txBuffer), m_PayloadSize, m_TxInterval);
    NN_ASSERT(result.IsSuccess(), "  - failed : StartPeriodicActionFrame()");
}

void Detector::CancelPeriodicActionFrame()
{
    nn::Result result;

    nn::wlan::WlanState state;
    result = nn::wlan::Detect::GetState(&state);
    NN_ASSERT(result.IsSuccess(), "  - failed : GetState()");

    if( state == nn::wlan::WlanState_Detect )
    {
        result = nn::wlan::Detect::CancelPeriodicActionFrame();
        NN_ASSERT(result.IsSuccess(), "  - failed : CancelPeriodicActionFrame()");
    }
}

nn::Result Detector::CreateRxEntry()
{
    nn::Result result;

    m_Cs.Lock();

    // RxId == 0 の場合に Rx エントリが未作成とみなし、新たに作成する
    if( m_AfRxId == 0 )
    {
        // 不要であるため他と重複しない適当な値
        uint16_t subType = nn::wlan::ActionFrameType_Detect;
        // ##fixme capacity を 15 にすると低層で Connect()/StartScan() ではまることがある
        result = WlanDetectCreateRxEntryForAf(&m_AfRxId, &subType, 1, 8);
        NN_ASSERT(result.IsSuccess());
    }

    m_Cs.Unlock();

    return nn::ResultSuccess();
}

nn::Result Detector::DeleteRxEntry()
{

    if( !m_IsInitialized )
    {
        return WlanTest::ResultNotInitialized();
    }

    m_Cs.Lock();
    NN_ABORT_UNLESS_RESULT_SUCCESS( WlanDetectDeleteRxEntryForAf(&m_AfRxId) );
    m_Cs.Unlock();

    return nn::ResultSuccess();
}

}
