﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DataGenerator.h"
#include "Log.h"

const uint32_t TxBufferSize = 20 * 1024 * 1024;
uint8_t txBuffer[TxBufferSize];

namespace WlanTest {

const char* DataGeneratorParam::ITEM_STR_TYPE            = "TYPE";
const char* DataGeneratorParam::ITEM_STR_PAYLOAD_SIZE    = "PAYLOAD_SIZE";
const char* DataGeneratorParam::ITEM_STR_SEND_INTERVAL   = "SEND_INTERVAL";
const char* DataGeneratorParam::ITEM_STR_NUM_OF_PACKETS  = "NUMBER_OF_FRAMES";

bool ContainsWlanNdHeader(const uint8_t* data, size_t size)
{
    if( size < sizeof(WlanNdHeader) )
    {
        return false;
    }

    return data[0] == 0x7f && data[1] == 0x00 &&
        data[2] == 0x22 && data[3] == 0xaa &&
        data[4] == nn::wlan::ActionFrameType_Detect;
}

bool ContainsWlanAfWithBeacon(const uint8_t* data, size_t size)
{
    if( size < 5 )
    {
        return false;
    }

    return data[0] == 0x7f && data[1] == 0x00 &&
        data[2] == 0x22 && data[3] == 0xaa &&
        data[4] == nn::wlan::ActionFrameType_Beacon;
}

/*---------------------------------------------------------------------------
　　　　　GenerateCommand
---------------------------------------------------------------------------*/
GenerateCommand::GenerateCommand(DataGenerator* pGenerator)
{
    m_pDataGenerator = pGenerator;
}

void GenerateCommand::CommandContent()
{
    m_pDataGenerator->Generate();
}

size_t GenerateCommand::GetInstanceSize() const
{
    return sizeof(GenerateCommand);
}

/*---------------------------------------------------------------------------
　　　　　FixedSizeGenerateCommand
---------------------------------------------------------------------------*/
FixedSizeGenerateCommand::FixedSizeGenerateCommand(FixedSizeDataTransmitter *pTransmitter)
{
    m_pTransmitter = pTransmitter;
}

void FixedSizeGenerateCommand::CommandContent()
{
    if( !m_pTransmitter->IsCompleted() )
    {
        m_pTransmitter->Generate();
    }
}

bool FixedSizeGenerateCommand::NeedReinvoke()
{
    if( m_pTransmitter->IsCompleted() )
    {
        return false;
    }
    return Command::NeedReinvoke();
}

size_t FixedSizeGenerateCommand::GetInstanceSize() const
{
    return sizeof(FixedSizeGenerateCommand);
}

/*---------------------------------------------------------------------------
　　　　　DataGenerator
---------------------------------------------------------------------------*/

DataGenerator::DataGenerator()
{
    m_Node = NULL;
    m_Count = 0;
    m_Invoker.Start(0);
    m_IeIndicator = 0;
}

DataGenerator::~DataGenerator()
{
    m_Invoker.Stop();
}

void DataGenerator::Generate()
{
    if( m_Node == NULL )
    {
        return;
    }

    nn::Result result;
    static nn::os::Tick last;
    nn::os::Tick now;

    // 送信準備
    //m_Frame.SetLength( m_Size + m_Frame.GetHeaderSize() - 4 );
    //m_Frame.SetPayload((uint8_t*)&m_Count, sizeof(uint64_t));

    uint16_t type = m_Frame.GetType();
    uint8_t buf[sizeof(uint64_t) * 2] = { 0 };
    uint32_t offset = 0;

    // シーケンス番号
    std::memcpy(buf, reinterpret_cast<uint8_t*>(&m_Count), sizeof(uint64_t));
    offset += sizeof(uint64_t);

    // インターバルまで待つ
    now = nn::os::GetSystemTick();
    nn::TimeSpan interval = m_Interval - (now - last).ToTimeSpan();
    if( interval >= 0 )
    {
        Sleep(interval);
    }

    // ローカル通信のデータの場合に送信時間を記録する
    if( type == PID_OUI_EXT )
    {
        int64_t tsfTime = 0;
        result = GetTsfTime(&tsfTime);
        if( result.IsSuccess() )
        {
            std::memcpy(buf + offset, reinterpret_cast<uint8_t*>(&tsfTime), sizeof(int64_t));
            offset += sizeof(int64_t);
        }
        else
        {
            NN_LOG(" - failed : GetDeltaTimeBetweenSystemAndTsf()\n");
        }
    }

    m_Frame.SetPayload(buf, sizeof(buf));

    // 送信
    size_t sentSize = 0;
    if( type == PID_PAYLOAD_ONLY )
    {
        // ヘッダーが不要な場合はペイロードのみ送信する
        result = m_Node->Send(m_Frame.GetPayload(), m_Size + m_Frame.GetHeaderSize(), m_IeIndicator, &sentSize);
    }
    else
    {
        result = m_Node->Send(reinterpret_cast<uint8_t*>(&m_Frame), m_Size + m_Frame.GetHeaderSize(), m_IeIndicator, &sentSize);
    }

    last = nn::os::GetSystemTick();
    if( result.IsSuccess() )
    {
        m_Count++;
    }
    else
    {
        //NN_LOG(" - failed : Send()\n");
    }
}

void DataGenerator::Generate(uint32_t num)
{
    GenerateCommand c(this);
    c.SetNumber(num);
    m_Invoker.AddCommand(c);
}

void DataGenerator::Cancel()
{
    m_Invoker.Stop();
    m_Invoker.Clear();
    m_Invoker.Start(0);
}

/*---------------------------------------------------------------------------
　　　　　FixedSizeDataTransmitter
---------------------------------------------------------------------------*/
FixedSizeDataTransmitter::FixedSizeDataTransmitter()
{
    m_DataSize = 0;
    m_SentDataSize = 0;
    std::memset(txBuffer, 0x0, sizeof(txBuffer));
}

FixedSizeDataTransmitter::~FixedSizeDataTransmitter()
{
}

void FixedSizeDataTransmitter::Generate()
{
    if(m_Node == NULL)
    {
        return;
    }

    nn::Result result;
    static nn::os::Tick last;
    nn::os::Tick now;

    // 送信準備
    size_t size = m_Size;
    if( size + m_SentDataSize > m_DataSize )
    {
        size = m_DataSize - m_SentDataSize;
    }

    // インターバルまで待つ
    now = nn::os::GetSystemTick();
    nn::TimeSpan interval = m_Interval - (now - last).ToTimeSpan();

    if(interval >= 0)
    {
        Sleep(interval);
    }

    // 送信
    size_t sentSize = 0;
    result = m_Node->Send(txBuffer, size, m_IeIndicator, &sentSize);

    last = nn::os::GetSystemTick();
    if(result.IsSuccess())
    {
        m_Count++;
        m_SentDataSize += sentSize;
        //NN_LOG(" total size : %u, total sent : %u, sent : %u, actual : %u\n", m_DataSize, m_SentDataSize, size, sentSize);
    }
    else
    {
        //NN_LOG(" - failed : Send()\n");
    }
}

void FixedSizeDataTransmitter::Generate(uint32_t num)
{
    // 規定のデータサイズになったら Generate() で送信終了する
    NN_UNUSED(num);

    NN_LOG(" === Transmitter Option ====================\n");
    NN_LOG("    Total Data Size : %d byte\n", m_DataSize);
    NN_LOG("    Tx Size         : %d byte\n", m_Size);
    NN_LOG("    Tx Interval     : %d ms\n", m_Interval.GetMilliSeconds());
    NN_LOG(" ======================================\n");

    FixedSizeGenerateCommand c(this);
    c.SetNumber(0);
    m_Invoker.AddCommand(c);
}


/*---------------------------------------------------------------------------
　　　　　DataSink
---------------------------------------------------------------------------*/

DataSink::DataSink()
{
    m_Frame = NULL;
}

/*---------------------------------------------------------------------------
　　　　　MultiDataSink
---------------------------------------------------------------------------*/

nn::TimeSpan MultiDataSink::Statistics::UpdateInterval = nn::TimeSpan::FromSeconds(1);
nn::os::Tick MultiDataSink::Statistics::LastUpdateTime = nn::os::Tick(0);

MultiDataSink::MultiDataSink() : m_Cs(false, 0)
{
    //m_Cs.Initialize();
}

MultiDataSink::~MultiDataSink()
{
    Clear();
}

void MultiDataSink::Sink(const EventArgs* pEventArgs)
{
    nn::os::Tick rxTick = nn::os::GetSystemTick();
    const ReceiveEventArgs* pArgs = reinterpret_cast<const ReceiveEventArgs*>(pEventArgs);

    m_Frame = reinterpret_cast<DixFrame*>(pArgs->Data);

    // NN_LOG("== packet (%d bytes) ==\n", args.Size);
    // // パケットの中身を表示する
    // {
    //     std::ostringstream oss;
    //     for(int i=0; i<args.Size; i++)
    //     {
    //         NN_LOG("%02x ", args.Data[i]);
    //         if( i % 8 == 7 )
    //         {
    //             NN_LOG("  ");
    //         }
    //         if( i % 16 == 15 )
    //         {
    //             NN_LOG("\n");
    //         }
    //     }
    //     NN_LOG("%s", oss.str().c_str());
    //     NN_LOG("\n\n");
    // }

    // データフレームの長さチェック
    // 16バイト(LLC/SNAP 8バイト、Payload 8バイト)
    // if(m_Frame->GetLength()<16)
    // {
    //     NN_LOG("Seq is not included.\n");
    //     return;
    // }

    uint64_t source = 0;
    memcpy(&source, m_Frame->GetSourceP(), nn::wlan::MacAddress::MacAddressSize);

    m_Cs.Lock();
    Statistics* s = m_Statistics[source];
    // 受信したことが無い端末からの送信ならば統計を追加
    if( s == nullptr )
    {
        char str[nn::wlan::MacAddress::MacStringSize];
        m_Sources[source] = m_Frame->GetSource().GetString(str);

        s = new Statistics();
        m_Statistics[source] = s;

        s->Throughput.Clear();
        s->WorkThroughput.Clear();
        s->FirstReceiveTime = rxTick;
        s->LastReceiveTime = nn::os::Tick(0);
        m_ReceiverInfos[source] = pArgs->Receiver;
    }
    else if( m_ReceiverInfos[source] != pArgs->Receiver )
    {// 異なる ReceiverInfo での受信ならば、統計を初期化する
        m_ReceiverInfos[source] = pArgs->Receiver;
        s->Clear();
    }

    const uint8_t* data = m_Frame->GetPayload();

    // 受信パケット内のシーケンス番号と送信時間の取り出し
    uint32_t offset = m_Frame->GetHeaderSize();

    // 先頭 1 byte が 0xff なら WitCommand, それ以外が古いバージョン
    bool isOldDataPacket = (data[offset] != 0xff);

    uint64_t seqno = 0;
    memcpy(&seqno, m_Frame->GetPayload() + offset, sizeof(uint64_t));

    int64_t delta = 0;
    int64_t txTime = 0;
    int64_t rxTime = 0;
    if( isOldDataPacket )
    {
        memcpy(&txTime, m_Frame->GetPayload() + offset + sizeof(uint64_t), sizeof(int64_t));

        nn::Result result = nn::wlan::Socket::GetDeltaTimeBetweenSystemAndTsf(&delta);
        if( result.IsFailure() )
        {
            NN_LOG(" - failed : GetDeltaTimeBetweenSystemAndTsf()\n");
        }
        rxTime = rxTick.ToTimeSpan().GetMicroSeconds() - delta;

        // us -> ms
        delta = (rxTime - txTime) / 1000;
        //NN_LOG("@@@@@%lld %lld %lld %lld %lld\n", rxTime, txTime, (rxTime - txTime), delta, seqno);
    }

    // 統計情報の更新
    s->Errors.Count(seqno);
    s->WorkErrors.Count(seqno);
    s->LastSequenceNo = seqno;
    s->Throughput.Count(pArgs->Size);
    s->WorkThroughput.Count(pArgs->Size);

    // RSSI が 0 以上の場合は不正な値とみなしてカウントしない (SIGLO-84435)
    auto rssi = pArgs->Rssi;
    if( rssi < 0 )
    {
        s->Rssi.Count(rssi);
    }

    if( s->LastReceiveTime != nn::os::Tick(0) )
    {
        uint64_t interval = (rxTick - s->LastReceiveTime).ToTimeSpan().GetMilliSeconds();
        s->RxInterval.Count(interval);
        s->WorkRxInterval.Count(interval);
    }

    // wlan FW -> wlan プロセスに TSF タイマ同期イベントが上がるのが 1 秒間隔のため、初期化されることを保障するため
    // 受信から 1500ms してから集計し始める
    bool isLatencyCounted = false;
    if( isOldDataPacket && (nn::os::GetSystemTick() - s->FirstReceiveTime).ToTimeSpan().GetMilliSeconds() >= 1500 )
    {
        isLatencyCounted = true;
        s->Latency.Count(delta);
        s->WorkLatency.Count(delta);
        s->ErrorsForLatency.Count(seqno);
        s->WorkErrorsForLatency.Count(seqno);
    }

    PacketLogger& logger = PacketLogger::GetInstance();
    if( logger.IsEnabled() )
    {
        static char buf[150];
        // source addr, size(byte), seqno, rx time(us), tx time(us), delta(us), latencyCounted?
        // (17+1)     + (4+1)     + (8+1)+ (12+1)     + (12+1)     + (12+1)   + (1)             = 72
        snprintf(buf, sizeof(buf), "%02x:%02x:%02x:%02x:%02x:%02x,%llu,%llu,%lld,%lld,%lld,%d\n",
                 m_Frame->GetSourceP()[0],
                 m_Frame->GetSourceP()[1],
                 m_Frame->GetSourceP()[2],
                 m_Frame->GetSourceP()[3],
                 m_Frame->GetSourceP()[4],
                 m_Frame->GetSourceP()[5],
                 static_cast<uint64_t>(pArgs->Size),
                 seqno,
                 rxTime,
                 txTime,
                 (rxTime - txTime),
                 isLatencyCounted);
        logger.Write(buf);

    }

    s->LastReceiveTime = rxTick;

    m_Cs.Unlock();
}

void MultiDataSink::Clear()
{
    m_Cs.Lock();

    for(auto stats : m_Statistics)
    {
        delete (stats.second);
    }

    m_Sources.clear();
    m_Statistics.clear();
    m_ReceiverInfos.clear();

    m_Cs.Unlock();
}

void MultiDataSink::ShowBurstStatistics()
{
    map<uint64_t, Statistics*>::iterator p = m_Statistics.begin();

    NN_LOG("|Brst| Cnt  |\n");
    while(p != m_Statistics.end())
    {
        for(int i=1; i<16; i++)
        {
            NN_LOG("| %2d | %4d |\n" , i, p->second->Errors.GetBurstErrorCount(i));
        }
        NN_LOG("| 16+| %4d | max:%4d\n" , p->second->Errors.GetBurstErrorCount(16), p->second->Errors.GetBurstErrorCount(0));
        p++;
    }
}

void MultiDataSink::UpdateLastStatistics()
{
    nn::os::Tick tick = nn::os::GetSystemTick();

    m_Cs.Lock();

    if( (tick - Statistics::LastUpdateTime).ToTimeSpan() >= Statistics::UpdateInterval )
    {
        for(auto& stats : GetStatistics())
        {
            Statistics* s = stats.second;
            if( s->LastReceiveTime != nn::os::Tick(0) )
            {
                s->LastErrors = s->WorkErrors;
                s->LastThroughput = s->WorkThroughput;
                s->LastRxInterval = s->WorkRxInterval;
                s->LastLatency = s->WorkLatency;
                s->LastErrorsForLatency = s->WorkErrorsForLatency;

                s->WorkErrors.Clear();
                s->WorkThroughput.Clear();
                s->WorkRxInterval.Clear();
                s->WorkLatency.Clear();
                s->WorkErrorsForLatency.Clear();
            }
        }
        Statistics::LastUpdateTime = tick;
    }

    m_Cs.Unlock();
}



/*---------------------------------------------------------------------------
　　　　　MultiAfSink
---------------------------------------------------------------------------*/

nn::TimeSpan MultiAfSink::Statistics::UpdateInterval = nn::TimeSpan::FromSeconds(1);
nn::os::Tick MultiAfSink::Statistics::LastUpdateTime = nn::os::Tick(0);

MultiAfSink::MultiAfSink() : m_Cs(false, 0)
{
    //m_Cs.Initialize();
}

MultiAfSink::~MultiAfSink()
{
    Clear();
}

void MultiAfSink::Sink(const EventArgs* pEventArgs)
{
    nn::os::Tick rxTick = nn::os::GetSystemTick();

    const AfReceiveEventArgs& args = *reinterpret_cast<const AfReceiveEventArgs*>(pEventArgs);
    uint64_t source = ToUint(args.Source);
    WitHeader header;
    ActionFrameSearchInfoCommand command;
    const uint8_t* data = nullptr;
    bool isValid = false;

    if( args.Size >= sizeof(WitHeader) + sizeof(command) )
    {
        data = args.Data;

        uint16_t offset = WlanActionFrameHeaderSize;
        auto size = sizeof(WitHeader);
        std::memcpy(&header, data + offset, size);
        offset += size;

        // 下記の条件にあえば WIT コマンドと判断して統計情報を処理する
        if( header.dummy == 0xffff &&
            header.command == WitCommand_ActionFrameSearchInfo &&
            header.length == sizeof(command) )
        {
            size = sizeof(command);
            std::memcpy(&command, data + offset, size);
            offset += size;
            isValid = true;
        }
    }

    // 想定外のパケットはチャネルを 0 とする
    auto channel = command.channel;
    auto seqno = command.sequenceNumber;
    if( !isValid )
    {
        channel = 0;
        seqno = 0;
    }

    m_Cs.Lock();
    Statistics* s = m_Statistics[source][channel];
    // 受信したことが無い端末からの送信ならば統計を追加
    if( s == nullptr )
    {
        char str[nn::wlan::MacAddress::MacStringSize];
        m_Sources[source] = ToString(args.Source);

        s = new Statistics();
        m_Statistics[source][channel] = s;

        s->Throughput.Clear();
        s->WorkThroughput.Clear();
        s->FirstReceiveTime = rxTick;
        s->LastReceiveTime = nn::os::Tick(0);
        m_ReceiverInfos[source] = args.Receiver;
    }
    else if( m_ReceiverInfos[source] != args.Receiver )
    {// 異なる ReceiverInfo での受信ならば、統計を初期化する
        m_ReceiverInfos[source] = args.Receiver;
        s->Clear();
    }

    int64_t delta = 0;
    int64_t txTime = 0;
    int64_t rxTime = 0;
    if( isValid )
    {
        txTime = command.tsfTime;
        nn::Result result = GetTsfTime(&rxTime);
        if( result.IsFailure() )
        {
            NN_LOG(" - failed : GetDeltaTimeBetweenSystemAndTsf()\n");
        }

        // us -> ms
        delta = (rxTime - txTime) / 1000;
    }

    // 統計情報の更新
    s->Errors.Count(seqno);
    s->WorkErrors.Count(seqno);
    s->LastSequenceNo = seqno;
    s->Throughput.Count(args.Size);
    s->WorkThroughput.Count(args.Size);

    if( s->LastReceiveTime != nn::os::Tick(0) )
    {
        uint64_t interval = (rxTick - s->LastReceiveTime).ToTimeSpan().GetMilliSeconds();
        s->RxInterval.Count(interval);
        s->WorkRxInterval.Count(interval);
    }

    // wlan FW -> wlan プロセスに TSF タイマ同期イベントが上がるのが 1 秒間隔のため、初期化されることを保障するため
    // 受信から 1500ms してから集計し始める
    bool isLatencyCounted = false;
    if( isValid && (nn::os::GetSystemTick() - s->FirstReceiveTime).ToTimeSpan().GetMilliSeconds() >= 1500 )
    {
        isLatencyCounted = true;
        s->Latency.Count(delta);
        s->WorkLatency.Count(delta);
    }

    s->LastReceiveTime = rxTick;

    m_Cs.Unlock();
}

void MultiAfSink::Clear()
{
    m_Cs.Lock();

    for(auto chStats : m_Statistics)
    {
        for(auto stats : chStats.second)
        {
            delete (stats.second);
        }
    }

    m_Sources.clear();
    m_Statistics.clear();
    m_ReceiverInfos.clear();

    m_Cs.Unlock();
}

void MultiAfSink::UpdateLastStatistics()
{
    nn::os::Tick tick = nn::os::GetSystemTick();

    m_Cs.Lock();

    if( (tick - Statistics::LastUpdateTime).ToTimeSpan() >= Statistics::UpdateInterval )
    {
        for(auto& chStats : GetStatistics())
        {
            for(auto& stats : chStats.second)
            {
                Statistics* s = stats.second;
                if( s->LastReceiveTime != nn::os::Tick(0) )
                {
                    s->LastErrors = s->WorkErrors;
                    s->LastThroughput = s->WorkThroughput;
                    s->LastRxInterval = s->WorkRxInterval;
                    s->LastLatency = s->WorkLatency;

                    s->WorkErrors.Clear();
                    s->WorkThroughput.Clear();
                    s->WorkRxInterval.Clear();
                    s->WorkLatency.Clear();
                }
            }
        }
        Statistics::LastUpdateTime = tick;
    }

    m_Cs.Unlock();
}

/*---------------------------------------------------------------------------
　　　　　MultiNdhpSink
---------------------------------------------------------------------------*/

nn::TimeSpan MultiNdhpSink::Statistics::UpdateInterval = nn::TimeSpan::FromSeconds(1);
nn::os::Tick MultiNdhpSink::Statistics::LastUpdateTime = nn::os::Tick(0);

MultiNdhpSink::MultiNdhpSink() : m_Cs(false, 0)
{
    //m_Cs.Initialize();
}

MultiNdhpSink::~MultiNdhpSink()
{
    Clear();
}

void MultiNdhpSink::Sink(const EventArgs* pEventArgs)
{
    nn::os::Tick rxTick = nn::os::GetSystemTick();

    const AfReceiveEventArgs& args = *reinterpret_cast<const AfReceiveEventArgs*>(pEventArgs);
    uint64_t source = ToUint(args.Source);
    const uint8_t* data = args.Data;
    uint16_t dataSize = args.Size;

    uint64_t hash = 0;
    if( ContainsWlanNdHeader(data, dataSize) )
    {
        nn::wlan::DetectHash ndHash;
        std::memcpy(ndHash.hash, data + WlanNdhpHeaderSize - sizeof(hash), sizeof(hash));
        data = args.Data + WlanNdhpHeaderSize;
        dataSize -= WlanNdhpHeaderSize;
        hash = ToUint(ndHash);
    }
    else
    {
        // 想定外のパケットは無視
        return;
    }

    WitHeader header;
    ActionFrameSearchInfoCommand command;
    bool isValid = false;
    if( dataSize >= sizeof(WitHeader) + sizeof(command) )
    {
        auto headerSize = sizeof(WitHeader);
        std::memcpy(&header, data, headerSize);
        auto offset = headerSize;

        // 下記の条件にあえば WIT コマンドと判断して統計情報を処理する
        if( header.dummy == 0xffff &&
            header.command == WitCommand_ActionFrameSearchInfo &&
            header.length == sizeof(command) )
        {
            auto commandSize = sizeof(command);
            std::memcpy(&command, data + offset, commandSize);
            offset += commandSize;
            isValid = true;
        }
    }

    auto channel = 0;
    auto seqno = 0;
    if( isValid )
    {
        channel = command.channel;
        seqno = command.sequenceNumber;
    }

    m_Cs.Lock();
    Statistics* s = m_Statistics[source][hash];
    // 受信したことが無い端末からの送信ならば統計を追加
    if( s == nullptr )
    {
        char str[nn::wlan::MacAddress::MacStringSize];
        m_Sources[source] = ToString(args.Source);

        s = new Statistics();
        s->Clear();
        m_Statistics[source][hash] = s;

        s->FirstReceiveTime = rxTick;
        s->LastReceiveTime = nn::os::Tick(0);
        m_ReceiverInfos[source] = args.Receiver;
    }
    else if( m_ReceiverInfos[source] != args.Receiver )
    {// 異なる ReceiverInfo での受信ならば、統計を初期化する
        m_ReceiverInfos[source] = args.Receiver;
        s->Clear();
    }

    int64_t delta = 0;
    int64_t txTime = 0;
    int64_t rxTime = 0;
    if( isValid )
    {
        txTime = command.tsfTime;
        nn::Result result = GetTsfTime(&rxTime);
        if( result.IsFailure() )
        {
            NN_LOG(" - failed : GetDeltaTimeBetweenSystemAndTsf()\n");
        }

        // us -> ms
        delta = (rxTime - txTime) / 1000;
    }

    // 統計情報の更新
    s->IsWitFormat = isValid;
    s->Errors.Count(seqno);
    s->WorkErrors.Count(seqno);
    s->LastSequenceNo = seqno;
    s->Throughput.Count(args.Size);
    s->WorkThroughput.Count(args.Size);

    // RSSI が 0 以上の場合は不正な値とみなしてカウントしない (SIGLO-84435)
    auto rssi = args.Rssi;
    if( rssi < 0 )
    {
        s->Rssi.Count(rssi);
        s->WorkRssi.Count(rssi);
    }

    if( s->LastReceiveTime != nn::os::Tick(0) )
    {
        uint64_t interval = (rxTick - s->LastReceiveTime).ToTimeSpan().GetMilliSeconds();
        s->RxInterval.Count(interval);
        s->WorkRxInterval.Count(interval);
    }

    // wlan FW -> wlan プロセスに TSF タイマ同期イベントが上がるのが 1 秒間隔のため、初期化されることを保障するため
    // 受信から 1500ms してから集計し始める
    bool isLatencyCounted = false;
    if( isValid && (nn::os::GetSystemTick() - s->FirstReceiveTime).ToTimeSpan().GetMilliSeconds() >= 1500 )
    {
        isLatencyCounted = true;
        s->Latency.Count(delta);
        s->WorkLatency.Count(delta);
    }

    s->LastReceiveTime = rxTick;

    m_Cs.Unlock();
} //NOLINT(impl/function_size)

void MultiNdhpSink::Clear()
{
    m_Cs.Lock();

    for(auto chStats : m_Statistics)
    {
        for(auto stats : chStats.second)
        {
            delete (stats.second);
        }
    }

    m_Sources.clear();
    m_Statistics.clear();
    m_ReceiverInfos.clear();

    m_Cs.Unlock();
}

void MultiNdhpSink::UpdateLastStatistics()
{
    nn::os::Tick tick = nn::os::GetSystemTick();

    m_Cs.Lock();

    if( (tick - Statistics::LastUpdateTime).ToTimeSpan() >= Statistics::UpdateInterval )
    {
        for(auto& chStats : GetStatistics())
        {
            for(auto& stats : chStats.second)
            {
                Statistics* s = stats.second;
                if( s->LastReceiveTime != nn::os::Tick(0) )
                {
                    s->LastErrors = s->WorkErrors;
                    s->LastThroughput = s->WorkThroughput;
                    s->LastRxInterval = s->WorkRxInterval;
                    s->LastLatency = s->WorkLatency;
                    s->LastRssi = s->WorkRssi;

                    s->WorkErrors.Clear();
                    s->WorkThroughput.Clear();
                    s->WorkRxInterval.Clear();
                    s->WorkLatency.Clear();
                    s->WorkRssi.Clear();
                }
            }
        }
        Statistics::LastUpdateTime = tick;
    }

    m_Cs.Unlock();
}

}
