﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifdef BT_STAT_SD_OUT
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <iomanip>
#include <sstream>

#include "WlanCommon.h"
#include "BtStatSdOut.h"
#include "Pad.h"

namespace
{

// For monitor threads
nn::os::ThreadType              g_BtRfMonitorThread;
NN_OS_ALIGNAS_THREAD_STACK char g_BtRfMonitorThreadStack[4096 * 2];
nn::os::EventType               g_BtRfMonitorFinalizeEvent;

// For SD card output
const size_t MemoryHeapSize = 2 * 1024 * 1024;
const char* RfStatSaveFile = "sd:/bluetoothDebug/WitRfStatistics.csv";

nn::Result SaveBtRfStat()
{
    nn::fs::FileHandle fileHandle;
    auto result = nn::fs::OpenFile(&fileHandle, RfStatSaveFile, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);

    if(nn::fs::ResultPathNotFound::Includes(result))
    {
        NN_LOG("File not found\n");
        return result;
    }

    nn::time::PosixTime posixTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime));
    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime) );

    nn::bluetooth::PlrStatistics    plr;
    nn::bluetooth::ChannelMap       chMap;
    nn::bluetooth::BluetoothAddress invalidBdAddr = {{0}};

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bluetooth::GetLatestPlr(&plr));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bluetooth::GetChannelMap(&chMap));

    for(int index = 0; index < nn::bluetooth::RfListSizeMax; ++index)
    {
        if(memcmp(invalidBdAddr.address, plr.plrList[index].bluetoothAddress.address, sizeof(invalidBdAddr.address)) == 0)
        {
            continue;
        }
        if(memcmp(plr.plrList[index].bluetoothAddress.address, chMap.chList[index].bluetoothAddress.address, sizeof(invalidBdAddr.address)) != 0)
        {
            NN_LOG("Bluetooth address does not match\n");
            for(int i = 0; i < sizeof(invalidBdAddr.address); ++i)
            {
                NN_LOG("0x%02X ", plr.plrList[index].bluetoothAddress.address[i]);
            }
            NN_LOG("\n");
            for(int i = 0; i < sizeof(invalidBdAddr.address); ++i)
            {
                NN_LOG("0x%02X ", chMap.chList[index].bluetoothAddress.address[i]);
            }
            NN_LOG("\n");
            continue;
        }

        std::ostringstream calenderOss;
        calenderOss << std::setw(2) << std::setfill('0') << std::right
                    << (uint32_t)calendarTime.year << "/" << (uint32_t)calendarTime.month << "/" << (uint32_t)calendarTime.day << " "
                    << (uint32_t)calendarTime.hour << ":" << (uint32_t)calendarTime.minute << ":" << (uint32_t)calendarTime.second << " "
                    << "(" << calendarAdditionalInfo.timeZone.standardTimeName << ")";

        std::ostringstream bdAddrOss;
        for(int j = 0; j < 6; ++j)
        {
            bdAddrOss << std::setw(2) << std::hex << std::uppercase << std::right
                      << (uint32_t)plr.plrList[index].bluetoothAddress.address[j];

            if(j != 5)
            {
                bdAddrOss << ":";
            }
        }

        std::ostringstream plrOss;
        plrOss << (uint32_t)plr.plrList[index].received << "," << (uint32_t)plr.plrList[index].plr;

        std::ostringstream channelMapOss;
        for(int j = nn::bluetooth::ARRAY_SIZE_OF_CHANNEL_INFO - 1; j >= 0 ; --j)
        {
            channelMapOss << std::hex << std::uppercase << std::setw(2) << std::setfill('0') << std::right
                          << (uint32_t)chMap.chList[index].channelInfo.channels[j];
        }

        std::ostringstream oss;
        oss << calenderOss.str() << "," << bdAddrOss.str() << "," << plrOss.str() << ","
            << channelMapOss.str() << "," << (uint32_t)chMap.chList[index].channelInfo.count << "\n";

        int64_t fileSize = 0;
        nn::fs::GetFileSize(&fileSize, fileHandle);

        char writeBuf[100];
        int length = nn::util::SNPrintf(writeBuf, sizeof(writeBuf), oss.str().c_str());
        result = nn::fs::WriteFile(fileHandle, fileSize, writeBuf, length, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        if(nn::fs::ResultUsableSpaceNotEnough::Includes(result))
        {
            NN_LOG("Usable space in SD not found.\n");
            return result;
        }
    }

    nn::fs::CloseFile(fileHandle);

    return result;
}

static void BtRfMonitorThreadFunc(void* arg)
{
    NN_UNUSED(arg);

    while(1)
    {
        if (nn::os::TimedWaitEvent(&g_BtRfMonitorFinalizeEvent, nn::TimeSpan::FromMilliSeconds(WlanTest::BluetoothPlrStatistics::UPDATE_INTERVAL_MS)))
        {
            break;
        }

        auto result = SaveBtRfStat();

        if(!result.IsSuccess())
        {
            NN_LOG("Error!! RF Monitor stop working.\n");
            break;
        }
    }

    nn::fs::Unmount("sd");
}

nn::Result InitializeRfMonitor()
{
    auto result = nn::fs::MountSdCardForDebug("sd");

    if(nn::fs::ResultSdCardAccessFailed::Includes(result))
    {
        NN_LOG("SD card not found.\n");
        return result;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // Craete directory
    result = nn::fs::CreateDirectory("sd:/bluetoothDebug");

    if(nn::fs::ResultUsableSpaceNotEnough::Includes(result))
    {
        NN_LOG("Usable space in SD not found.\n");
        return result;
    }

    nn::fs::DirectoryEntryType directoryEntryType;
    result = nn::fs::GetEntryType(&directoryEntryType, RfStatSaveFile);
    // File does not exist, then create new one
    if(nn::fs::ResultPathNotFound().Includes(result))
    {
        result = nn::fs::CreateFile(RfStatSaveFile, 0);

        if(nn::fs::ResultUsableSpaceNotEnough::Includes(result))
        {
            NN_ASSERT("Usable space in SD not found.");
            return result;
        }
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());

    return result;
}

}       // namespace

void startBtRfMonitor()
{
    nn::Result result = InitializeRfMonitor();

    if(!result.IsSuccess())
    {
        NN_LOG("Failed to initialize RF monitor.\n");
        return;
    }

    nn::os::InitializeEvent(&g_BtRfMonitorFinalizeEvent, false, nn::os::EventClearMode_AutoClear);
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&g_BtRfMonitorThread, BtRfMonitorThreadFunc, NULL, g_BtRfMonitorThreadStack, sizeof(g_BtRfMonitorThreadStack), nn::os::DefaultThreadPriority - 1)
    );
    nn::os::StartThread(&g_BtRfMonitorThread);
}

void stopBtRfMonitor()
{
    nn::os::SignalEvent(&g_BtRfMonitorFinalizeEvent);
    nn::os::WaitThread(&g_BtRfMonitorThread);
    nn::os::DestroyThread(&g_BtRfMonitorThread);
    nn::os::FinalizeEvent(&g_BtRfMonitorFinalizeEvent);
}

#endif      // #ifdef BT_STAT_SD_OUT
