﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <sstream>
#include <iostream>
#include <iomanip>
#include <string>
#include <vector>

#include "Util.h"

namespace WlanTest {

using namespace std;

/*!--------------------------------------------------------------------------*
  @brief        選択可能値のインタフェース
  *---------------------------------------------------------------------------*/
class ISelectableValue
{
public:

    virtual void Next(bool loop=false) = 0;
    virtual void Back(bool loop=false) = 0;

    virtual bool Select(const int& index) = 0;
    virtual bool Deselect(const int& index) = 0;

    // 選択可能なアイテム数
    virtual size_t GetSelectableItemCount() const = 0;
    // 選択済みアイテム数
    virtual size_t GetSelectingItemCount() const = 0;
    virtual size_t GetSelectingItemCountMax() const = 0;
    virtual size_t GetCurrentIndex() const = 0;

    virtual string ToStringSelectableItem() = 0;
    virtual string ToStringSelectingItem() = 0;
};

inline ostream& operator<<(ostream& stream, ISelectableValue* svalue)
{
    stream << svalue->ToStringSelectableItem();
    return stream;
}


/*!--------------------------------------------------------------------------*
  @brief        汎用な選択可能値
  *---------------------------------------------------------------------------*/
template<typename T>
class SelectableValue : public ISelectableValue
{
public:

    SelectableValue()
    {
        currentIndex = 0;
    }

    virtual ~SelectableValue()
    {
        Clear();
    }

    // 選択可能なアイテムから検索
    virtual int FindInSelectableItems(const T& value)
    {
        return Find(value, values);
    }

    // 選択済みアイテムから検索
    virtual int FindInSelectingItems(const T& value)
    {
        return -1;
    }

    virtual bool Select(const int& index)
    {
        if( index < 0 || index >= GetSelectableItemCount() )
        {
            return false;
        }

        currentIndex = index;
        return true;
    }

    virtual bool Deselect(const int& index)
    {
        // 常に一つしか値を保持しないため選択しないことは不可
        return false;
    }

    virtual size_t GetSelectableItemCount() const
    {
        return values.size();
    }

    virtual size_t GetSelectingItemCount() const
    {
        return 1;
    }

    virtual size_t GetSelectingItemCountMax() const
    {
        return 1;
    }

    virtual size_t GetCurrentIndex() const
    {
        return currentIndex;
    }

    virtual void Add(T value, bool isDefault = false)
    {
        if(isDefault)
        {
            currentIndex = values.size();
        }

        values.push_back(value);
    }

    virtual void Clear()
    {
        currentIndex = 0;
        values.clear();
    }

    virtual T GetValue() const
    {
        NN_ASSERT( values.size() != 0 )
        return values[currentIndex];
    }

    virtual bool GetValue(T& v) const
    {
        if( values.size() == 0 )
        {
            return false;
        }

        v = values[currentIndex];
        return true;
    }

    virtual void Next(bool loop=false)
    {
        currentIndex++;

        if(currentIndex == values.size())
        {
            if(loop)
            {
                currentIndex = 0;
            }
            else
            {
                currentIndex--;
            }
        }
    }

    virtual void Back(bool loop=false)
    {
        currentIndex--;

        if(currentIndex < 0)
        {
            if(loop)
            {
                currentIndex = values.size() - 1;
            }
            else
            {
                currentIndex++;
            }
        }
    }

    virtual string ToStringSelectableItem()
    {
        ostringstream oss;
        oss << values[currentIndex];
        return oss.str();
    }

    virtual string ToStringSelectingItem()
    {
        return ToStringSelectableItem();
    }

    size_t Size()
    {
        return values.size();
    }

    // 暗黙の型変換演算子をオーバーライド
    NN_IMPLICIT operator T() const
    {
        return GetValue();
    }

protected:

    vector<T> values;
    int currentIndex;

    int Find(const T& value, const vector<T>& list)
    {
        for(int i=0; i<list.size(); ++i)
        {
            if( list[i] == value )
            {
                return i;
            }
        }

        return -1;
    }

};


/*!--------------------------------------------------------------------------*
  @brief        実行可能なインターフェース
  *---------------------------------------------------------------------------*/
class IExecutable
{
public:

    virtual ~IExecutable(){}
    virtual int Execute() = 0;
    virtual int Cancel() = 0;

};


/*!--------------------------------------------------------------------------*
  @brief        リセット可能なインターフェース
  *---------------------------------------------------------------------------*/
class IResettable
{
public:

    virtual ~IResettable(){}
    virtual int Reset() = 0;

};


/*!--------------------------------------------------------------------------*
  @brief        設定コマンドのインターフェース
  *---------------------------------------------------------------------------*/
class IConfigCommand : public IExecutable, public IResettable
{
};


/*!--------------------------------------------------------------------------*
  @brief        汎用な選択可能コマンドインターフェース
  *---------------------------------------------------------------------------*/
class ISelectableConfigCommand : public ISelectableValue, public IConfigCommand
{
};

/*!--------------------------------------------------------------------------*
  @brief        汎用な選択可能コマンド
  *---------------------------------------------------------------------------*/
template<typename T>
class SelectableConfigCommand : public ISelectableConfigCommand
{

public:

    SelectableConfigCommand()
    {
    }

    virtual ~SelectableConfigCommand()
    {
        m_Value.Clear();
    }

    // 選択可能なアイテムから検索
    virtual int FindInSelectableItems(const T& value)
    {
        return m_Value.FindInSelectableItems(value);
    }

    // 選択済みアイテムから検索
    virtual int FindInSelectingItems(const T& value)
    {
        return -1;
    }

    virtual bool Select(const int& index)
    {
        return m_Value.Select(index);
    }

    virtual bool Deselect(const int& index)
    {
        // 常に一つしか値を保持しないため選択しないことは不可
        return false;
    }

    virtual size_t GetSelectableItemCount() const
    {
        return m_Value.GetSelectableItemCount();
    }

    virtual size_t GetSelectingItemCount() const
    {
        return 1;
    }

    virtual size_t GetSelectingItemCountMax() const
    {
        return 1;
    }

    virtual size_t GetCurrentIndex() const
    {
        return m_Value.GetCurrentIndex();
    }

    virtual void Add(T value, bool isDefault = false)
    {
        m_Value.Add(value, isDefault);
    }

    virtual void Clear()
    {
        m_Value.Clear();
    }

    virtual T GetValue() const
    {
        return m_Value.GetValue();
    }

    virtual bool GetValue(T& v) const
    {
        return m_Value.GetValue(v);
    }

    virtual void Next(bool loop=false)
    {
        m_Value.Next(loop);
    }

    virtual void Back(bool loop=false)
    {
        m_Value.Back(loop);
    }

    virtual string ToStringSelectableItem()
    {
        return m_Value.ToStringSelectableItem();
    }

    virtual string ToStringSelectingItem()
    {
        return m_Value.ToStringSelectingItem();
    }

    size_t Size()
    {
        return m_Value.Size();
    }

    // 暗黙の型変換演算子をオーバーライド
    NN_IMPLICIT operator T() const
    {
        return m_Value.GetValue();
    }

    virtual int Execute() = 0;

protected:

    SelectableValue<T> m_Value;

};

/*!--------------------------------------------------------------------------*
  @brief        複数選択可能値
  *---------------------------------------------------------------------------*/
template<typename T>
class MultiSelectableValue : public SelectableValue<T>
{
public:

    MultiSelectableValue()
    {
        itemCountMax = 1;
    }

    virtual ~MultiSelectableValue()
    {
        Clear();
    }

    virtual int FindInSelectableItems(const T& value)
    {
        return SelectableValue<T>::FindInSelectableItems(value);
    }

    virtual int FindInSelectingItems(const T& value)
    {
        return Find(value, selectingItemList);
    }

    virtual bool Select(const int& index)
    {
        if( index < 0 || index >= GetSelectableItemCount() )
        {
            return false;
        }

        const T& value = SelectableValue<T>::values[index];
        // すでに選択しているか、選択上限に達していたら何もしない
        if( FindInSelectingItems(value) >= 0 || itemCountMax <= GetSelectingItemCount() )
        {
            return false;
        }

        selectingItemList.push_back(value);
        return true;
    }

    virtual bool Deselect(const int& index)
    {
        if( index < 0 || index >= GetSelectableItemCount() )
        {
            return false;
        }

        // 選択していないか、選択下限の場合は何もしない
        const T& value = SelectableValue<T>::values[index];
        if( FindInSelectingItems(value) >= 0 || GetSelectingItemCount() <= 1 )
        {
            return false;
        }

        selectingItemList.erase(selectingItemList.begin() + index);
        return true;
    }

    virtual void Clear()
    {
        selectingItemList.clear();
        SelectableValue<T>::Clear();
    }

    virtual const vector<T>* GetSelectingItems() const
    {
        return &selectingItemList;
    }

    virtual size_t GetSelectableItemCount() const
    {
        return SelectableValue<T>::GetSelectableItemCount();
    }

    virtual size_t GetSelectingItemCount() const
    {
        return selectingItemList.size();
    }

    virtual size_t GetSelectingItemCountMax() const
    {
        return itemCountMax;
    }

    virtual void SetSelectableItemCountMax(size_t countMax)
    {
        itemCountMax = countMax;
    }

    virtual string ToStringSelectingItem()
    {
        ostringstream oss;

        for(int i=0; i<selectingItemList.size(); ++i)
        {
            if( i != 0 )
            {
                oss << ", ";
            }

            oss << selectingItemList[i];
        }
        return oss.str();
    }

    // 暗黙の型変換演算子は使用しない
    NN_IMPLICIT operator T();

private:

    uint8_t itemCountMax;
    vector<T> selectingItemList;

};

/*!--------------------------------------------------------------------------*
  @brief        OUIクラス
  *---------------------------------------------------------------------------*/
class Oui
{
private:
    uint8_t* m_Data;
    string m_Separator;

public:
    Oui()
    {
        m_Data = new uint8_t[GetLength()];
        for(int i=0; i<GetLength(); i++)
            m_Data[i] = 0;
        m_Separator = ":";
    }

    NN_IMPLICIT Oui(uint8_t* oui)
    {
        m_Data = new uint8_t[GetLength()];
        memcpy(m_Data, oui, GetLength());
        m_Separator = ":";
    }

    Oui(const Oui &obj)
    {
        m_Data = new uint8_t[GetLength()];
        memcpy(m_Data, obj.m_Data, GetLength());
        m_Separator = obj.m_Separator;
    }

    ~Oui()
    {
        delete [] m_Data;
    }

    uint8_t operator[](int i)
    {
        return m_Data[i];
    }

    Oui& operator=(const Oui& oui)
    {
        SetRawData(oui.m_Data);
        SetSeparator(oui.m_Separator);

        return *this;
    }

    bool operator==(const Oui& oui) const
    {
        if( GetLength() != oui.GetLength() )
        {
            return false;
        }

        for(int i=0; i<GetLength(); ++i)
        {
            if( m_Data[i] != oui.m_Data[i] )
            {
                return false;
            }
        }

        return true;
    }

    bool operator!=(const Oui& oui) const
    {
        return !(*this == oui);
    }

    friend ostream& operator<<(ostream& stream, Oui& value);

    string ToString()
    {
        stringstream ss;
        ss << this;
        return ss.str();
    }

    static uint8_t GetLength()
    {
        return 3;
    }

    void SetSeparator(string c)
    {
        m_Separator = c;
    }


    uint8_t* GetRawData()
    {
        return m_Data;
    }


    void SetRawData(uint8_t* data)
    {
        memcpy(m_Data, data, GetLength());
    }


    void SetRawDataFromU32(uint32_t data)
    {
        m_Data[0] = (uint8_t)(data >> (8 * 2));
        m_Data[1] = (uint8_t)(data >> (8 * 1));
        m_Data[2] = (uint8_t)(data >> (8 * 0));
    }


};

inline ostream& operator<<(ostream& stream, Oui& value)
{
    stream << hex;
    for(int i=0; i<value.GetLength(); i++)
    {
        if(i != 0)
        {
            stream << value.m_Separator;
        }
        stream << setw(2) << setfill('0') << (int)(value[i]);
    }
    stream << dec;
    return stream;
}


/*!--------------------------------------------------------------------------*
  @brief        名前つき値
  *---------------------------------------------------------------------------*/
template<typename T>
class NamedValue
{
public:

    string Name;
    T Value;

    NamedValue(string name, T value)
    {
        Name = name;
        Value = value;
    }

    NamedValue(const NamedValue<T>& nv)
    {
        Name = nv.Name;
        Value = nv.Value;
    }

    virtual ~NamedValue()
    {
    }

    virtual NamedValue operator=(const NamedValue<T>& nv)
    {
        Name = nv.Name;
        Value = nv.Value;

        return *this;
    }

    NN_IMPLICIT virtual operator T()
    {
        return Value;
    }
};

template<typename T>
ostream& operator<<(ostream& stream, NamedValue<T>& nv)
{
    stream << nv.Name;
    return stream;
}



/*!--------------------------------------------------------------------------*
  @brief        NamedValue<T>専用の選択可能値
  　　　　　必要なのはoperator Tのオーバーロードのみ。
  　　 ただ、コンパイラがテンプレートの部分特殊化に未対応なので、
  全体を特殊化している。

  *---------------------------------------------------------------------------*/
template<typename T>
class SelectableValue<NamedValue<T> > : public ISelectableValue
{
public:

    SelectableValue()
    {
        currentIndex = 0;
    }

    virtual ~SelectableValue()
    {
        Clear();
    }

    // 選択可能なアイテムから検索
    virtual int FindInSelectableItems(const T& value)
    {
        return Find(value, values);
    }

    // 選択済みアイテムから検索
    virtual int FindInSelectingItems(const T& value)
    {
        return -1;
    }

    virtual size_t GetCurrentIndex() const
    {
        return currentIndex;
    }

    virtual bool Select(const int& index)
    {
        if( index < 0 || index >= GetSelectableItemCount() )
        {
            return false;
        }

        currentIndex = index;
        return true;
    }

    virtual bool Deselect(const int& index)
    {
        // 常に一つしか値を保持しないため選択しないことは不可
        return false;
    }

    virtual size_t GetSelectableItemCount() const
    {
        return values.size();
    }

    virtual size_t GetSelectingItemCount() const
    {
        return 1;
    }

    virtual size_t GetSelectingItemCountMax() const
    {
        return 1;
    }

    virtual void Add(NamedValue<T> value, bool isDefault = false)
    {
        if(isDefault)
        {
            currentIndex = values.size();
        }

        values.push_back(value);
    }

    virtual void Clear()
    {
        currentIndex = 0;
        values.clear();
    }

    virtual NamedValue<T> GetValue() const
    {
        NN_ASSERT( values.size() != 0 )
        return values[currentIndex];
    }

    virtual void Next(bool loop=false)
    {
        currentIndex++;

        if(currentIndex == values.size())
        {
            if(loop)
            {
                currentIndex = 0;
            }
            else
            {
                currentIndex--;
            }
        }
    }

    virtual void Back(bool loop=false)
    {
        currentIndex--;

        if(currentIndex < 0)
        {
            if(loop)
            {
                currentIndex = values.size() - 1;
            }
            else
            {
                currentIndex++;
            }
        }
    }

    virtual string ToStringSelectableItem()
    {
        ostringstream oss;
        oss << values[currentIndex];
        return oss.str();
    }

    virtual string ToStringSelectingItem()
    {
        return ToStringSelectableItem();
    }

    size_t Size()
    {
        return values.size();
    }

    NN_IMPLICIT operator T() const
    {
        return (T)(NamedValue<T>)GetValue();
    } // 暗黙の型変換演算子をオーバーライド

protected:

    vector<NamedValue<T> > values;
    int currentIndex;

    int Find(const T& value, const vector<NamedValue<T> >& list)
    {
        for(int i=0; i<list.size(); ++i)
        {
            if( list[i].Value == value )
            {
                return i;
            }
        }

        return -1;
    }

};

/*!--------------------------------------------------------------------------*
  @brief        複数選択可能値
  *---------------------------------------------------------------------------*/
template<typename T>
class MultiSelectableValue<NamedValue<T> > : public SelectableValue<NamedValue<T> >
{
public:

    MultiSelectableValue()
    {
        itemCountMax = 1;
    }

    virtual ~MultiSelectableValue()
    {
        Clear();
    }

    virtual int FindInSelectableItems(const T& value)
    {
        return SelectableValue<NamedValue<T> >::FindInSelectableItems(value);
    }

    virtual int FindInSelectingItems(const T& value)
    {
        return SelectableValue<NamedValue<T> >::Find(value, selectingItemList);
    }

    virtual bool Select(const int& index)
    {
        if( index < 0 || index >= GetSelectableItemCount() )
        {
            return false;
        }

        NamedValue<T>& value = SelectableValue<NamedValue<T> >::values[index];
        // すでに選択しているか、選択上限に達していたら何もしない
        if( FindInSelectingItems(value.Value) >= 0 || itemCountMax <= GetSelectingItemCount() )
        {
            return false;
        }

        selectingItemList.push_back(value);
        return true;
    }

    virtual bool Deselect(const int& index)
    {
        if( index < 0 || index >= GetSelectingItemCount() )
        {
            return false;
        }

        // 選択していないか、選択下限の場合は何もしない
        const NamedValue<T>& value = selectingItemList[index];
        if( FindInSelectingItems(value.Value) < 0 || GetSelectingItemCount() == 0 )
        {
            return false;
        }

        selectingItemList.erase(selectingItemList.begin() + index);
        return true;
    }

    virtual void Clear()
    {
        selectingItemList.clear();
        SelectableValue<NamedValue<T> >::Clear();
    }

    virtual const vector<NamedValue<T> >* GetSelectingItems() const
    {
        return &selectingItemList;
    }

    virtual size_t GetSelectableItemCount() const
    {
        return SelectableValue<NamedValue<T> >::GetSelectableItemCount();
    }

    virtual size_t GetSelectingItemCount() const
    {
        return selectingItemList.size();
    }

    virtual size_t GetSelectingItemCountMax() const
    {
        return itemCountMax;
    }

    virtual void SetSelectableItemCountMax(size_t countMax)
    {
        itemCountMax = countMax;
    }

    virtual string ToStringSelectingItem()
    {
        ostringstream oss;

        for(int i=0; i<selectingItemList.size(); ++i)
        {
            if( i != 0 )
            {
                oss << ", ";
            }

            oss << selectingItemList[i].Name;
        }
        return oss.str();
    }

    // 暗黙の型変換演算子は使用しない
    NN_IMPLICIT operator T();

private:

    uint8_t itemCountMax;
    vector<NamedValue<T> > selectingItemList;

};


/*!--------------------------------------------------------------------------*
  @brief        汎用な選択可能コマンド
  *---------------------------------------------------------------------------*/
template<typename T>
class SelectableConfigCommand<NamedValue<T>> : public ISelectableConfigCommand
{

public:

    SelectableConfigCommand()
    {
    }

    virtual ~SelectableConfigCommand()
    {
        m_Value.Clear();
    }

    // 選択可能なアイテムから検索
    virtual int FindInSelectableItems(const T& value)
    {
        return m_Value.FindInSelectableItems(value);
    }

    // 選択済みアイテムから検索
    virtual int FindInSelectingItems(const T& value)
    {
        return -1;
    }

    virtual bool Select(const int& index)
    {
        return m_Value.Select(index);
    }

    virtual bool Deselect(const int& index)
    {
        // 常に一つしか値を保持しないため選択しないことは不可
        return false;
    }

    virtual size_t GetSelectableItemCount() const
    {
        return m_Value.GetSelectableItemCount();
    }

    virtual size_t GetSelectingItemCount() const
    {
        return 1;
    }

    virtual size_t GetSelectingItemCountMax() const
    {
        return 1;
    }

    virtual size_t GetCurrentIndex() const
    {
        return m_Value.GetCurrentIndex();
    }

    virtual void Add(NamedValue<T> value, bool isDefault = false)
    {
        m_Value.Add(value, isDefault);
    }

    virtual void Clear()
    {
        m_Value.Clear();
    }

    virtual T GetValue() const
    {
        return m_Value.GetValue();
    }

    virtual void Next(bool loop=false)
    {
        m_Value.Next(loop);
    }

    virtual void Back(bool loop=false)
    {
        m_Value.Back(loop);
    }

    virtual string ToStringSelectableItem()
    {
        return m_Value.ToStringSelectableItem();
    }

    virtual string ToStringSelectingItem()
    {
        return m_Value.ToStringSelectingItem();
    }

    size_t Size()
    {
        return m_Value.Size();
    }

    // 暗黙の型変換演算子をオーバーライド
    NN_IMPLICIT operator T() const
    {
        return (T)(NamedValue<T>)GetValue();
    }

    virtual int Execute() = 0;

protected:

    SelectableValue<NamedValue<T>> m_Value;

};



}

