﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <iomanip>
#include <vector>
#include <list>
#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include "Display.h"
#include "Pad.h"
#include "RingBuffer.h"
#include "TextWriter.h"
#include "Counter.h"

namespace WlanTest {



class UIControl
{
/*---------------------------------------------------------------------------
　　　　　定数・型定義など
---------------------------------------------------------------------------*/
public:
private:

/*---------------------------------------------------------------------------
　　　　　静的変数
---------------------------------------------------------------------------*/
public:
private:

/*---------------------------------------------------------------------------
　　　　　静的メソッド
---------------------------------------------------------------------------*/
public:
private:

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
    float X;
    float Y;
    float Width;
    float Height;
    UIControl* Parent;
protected:
private:

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void Show(Display& display);
    virtual void ShowImpl(Display&) = 0;
    virtual void InputPad(Pad&);
protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    UIControl();
    virtual ~UIControl();
private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    virtual int GetX()
    {
        if(Parent == NULL){ return X; }
        return (Parent->GetX() + X);
    }
    virtual int GetY()
    {
        if(Parent == NULL){ return Y; }
        return (Parent->GetY() + Y);
    }
};


class LineGraph : public UIControl
{
public:
    float MaxX;
    float MaxY;
    float MinX;
    float MinY;
    float ScaleX;
    float ScaleY;
    float ScaleOffsetX;
    float ScaleOffsetY;
    bool ShowData;

protected:
    Color m_Color;
    Color m_ScaleColor;
    float m_PointSize;
    uint8_t m_Points;
    float* m_PointsX;
    float* m_PointsY;

    virtual void ShowImpl(Display& display);

public:
    NN_IMPLICIT LineGraph(uint8_t points);
    virtual ~LineGraph();

    void SetColor(Color color){ m_Color = color; }
    void SetPointSize(float size){ m_PointSize = size; }
    void SetPoint(uint8_t index, float x, float y)
    {
        m_PointsX[index] = x;
        m_PointsY[index] = y;
    }
    void SetPoint(uint8_t index, float y)
    {
        m_PointsX[index] = index + 1;
        m_PointsY[index] = y;
    }

private:
    LineGraph(const LineGraph& lineGraph);
    LineGraph& operator=(const LineGraph& lineGraph);

};



class Rectangle : public UIControl
{
protected:
    Color m_Color;
    virtual void ShowImpl(Display& display);

public:
    Rectangle();
    virtual ~Rectangle();
    void SetColor(Color color){ m_Color = color; }
};





class PercentageBar : public Rectangle
{
protected:
    double m_Percentage;
    string m_Text;
    bool m_IsHorizontal;

    virtual void ShowImpl(Display& display);

public:
    PercentageBar();
    virtual ~PercentageBar();

    virtual void SetValue(double percentage)
    {
        m_Percentage = percentage;
        ostringstream oss;
        oss << (int)(m_Percentage * 100 + 0.5);
        m_Text = oss.str();
    }
    virtual void Clear()
    {
        m_Percentage = 0;
        m_Text = "0";
    }
    void SetHorizontal(bool horizontal) { m_IsHorizontal = horizontal; };
    double GetPercentage() const
    {
        return m_Percentage;
    }
};




class PercentageBar2 : public PercentageBar
{
protected:
    double m_Peak;
    Average m_Average;
    string m_PeakText;
    string m_AverageText;
    Color m_PeakColor;
    Color m_AverageColor;

    virtual void ShowImpl(Display& display);

public:
    PercentageBar2();
    virtual ~PercentageBar2();

    virtual void SetValue(double percentage)
    {
        ostringstream oss;

        PercentageBar::SetValue(percentage);
        if(percentage > m_Peak)
        {
            m_Peak = percentage;

            oss << (int)(m_Peak * 100 + 0.5);
            m_PeakText = oss.str();
            oss.str("");
        }

        m_Average.Count(percentage);
        oss << (int)(m_Average.GetValue() * 100 + 0.5);
        m_AverageText = oss.str();
    }
    virtual void Clear()
    {
        PercentageBar::Clear();
        m_PeakText = "0";
        m_Peak = 0;
        m_AverageText = "0";
        m_Average.Clear();
    }
    virtual void SetPeakColor(Color color)
    {
        m_PeakColor = color;
    }
    virtual void SetAverageColor(Color color)
    {
        m_AverageColor = color;
    }
    double GetAverage() const
    {
        return m_Average.GetValue();
    }
    double GetPeak() const
    {
        return m_Peak;
    }

private:
    PercentageBar2(const PercentageBar2& percentageBar2);
    PercentageBar2& operator=(const PercentageBar2& percentageBar2);

};





/*!--------------------------------------------------------------------------*
  @brief        ページ

 *---------------------------------------------------------------------------*/
class Page : public UIControl
{
public:
    Page(){}
    virtual ~Page(){}

    void InvokeInput();
    void InvokeShow(Display& display);

    virtual void ShowImpl(Display& display);
    virtual void Add(UIControl& ui);
    virtual void Remove(UIControl& ui);
    virtual void Clear();
    virtual void UpdateElements();

private:
    list<UIControl*> controls;
};



/*!--------------------------------------------------------------------------*
  @brief        切り替え可能ページ

 *---------------------------------------------------------------------------*/
class SwitchablePage : public UIControl
{
protected:

    Button NextKey;
    Button BackKey;
    vector<string> m_Names;
    vector<Page*>  m_Pages;
    vector<string>::iterator m_NamesIterator;
    vector<Page*>::iterator  m_PagesIterator;
    Page*          m_Current;
    float          m_CurrentX;
    Page*          m_Next;
    float          m_NextX;
    bool           m_Switching;
    float          m_Smoothness;

public:

    SwitchablePage();

    virtual void ShowImpl(Display& display);
    virtual void InputPad(Pad& key);
    virtual void AddPage(string name, Page& page);

    virtual Page& GetCurrentPage();

    void Clear();
    void SetSmoothness(float val);
    void ChangeNextKey(const Button& button);
    void ChangeBackKey(const Button& button);

private:

    SwitchablePage(const SwitchablePage& switchablePage);
    SwitchablePage& operator=(const SwitchablePage& switchablePage);

};

/*!--------------------------------------------------------------------------*
  @brief        コンテンツ配置方法

 *---------------------------------------------------------------------------*/
enum ContentAlignment
{
    TopLeft,
    TopCenter,
    TopRight,
    MiddleLeft,
    MiddleCenter,
    MiddleRight,
    BottomLeft,
    BottomCenter,
    BottomRight
};



/*!--------------------------------------------------------------------------*
  @brief        ラベル

 *---------------------------------------------------------------------------*/
class Label : public UIControl
{
public:
    string Text;
    float FontWidth;
    float FontHeight;
    float FixedWidth;
    Color BackgroundColor;
    Color TextColor;
    ContentAlignment Alignment;

    virtual void ShowImpl(Display& display);
    virtual void FitSize();
    Label();

public:
    ContentAlignment GetAlignment() { return Alignment; }
    void SetAlignment(ContentAlignment a) { Alignment = a; }

    void SetFgColor(Color c){ TextColor = c; }
    void SetBgColor(Color c){ BackgroundColor = c; }

    Color GetFgColor(){ return TextColor; }
    Color GetBgColor(){ return BackgroundColor; }
};




/*!--------------------------------------------------------------------------*
  @brief        アクティブインジケータ
 *---------------------------------------------------------------------------*/
class ArrowIndicator : public UIControl
{
private:

    Label    label;
    uint8_t  animCount;
    char     ch;
    bool     m_IsActive;
    nn::TimeSpan m_Interval;
    nn::os::Tick m_LastTick;
    float    x[10];
    float    y[10];
    uint8_t  index;

public:

    virtual void ShowImpl(Display& display);
    ArrowIndicator();
    void Activate()
    {
        m_IsActive = true;
    }

    void Inactivate()
    {
        m_IsActive=false;
    }

    void SetIndicator(char c){ ch = c; }
    char GetIndicator(){ return ch; }

    void SetAnimCount(uint8_t size){ animCount = size; }
    uint8_t GetAnimCount(){ return animCount; }

    void SetPosition(const float px[], const float py[], const uint8_t size)
    {
        animCount = size;
        if( size > 10 )
        {
            animCount = 10;
        }

        memcpy(x, px, sizeof(x[0]) * animCount);
        memcpy(y, py, sizeof(y[0]) * animCount);
    }
    void GetPositionX(float *px, uint8_t *size)
    {
        NN_ASSERT(px);
        NN_ASSERT(size);
        px = x;
        *size = animCount;
    }
    void GetPositionY(float *py, uint8_t *size)
    {
        NN_ASSERT(py);
        NN_ASSERT(size);
        py = x;
        *size = animCount;
    }

    void SetFgColor(Color c){ label.SetFgColor(c); }
    void SetBgColor(Color c){ label.SetBgColor(c); }

    Color GetFgColor(){ return label.GetFgColor(); }
    Color GetBgColor(){ return label.GetBgColor(); }

private:

    ArrowIndicator(const ArrowIndicator& activeIndicator);
    ArrowIndicator& operator=(const ArrowIndicator& activeIndicator);
};


/*!--------------------------------------------------------------------------*
  @brief        スクロール可能テキストページ

 *---------------------------------------------------------------------------*/
class ScrollableTextPage : public UIControl
{
public:

    typedef vector<Label*> TextPage;

protected:

    Button m_NextKey;
    Button m_BackKey;

    float  m_DisplayRangeLeft;
    float  m_DisplayRangeRight;

    vector<TextPage*> m_Pages;
    vector<TextPage*>::iterator m_PagesIterator;
    TextPage* m_Current;
    TextPage* m_Next;
    float     m_CurrentX;
    float     m_NextX;
    bool      m_Switching;
    float     m_Smoothness;
    bool      m_IsLoopEnabled;

    ArrowIndicator m_LeftIndicator;
    ArrowIndicator m_RightIndicator;

    string Trim(const string& text, const float& textX, const float& fontWidth, const float& left, const float& right);
    void ShowIndicator(Display& display);
    void ShowTextPage(Display& display);

public:

    ScrollableTextPage();

    virtual void ShowImpl(Display& display);
    virtual void InputPad(Pad& key);
    virtual void AddPage(TextPage& page);

    virtual TextPage& GetCurrentPage();

    void Clear();
    void SetSmoothness(float val);
    void ChangeNextKey(const Button& button);
    void ChangeBackKey(const Button& button);

    void SetDisplayRange(const float& left, const float& right);
    void SetDisplayRangeLeft(const float& left);
    void SetDisplayRangeRight(const float& right);

    void SetLoop(const bool& isEnabled);

    explicit ScrollableTextPage(const SwitchablePage& switchablePage);
    ScrollableTextPage& operator=(const SwitchablePage& switchablePage);

};



/*!--------------------------------------------------------------------------*
  @brief        ビルド情報

 *---------------------------------------------------------------------------*/
class BuildInfo : public UIControl
{
private:
    Page  m_Page;
    Label m_Date;
    Label m_Time;

public:
    BuildInfo();
    virtual void ShowImpl(WlanTest::Display& display);
};


/*!--------------------------------------------------------------------------*
  @brief        ラベル棒

 *---------------------------------------------------------------------------*/
class LabelBar : public UIControl
{
/*---------------------------------------------------------------------------
　　　　　定数・型定義など
---------------------------------------------------------------------------*/
public:
    static const uint8_t Size = 20;
private:
    float  m_Space;
    bool m_IsHorizontal;

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:
    Label* labels[Size];
    uint16_t   next;
/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    void AddLabel(Label& label);

protected:
    virtual void ShowImpl(Display& display);

private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    LabelBar();

private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    void ToHorizontal()
    {
        m_IsHorizontal = true;
    }

    void ToVertical()
    {
        m_IsHorizontal = false;
    }

    void SetSpace(float f)
    {
        m_Space = f;
    }

private:
    LabelBar(const LabelBar& labelBar);
    LabelBar& operator=(const LabelBar& labelBar);

};



/*!--------------------------------------------------------------------------*
  @brief        コンソール

 *---------------------------------------------------------------------------*/
class Console : public UIControl, public TextWriter
{
public:
    float FontWidth;
    float FontHeight;
    Color BackgroundColor;
    Color TextColor;

    Console();
    virtual ~Console();

    // 改行コードの後に文字列を含まない文字列の描画（ただし、はみ出して改行になることはある）
    virtual bool WriteOneLine(const string& text);
    virtual bool Write(const string& text);
    virtual bool WriteLine(const string& text);
    void Clear();
    virtual void ShowImpl(Display& display);

private:
    uint8_t Lines;          // 行数
    uint8_t Charactors;     // 文字数（一行の）
    string m_Line;
    RingBuffer<string>* m_Buffer;
    nn::os::Mutex m_Mutex;

private:
    Console(const Console& console);
    Console& operator=(const Console& console);
};



/*!--------------------------------------------------------------------------*
  @brief        アクティブインジケータ
 *---------------------------------------------------------------------------*/
class ActiveIndicator : public UIControl
{
private:

    Label    label;
    uint8_t  index;
    char     ch[5];
    bool     m_IsActive;
    nn::TimeSpan m_Interval;
    nn::os::Tick m_LastTick;

public:

    virtual void ShowImpl(Display& display);
    ActiveIndicator();
    void Activate()
    {
        //index = 0;
        m_IsActive = true;
    }

    void Inactivate()
    {
        index = 4;
        m_IsActive=false;
    }

    void SetFgColor(Color c){ label.SetFgColor(c); }
    void SetBgColor(Color c){ label.SetBgColor(c); }

    Color GetFgColor(){ return label.GetFgColor(); }
    Color GetBgColor(){ return label.GetBgColor(); }

private:

    ActiveIndicator(const ActiveIndicator& activeIndicator);
    ActiveIndicator& operator=(const ActiveIndicator& activeIndicator);
};

}    // WlanTest
