﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "ModelViewer.h"
#include "Tester.h"
#include "Util.h"

namespace WlanTest {

extern const char* TEST_SUMMARY_TITLE_LABEL;
extern const char* ELAPSED_TIME_LABEL;
extern const char* TEST_TIME_LABEL;
extern const char* TARGET_TX_RATE_LABEL;
extern const char* TARGET_RX_RATE_LABEL;
extern const char* TARGET_PLR_LABEL;
extern const char* TARGET_MAX_LATENCY_LABEL;
extern const char* TARGET_AVG_LATENCY_LABEL;
extern const char* RESULT_LABEL;

void CreateTitleLabel(Label* pLabel, const char* title, const uint32_t x = DISPLAY_TITLE_START_X, const uint32_t y = DISPLAY_TITLE_START_Y);
void CreateContentLabel(Label* pLabel, const char* name, const uint32_t x, const uint32_t y, const uint32_t width = 1000);

void UpdateElapsedTimeLabel(Label* pLabel, const nn::TimeSpan& ts);
void UpdateTestTimeLabel(Label* pLabel, const nn::TimeSpan& ts);
void UpdateTargetThroughput(Label* pLabel, const char* label, const uint32_t tps, const TestResult result);
void UpdateTargetPlr(Label* pLabel, const char* label, const double plr, const TestResult result);
void UpdateTargetLatency(Label* pLabel, const char* label, const uint32_t latency, const TestResult result);
void UpdateTotalResult(Label* pLabel, const char* label, const TestResult result, const bool isTesting);

/*---------------------------------------------------------------------------
           TimerViewer
---------------------------------------------------------------------------*/
template<typename T>
class TimerViewer : public SceneViewer<AgingTester<T>>
{
public:
protected:
private:

    Label timeLabel;

public:

    TimerViewer()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        SceneViewer<AgingTester<T>>::Add(timeLabel);
        timeLabel.Text = "000:00:00";
        timeLabel.X = 1100;
        timeLabel.Y = 5;

        SceneViewer<AgingTester<T>>::ErasePadText(Button::A);
        SceneViewer<AgingTester<T>>::ErasePadText(Button::START);
    }

    virtual void ShowImpl(Display& display)
    {
        ostringstream oss;
        nn::TimeSpan ts = SceneViewer<AgingTester<T>>::target->GetElapsedTime();

        oss << setw(2) << setfill('0') << ts.GetHours() << ":";
        ts -= nn::TimeSpan::FromHours(ts.GetHours());

        oss << setw(2) << setfill('0') << ts.GetMinutes() << ":";
        ts -= nn::TimeSpan::FromMinutes(ts.GetMinutes());

        oss << setw(2) << setfill('0') << ts.GetSeconds();

        timeLabel.Text = oss.str();
        oss.str("");

        SceneViewer<AgingTester<T>>::ShowImpl(display);
    }

protected:
private:

};

/*---------------------------------------------------------------------------
           LocalmasterTesterViewer
---------------------------------------------------------------------------*/
class LocalMasterTesterViewer : public SceneViewer<LocalMasterPerformanceTester>
{
public:
protected:
private:

    Label title;
    Label currentTimeLabel;
    Label testTimeLabel;
    Label blanklineLabel;

    Label txRateLabel;
    Label rxRateLabel;
    Label plrLabel;
    Label maxLatencyLabel;
    Label avgLatencyLabel;

    Label resultLabel;

public:

    LocalMasterTesterViewer()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        CreateTitleLabel(&title, TEST_SUMMARY_TITLE_LABEL);
        CreateContentLabel(&currentTimeLabel, ELAPSED_TIME_LABEL, x, y);
        CreateContentLabel(&testTimeLabel, TEST_TIME_LABEL, x, (y += h));

        CreateContentLabel(&txRateLabel, TARGET_TX_RATE_LABEL, x, (y += 2 * h));
        CreateContentLabel(&rxRateLabel, TARGET_RX_RATE_LABEL, x, (y += h));
        CreateContentLabel(&plrLabel, TARGET_PLR_LABEL, x, (y += h));
        CreateContentLabel(&avgLatencyLabel, TARGET_AVG_LATENCY_LABEL, x, (y += h));
        CreateContentLabel(&maxLatencyLabel, TARGET_MAX_LATENCY_LABEL, x, (y += h));

        CreateContentLabel(&resultLabel, RESULT_LABEL, x, (y += 2 * h));

        blanklineLabel.Text = " ";

        Add(title);
        Add(currentTimeLabel);
        Add(testTimeLabel);
        Add(blanklineLabel);

        Add(txRateLabel);
        Add(rxRateLabel);
        Add(plrLabel);
        Add(maxLatencyLabel);
        Add(avgLatencyLabel);
        Add(blanklineLabel);

        Add(resultLabel);
        Add(blanklineLabel);
    }

    virtual void ShowImpl(Display& display)
    {
        UpdateElapsedTimeLabel(&currentTimeLabel, target->GetElapsedTime());
        UpdateTestTimeLabel(&testTimeLabel, target->GetTestTime());

        UpdateTargetThroughput(&txRateLabel, TARGET_TX_RATE_LABEL, target->GetMinTxRate(), target->GetTxRateResult());
        UpdateTargetThroughput(&rxRateLabel, TARGET_RX_RATE_LABEL, target->GetMinRxRate(), target->GetRxRateResult());
        UpdateTargetPlr(&plrLabel, TARGET_PLR_LABEL, target->GetMaxPlr(), target->GetPlrResult());
        UpdateTargetLatency(&avgLatencyLabel, TARGET_AVG_LATENCY_LABEL, target->GetMaxLatencyAvg(), target->GetAvgLatencyResult());
        UpdateTargetLatency(&maxLatencyLabel, TARGET_MAX_LATENCY_LABEL, target->GetMaxLatencyMax(), target->GetMaxLatencyResult());

        UpdateTotalResult(&resultLabel, RESULT_LABEL, target->GetResult(), target->IsTesting());

        SceneViewer<LocalMasterPerformanceTester>::ShowImpl(display);
    }

protected:
private:

};

/*---------------------------------------------------------------------------
           LocalClientPerformanceTesterViewer
---------------------------------------------------------------------------*/
class LoalClientTesterViewer : public SceneViewer<LocalClientPerformanceTester>
{
public:
protected:
private:

    Label title;
    Label currentTimeLabel;
    Label testTimeLabel;
    Label blanklineLabel;

    Label txRateLabel;
    Label rxRateLabel;
    Label plrLabel;
    Label maxLatencyLabel;
    Label avgLatencyLabel;

    Label resultLabel;

public:

    LoalClientTesterViewer()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        CreateTitleLabel(&title, TEST_SUMMARY_TITLE_LABEL);
        CreateContentLabel(&currentTimeLabel, ELAPSED_TIME_LABEL, x, y);
        CreateContentLabel(&testTimeLabel, TEST_TIME_LABEL, x, (y += h));

        CreateContentLabel(&txRateLabel, TARGET_TX_RATE_LABEL, x, (y += 2 * h));
        CreateContentLabel(&rxRateLabel, TARGET_RX_RATE_LABEL, x, (y += h));
        CreateContentLabel(&plrLabel, TARGET_PLR_LABEL, x, (y += h));
        CreateContentLabel(&avgLatencyLabel, TARGET_AVG_LATENCY_LABEL, x, (y += h));
        CreateContentLabel(&maxLatencyLabel, TARGET_MAX_LATENCY_LABEL, x, (y += h));

        CreateContentLabel(&resultLabel, RESULT_LABEL, x, (y += 2 * h));

        blanklineLabel.Text = " ";

        Add(title);
        Add(currentTimeLabel);
        Add(testTimeLabel);
        Add(blanklineLabel);

        Add(txRateLabel);
        Add(rxRateLabel);
        Add(plrLabel);
        Add(maxLatencyLabel);
        Add(avgLatencyLabel);
        Add(blanklineLabel);

        Add(resultLabel);
        Add(blanklineLabel);
    }

    virtual void ShowImpl(Display& display)
    {
        UpdateElapsedTimeLabel(&currentTimeLabel, target->GetElapsedTime());
        UpdateTestTimeLabel(&testTimeLabel, target->GetTestTime());

        UpdateTargetThroughput(&txRateLabel, TARGET_TX_RATE_LABEL, target->GetMinTxRate(), target->GetTxRateResult());
        UpdateTargetThroughput(&rxRateLabel, TARGET_RX_RATE_LABEL, target->GetMinRxRate(), target->GetRxRateResult());
        UpdateTargetPlr(&plrLabel, TARGET_PLR_LABEL, target->GetMaxPlr(), target->GetPlrResult());
        UpdateTargetLatency(&avgLatencyLabel, TARGET_AVG_LATENCY_LABEL, target->GetMaxLatencyAvg(), target->GetAvgLatencyResult());
        UpdateTargetLatency(&maxLatencyLabel, TARGET_MAX_LATENCY_LABEL, target->GetMaxLatencyMax(), target->GetMaxLatencyResult());

        UpdateTotalResult(&resultLabel, RESULT_LABEL, target->GetResult(), target->IsTesting());

        SceneViewer<LocalClientPerformanceTester>::ShowImpl(display);
    }

protected:
private:

};

/*---------------------------------------------------------------------------
           SocketTesterViewer
---------------------------------------------------------------------------*/
class SocketTesterViewer : public SceneViewer<SocketPerformanceTester>
{
public:
protected:
private:

    Label title;
    Label currentTimeLabel;
    Label testTimeLabel;
    Label blanklineLabel;

    Label txRateLabel;
    Label rxRateLabel;
    Label plrLabel;

    Label resultLabel;

public:

    SocketTesterViewer()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        CreateTitleLabel(&title, TEST_SUMMARY_TITLE_LABEL);
        CreateContentLabel(&currentTimeLabel, ELAPSED_TIME_LABEL, x, y);
        CreateContentLabel(&testTimeLabel, TEST_TIME_LABEL, x, (y += h));

        CreateContentLabel(&txRateLabel, TARGET_TX_RATE_LABEL, x, (y += 2 * h));
        CreateContentLabel(&rxRateLabel, TARGET_RX_RATE_LABEL, x, (y += h));
        CreateContentLabel(&plrLabel, TARGET_PLR_LABEL, x, (y += h));

        CreateContentLabel(&resultLabel, RESULT_LABEL, x, (y += 2 * h));

        blanklineLabel.Text = " ";

        Add(title);
        Add(currentTimeLabel);
        Add(testTimeLabel);
        Add(blanklineLabel);

        Add(txRateLabel);
        Add(rxRateLabel);
        Add(plrLabel);
        Add(blanklineLabel);

        Add(resultLabel);
        Add(blanklineLabel);
    }

    virtual void ShowImpl(Display& display)
    {
        UpdateElapsedTimeLabel(&currentTimeLabel, target->GetElapsedTime());
        UpdateTestTimeLabel(&testTimeLabel, target->GetTestTime());

        UpdateTargetThroughput(&txRateLabel, TARGET_TX_RATE_LABEL, target->GetMinTxRate(), target->GetTxRateResult());
        UpdateTargetThroughput(&rxRateLabel, TARGET_RX_RATE_LABEL, target->GetMinRxRate(), target->GetRxRateResult());
        UpdateTargetPlr(&plrLabel, TARGET_PLR_LABEL, target->GetMaxPlr(), target->GetPlrResult());

        UpdateTotalResult(&resultLabel, RESULT_LABEL, target->GetResult(), target->IsTesting());

        SceneViewer<SocketPerformanceTester>::ShowImpl(display);
    }

protected:
private:

};

/*---------------------------------------------------------------------------
           LocalMasterResultViewer
---------------------------------------------------------------------------*/
class LocalMasterResultViewer : public SceneViewer<LocalMasterPerformanceTester>
{
public:
protected:
private:
    Label titleLabel;

    Label elapsedTimeLabel;
    Label testTimeLabel;
    Label clientLabel;
    Label connectionTimeLabel;
    Label connectionLabel;

    Label minTxRateLabel;
    Label minRxRateLabel;
    Label detailedRxLabel;
    Label maxRxPlrLabel;
    Label maxLatencyMaxLabel;
    Label maxLatencyAvgLabel;
    Label performanceLabel;

public:
    LocalMasterResultViewer()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(titleLabel);
        titleLabel.X = DISPLAY_TITLE_START_X;
        titleLabel.Y = DISPLAY_TITLE_START_Y;
        titleLabel.Text = "Result\n";
        titleLabel.FitSize();

        uint32_t py = y;
        Add(elapsedTimeLabel);
        elapsedTimeLabel.X = x;
        elapsedTimeLabel.Y = y;
        elapsedTimeLabel.Text = "Elapsed Time   : ";
        elapsedTimeLabel.FitSize();
        elapsedTimeLabel.Width += 1000;

        Add(testTimeLabel);
        testTimeLabel.X = x;
        testTimeLabel.Y = (y += h);
        testTimeLabel.Text = "Test Time      : ";
        testTimeLabel.FitSize();
        testTimeLabel.Width += 1000;

        y = py;
        Add(clientLabel);
        clientLabel.X = x + 600;
        clientLabel.Y = y;
        clientLabel.Text = "Num of Clients : ";
        clientLabel.FitSize();
        clientLabel.Width += 100;

        Add(connectionTimeLabel);
        connectionTimeLabel.X = x + 600;
        connectionTimeLabel.Y = (y += h);
        connectionTimeLabel.Text = "CreateBSS()    : ";
        connectionTimeLabel.FitSize();
        connectionTimeLabel.Width += 400;;

        Add(connectionLabel);
        connectionLabel.X = x;
        connectionLabel.Y = (y += h);
        connectionLabel.Text = "<< >>";
        connectionLabel.FitSize();
        connectionLabel.Width += 1000;

        py = y;
        Add(minTxRateLabel);
        minTxRateLabel.X = x;
        minTxRateLabel.Y = (y += 3 * h);
        minTxRateLabel.Text = "Tx Throughput : ";
        minTxRateLabel.FitSize();
        minTxRateLabel.Width += 3000;

        y = py;
        Add(minRxRateLabel);
        minRxRateLabel.X = x + 600;
        minRxRateLabel.Y = (y += 3 * h);
        minRxRateLabel.Text = "Rx Throughput : ";
        minRxRateLabel.FitSize();
        minRxRateLabel.Width += 3000;

        Add(detailedRxLabel);
        detailedRxLabel.X = x;
        detailedRxLabel.Y = (y += 7 * h);
        detailedRxLabel.Text = "Rx : ";
        detailedRxLabel.FitSize();
        detailedRxLabel.Width += 7 * 5000;

        /*
        Add(maxRxPlrLabel);
        maxRxPlrLabel.X = 10;
        maxRxPlrLabel.Y = 10;
        maxRxPlrLabel.Text = "PLR           : ";
        maxRxPlrLabel.FitSize();

        Add(maxLatencyMaxLabel);
        maxLatencyMaxLabel.X = 10;
        maxLatencyMaxLabel.Y = 10;
        maxLatencyMaxLabel.Text = "Latency(Max)  : ";
        maxLatencyMaxLabel.FitSize();

        Add(maxLatencyAvgLabel);
        maxLatencyAvgLabel.X = 10;
        maxLatencyAvgLabel.Y = 10;
        maxLatencyAvgLabel.Text = "Latency(Avg)  : ";
        maxLatencyAvgLabel.FitSize();
        */

        Add(performanceLabel);
        performanceLabel.X = x;
        performanceLabel.Y = (y += 5 * h);
        performanceLabel.Text = "<< >>";
        performanceLabel.FitSize();
        performanceLabel.Width += 1000;
    }

    virtual void ShowImpl(Display& display)
    {
        ostringstream oss;

        string str;
        nn::TimeSpan ts = target->GetElapsedTime();
        if(ts == nn::TimeSpan(0))
        {
            str = "-";
        }
        else
        {
            str = ToString(ts);
        }
        oss << "Elapsed Time   : " << str;
        elapsedTimeLabel.Text = oss.str();
        oss.str("");

        oss << "Test Time      : " << ToString(target->GetTestTime());
        testTimeLabel.Text = oss.str();
        oss.str("");

        oss << "Num of Clients : " << target->GetConnectedCount();
        clientLabel.Text = oss.str();
        oss.str("");

        oss << "CreateBss()    : " << target->GetConnectionTime() << " ms";
        connectionTimeLabel.Text = oss.str();
        oss.str("");

        str = (target->GetConnectionTestSuccess()) ?
            "\n  << CONNECTION TEST : OK >>\n" :
            "\n  << CONNECTION TEST : NG >>\n";

        oss << str;
        connectionLabel.Text = oss.str();
        oss.str("");

        // ---

        NodeStatistics& txStats = target->GetTxStatistics();
        //ReceiverStatistics& rxStats = target->GetRxStatistics();
        map<uint64_t, MultiDataSink::Statistics*>& rxDetailedStats = target->GetRxDetailedStatistics();

        // Mbit
        double txSize = txStats.SendSize * 8.0f / (1024.0f * 1024.0f);
        // msec
        uint64_t txTime = (txStats.LastSendTime - txStats.FirstSendTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
        double txTps = 1.0f * txSize / txTime;

        /*
          // 将来的な表示
          Tx Throughput > 500.0Mbps
            Result : [OK] 550.0Mbps

          Rx Throughput > 500.0Mbps
            Result : [NG] 20.0Mbps [aa:aa:aa:aa:aa:aa]
            Result : [OK] 503.0Mbps [bb:aa:aa:aa:aa:aa]
        */

        oss << "[Tx]\n";
        oss << "  Tx Count   : " << txStats.SendCount << " packets\n";
        oss << "  Tx Error   : " << txStats.SendErrorCount << "\n";
        oss << "  Tx Size    : " << txStats.SendSize << " byte\n";
        oss << "  Throughput : " << setw(7) << setprecision(3) << txTps << " Mbps\n";//   (" << GetMinTxRate << ")";
        minTxRateLabel.Text = oss.str();
        oss.str("");

        // Mbit
        double rxSize = txStats.ReceiveSize * 8.0f / (1024.0f * 1024.0f);
        // msec
        uint64_t rxTime = (txStats.LastReceiveTime - txStats.FirstReceiveTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
        double rxTps = 1.0f * rxSize / rxTime;
        oss << "[Rx]\n";
        oss << "  Rx Count   : " << txStats.ReceiveCount << " packets\n";
        oss << "  Rx Error   : " << txStats.ReceiveErrorCount << "\n";
        oss << "  Rx Size    : " << txStats.ReceiveSize << " byte\n";
        oss << "  Throughput : " << setw(7) << setprecision(3) << rxTps << " Mbps\n";

        minRxRateLabel.Text = oss.str();
        oss.str("");

        map<uint64_t, string>& sources = target->GetSources();
        map<uint64_t, string>::iterator iter = sources.begin();
        while(iter != sources.end())
        {
            // MACAddress, Throughput, PLR, Latency(Avg), Latency(Max), Latency(Histogram)
            oss << "   [" << iter->second << "] "
                << "Last No. " << setw(5) << rxDetailedStats[iter->first]->LastSequenceNo << ", "
                << "Lost " << setw(4) << rxDetailedStats[iter->first]->Errors.GetErrorCount() << ", "
                << "TPS " << setw(7) << setprecision(3) << rxDetailedStats[iter->first]->Throughput.GetThroughput() << " Mbps, "
                << "PLR " << setw(7) << setprecision(3) << rxDetailedStats[iter->first]->Errors.GetErrorRate() * 100.f << " %%, "
                << "Avg lat. " << fixed << setw(4) << setprecision(1) << rxDetailedStats[iter->first]->Latency.GetAvgLatency() << " ms, "
                << "Max lat. " << setw(3) << rxDetailedStats[iter->first]->Latency.GetMaxLatency() << " ms, "
                << "[";

            LatencyCounter& latencyCounter = rxDetailedStats[iter->first]->Latency;
            uint8_t lastIndex = latencyCounter.GetLastIndex();
            for(int i=0; i<lastIndex - 1; ++i)
            {
                oss << "(<" << i + 1 << "ms):" << latencyCounter.GetBurstLatencyCount(i) << ", ";
            }
            oss << "(>=" << lastIndex - 1 << "ms):" << latencyCounter.GetBurstLatencyCount(lastIndex - 1);
            oss << "]\n";

            iter++;
        }

        detailedRxLabel.Text = oss.str();
        oss.str("");

        str = (target->GetPerformanceTestSuccess()) ?
            "  << PERFORMANCE TEST : OK >>\n" :
            "  << PERFORMANCE TEST : NG >>\n";
        oss << str;
        performanceLabel.Text = oss.str();
        oss.str("");

        SceneViewer<LocalMasterPerformanceTester>::ShowImpl(display);
    }

    void PrintResult(string& key, string& value)
    {
        ostringstream osk;
        ostringstream osv;

        osk << "Target Test Time(sec), ";
        osv << target->GetTestTime().GetSeconds() << ", ";

        osk << "Target Test Margin(sec), ";
        osv << target->GetMarginTime().GetSeconds() << ", ";

        osk << "Target Num of clients, ";
        osv << target->GetConnectableCount() << ", ";

        osk << "Target Tx Throughput(bps), ";
        osv << target->GetMinTxRate() << ", ";

        osk << "Target Rx Throughput(bps), ";
        osv << target->GetMinRxRate() << ", ";

        osk << "Target PLR(%%), ";
        osv << target->GetMaxPlr() * 100.0f << ", ";

        osk << "Target Max Latency(ms), ";
        osv << target->GetMaxLatencyMax() << ", ";

        osk << "Target Avg Latency(ms), ";
        osv << target->GetMaxLatencyAvg() << ", ";


        osk << "Test Time(sec), ";
        osv << target->GetElapsedTime().GetSeconds() << ", ";

        osk << "CreateBss() (ms), ";
        osv << target->GetConnectionTime() << ", ";

        NodeStatistics& txStats = target->GetTxStatistics();
        map<uint64_t, MultiDataSink::Statistics*>& rxDetailedStats = target->GetRxDetailedStatistics();

        // bit
        double txSize = txStats.SendSize * 8.0f;
        // msec
        uint64_t txTime = (txStats.LastSendTime - txStats.FirstSendTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
        double txTps = 1.0f * txSize / txTime;

        osk << "Tx Count, ";
        osv << txStats.SendCount << ", ";

        osk << "Tx Size(byte), ";
        osv << txStats.SendSize << ", ";

        osk << "Tx Error, ";
        osv << txStats.SendErrorCount << ", ";

        osk << "Tx Throughput(bps), ";
        osv << txTps << ", ";


        map<uint64_t, string>& sources = target->GetSources();
        map<uint64_t, string>::iterator iter = sources.begin();
        osk << "Rx Nodes, ";
        osv << sources.size() << ", ";

        while(iter != sources.end())
        {
            osk << "MAC Address, Rx Throughput(bps), Rx PLR(%%), Avg Latency(ms), Max Latency(ms),";
            LatencyCounter& latencyCounter = rxDetailedStats[iter->first]->Latency;
            uint8_t lastIndex = latencyCounter.GetLastIndex();
            for(int i=0; i<lastIndex - 1; ++i)
            {
                osk << "Rx Latency(<" << i + 1<< "ms), ";
            }
            osk << "Rx Latency(>=" << lastIndex - 1 << "ms), ";

            // MACAddress, Throughput, PLR, Latency(Avg), Latency(Max), Latency(Histogram)
            osv << iter->second << ", "
                << rxDetailedStats[iter->first]->Throughput.GetThroughput() << ", "
                << rxDetailedStats[iter->first]->Errors.GetErrorRate() * 100.f << ", "
                << rxDetailedStats[iter->first]->Latency.GetAvgLatency() << ", "
                << rxDetailedStats[iter->first]->Latency.GetMaxLatency() << ", ";

            for(int i=0; i<lastIndex; ++i)
            {
                osv << latencyCounter.GetBurstLatencyCount(i) << ", ";
            }

            iter++;
        }

        osk << "Connectivity Result, ";
        osv << ((target->GetConnectionTestSuccess()) ? "Success" : "Failure") << ", ";

        osk << "Performance Result";
        osv << ((target->GetPerformanceTestSuccess()) ? "Success" : "Failure");

        // NN_LOG(osk.str().c_str());
        // NN_LOG("\n");
        // NN_LOG(osv.str().c_str());
        // NN_LOG("\n");
        key = osk.str();
        value = osv.str();
    }

protected:
private:

};

/*---------------------------------------------------------------------------
           LocalClientResultViewer
---------------------------------------------------------------------------*/
class LocalClientResultViewer : public SceneViewer<LocalClientPerformanceTester>
{
public:
protected:
private:
    Label titleLabel;

    Label elapsedTimeLabel;
    Label testTimeLabel;
    Label connectionLabel;
    Label connectionTimeLabel;

    Label minTxRateLabel;
    Label minRxRateLabel;
    Label detailedRxLabel;
    Label maxRxPlrLabel;
    Label maxLatencyMaxLabel;
    Label maxLatencyAvgLabel;
    Label performanceLabel;

public:
    LocalClientResultViewer()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(titleLabel);
        titleLabel.X = DISPLAY_TITLE_START_X;
        titleLabel.Y = DISPLAY_TITLE_START_Y;
        titleLabel.Text = "Result\n";
        titleLabel.FitSize();

        uint32_t py = y;
        Add(elapsedTimeLabel);
        elapsedTimeLabel.X = x;
        elapsedTimeLabel.Y = y;
        elapsedTimeLabel.Text = "Elapsed Time   : ";
        elapsedTimeLabel.FitSize();
        elapsedTimeLabel.Width += 1000;

        Add(testTimeLabel);
        testTimeLabel.X = x;
        testTimeLabel.Y = (y += h);
        testTimeLabel.Text = "Test Time      : ";
        testTimeLabel.FitSize();
        testTimeLabel.Width += 1000;

        y = py;
        Add(connectionTimeLabel);
        connectionTimeLabel.X = x + 600;
        connectionTimeLabel.Y = y;
        connectionTimeLabel.Text = "Connect()      : ";
        connectionTimeLabel.FitSize();
        connectionTimeLabel.Width += 400;

        y += h;

        Add(connectionLabel);
        connectionLabel.X = x;
        connectionLabel.Y = (y += h);
        connectionLabel.Text = "<< >>";
        connectionLabel.FitSize();
        connectionLabel.Width += 1000;

        py = y;
        Add(minTxRateLabel);
        minTxRateLabel.X = x;
        minTxRateLabel.Y = (y += 3 * h);
        minTxRateLabel.Text = "Tx Throughput : ";
        minTxRateLabel.FitSize();
        minTxRateLabel.Width += 3000;

        y = py;
        Add(minRxRateLabel);
        minRxRateLabel.X = x + 600;
        minRxRateLabel.Y = (y += 3 * h);
        minRxRateLabel.Text = "Rx Throughput : ";
        minRxRateLabel.FitSize();
        minRxRateLabel.Width += 3000;

        Add(detailedRxLabel);
        detailedRxLabel.X = x;
        detailedRxLabel.Y = (y += 7 * h);
        detailedRxLabel.Text = "Rx : ";
        detailedRxLabel.FitSize();
        detailedRxLabel.Width += 7 * 5000;

        /*
        Add(maxRxPlrLabel);
        maxRxPlrLabel.X = 10;
        maxRxPlrLabel.Y = 10;
        maxRxPlrLabel.Text = "PLR           : ";
        maxRxPlrLabel.FitSize();

        Add(maxLatencyMaxLabel);
        maxLatencyMaxLabel.X = 10;
        maxLatencyMaxLabel.Y = 10;
        maxLatencyMaxLabel.Text = "Latency(Max)  : ";
        maxLatencyMaxLabel.FitSize();

        Add(maxLatencyAvgLabel);
        maxLatencyAvgLabel.X = 10;
        maxLatencyAvgLabel.Y = 10;
        maxLatencyAvgLabel.Text = "Latency(Avg)  : ";
        maxLatencyAvgLabel.FitSize();
        */

        Add(performanceLabel);
        performanceLabel.X = x;
        performanceLabel.Y = (y += 5 * h);
        performanceLabel.Text = "<< >>";
        performanceLabel.FitSize();
        performanceLabel.Width += 1000;
    }

    virtual void ShowImpl(Display& display)
    {
        ostringstream oss;

        string str;
        nn::TimeSpan ts = target->GetElapsedTime();
        if(ts == nn::TimeSpan(0))
        {
            str = "-";
        }
        else
        {
            str = ToString(ts);
        }
        oss << "Elapsed Time   : " << str;
        elapsedTimeLabel.Text = oss.str();
        oss.str("");

        oss << "Test Time      : " << ToString(target->GetTestTime());
        testTimeLabel.Text = oss.str();
        oss.str("");

        oss << "Connect()      : " << target->GetConnectionTime() << " ms";
        connectionTimeLabel.Text = oss.str();
        oss.str("");

        str = (target->GetConnectionTestSuccess()) ?
            "\n  << CONNECTION TEST : OK >>\n" :
            "\n  << CONNECTION TEST : NG >>\n";

        oss << str;
        connectionLabel.Text = oss.str();
        oss.str("");

        // ---

        NodeStatistics& txStats = target->GetTxStatistics();
        //ReceiverStatistics& rxStats = target->GetRxStatistics();
        map<uint64_t, MultiDataSink::Statistics*>& rxDetailedStats = target->GetRxDetailedStatistics();

        // Mbit
        double txSize = txStats.SendSize * 8.0f / (1024.0f * 1024.0f);
        // msec
        uint64_t txTime = (txStats.LastSendTime - txStats.FirstSendTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
        double txTps = 1.0f * txSize / txTime;

        /*
          // 将来的な表示
          Tx Throughput > 500.0Mbps
            Result : [OK] 550.0Mbps

          Rx Throughput > 500.0Mbps
            Result : [NG] 20.0Mbps [aa:aa:aa:aa:aa:aa]
            Result : [OK] 503.0Mbps [bb:aa:aa:aa:aa:aa]
        */

        oss << "[Tx]\n";
        oss << "  Tx Count   : " << txStats.SendCount << " packets\n";
        oss << "  Tx Error   : " << txStats.SendErrorCount << "\n";
        oss << "  Tx Size    : " << txStats.SendSize << " byte\n";
        oss << "  Throughput : " << setw(7) << setprecision(3) << txTps << " Mbps\n";//   (" << GetMinTxRate << ")";
        minTxRateLabel.Text = oss.str();
        oss.str("");

        // Mbit
        double rxSize = txStats.ReceiveSize * 8.0f / (1024.0f * 1024.0f);
        // msec
        uint64_t rxTime = (txStats.LastReceiveTime - txStats.FirstReceiveTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
        double rxTps = 1.0f * rxSize / rxTime;
        oss << "[Rx]\n";
        oss << "  Rx Count   : " << txStats.ReceiveCount << " packets\n";
        oss << "  Rx Error   : " << txStats.ReceiveErrorCount << "\n";
        oss << "  Rx Size    : " << txStats.ReceiveSize << " byte\n";
        oss << "  Throughput : " << setw(7) << setprecision(3) << rxTps << " Mbps\n";

        minRxRateLabel.Text = oss.str();
        oss.str("");

        map<uint64_t, string>& sources = target->GetSources();
        map<uint64_t, string>::iterator iter = sources.begin();
        while(iter != sources.end())
        {
            // MACAddress, Throughput, PLR, Latency(Avg), Latency(Max), Latency(Histogram)
            oss << "   [" << iter->second << "] "
                << "Last no. " << setw(5) << rxDetailedStats[iter->first]->LastSequenceNo << ", "
                << "Lost " << setw(4) << rxDetailedStats[iter->first]->Errors.GetErrorCount() << ", "
                << "TPS " << setw(7) << setprecision(3) << rxDetailedStats[iter->first]->Throughput.GetThroughput() << " Mbps, "
                << "PLR " << setw(7) << setprecision(3) << rxDetailedStats[iter->first]->Errors.GetErrorRate() * 100.f << " %%, "
                << "Avg lat. " << fixed << setw(4) << setprecision(1) << rxDetailedStats[iter->first]->Latency.GetAvgLatency() << " ms, "
                << "Max lat. " << setw(3) << rxDetailedStats[iter->first]->Latency.GetMaxLatency() << " ms, "
                << "[";

            LatencyCounter& latencyCounter = rxDetailedStats[iter->first]->Latency;
            uint8_t lastIndex = latencyCounter.GetLastIndex();
            for(int i=0; i<lastIndex - 1; ++i)
            {
                oss << "(<" << i + 1 << "ms):" << latencyCounter.GetBurstLatencyCount(i) << ", ";
            }
            oss << "(>=" << lastIndex - 1 << "ms):" << latencyCounter.GetBurstLatencyCount(lastIndex - 1);
            oss << "]\n";

            iter++;
        }

        detailedRxLabel.Text = oss.str();
        oss.str("");

        str = (target->GetPerformanceTestSuccess()) ?
            "  << PERFORMANCE TEST : OK >>\n" :
            "  << PERFORMANCE TEST : NG >>\n";
        oss << str;
        performanceLabel.Text = oss.str();
        oss.str("");

        SceneViewer<LocalClientPerformanceTester>::ShowImpl(display);
    }

    void PrintResult(string& key, string& value)
    {
        ostringstream osk;
        ostringstream osv;

        osk << "Test Time(sec), ";
        osv << target->GetTestTime().GetSeconds() << ", ";

        osk << "Test Margin(sec), ";
        osv << target->GetMarginTime().GetSeconds() << ", ";

        osk << "Target Tx Throughput(bps), ";
        osv << target->GetMinTxRate() << ", ";

        osk << "Target Rx Throughput(bps), ";
        osv << target->GetMinRxRate() << ", ";

        osk << "Target PLR(%%), ";
        osv << target->GetMaxPlr() * 100.0f << ", ";

        osk << "Target Max Latency(ms), ";
        osv << target->GetMaxLatencyMax() << ", ";

        osk << "Target Avg Latency(ms), ";
        osv << target->GetMaxLatencyAvg() << ", ";



        osk << "Test Time(sec), ";
        osv << target->GetElapsedTime().GetSeconds() << ", ";

        osk << "Connect() (ms), ";
        osv << target->GetConnectionTime() << ", ";

        NodeStatistics& txStats = target->GetTxStatistics();
        map<uint64_t, MultiDataSink::Statistics*>& rxDetailedStats = target->GetRxDetailedStatistics();

        // bit
        double txSize = txStats.SendSize * 8.0f;
        // msec
        uint64_t txTime = (txStats.LastSendTime - txStats.FirstSendTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
        double txTps = 1.0f * txSize / txTime;

        osk << "Tx Count, ";
        osv << txStats.SendCount << ", ";

        osk << "Tx Size(byte), ";
        osv << txStats.SendSize << ", ";

        osk << "Tx Error, ";
        osv << txStats.SendErrorCount << ", ";

        osk << "Tx Throughput(bps), ";
        osv << txTps << ", ";


        map<uint64_t, string>& sources = target->GetSources();
        map<uint64_t, string>::iterator iter = sources.begin();
        osk << "Rx Nodes, ";
        osv << sources.size() << ", ";

        while(iter != sources.end())
        {
            osk << "MAC Address, Rx Throughput(bps), Rx PLR(%%), Avg Latency(ms), Max Latency(ms),";
            LatencyCounter& latencyCounter = rxDetailedStats[iter->first]->Latency;
            uint8_t lastIndex = latencyCounter.GetLastIndex();
            for(int i=0; i<lastIndex - 1; ++i)
            {
                osk << "Rx Latency(<" << i + 1<< "ms), ";
            }
            osk << "Rx Latency(>=" << lastIndex - 1 << "ms), ";

            // MACAddress, Throughput, PLR, Latency(Avg), Latency(Max), Latency(Histogram)
            osv << iter->second << ", "
                << rxDetailedStats[iter->first]->Throughput.GetThroughput() << ", "
                << rxDetailedStats[iter->first]->Errors.GetErrorRate() * 100.f << ", "
                << rxDetailedStats[iter->first]->Latency.GetAvgLatency() << ", "
                << rxDetailedStats[iter->first]->Latency.GetMaxLatency() << ", ";

            for(int i=0; i<lastIndex; ++i)
            {
                osv << latencyCounter.GetBurstLatencyCount(i) << ", ";
            }

            iter++;
        }

        osk << "Connectivity Result, ";
        osv << ((target->GetConnectionTestSuccess()) ? "Success" : "Failure") << ", ";

        osk << "Performance Result";
        osv << ((target->GetPerformanceTestSuccess()) ? "Success" : "Failure");

        // NN_LOG(osk.str().c_str());
        // NN_LOG("\n");
        // NN_LOG(osv.str().c_str());
        // NN_LOG("\n");
        key = osk.str();
        value = osv.str();
    }

protected:
private:

};


/*---------------------------------------------------------------------------
           SocketResultViewer
---------------------------------------------------------------------------*/
class SocketResultViewer : public SceneViewer<SocketPerformanceTester>
{
public:
protected:
private:
    Label titleLabel;

    Label elapsedTimeLabel;
    Label testTimeLabel;
    Label connectionLabel;

    Label minTxRateLabel;
    Label minRxRateLabel;
    Label maxRxPlrLabel;
    Label performanceLabel;

public:
    SocketResultViewer()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(titleLabel);
        titleLabel.X = DISPLAY_TITLE_START_X;
        titleLabel.Y = DISPLAY_TITLE_START_Y;
        titleLabel.Text = "Result\n";
        titleLabel.FitSize();

        uint32_t py = y;
        Add(elapsedTimeLabel);
        elapsedTimeLabel.X = x;
        elapsedTimeLabel.Y = y;
        elapsedTimeLabel.Text = "Elapsed Time   : ";
        elapsedTimeLabel.FitSize();
        elapsedTimeLabel.Width += 1000;

        Add(testTimeLabel);
        testTimeLabel.X = x;
        testTimeLabel.Y = (y += h);
        testTimeLabel.Text = "Test Time      : ";
        testTimeLabel.FitSize();
        testTimeLabel.Width += 1000;

        y += h;

        Add(connectionLabel);
        connectionLabel.X = x;
        connectionLabel.Y = (y += h);
        connectionLabel.Text = "<< >>";
        connectionLabel.FitSize();
        connectionLabel.Width += 1000;

        py = y;
        Add(minTxRateLabel);
        minTxRateLabel.X = x;
        minTxRateLabel.Y = (y += 3 * h);
        minTxRateLabel.Text = "Tx Throughput : ";
        minTxRateLabel.FitSize();
        minTxRateLabel.Width += 3000;

        y = py;
        Add(minRxRateLabel);
        minRxRateLabel.X = x + 600;
        minRxRateLabel.Y = (y += 3 * h);
        minRxRateLabel.Text = "Rx Throughput : ";
        minRxRateLabel.FitSize();
        minRxRateLabel.Width += 3000;

        Add(performanceLabel);
        performanceLabel.X = x;
        performanceLabel.Y = (y += 5 * h);
        performanceLabel.Text = "<< >>";
        performanceLabel.FitSize();
        performanceLabel.Width += 1000;
    }

    virtual void ShowImpl(Display& display)
    {
        ostringstream oss;

        string str;
        nn::TimeSpan ts = target->GetElapsedTime();
        if(ts == nn::TimeSpan(0))
        {
            str = "-";
        }
        else
        {
            str = ToString(ts);
        }
        oss << "Elapsed Time   : " << str;
        elapsedTimeLabel.Text = oss.str();
        oss.str("");

        oss << "Test Time      : " << ToString(target->GetTestTime());
        testTimeLabel.Text = oss.str();
        oss.str("");

        str = (target->GetConnectionTestSuccess()) ?
            "\n  << CONNECTION TEST : OK >>\n" :
            "\n  << CONNECTION TEST : NG >>\n";

        oss << str;
        connectionLabel.Text = oss.str();
        oss.str("");

        // ---

        NodeStatistics& txStats = target->GetTxStatistics();
        //ReceiverStatistics& rxStats = target->GetRxStatistics();
        //map<uint64_t, MultiDataSink::Statistics*>& rxDetailedStats = target->GetRxDetailedStatistics();

        // Mbit
        double txSize = txStats.SendSize * 8.0f / (1024.0f * 1024.0f);
        // msec
        uint64_t txTime = (txStats.LastSendTime - txStats.FirstSendTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
        double txTps = 1.0f * txSize / txTime;

        /*
          // 将来的な表示
          Tx Throughput > 500.0Mbps
            Result : [OK] 550.0Mbps

          Rx Throughput > 500.0Mbps
            Result : [NG] 20.0Mbps [aa:aa:aa:aa:aa:aa]
            Result : [OK] 503.0Mbps [bb:aa:aa:aa:aa:aa]
        */

        oss << "[Tx]\n";
        oss << "  Tx Count   : " << txStats.SendCount << " packets\n";
        oss << "  Tx Error   : " << txStats.SendErrorCount << "\n";
        oss << "  Tx Size    : " << txStats.SendSize << " byte\n";
        oss << "  Throughput : " << setw(7) << setprecision(3) << txTps << " Mbps\n";//   (" << GetMinTxRate << ")";
        minTxRateLabel.Text = oss.str();
        oss.str("");

        // Mbit
        double rxSize = txStats.ReceiveSize * 8.0f / (1024.0f * 1024.0f);
        // msec
        uint64_t rxTime = (txStats.LastReceiveTime - txStats.FirstReceiveTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
        double rxTps = 1.0f * rxSize / rxTime;
        oss << "[Rx]\n";
        oss << "  Rx count   : " << txStats.ReceiveCount << " packets\n";
        oss << "  Rx lost    : " << target->GetError().GetErrorCount() << "\n";
        oss << "  Rx error   : " << txStats.ReceiveErrorCount << "\n";
        oss << "  Rx size    : " << txStats.ReceiveSize << " byte\n";
        oss << "  Throughput : " << setw(7) << setprecision(3) << rxTps << " Mbps\n";
        oss << "  PLR        : " << setw(7) << setprecision(3) << target->GetError().GetErrorRate() * 100.f << " %%\n";

        minRxRateLabel.Text = oss.str();
        oss.str("");

        str = (target->GetPerformanceTestSuccess()) ?
            "  << PERFORMANCE TEST : OK >>\n" :
            "  << PERFORMANCE TEST : NG >>\n";
        oss << str;
        performanceLabel.Text = oss.str();
        oss.str("");

        SceneViewer<SocketPerformanceTester>::ShowImpl(display);
    }

    void PrintResult(string& key, string& value)
    {
        ostringstream osk;
        ostringstream osv;

        osk << "Test Time(sec), ";
        osv << target->GetTestTime().GetSeconds() << ", ";

        osk << "Test Margin(sec), ";
        osv << target->GetMarginTime().GetSeconds() << ", ";

        osk << "Target Tx Throughput(bps), ";
        osv << target->GetMinTxRate() << ", ";

        osk << "Target Rx Throughput(bps), ";
        osv << target->GetMinRxRate() << ", ";

        osk << "Target PLR(%%), ";
        osv << target->GetMaxPlr() * 100.0f << ", ";


        osk << "Test Time(sec), ";
        osv << target->GetElapsedTime().GetSeconds() << ", ";

        NodeStatistics& txStats = target->GetTxStatistics();

        // bit
        double txSize = txStats.SendSize * 8.0f;
        // msec
        uint64_t txTime = (txStats.LastSendTime - txStats.FirstSendTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
        double txTps = 1.0f * txSize / txTime;

        osk << "Tx Count, ";
        osv << txStats.SendCount << ", ";

        osk << "Tx Size(byte), ";
        osv << txStats.SendSize << ", ";

        osk << "Tx Error, ";
        osv << txStats.SendErrorCount << ", ";

        osk << "Tx Throughput(bps), ";
        osv << txTps << ", ";


        // Mbit
        double rxSize = txStats.ReceiveSize * 8.0f / (1024.0f * 1024.0f);
        // msec
        uint64_t rxTime = (txStats.LastReceiveTime - txStats.FirstReceiveTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
        double rxTps = 1.0f * rxSize / rxTime;
        osk << "Rx Count, ";
        osv << txStats.ReceiveCount << ", ";

        osk << "Rx Lost Count, ";
        osv << target->GetError().GetErrorCount() << ", ";

        osk << "Rx Error, ";
        osv << txStats.ReceiveErrorCount << ", ";

        osk << "Rx Size(byte), ";
        osv << txStats.ReceiveSize << ", ";

        osk << "Rx Error, ";
        osv << txStats.ReceiveErrorCount << ", ";

        osk << "Rx Throughput(bps), ";
        osv << txTps << ", ";

        osk << "Rx PLR(%%), ";
        osv << target->GetError().GetErrorRate() * 100.f << ", ";


        osk << "Connectivity Result, ";
        osv << ((target->GetConnectionTestSuccess()) ? "Success" : "Failure") << ", ";

        osk << "Performance Result";
        osv << ((target->GetPerformanceTestSuccess()) ? "Success" : "Failure");

        // NN_LOG(osk.str().c_str());
        // NN_LOG("\n");
        // NN_LOG(osv.str().c_str());
        // NN_LOG("\n");
        key = osk.str();
        value = osv.str();
    }

protected:
private:

};

}    // WlanTest
