﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "ModelController.h"
#include "Tester.h"
#include "Counter.h"

namespace WlanTest {

void GenerateLoggingInterval(SelectableValue<NamedValue<nn::TimeSpan> >* sv, const nn::TimeSpan& initialValue);
void GenerateTestTime(SelectableValue<NamedValue<nn::TimeSpan> >* sv, const nn::TimeSpan& initialValue);
void GenerateTestMargin(SelectableValue<NamedValue<nn::TimeSpan> >* sv, const nn::TimeSpan& initialValue);
void GenerateTxStartDelay(SelectableValue<NamedValue<nn::TimeSpan> >* sv, const nn::TimeSpan& initialValue);
void GenerateMinTxRxRate(SelectableValue<NamedValue<uint32_t> >* sv, const uint32_t& initialValue);
void GenerateMaxPlr(SelectableValue<NamedValue<double> >* sv, const double& initialValue);
void GenerateMaxLatency(SelectableValue<NamedValue<uint32_t> >* sv, const double& initialValue);
void GeneratePacketLifetime(SelectableValue<NamedValue<uint32_t> >* sv, const uint32_t& initialValue);
void GenerateUpdateInterval(SelectableValue<NamedValue<nn::TimeSpan> >* sv, const nn::TimeSpan& initialValue);
void GenerateWlanMode(SelectableValue<NamedValue<nn::btm::WlanMode>>* sv, const nn::btm::WlanMode& initialValue);

class Test;

/*---------------------------------------------------------------------------
          TestController
---------------------------------------------------------------------------*/
class TestController : public SceneController<Test>
{
public:
protected:

    Label title;
    ValueSelector selector;

private:

    SelectableValue<NamedValue<nn::TimeSpan> > loggingInterval;
    SelectableValue<NamedValue<bool> > packetDump;

public:

    TestController()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Log Parameters";
        title.FitSize();

        Add(selector);
        selector.X = x;
        selector.Y = y;
        selector.Height = 450;

        selector.Register("Logging Interval", &loggingInterval);
        GenerateLoggingInterval(&loggingInterval, nn::TimeSpan::FromSeconds(1));

#ifdef ENABLE_DUMP_PACKET
        selector.Register("Packet Dump(SD Card)", &packetDump);
#endif
        GenerateSelectableOnOffValue(&packetDump, false);
    }

    virtual ~TestController()
    {}

    virtual void Set();
};

/*---------------------------------------------------------------------------
          LocalMasterConnectivityTesterController
---------------------------------------------------------------------------*/
class LocalMasterConnectivityTesterController : public SceneController<LocalMasterConnectivityTester>
{
public:
protected:

    Label title;
    ValueSelector selector;

private:

    SelectableValue<NamedValue<nn::TimeSpan> > testTime;
    SelectableValue<NamedValue<nn::TimeSpan> > marginTime;

public:

    LocalMasterConnectivityTesterController()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Test Parameters";
        title.FitSize();

        Add(selector);
        selector.X = x;
        selector.Y = y;
        selector.Height = 450;

        selector.Register("Test Time", &testTime);
        GenerateTestTime(&testTime, InfinityTime);

        selector.Register("Test Margin", &marginTime);
        GenerateTestMargin(&marginTime, nn::TimeSpan::FromSeconds(3));
    }

    virtual ~LocalMasterConnectivityTesterController()
    {}

    virtual void Set()
    {
        target->SetTestTime(testTime);
        target->SetMarginTime(marginTime);
    }
};

/*---------------------------------------------------------------------------
          LocalClientConnectivityTesterController
---------------------------------------------------------------------------*/
class LocalClientConnectivityTesterController : public SceneController<LocalClientConnectivityTester>
{
public:
protected:
    Label title;
    ValueSelector selector;

private:
    SelectableValue<NamedValue<nn::TimeSpan> > testTime;
    SelectableValue<NamedValue<nn::TimeSpan> > marginTime;

public:

    LocalClientConnectivityTesterController()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Test Parameters";
        title.FitSize();

        Add(selector);
        selector.X = x;
        selector.Y = y;
        selector.Height = 450;

        selector.Register("Test Time", &testTime);
        GenerateTestTime(&testTime, InfinityTime);

        selector.Register("Test Margin", &marginTime);
        GenerateTestMargin(&marginTime, nn::TimeSpan::FromSeconds(3));
    };

    virtual ~LocalClientConnectivityTesterController()
    {}

    virtual void Set()
    {
        target->SetTestTime(testTime);
        target->SetMarginTime(marginTime);
    }
};

/*---------------------------------------------------------------------------
          LocalMasterPerformanceTesterController
---------------------------------------------------------------------------*/
class LocalMasterPerformanceTesterController : public SceneController<LocalMasterPerformanceTester>
{
public:
protected:

    Label title;
    ValueSelector selector;

private:

    SelectableValue<NamedValue<nn::TimeSpan> > testTime;
    SelectableValue<NamedValue<nn::TimeSpan> > marginTime;

    SelectableValue<NamedValue<uint32_t> > minTxRate;
    SelectableValue<NamedValue<uint32_t> > minRxRate;
    SelectableValue<NamedValue<double> >   maxPlr;
    SelectableValue<NamedValue<uint32_t> > maxLatencyMax;
    SelectableValue<NamedValue<uint32_t> > maxLatencyAvg;

    SelectableValue<NamedValue<double> >   maxBtPlr;

    SelectableValue<NamedValue<uint32_t> > packetLifetime;
    SelectableValue<NamedValue<nn::TimeSpan> > updateInterval;
    SelectableValue<NamedValue<nn::btm::WlanMode> > wlanMode;

    SelectableValue<NamedValue<bool> >     controllerOff;

public:

    LocalMasterPerformanceTesterController()
    {
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Test Parameters";
        title.FitSize();

        Add(selector);
        selector.X = x;
        selector.Y = y;
        selector.Height = 450;

        selector.Register("Test Time", &testTime);
        GenerateTestTime(&testTime, InfinityTime);

        selector.Register("Test Margin", &marginTime);
        GenerateTestMargin(&marginTime, nn::TimeSpan::FromSeconds(3));

        selector.Register("RC Disconnection", &controllerOff);
        GenerateSelectableOnOffValue(&controllerOff, false);

        selector.Register("Min Tx Rate", &minTxRate);
        GenerateMinTxRxRate(&minTxRate, 0);

        selector.Register("Min Rx Rate", &minRxRate);
        GenerateMinTxRxRate(&minRxRate, 0);

        selector.Register("Max PLR", &maxPlr);
        GenerateMaxPlr(&maxPlr, -100.0f);

        selector.Register("Max Latency(Avg)", &maxLatencyAvg);
        GenerateMaxLatency(&maxLatencyAvg, 0);

        selector.Register("Max Latency(Max)", &maxLatencyMax);
        GenerateMaxLatency(&maxLatencyMax, 0);

        selector.Register("Packet Lifetime(TTL)", &packetLifetime);
        GeneratePacketLifetime(&packetLifetime, LatencyCounter::Lifetime_Infinity);

        selector.Register("Update Interval (WLAN Rx)", &updateInterval);
        GenerateUpdateInterval(&updateInterval, nn::TimeSpan::FromSeconds(1));

        selector.Register("Btm Wlan Mode", &wlanMode);
        GenerateWlanMode(&wlanMode, nn::btm::WlanMode_Local4);

        //BT は表示しないが、自動テストのために値の設定は行っておく
        //selector.Register("Max BT PLR", &maxBtPlr);
        GenerateMaxPlr(&maxBtPlr, -100.0f);
    }

    virtual ~LocalMasterPerformanceTesterController()
    {}

    virtual void Set()
    {
        TestParam param;

        param.testTime        = testTime;
        param.testMargin      = marginTime;
        param.controllerOff   = controllerOff;
        param.wlanMode        = wlanMode;

        param.minTxRate       = minTxRate;
        param.minRxRate       = minRxRate;
        param.maxPlr          = maxPlr;
        param.maxLatencyMax   = maxLatencyMax;
        param.maxLatencyAvg   = maxLatencyAvg;

        param.maxBtPlr        = maxBtPlr;

        target->SetTestParam(param);
        target->SetTargetParam();
        target->SetTxParam();
        LatencyCounter::SetLifetime(packetLifetime);
        MultiDataSink::Statistics::UpdateInterval = updateInterval;
    }

    virtual void Set(const TestParam& testParam, const MasterParam& masterParam, const DataGeneratorParam& txParam)
    {
        target->SetTestParam(testParam);
        target->SetTargetParam(masterParam);
        target->SetTxParam(txParam);
    }
};

/*---------------------------------------------------------------------------
          LocalClientPerformanceTesterController
---------------------------------------------------------------------------*/
class LocalClientPerformanceTesterController : public SceneController<LocalClientPerformanceTester>
{
public:
protected:
    Label title;
    ValueSelector selector;

private:
    SelectableValue<NamedValue<nn::TimeSpan> > testTime;
    SelectableValue<NamedValue<nn::TimeSpan> > marginTime;

    SelectableValue<NamedValue<uint32_t> > minTxRate;
    SelectableValue<NamedValue<uint32_t> > minRxRate;
    SelectableValue<NamedValue<double> >   maxPlr;
    SelectableValue<NamedValue<uint32_t> > maxLatencyMax;
    SelectableValue<NamedValue<uint32_t> > maxLatencyAvg;
    SelectableValue<NamedValue<nn::TimeSpan> > txStartDelay;

    SelectableValue<NamedValue<double> >   maxBtPlr;

    SelectableValue<NamedValue<uint32_t> > packetLifetime;
    SelectableValue<NamedValue<nn::TimeSpan> > updateInterval;
    SelectableValue<NamedValue<nn::btm::WlanMode> > wlanMode;

    SelectableValue<NamedValue<bool> >     controllerOff;

public:

    LocalClientPerformanceTesterController()
    {
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Test Parameters";
        title.FitSize();

        Add(selector);
        selector.X = x;
        selector.Y = y;
        selector.Height = 450;

        selector.Register("Test Time", &testTime);
        GenerateTestTime(&testTime, InfinityTime);

        selector.Register("Test Margin", &marginTime);
        GenerateTestMargin(&marginTime, nn::TimeSpan::FromSeconds(3));

        selector.Register("Tx Start Delay", &txStartDelay);
        GenerateTxStartDelay(&txStartDelay, 0);

        selector.Register("RC Disconnection", &controllerOff);
        GenerateSelectableOnOffValue(&controllerOff, false);

        selector.Register("Min Tx Rate", &minTxRate);
        GenerateMinTxRxRate(&minTxRate, 0);

        selector.Register("Min Rx Rate", &minRxRate);
        GenerateMinTxRxRate(&minRxRate, 0);

        selector.Register("Max PLR", &maxPlr);
        GenerateMaxPlr(&maxPlr, -100.0f);

        selector.Register("Max Latency(Avg)", &maxLatencyAvg);
        GenerateMaxLatency(&maxLatencyAvg, 0);

        selector.Register("Max Latency(Max)", &maxLatencyMax);
        GenerateMaxLatency(&maxLatencyMax, 0);

        selector.Register("Packet Lifetime(TTL)", &packetLifetime);
        GeneratePacketLifetime(&packetLifetime, LatencyCounter::Lifetime_Infinity);

        selector.Register("Update Interval (WLAN Rx)", &updateInterval);
        GenerateUpdateInterval(&updateInterval, nn::TimeSpan::FromSeconds(1));

        selector.Register("Btm Wlan Mode", &wlanMode);
        GenerateWlanMode(&wlanMode, nn::btm::WlanMode_Local4);

        //BT は表示しないが、自動テストのために値の設定は行っておく
        //selector.Register("Max BT PLR", &maxBtPlr);
        GenerateMaxPlr(&maxBtPlr, -100.0f);
    }

    virtual ~LocalClientPerformanceTesterController()
    {}

    virtual void Set()
    {
        TestParam param;

        param.testTime        = testTime;
        param.testMargin      = marginTime;
        param.txStartDelay    = txStartDelay;
        param.controllerOff   = controllerOff;
        param.wlanMode        = wlanMode;

        param.minTxRate       = minTxRate;
        param.minRxRate       = minRxRate;
        param.maxPlr          = maxPlr;
        param.maxLatencyMax   = maxLatencyMax;
        param.maxLatencyAvg   = maxLatencyAvg;

        param.maxBtPlr        = maxBtPlr;

        target->SetTestParam(param);
        target->SetTargetParam();
        target->SetTxParam();
        LatencyCounter::SetLifetime(packetLifetime);
        MultiDataSink::Statistics::UpdateInterval = updateInterval;
    }

    virtual void Set(const TestParam& testParam, const ClientParam& clientParam, const DataGeneratorParam& txParam)
    {
        target->SetTestParam(testParam);
        target->SetTargetParam(clientParam);
        target->SetTxParam(txParam);
    }
};

/*---------------------------------------------------------------------------
          SocketPerformanceTesterController
---------------------------------------------------------------------------*/
class SocketPerformanceTesterController : public SceneController<SocketPerformanceTester>
{
public:
protected:
    Label title;
    ValueSelector selector;

private:
    SelectableValue<NamedValue<nn::TimeSpan> > testTime;
    SelectableValue<NamedValue<nn::TimeSpan> > marginTime;

    SelectableValue<NamedValue<uint32_t> > minTxRate;
    SelectableValue<NamedValue<uint32_t> > minRxRate;
    SelectableValue<NamedValue<double> >   maxPlr;

    SelectableValue<NamedValue<double> >   maxBtPlr;

    SelectableValue<NamedValue<bool> >     controllerOff;

public:

    SocketPerformanceTesterController()
    {
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Test Parameters";
        title.FitSize();

        Add(selector);
        selector.X = x;
        selector.Y = y;
        selector.Height = 450;

        selector.Register("Test Time", &testTime);
        GenerateTestTime(&testTime, InfinityTime);

        selector.Register("Test Margin", &marginTime);
        GenerateTestMargin(&marginTime, nn::TimeSpan::FromSeconds(3));

        selector.Register("RC Disconnection", &controllerOff);
        GenerateSelectableOnOffValue(&controllerOff, false);

        selector.Register("Min Tx Rate", &minTxRate);
        GenerateMinTxRxRate(&minTxRate, 0);

        selector.Register("Min Rx Rate", &minRxRate);
        GenerateMinTxRxRate(&minRxRate, 0);

        selector.Register("Max PLR", &maxPlr);
        GenerateMaxPlr(&maxPlr, -100.0f);

        //BT は表示しないが、自動テストのために値の設定は行っておく
        //selector.Register("Max BT PLR", &maxBtPlr);
        GenerateMaxPlr(&maxBtPlr, -100.0f);
    }

    virtual ~SocketPerformanceTesterController()
    {}

    virtual void Set()
    {
        TestParam param;

        param.testTime        = testTime;
        param.testMargin      = marginTime;
        param.controllerOff   = controllerOff;

        param.minTxRate       = minTxRate;
        param.minRxRate       = minRxRate;
        param.maxPlr          = maxPlr;
        param.maxLatencyMax   = 0;
        param.maxLatencyAvg   = 0;

        param.maxBtPlr        = maxBtPlr;

        target->SetTestParam(param);
        target->SetTargetParam();
        target->SetTxParam();
    }

    virtual void Set(const TestParam& testParam, const SocketParam& socketParam, const DataGeneratorParam& txParam)
    {
        target->SetTestParam(testParam);
        target->SetTargetParam(socketParam);
        target->SetTxParam(txParam);
    }
};

/*---------------------------------------------------------------------------
          HdDetectorPerformanceTesterController
---------------------------------------------------------------------------*/
class HdDetectorPerformanceTesterController : public SceneController<HostDrivenDetectorPerformanceTester>
{
public:
protected:
    Label title;
    ValueSelector selector;

private:
    SelectableValue<NamedValue<nn::TimeSpan> > testTime;

public:

    HdDetectorPerformanceTesterController()
    {
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Test Parameters";
        title.FitSize();

        Add(selector);
        selector.X = x;
        selector.Y = y;
        selector.Height = 450;

        selector.Register("Test Time", &testTime);
        GenerateTestTime(&testTime, InfinityTime);
    }

    virtual ~HdDetectorPerformanceTesterController()
    {}

    virtual void Set()
    {
        target->SetTestTime(testTime);
        target->SetMarginTime(0);
    }
};



}        // WlanTest
