﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>

#include "ModelController.h"

namespace WlanTest {

enum TestResult
{
    RESULT_NOT_COMPLETED = 0x0,
    RESULT_SUCCESS,
    RESULT_FAILURE
};

TestResult ToTestResult(bool isSuccess);
TestResult operator&(const TestResult& t1, const TestResult& t2);
TestResult operator|(const TestResult& t1, const TestResult& t2);
TestResult& operator&=(TestResult& t1, const TestResult& t2);
TestResult& operator|=(TestResult& t1, const TestResult& t2);

/*---------------------------------------------------------------------------
          TestParam
---------------------------------------------------------------------------*/
class TestParam
{
public :
    enum ItemId
    {
        ITEM_ID_UNDEFINED = 0,

        // common
        ITEM_ID_TEST_TIME,
        ITEM_ID_TEST_MARGIN,

        // master & client
        ITEM_ID_MIN_RX_RATE,
        ITEM_ID_MAX_PLR,
        ITEM_ID_MAX_LATENCY_MAX,
        ITEM_ID_MAX_LATENCY_AVG,
        ITEM_ID_WLAN_MODE,

        ITEM_ID_MIN_TX_RATE,

        // client
        ITEM_ID_TX_START_DELAY,
    };

    static const char* ITEM_STR_TEST_TIME;
    static const char* ITEM_STR_TEST_MARGIN;
    static const char* ITEM_STR_MIN_TX_RATE;
    static const char* ITEM_STR_MIN_RX_RATE;
    static const char* ITEM_STR_MAX_PLR;
    static const char* ITEM_STR_MAX_LATENCY_MAX;
    static const char* ITEM_STR_MAX_LATENCY_AVG;
    static const char* ITEM_STR_WLAN_MODE;
    static const char* ITEM_STR_TX_START_DELAY;

    bool         isAutoTest;

    nn::TimeSpan testTime;
    nn::TimeSpan testMargin;
    uint32_t     minTxRate;
    uint32_t     minRxRate;
    double       maxPlr;
    uint32_t     maxLatencyMax;
    uint32_t     maxLatencyAvg;
    double       maxBtPlr;
    bool         controllerOff;

    nn::btm::WlanMode  wlanMode;
    nn::TimeSpan       txStartDelay;

    TestParam()
    {
        isAutoTest       = false;
        testTime         = nn::TimeSpan::FromMinutes(10);
        testMargin       = nn::TimeSpan::FromSeconds(5);
        minTxRate        = 0;
        minRxRate        = 0;
        maxPlr           = 0;
        maxLatencyMax    = 0;
        maxLatencyAvg    = 0;
        wlanMode         = nn::btm::WlanMode_None;
        txStartDelay     = nn::TimeSpan::FromMilliSeconds(0);
        controllerOff    = false;
    }

    void Print()
    {
        NN_LOG("  Test time                : %s\n", ToString(testTime).c_str());
        NN_LOG("  Test margin              : %s\n", ToString(testMargin).c_str());
        NN_LOG("  Min TX rate              : %sbps\n", ToStringForBinaryPrefix(minTxRate).c_str());
        NN_LOG("  Min RX rate              : %sbps\n", ToStringForBinaryPrefix(minRxRate).c_str());
        NN_LOG("  Max PLR                  : %lf \n", maxPlr);
        NN_LOG("  Max latency(max)         : %u ms\n", maxLatencyMax);
        NN_LOG("  Max latency(avg)         : %u ms\n", maxLatencyAvg);
        NN_LOG("  Wlan Mode                : %d (0:Local4, 1:Local8, 2:None, 3:User8)\n", wlanMode);
        NN_LOG("  Tx start delay           : %u ms\n", txStartDelay);
    }

    ItemId GetItemId(const string& item)
    {
        if(strcmp(item.c_str(), ITEM_STR_TEST_TIME) == 0)       return ITEM_ID_TEST_TIME;
        if(strcmp(item.c_str(), ITEM_STR_TEST_MARGIN) == 0)     return ITEM_ID_TEST_MARGIN;
        if(strcmp(item.c_str(), ITEM_STR_MIN_TX_RATE) == 0)     return ITEM_ID_MIN_TX_RATE;
        if(strcmp(item.c_str(), ITEM_STR_MIN_RX_RATE) == 0)     return ITEM_ID_MIN_RX_RATE;
        if(strcmp(item.c_str(), ITEM_STR_MAX_PLR) == 0)         return ITEM_ID_MAX_PLR;
        if(strcmp(item.c_str(), ITEM_STR_MAX_LATENCY_MAX) == 0) return ITEM_ID_MAX_LATENCY_MAX;
        if(strcmp(item.c_str(), ITEM_STR_MAX_LATENCY_AVG) == 0) return ITEM_ID_MAX_LATENCY_AVG;
        if(strcmp(item.c_str(), ITEM_STR_WLAN_MODE) == 0)  return ITEM_ID_WLAN_MODE;
        if(strcmp(item.c_str(), ITEM_STR_TX_START_DELAY) == 0)  return ITEM_ID_TX_START_DELAY;

        NN_LOG("  - failed : Unrecognized item name (%s)\n", item.c_str());

        return ITEM_ID_UNDEFINED;
    }

    bool SetParam(const string& item, const string& value)
    {
        ItemId id = GetItemId(item);
        if(id == ITEM_ID_UNDEFINED)
        {
            return false;
        }

        switch(id)
        {
        case ITEM_ID_TEST_TIME       : return SetTestTime(value);
        case ITEM_ID_TEST_MARGIN     : return SetTestMargin(value);
        case ITEM_ID_MIN_TX_RATE     : return SetMinTxRate(value);
        case ITEM_ID_MIN_RX_RATE     : return SetMinRxRate(value);
        case ITEM_ID_MAX_PLR         : return SetMaxPlr(value);
        case ITEM_ID_MAX_LATENCY_MAX : return SetMaxLatencyMax(value);
        case ITEM_ID_MAX_LATENCY_AVG : return SetMaxLatencyAvg(value);
        case ITEM_ID_WLAN_MODE       : return SetWlanMode(value);
        case ITEM_ID_TX_START_DELAY  : return SetTxStartDelay(value);
        default : return false;
        }

        return false;
    }

    bool SetTestTime(const string& valueStr)
    {
        int64_t value;
        char unit;

        if(sscanf(valueStr.c_str(), "%lld %c", &value, &unit) == 2)
        {
            return GenerateTimeSpan(testTime, value, unit);
        }
        else
        {
            return false;
        }
    }

    bool SetTestMargin(const string& valueStr)
    {
        int64_t value;

        if(sscanf(valueStr.c_str(), "%lld", &value) == 1)
        {
            return GenerateTimeSpan(testMargin, value, TIME_UNIT_SECONDS);
        }
        else
        {
            return false;
        }
    }

    bool SetMinTxRate(const string& valueStr)
    {
        int64_t value;
        char unit;

        if(sscanf(valueStr.c_str(), "%lld %c", &value, &unit) == 2)
        {
            return GenerateByteScale(minTxRate, value, unit);
        }
        else
        {
            return false;
        }
    }

    bool SetMinRxRate(const string& valueStr)
    {
        int64_t value;
        char unit;

        if(sscanf(valueStr.c_str(), "%lld %c", &value, &unit) == 2)
        {
            return GenerateByteScale(minRxRate, value, unit);
        }
        else
        {
            return false;
        }
    }

    bool SetMaxPlr(const string& valueStr)
    {
        if(sscanf(valueStr.c_str(), "%lf", &maxPlr) == 1)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    bool SetMaxLatencyAvg(const string& valueStr)
    {
        if(sscanf(valueStr.c_str(), "%u", &maxLatencyAvg) == 1)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    bool SetMaxLatencyMax(const string& valueStr)
    {
        if(sscanf(valueStr.c_str(), "%u", &maxLatencyMax) == 1)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    bool SetWlanMode(const string& valueStr)
    {
        int value;

        if( sscanf(valueStr.c_str(), "%d", &value) == 1 &&
            nn::btm::WlanMode_None <= value && value <= nn::btm::WlanMode_User8 )
        {
            wlanMode = static_cast<nn::btm::WlanMode>(value);
            return true;
        }
        else
        {
            return false;
        }
    }

    bool SetTxStartDelay(const string& valueStr)
    {
        int64_t value;

        if(sscanf(valueStr.c_str(), "%lld", &value) == 1)
        {
            txStartDelay = nn::TimeSpan::FromMilliSeconds(value);
            return true;
        }
        else
        {
            return false;
        }
    }
};

/*---------------------------------------------------------------------------
          ITester
---------------------------------------------------------------------------*/
class ITester
{
private:

protected:

    virtual bool Test()     = 0;

    virtual bool Open()     = 0;
    virtual bool Ready()    = 0;
    virtual bool Start()    = 0;
    virtual bool TestImpl() = 0;
    virtual bool Close()    = 0;
    virtual bool Evaluate() = 0;

public:

    virtual bool Execute()     = 0;
    virtual bool Exit()        = 0;
    virtual bool IsTesting()   = 0;
    virtual TestResult GetResult() = 0;
};

/*---------------------------------------------------------------------------
          Tester
---------------------------------------------------------------------------*/
template<typename T>
class Tester : public ITester, public ModelController<T>
{
private:

    bool m_NeedExit;
    bool m_IsAllFinished;

    nn::os::ThreadType m_TestThread;

    static void TestThreadFunction(void* arg)
    {
        DBG_VLOG("Thread start - TestThread");
        reinterpret_cast<Tester*>(arg)->Test();
        DBG_VLOG("Thread stop - TestThread");
    }

protected:

    bool m_IsAutoTest;
    bool m_IsFinished;
    TestResult m_Result;

    virtual bool Test()
    {
        bool result = true;
        DBG_LOG();

        Initialize();

        Open();
        DBG_LOG();

        while(!Ready())
        {
            if(m_NeedExit)
            {
                Close();
                result = false;
                goto __END__;
            }

            Sleep(nn::TimeSpan::FromMilliSeconds(5));
        }

        DBG_LOG();
        if(!Start())
        {
            NN_LOG("  - failed : Start()\n");
            Close();
            result = false;
            goto __END__;
        }

        DBG_LOG();
        while(!IsFinished())
        {
            if(m_NeedExit)
            {
                Close();
                result = false;
                goto __END__;
            }

            if(!TestImpl())
            {
                break;
            }
            Sleep(nn::TimeSpan::FromMilliSeconds(5));
        }
        DBG_LOG();
        Close();

      __END__:

        DBG_LOG();
        Evaluate();

        m_IsAllFinished = true;

        DBG_LOG();

        return result;
    }

    virtual bool Initialize() = 0; // 変数の初期化など軽い処理を行う
    virtual bool Open() = 0;
    virtual bool Ready() = 0;
    virtual bool Start() = 0;
    virtual bool TestImpl() = 0;
    virtual bool Close() = 0;
    virtual bool Evaluate() = 0;

public:
    Tester()
    {
        m_IsAutoTest = false;
        m_NeedExit = false;
        m_IsFinished = true;
        m_IsAllFinished = true;
        m_Result = RESULT_NOT_COMPLETED;
    }

    virtual ~Tester()
    {
        Exit();
    }

    virtual bool Execute()
    {
        static NN_ALIGNAS(4096) char s_TestThreadStack[4096 * 5];

        if( IsTesting() )
        {
            return false;
        }

        ClearTestFlag();
        nn::Result result = nn::os::CreateThread(&m_TestThread, TestThreadFunction, this, s_TestThreadStack, sizeof(s_TestThreadStack), nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()));
        if( result.IsFailure() )
        {
            NN_LOG("TestThread can not run.\n");
            return false;
        }

        nn::os::SetThreadNamePointer(&m_TestThread, "WitTestThread");
        nn::os::StartThread(&m_TestThread);

        return true;
    }

    virtual bool Exit()
    {
        m_NeedExit = true;

        if(nn::os::ThreadType::State_Initialized <= m_TestThread._state && m_TestThread._state <= nn::os::ThreadType::State_Exited)
        {
            nn::os::DestroyThread(&m_TestThread);
        }

        m_IsFinished = true;
        m_IsAllFinished = true;

        return true;
    }

    bool IsExitNeeded()
    {
        return m_NeedExit;
    }

    bool IsAutoTest()
    {
        return m_IsAutoTest;
    }

    void SetAutoTestFlag(bool flag)
    {
        m_IsAutoTest = flag;
    }

    virtual void SetTestParam(const TestParam& param)
    {
        SetAutoTestFlag(param.isAutoTest);
    }

    virtual bool IsTesting()
    {
        return !m_IsAllFinished;
    }

    virtual TestResult GetResult()
    {
        return m_Result;
    }

    virtual bool IsFinished()
    {
        return m_IsFinished;
    }

    virtual void ClearTestFlag()
    {
        m_NeedExit = false;
        m_IsFinished = false;
        m_IsAllFinished = false;
        m_Result = RESULT_NOT_COMPLETED;
    }
};


/*---------------------------------------------------------------------------
          AgingTester
---------------------------------------------------------------------------*/
template<typename T>
class AgingTester : public Tester<T>
{
private:

protected:

    nn::TimeSpan      m_TestTime;    // テスト時間
    nn::TimeSpan      m_MarginTime;  // テスト時間のマージン(全ノードが余裕をもって指定時間テストするため)
    nn::os::Tick      m_StartTick;   // テスト開始チック
    nn::os::Tick      m_EndTick;     // テスト終了チック
    bool              m_ControllerOff; // テスト開始時のコントローラの状態
    nn::btm::WlanMode m_WlanMode;      // WlanMode の設定

    virtual bool Initialize()
    {
        m_StartTick = nn::os::Tick(0);
        return true;
    }

    virtual bool Open()
    {
        return true;
    }

    virtual bool Ready()
    {
        return true;
    }

    virtual bool Start()
    {
        if( m_ControllerOff )
        {
            // 操作用の Joy-Con を切断する
            Pad::GetInstance().Disconnect();
        }

        m_StartTick = nn::os::GetSystemTick();
        return true;
    }

    virtual bool TestImpl()
    {
        nn::os::Tick tick = nn::os::GetSystemTick();
        if((tick - m_StartTick).ToTimeSpan() >= m_TestTime + m_MarginTime)
        {
            m_EndTick = tick;
            Tester<T>::m_IsFinished = true;
            return true;
        }

        return true;
    }

    virtual bool Close()
    {
        return true;
    }

    virtual bool Evaluate()
    {
        // TestTime 経過していれば OK
        if((m_EndTick - m_StartTick).ToTimeSpan() >= m_TestTime)
        {
            Tester<T>::m_Result = RESULT_SUCCESS;
        }
        else
        {
            Tester<T>::m_Result = RESULT_FAILURE;
        }

        return true;
    }

public:

    AgingTester()
    {
        m_TestTime = nn::TimeSpan::FromSeconds(3);
        m_MarginTime = nn::TimeSpan::FromSeconds(2);
        m_StartTick = nn::os::Tick(0);
        m_EndTick = nn::os::Tick(0);
        m_ControllerOff = false;
    }

    virtual ~AgingTester()
    {
    }

    nn::TimeSpan GetElapsedTime()
    {
        if(m_StartTick == nn::os::Tick(0)) return 0;
        if(Tester<T>::m_IsFinished) return (m_EndTick - m_StartTick).ToTimeSpan();
        return (nn::os::GetSystemTick() - m_StartTick).ToTimeSpan();
    }

    virtual void SetTestParam(const TestParam& param)
    {
        Tester<T>::SetTestParam(param);
        SetTestTime(param.testTime);
        SetMarginTime(param.testMargin);
        SetControllerOff(param.controllerOff);
        SetWlanMode(param.wlanMode);
    }
    void SetTestTime(nn::TimeSpan time) { m_TestTime = time; }
    void SetMarginTime(nn::TimeSpan time) { m_MarginTime = time; }
    void SetControllerOff(bool v) { m_ControllerOff = v; }
    void SetWlanMode(nn::btm::WlanMode wlanMode) { m_WlanMode = wlanMode; }

    nn::TimeSpan GetTestTime() { return m_TestTime; }
    nn::TimeSpan GetMarginTime() { return m_MarginTime; }
    bool GetControllerOff() { return m_ControllerOff; }
};


/*---------------------------------------------------------------------------
           LocalMasterConnectivityTester
---------------------------------------------------------------------------*/
class LocalMasterConnectivityTester : public AgingTester<Master>
{
protected:

    MasterController    m_MasterController;

    bool m_ConnectionTestSuccess;   // テスト中の接続エラーの有無

    bool         m_IsTxReady;
    nn::os::Tick m_ConnectionStartTick;   // 接続開始チック

protected:

    virtual bool Initialize()
    {
        AgingTester<Master>::Initialize();
        m_ConnectionTestSuccess = true;
        //m_MasterController.Set();
        m_ConnectionStartTick = nn::os::Tick(0);
        m_IsTxReady = false;
        NPad::GetInstance().ClearStatistics();
        return true;
    }

    virtual bool Open()
    {
        bool rtn = true;
        nn::Result result;

        // WlanMode の設定を行います
        if( !NPad::GetInstance().SetBtmWlanMode(m_WlanMode) )
        {
            NN_ASSERT(false, " - failed : cannot enter WlanMode(%d)\n", m_WlanMode);
            rtn = false;
        }

        AgingTester<Master>::Open();

        // ノードの初期化
        result = target->Initialize();
        if( result.IsFailure() )
        {
            NN_ASSERT(false, " - failed : cannot initialize...\n");
            rtn = false;
        }

        // 接続監視スレッド開始
        m_MasterController.Start();

        return rtn;
    }

    virtual bool Ready()
    {
        bool isTxReady = false;
        bool isButtonPressed = false;

        if(!AgingTester<Master>::Ready())
        {
            return false;
        }

        // 既接続クライアント数が規定値に達しているかを確認する
        if( target->IsConnectableCountConnected() )
        {
            if(m_ConnectionStartTick == nn::os::Tick(0))
            {
                m_ConnectionStartTick = nn::os::GetSystemTick();
            }

            if(IsAutoTest())
            {
                // 全クライアントが接続してから少し待ったのちに次のステートへ遷移する
                if((nn::os::GetSystemTick() - m_ConnectionStartTick).ToTimeSpan().GetMilliSeconds() > 3000)
                {
                    isButtonPressed = true;
                }
            }
            else
            {
                isTxReady = true;
                // 全クライアントが接続後、A ボタンが押下されたら次のステートへ遷移する
                if( Pad::GetInstance().IsTrigger(Button::A) )
                {
                    isButtonPressed = true;
                }
            }
        }
        else
        {
            m_ConnectionStartTick = nn::os::Tick(0);
        }

        m_IsTxReady = isTxReady;

        return isButtonPressed;
    }

    virtual bool Start()
    {
        m_IsTxReady = false;
        return AgingTester<Master>::Start();
    }

    virtual bool TestImpl()
    {
        AgingTester<Master>::TestImpl();
        if( m_IsFinished )
        {
            return true;
        }

        if( !target->IsConnectableCountConnected() )
        {
            NN_LOG(" - failed : %d connected clients found (expected %d)\n",
                   target->GetConnectedCount(), target->GetConnectableCount());
            m_EndTick = nn::os::GetSystemTick();
            m_IsFinished = true;
            return true;
        }

        return true;
    }

    virtual bool Close()
    {
        bool rtn = true;

        // 10 秒間 or クライアントがいなくなるまで待つ
        nn::os::Tick tick = nn::os::GetSystemTick();
        while(( nn::os::GetSystemTick() - tick).ToTimeSpan().GetMilliSeconds() < 10000 && !IsExitNeeded() )
        {
            if( target->GetConnectedCount() == 0 )
            {
                NN_LOG("All clients are left\n");
                break;
            }
            Sleep(nn::TimeSpan::FromMilliSeconds(10));
        }

        // 接続監視スレッドの終了
        m_MasterController.Stop();

        // 切断
        target->Finalize();

        AgingTester<Master>::Close();

        // Static Local Mode を抜ける
        if( !NPad::GetInstance().SetBtmWlanMode(nn::btm::WlanMode_None) )
        {
            NN_ASSERT(false, " - failed : cannot enter WlanMode\n");
            rtn = false;
        }

        return rtn;
    }

    virtual bool Evaluate()
    {
        AgingTester<Master>::Evaluate();

        // TestTime の間、接続できていれば OK
        if( m_Result == RESULT_SUCCESS )
        {
            m_ConnectionTestSuccess = true;
        }
        else
        {
            m_ConnectionTestSuccess = false;
            m_Result = RESULT_FAILURE;
        }

        return true;
    }

public:
    LocalMasterConnectivityTester()
    {
    }

    virtual ~LocalMasterConnectivityTester() {}

    bool SetTargetParam()
    {
        m_MasterController.Set();
        return true;
    }
    bool SetTargetParam(const MasterParam& param)
    {
        m_MasterController.Set(param);
        return true;
    }

    Page& GetNodeControllerPage() { return m_MasterController; }

    void SetTarget(Master& master)
    {
        ModelController<Master>::SetTarget(master);
        m_MasterController.SetTarget(master);
    }

    int64_t GetConnectionTime() { return target->GetConnectionTime(); }
    uint32_t GetConnectableCount() { return target->GetConnectableCount(); }
    uint32_t GetConnectedCount() { return target->GetConnectedCount(); }
    bool GetConnectionTestSuccess(){ return m_ConnectionTestSuccess; }
    bool IsTxReady(){ return m_IsTxReady; }
};

/*---------------------------------------------------------------------------
           LocalClientConnectivityTester
---------------------------------------------------------------------------*/
class LocalClientConnectivityTester : public AgingTester<Client>
{
protected:
    ClientController m_ClientController;

    bool m_ConnectionTestSuccess;   // テスト中の接続エラーの有無

protected:

    virtual bool Initialize()
    {
        AgingTester<Client>::Initialize();
        m_ConnectionTestSuccess = true;
        NPad::GetInstance().ClearStatistics();
        //m_ClientController.Set();
        return true;
    }

    virtual bool Open()
    {
        bool rtn = true;
        nn::Result result;

        AgingTester<Client>::Open();

        // ノードの初期化
        result = target->Initialize();
        if(result.IsFailure())
        {
            NN_ASSERT(false, " - failed : cannot initialize...\n");
            return false;
        }

        // WlanMode の設定を行います
        if( !NPad::GetInstance().SetBtmWlanMode(m_WlanMode) )
        {
            NN_ASSERT(false, " - failed : cannot enter WlanMode(%d)\n", m_WlanMode);
            rtn = false;
        }

        // 接続監視スレッド開始
        m_ClientController.Start();

        return rtn;
    }

    virtual bool Ready()
    {
        if(!AgingTester<Client>::Ready())
        {
            return false;
        }

        return target->IsConnected();
    }

    virtual bool Start()
    {
        return AgingTester<Client>::Start();
    }

    virtual bool TestImpl()
    {
        AgingTester<Client>::TestImpl();
        if(m_IsFinished)
        {
            return true;
        }

        if(!target->IsConnected())
        {
            NN_LOG(" - failed : connection broken..\n");
            m_EndTick = nn::os::GetSystemTick();
            m_IsFinished = true;
            return true;
        }

        return true;
    }

    virtual bool Close()
    {
        bool rtn = true;

        // 接続監視スレッドの終了
        m_ClientController.Stop();

        // Static Local Mode を抜ける
        if( !NPad::GetInstance().SetBtmWlanMode(nn::btm::WlanMode_None) )
        {
            NN_ASSERT(false, " - failed : cannot enter WlanMode\n");
            rtn = false;
        }

        // 切断
        target->Finalize();

        AgingTester<Client>::Close();

        return rtn;
    }

    virtual bool Evaluate()
    {
        AgingTester<Client>::Evaluate();

        // TestTime の間、接続できていれば OK
        if(m_Result == RESULT_SUCCESS)
        {
            m_ConnectionTestSuccess = true;
        }
        else
        {
            m_ConnectionTestSuccess = false;
            m_Result = RESULT_FAILURE;
        }

        return true;
    }

public:
    LocalClientConnectivityTester()
    {
    }

    virtual ~LocalClientConnectivityTester() {}

    bool SetTargetParam()
    {
        m_ClientController.Set();
        return true;
    }
    bool SetTargetParam(const ClientParam& param)
    {
        m_ClientController.Set(param);
        return true;
    }

    Page& GetNodeControllerPage() { return m_ClientController; }

    void SetTarget(Client& client)
    {
        ModelController<Client>::SetTarget(client);
        m_ClientController.SetTarget(client);
    }

    int64_t GetConnectionTime() { return target->GetConnectionTime(); }
    bool GetConnectionTestSuccess(){ return m_ConnectionTestSuccess; }
};

/*---------------------------------------------------------------------------
           SocketConnectivityTester
---------------------------------------------------------------------------*/
class SocketConnectivityTester : public AgingTester<SocketNode>
{
protected:

    SocketNodeController m_SocketController;

    bool m_ConnectionTestSuccess;   // テスト中の接続エラーの有無

    bool         m_IsTxReady;
    nn::os::Tick m_ConnectionStartTick;   // 接続開始チック

protected:

    virtual bool Initialize()
    {
        AgingTester<SocketNode>::Initialize();
        m_ConnectionTestSuccess = true;
        m_ConnectionStartTick = nn::os::Tick(0);
        m_IsTxReady = false;
        NPad::GetInstance().ClearStatistics();
        NPad::GetInstance().EnableIrConfiguration(true);
        //m_ClientController.Set();
        return true;
    }

    virtual bool Open()
    {
        bool rtn = true;
        nn::Result result;

        AgingTester<SocketNode>::Open();

        // ノードの初期化
        result = target->Initialize();
        if(result.IsFailure())
        {
            NN_ASSERT(false, " - failed : cannot initialize...\n");
            rtn = false;
        }

        // Dynamic 状態に遷移する
        if( !NPad::GetInstance().SetBtmWlanMode(nn::btm::WlanMode_None) )
        {
            NN_ASSERT(false, " - failed : cannot enter WlanMode\n");
            rtn = false;
        }

        // 接続監視スレッド開始
        m_SocketController.Start();

        return rtn;
    }

    virtual bool Ready()
    {
        bool isTxReady = false;
        bool isButtonPressed = false;


        if( !AgingTester<SocketNode>::Ready() )
        {
            return false;
        }

        if( target->IsConnected() )
        {
            if(m_ConnectionStartTick == nn::os::Tick(0))
            {
                m_ConnectionStartTick = nn::os::GetSystemTick();
            }

            if(IsAutoTest())
            {
                // 少し待ったのちに次のステートへ遷移する
                if((nn::os::GetSystemTick() - m_ConnectionStartTick).ToTimeSpan().GetMilliSeconds() > 3000)
                {
                    isButtonPressed = true;
                }
            }
            else
            {
                isTxReady = true;
                // A ボタンが押下されたら次のステートへ遷移する
                if( Pad::GetInstance().IsTrigger(Button::A) )
                {
                    isButtonPressed = true;
                }
            }
        }

        m_IsTxReady = isTxReady;

        return isButtonPressed;
    }

    virtual bool Start()
    {
        m_IsTxReady = false;
        return AgingTester<SocketNode>::Start();
    }

    virtual bool TestImpl()
    {
        AgingTester<SocketNode>::TestImpl();
        if(m_IsFinished)
        {
            return true;
        }

        if(!target->IsConnected())
        {
            NN_LOG(" - failed : connection broken..\n");
            m_EndTick = nn::os::GetSystemTick();
            m_IsFinished = true;
            return true;
        }

        return true;
    }

    virtual bool Close()
    {
        bool rtn = true;

        // 接続監視スレッドの終了
        m_SocketController.Stop();

        // Dynamic 状態に遷移する
        if( !NPad::GetInstance().SetBtmWlanMode(nn::btm::WlanMode_None) )
        {
            NN_ASSERT(false, " - failed : cannot enter WlanMode\n");
            rtn = false;
        }

        // 切断
        target->Finalize();

        AgingTester<SocketNode>::Close();

        NPad::GetInstance().EnableIrConfiguration(false);

        return rtn;
    }

    virtual bool Evaluate()
    {
        AgingTester<SocketNode>::Evaluate();

        // TestTime の間、接続できていれば OK
        if(m_Result == RESULT_SUCCESS)
        {
            m_ConnectionTestSuccess = true;
        }
        else
        {
            m_ConnectionTestSuccess = false;
            m_Result = RESULT_FAILURE;
        }

        return true;
    }

public:

    SocketConnectivityTester()
    {
    }

    virtual ~SocketConnectivityTester() {}

    virtual bool Exit()
    {
        if(target)
        {
            target->CancelConnect();
        }

        return AgingTester::Exit();
    }

    bool SetTargetParam()
    {
        m_SocketController.Set();
        return true;
    }
    bool SetTargetParam(const SocketParam& param)
    {
        m_SocketController.Set(param);
        return true;
    }

    Page& GetNodeControllerPage() { return m_SocketController; }

    void SetTarget(SocketNode& socket)
    {
        ModelController<SocketNode>::SetTarget(socket);
        m_SocketController.SetTarget(socket);
    }

    bool GetConnectionTestSuccess(){ return m_ConnectionTestSuccess; }
    bool IsTxReady(){ return m_IsTxReady; }
    SocketType GetSelectingSocketType(){ return m_SocketController.GetSelectingSocketType(); }
};

/*---------------------------------------------------------------------------
           LocalMasterPerformanceTester
---------------------------------------------------------------------------*/
class LocalMasterPerformanceTester : public LocalMasterConnectivityTester
{

protected:

    DataGeneratorController m_TxController;
    bool m_PerformanceTestSuccess;

    uint32_t m_MinTxRate;
    uint32_t m_MinRxRate;
    double   m_MaxPlr;
    uint32_t m_MaxLatencyMax;
    uint32_t m_MaxLatencyAvg;

    TestResult m_TxRateResult;
    TestResult m_RxRateResult;
    TestResult m_PlrResult;
    TestResult m_MaxLatencyResult;
    TestResult m_AvgLatencyResult;

protected:

    DataGenerator* m_pGenerator;
    MultiDataSink* m_pSink;
    AfTxScanner*   m_pScanner;

    AfScannerController m_AfScannerController;

    virtual bool Initialize()
    {
        LocalMasterConnectivityTester::Initialize();
        m_AfScannerController.Set();
        m_PerformanceTestSuccess = true;
        m_TxRateResult = RESULT_NOT_COMPLETED;
        m_RxRateResult = RESULT_NOT_COMPLETED;
        m_PlrResult = RESULT_NOT_COMPLETED;
        m_MaxLatencyResult = RESULT_NOT_COMPLETED;
        m_AvgLatencyResult = RESULT_NOT_COMPLETED;
        return true;
    }

    virtual bool Open()
    {
        bool opened = LocalMasterConnectivityTester::Open();

        if( opened )
        {
            nn::wlan::WlanState state = nn::wlan::WlanState_Stop;

            // ##wa SIGLO-71665 のワークアラウンド
            // LocalMasterIdle 状態で PutActionFrameOneShot() をして I/F を up させないように
            // LocalMasterBss 状態に遷移するまで待つ
            while( !IsExitNeeded() )
            {
                nn::Result result = nn::wlan::Local::GetState(&state);
                NN_ASSERT(result.IsSuccess());

                if( state == nn::wlan::WlanState_LocalMasterBss )
                {
                    // サーチ開始
                    m_AfScannerController.Start();
                    break;
                }
                Sleep(nn::TimeSpan::FromMilliSeconds(100));
            }
        }

        return opened;
    }

    virtual bool Ready()
    {
        return LocalMasterConnectivityTester::Ready();
    }

    virtual bool Start()
    {
        if(LocalMasterConnectivityTester::Start())
        {
            m_TxController.Start();

            return true;
        }

        return false;
    }

    virtual bool TestImpl()
    {
        return LocalMasterConnectivityTester::TestImpl();
    }

    virtual bool Close()
    {
        m_TxController.Stop();

        // サーチ終了
        m_AfScannerController.Stop();

        nn::os::Tick start = nn::os::GetSystemTick();
        while(!m_TxController.IsIdle())
        {
            Sleep(nn::TimeSpan::FromMilliSeconds(10));
            NN_ASSERT((nn::os::GetSystemTick() - start).ToTimeSpan().GetMilliSeconds() <= 2000);
        }

        return LocalMasterConnectivityTester::Close();
    }

    virtual bool Evaluate()
    {
        LocalMasterConnectivityTester::Evaluate();

        if( m_Result == RESULT_SUCCESS )
        {
            bool success = true;
            NodeStatistics& txStats = GetTxStatistics();
            // bit
            double txSize = txStats.SendSize * 8.0f;
            // msec
            uint64_t txTime = (txStats.LastSendTime - txStats.FirstSendTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
            double txTps = 1.0f * txSize / txTime;

            // min Tx rate
            if( m_MinTxRate != 0 )
            {
                bool result = (1.0f * m_MinTxRate <= txTps);
                success &= result;
                m_TxRateResult = ToTestResult(result);
                NN_LOG("  Tx rate [>= %lf] : %lf  success? %d\n",
                       1.0f * m_MinTxRate,
                       txTps,
                       (1.0f * m_MinRxRate <= txTps));
            }

            map<uint64_t, MultiDataSink::Statistics*>& rxDetailedStats = GetRxDetailedStatistics();
            map<uint64_t, string>& sources = GetSources();
            map<uint64_t, string>::iterator iter = sources.begin();
            while( iter != sources.end() )
            {
                // min Rx rate
                if( m_MinRxRate != 0 )
                {
                    double tps = rxDetailedStats[iter->first]->Throughput.GetThroughput() * 1024.0f * 1024.0f;
                    bool result = (1.0f * m_MinRxRate <= tps);
                    success &= result;
                    m_RxRateResult &= ToTestResult(result);
                    NN_LOG("  Rx rate [>= %lf] : %lf  success? %d\n",
                           1.0f * m_MinRxRate,
                           tps,
                           (m_MinRxRate <= tps));
                }

                // max Rx PLR
                if( m_MaxPlr >= 0.0f )
                {
                    bool result = (m_MaxPlr > rxDetailedStats[iter->first]->Errors.GetErrorRate());
                    success &= result;
                    m_PlrResult &= ToTestResult(result);
                    NN_LOG("  PLR [< %lf] : %lf  success? %d\n",
                           m_MaxPlr,
                           rxDetailedStats[iter->first]->Errors.GetErrorRate(),
                           (m_MaxPlr > rxDetailedStats[iter->first]->Errors.GetErrorRate()));
                }

                // max Latency(Max)
                if( m_MaxLatencyMax != 0 )
                {
                    bool result = (m_MaxLatencyMax > rxDetailedStats[iter->first]->Latency.GetMaxLatency());
                    success &= result;
                    m_MaxLatencyResult &= ToTestResult(result);
                    NN_LOG("  Latency(Max) [< %u] : %u  success? %d\n",
                           m_MaxLatencyMax,
                           rxDetailedStats[iter->first]->Latency.GetMaxLatency(),
                           (m_MaxLatencyMax > rxDetailedStats[iter->first]->Latency.GetMaxLatency()));
                }

                // max Latency(Avg)
                if( m_MaxLatencyAvg != 0 )
                {
                    bool result = (1.0f * m_MaxLatencyAvg > rxDetailedStats[iter->first]->Latency.GetAvgLatency());
                    success &= result;
                    m_AvgLatencyResult &= ToTestResult(result);
                    NN_LOG("  Latency(Avg) [< %lf] : %lf  success? %d\n",
                           1.0f * m_MaxLatencyAvg,
                           rxDetailedStats[iter->first]->Latency.GetAvgLatency(),
                           (1.0f * m_MaxLatencyAvg > rxDetailedStats[iter->first]->Latency.GetAvgLatency()));
                }


                iter++;
            }

            if( success )
            {
                m_Result = RESULT_SUCCESS;
            }
            else
            {
                m_Result = RESULT_FAILURE;
                m_PerformanceTestSuccess = false;
            }
        }
        else
        {
            m_PerformanceTestSuccess = false;
            m_Result = RESULT_FAILURE;
        }

        return true;
    }

public:

    LocalMasterPerformanceTester()
    {
        m_TxController.SelectType(PID_OUI_EXT);
    }
    virtual ~LocalMasterPerformanceTester() {}

    bool SetTxParam()
    {
        m_TxController.Set();
        return true;
    }
    bool SetTxParam(const DataGeneratorParam& param)
    {
        m_TxController.Set(param);
        return true;
    }

    virtual void SetTestParam(const TestParam& param)
    {
        LocalMasterConnectivityTester::SetTestParam(param);

        SetMinTxRate(param.minTxRate);
        SetMinRxRate(param.minRxRate);
        SetMaxPlr(param.maxPlr);
        SetMaxLatencyMax(param.maxLatencyMax);
        SetMaxLatencyAvg(param.maxLatencyAvg);
    }

    Page& GetGeneratorControllerPage() { return m_TxController; }
    Page& GetSearchControllerPage() { return m_AfScannerController; }

    void SetGenerator(DataGenerator& d)
    {
        m_pGenerator = &d;
        m_pGenerator->SetTarget(*target);
        m_TxController.SetTarget(*m_pGenerator);
    }
    void SetSink(MultiDataSink& m)
    {
        m_pSink = &m;
    }
    void SetScanner(AfTxScanner& s)
    {
        m_pScanner = &s;
        m_AfScannerController.SetTarget(*m_pScanner);
    }

    void SetMinTxRate(uint32_t v) { m_MinTxRate = v; }
    void SetMinRxRate(uint32_t v) { m_MinRxRate = v; }
    void SetMaxPlr(double v) { m_MaxPlr = v; }
    void SetMaxLatencyMax(uint32_t v) { m_MaxLatencyMax = v; }
    void SetMaxLatencyAvg(uint32_t v) { m_MaxLatencyAvg = v; }

    uint32_t GetMinTxRate() { return m_MinTxRate; }
    uint32_t GetMinRxRate() { return m_MinRxRate; }
    double GetMaxPlr() { return m_MaxPlr; }
    uint32_t GetMaxLatencyMax() { return m_MaxLatencyMax; }
    uint32_t GetMaxLatencyAvg() { return m_MaxLatencyAvg; }

    TestResult GetTxRateResult() { return m_TxRateResult; }
    TestResult GetRxRateResult() { return m_RxRateResult; }
    TestResult GetPlrResult() { return m_PlrResult; }
    TestResult GetMaxLatencyResult() { return m_MaxLatencyResult; }
    TestResult GetAvgLatencyResult() { return m_AvgLatencyResult; }

    virtual NodeStatistics& GetTxStatistics()
    {
        NN_ASSERT(target);
        return target->GetTxStatistics();
    }

    virtual ReceiverStatistics& GetRxStatistics()
    {
        NN_ASSERT(target);
        return target->GetRxStatistics();
    }

    virtual map<uint64_t, MultiDataSink::Statistics*>& GetRxDetailedStatistics()
    {
        NN_ASSERT(m_pSink);
        return m_pSink->GetStatistics();
    }

    map<uint64_t, string>& GetSources()
    {
        return m_pSink->GetSources();
    }

    bool GetPerformanceTestSuccess(){ return m_PerformanceTestSuccess; }
};


/*---------------------------------------------------------------------------
           LocalClientPerformanceTester
---------------------------------------------------------------------------*/
class LocalClientPerformanceTester : public LocalClientConnectivityTester
{
protected:

    DataGeneratorController m_TxController;
    bool m_PerformanceTestSuccess;

    uint32_t m_MinTxRate;
    uint32_t m_MinRxRate;
    double   m_MaxPlr;
    uint32_t m_MaxLatencyMax;
    uint32_t m_MaxLatencyAvg;
    nn::TimeSpan m_TxStartDelay;

    TestResult m_TxRateResult;
    TestResult m_RxRateResult;
    TestResult m_PlrResult;
    TestResult m_MaxLatencyResult;
    TestResult m_AvgLatencyResult;

protected:

    DataGenerator* m_pGenerator;
    MultiDataSink* m_pSink;
    AfTxScanner*   m_pScanner;

    AfScannerController m_AfScannerController;

    virtual bool Initialize()
    {
        LocalClientConnectivityTester::Initialize();
        m_AfScannerController.Set();
        m_PerformanceTestSuccess = true;
        m_TxRateResult = RESULT_NOT_COMPLETED;
        m_RxRateResult = RESULT_NOT_COMPLETED;
        m_PlrResult = RESULT_NOT_COMPLETED;
        m_MaxLatencyResult = RESULT_NOT_COMPLETED;
        m_AvgLatencyResult = RESULT_NOT_COMPLETED;
        return true;
    }

    virtual bool Open()
    {
        bool opened = LocalClientConnectivityTester::Open();

        if( opened )
        {
            nn::wlan::WlanState state = nn::wlan::WlanState_Stop;

            // ##wa SIGLO-71213 のワークアラウンド
            // LocalClientIdle 状態で Connect() と同時に PutActionFrameOneShot() を行わないように
            // LocalClient 状態に遷移するまで待つ
            while( !IsExitNeeded() )
            {
                nn::Result result = nn::wlan::Local::GetState(&state);
                NN_ASSERT(result.IsSuccess());

                if( state == nn::wlan::WlanState_LocalClient )
                {
                    // サーチ開始
                    m_AfScannerController.Start();
                    break;
                }
                Sleep(nn::TimeSpan::FromMilliSeconds(100));
            }
        }

        return opened;
    }

    virtual bool Ready()
    {
        bool ready = false;

        if(LocalClientConnectivityTester::Ready() )
        {
            ready = (m_pSink->GetSources().size() > 0);
        }

        return ready;
    }

    virtual bool Start()
    {
        if(LocalClientConnectivityTester::Start())
        {
            Sleep(m_TxStartDelay);
            m_TxController.Start();

            return true;
        }

        return false;
    }

    virtual bool TestImpl()
    {
        return LocalClientConnectivityTester::TestImpl();
    }

    virtual bool Close()
    {
        m_TxController.Stop();

        // サーチ終了
        m_AfScannerController.Stop();

        nn::os::Tick start = nn::os::GetSystemTick();
        while(!m_TxController.IsIdle())
        {
            Sleep(nn::TimeSpan::FromMilliSeconds(10));
            NN_ASSERT((nn::os::GetSystemTick() - start).ToTimeSpan().GetMilliSeconds() <= 2000);
        }

        return LocalClientConnectivityTester::Close();
    }

    virtual bool Evaluate()
    {
        LocalClientConnectivityTester::Evaluate();

        if( m_Result == RESULT_SUCCESS )
        {
            bool success = true;
            NodeStatistics& txStats = GetTxStatistics();
            // bit
            double txSize = txStats.SendSize * 8.0f;
            // msec
            uint64_t txTime = (txStats.LastSendTime - txStats.FirstSendTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
            double txTps = 1.0f * txSize / txTime;

            // min Tx rate
            if( m_MinTxRate != 0 )
            {
                bool result = (1.0f * m_MinTxRate <= txTps);
                success &= result;
                m_TxRateResult = ToTestResult(result);
                NN_LOG("  Tx rate [>= %lf] : %lf  success? %d\n",
                       1.0f * m_MinTxRate,
                       txTps,
                       (1.0f * m_MinRxRate <= txTps));
            }

            map<uint64_t, MultiDataSink::Statistics*>& rxDetailedStats = GetRxDetailedStatistics();
            map<uint64_t, string>& sources = GetSources();
            map<uint64_t, string>::iterator iter = sources.begin();
            while( iter != sources.end() )
            {
                // min Rx rate
                if( m_MinRxRate != 0 )
                {
                    double tps = rxDetailedStats[iter->first]->Throughput.GetThroughput() * 1024.0f * 1024.0f;
                    bool result = (1.0f * m_MinRxRate <= tps);
                    success &= result;
                    m_RxRateResult &= ToTestResult(result);
                    NN_LOG("  Rx rate [>= %lf] : %lf  success? %d\n",
                           1.0f * m_MinRxRate,
                           tps,
                           (m_MinRxRate <= tps));
                }

                // max Rx PLR
                if( m_MaxPlr >= 0.0f )
                {
                    bool result = (m_MaxPlr > rxDetailedStats[iter->first]->Errors.GetErrorRate());
                    success &= result;
                    m_PlrResult &= ToTestResult(result);
                    NN_LOG("  PLR [<= %lf] : %lf  success? %d\n",
                           m_MaxPlr,
                           rxDetailedStats[iter->first]->Errors.GetErrorRate(),
                           (m_MaxPlr > rxDetailedStats[iter->first]->Errors.GetErrorRate()));
                }

                // max Latency(Max)
                if( m_MaxLatencyMax != 0 )
                {
                    bool result = (m_MaxLatencyMax > rxDetailedStats[iter->first]->Latency.GetMaxLatency());
                    success &= result;
                    m_MaxLatencyResult &= ToTestResult(result);
                    NN_LOG("  Latency(Max) [< %u] : %u  success? %d\n",
                           m_MaxLatencyMax,
                           rxDetailedStats[iter->first]->Latency.GetMaxLatency(),
                           (m_MaxLatencyMax > rxDetailedStats[iter->first]->Latency.GetMaxLatency()));
                }

                // max Latency(Avg)
                if( m_MaxLatencyAvg != 0 )
                {
                    bool result = (1.0f * m_MaxLatencyAvg > rxDetailedStats[iter->first]->Latency.GetAvgLatency());
                    success &= result;
                    m_AvgLatencyResult &= ToTestResult(result);
                    NN_LOG("  Latency(Avg) [< %lf] : %lf  success? %d\n",
                           1.0f * m_MaxLatencyAvg,
                           rxDetailedStats[iter->first]->Latency.GetAvgLatency(),
                           (1.0f * m_MaxLatencyAvg > rxDetailedStats[iter->first]->Latency.GetAvgLatency()));
                }

                iter++;
            }

            if( success )
            {
                m_Result = RESULT_SUCCESS;
            }
            else
            {
                m_Result = RESULT_FAILURE;
                m_PerformanceTestSuccess = false;
            }
        }
        else
        {
            m_PerformanceTestSuccess = false;
            m_Result = RESULT_FAILURE;
        }

        return true;
    }

public:

    LocalClientPerformanceTester()
    {
        m_TxController.SelectType(PID_OUI_EXT);
    }

    virtual ~LocalClientPerformanceTester() {}

    bool SetTxParam()
    {
        m_TxController.Set();
        return true;
    }
    bool SetTxParam(const DataGeneratorParam& param)
    {
        m_TxController.Set(param);
        return true;
    }

    virtual void SetTestParam(const TestParam& param)
    {
        LocalClientConnectivityTester::SetTestParam(param);

        SetTxStartDelay(param.txStartDelay);

        SetMinTxRate(param.minTxRate);
        SetMinRxRate(param.minRxRate);
        SetMaxPlr(param.maxPlr);
        SetMaxLatencyMax(param.maxLatencyMax);
        SetMaxLatencyAvg(param.maxLatencyAvg);
    }

    Page& GetGeneratorControllerPage() { return m_TxController; }
    Page& GetSearchControllerPage() { return m_AfScannerController; }

    void SetGenerator(DataGenerator& d)
    {
        m_pGenerator = &d;
        m_pGenerator->SetTarget(*target);
        m_TxController.SetTarget(*m_pGenerator);
    }
    void SetSink(MultiDataSink& m)
    {
        m_pSink = &m;
    }
    void SetScanner(AfTxScanner& s)
    {
        m_pScanner = &s;
        m_AfScannerController.SetTarget(*m_pScanner);
    }

    void SetMinTxRate(uint32_t v) { m_MinTxRate = v; }
    void SetMinRxRate(uint32_t v) { m_MinRxRate = v; }
    void SetMaxPlr(double v) { m_MaxPlr = v; }
    void SetMaxLatencyMax(uint32_t v) { m_MaxLatencyMax = v; }
    void SetMaxLatencyAvg(uint32_t v) { m_MaxLatencyAvg = v; }
    void SetTxStartDelay(nn::TimeSpan v) { m_TxStartDelay = v; }

    uint32_t GetMinTxRate() { return m_MinTxRate; }
    uint32_t GetMinRxRate() { return m_MinRxRate; }
    double GetMaxPlr() { return m_MaxPlr; }
    uint32_t GetMaxLatencyMax() { return m_MaxLatencyMax; }
    uint32_t GetMaxLatencyAvg() { return m_MaxLatencyAvg; }
    nn::TimeSpan GetTxStartDelay() { return m_TxStartDelay; }

    TestResult GetTxRateResult() { return m_TxRateResult; }
    TestResult GetRxRateResult() { return m_RxRateResult; }
    TestResult GetPlrResult() { return m_PlrResult; }
    TestResult GetMaxLatencyResult() { return m_MaxLatencyResult; }
    TestResult GetAvgLatencyResult() { return m_AvgLatencyResult; }

    virtual NodeStatistics& GetTxStatistics()
    {
        NN_ASSERT(target);
        return target->GetTxStatistics();
    }

    virtual ReceiverStatistics& GetRxStatistics()
    {
        NN_ASSERT(target);
        return target->GetRxStatistics();
    }

    virtual map<uint64_t, MultiDataSink::Statistics*>& GetRxDetailedStatistics()
    {
        NN_ASSERT(m_pSink);
        return m_pSink->GetStatistics();
    }

    map<uint64_t, string>& GetSources()
    {
        return m_pSink->GetSources();
    }

    bool GetPerformanceTestSuccess(){ return m_PerformanceTestSuccess; }
};


/*---------------------------------------------------------------------------
           LcsMasterPerformanceTester
---------------------------------------------------------------------------*/
class LcsMasterPerformanceTester : public LocalMasterConnectivityTester
{

protected:

    FixedSizeDataTransmitterController m_TxController;
    bool m_PerformanceTestSuccess;

    uint32_t m_MinTxRate;
    uint32_t m_MinRxRate;
    double   m_MaxPlr;
    uint32_t m_MaxLatencyMax;
    uint32_t m_MaxLatencyAvg;

    TestResult m_TxRateResult;
    TestResult m_RxRateResult;
    TestResult m_PlrResult;
    TestResult m_MaxLatencyResult;
    TestResult m_AvgLatencyResult;

protected:

    FixedSizeDataTransmitter* m_pGenerator;
    MultiDataSink* m_pSink;

    virtual bool Initialize()
    {
        LocalMasterConnectivityTester::Initialize();
        m_PerformanceTestSuccess = true;
        m_TxRateResult = RESULT_NOT_COMPLETED;
        m_RxRateResult = RESULT_NOT_COMPLETED;
        m_PlrResult = RESULT_NOT_COMPLETED;
        m_MaxLatencyResult = RESULT_NOT_COMPLETED;
        m_AvgLatencyResult = RESULT_NOT_COMPLETED;
        SetTestTime(InfinityTime);

        // LCS モードでは Local4 固定
        SetWlanMode(nn::btm::WlanMode_Local4);

        return true;
    }

    virtual bool Open()
    {
        bool opened = LocalMasterConnectivityTester::Open();

        if(opened)
        {
            //m_TxController.Set();
        }

        return opened;
    }

    virtual bool Ready()
    {
        return LocalMasterConnectivityTester::Ready();
    }

    virtual bool Start()
    {
        if(LocalMasterConnectivityTester::Start())
        {
            return true;
        }

        return false;
    }

    virtual bool TestImpl()
    {
        return LocalMasterConnectivityTester::TestImpl();
    }

    virtual bool Close()
    {
        m_TxController.Stop();

        nn::os::Tick start = nn::os::GetSystemTick();
        while(!m_TxController.IsIdle())
        {
            Sleep(nn::TimeSpan::FromMilliSeconds(10));
            NN_ASSERT((nn::os::GetSystemTick() - start).ToTimeSpan().GetMilliSeconds() <= 2000);
        }

        return LocalMasterConnectivityTester::Close();
    }

    virtual bool Evaluate()
    {
        LocalMasterConnectivityTester::Evaluate();

        if( m_Result == RESULT_SUCCESS )
        {
            bool success = true;
            NodeStatistics& txStats = GetTxStatistics();
            // bit
            double txSize = txStats.SendSize * 8.0f;
            // msec
            uint64_t txTime = (txStats.LastSendTime - txStats.FirstSendTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
            double txTps = 1.0f * txSize / txTime;

            // min Tx rate
            if( m_MinTxRate != 0 )
            {
                bool result = (1.0f * m_MinTxRate <= txTps);
                success &= result;
                m_TxRateResult = ToTestResult(result);
                NN_LOG("  Tx rate [>= %lf] : %lf  success? %d\n",
                       1.0f * m_MinTxRate,
                       txTps,
                       (1.0f * m_MinRxRate <= txTps));
            }

            map<uint64_t, MultiDataSink::Statistics*>& rxDetailedStats = GetRxDetailedStatistics();
            map<uint64_t, string>& sources = GetSources();
            map<uint64_t, string>::iterator iter = sources.begin();
            while( iter != sources.end() )
            {
                // min Rx rate
                if( m_MinRxRate != 0 )
                {
                    double tps = rxDetailedStats[iter->first]->Throughput.GetThroughput() * 1024.0f * 1024.0f;
                    bool result = (1.0f * m_MinRxRate <= tps);
                    success &= result;
                    m_RxRateResult &= ToTestResult(result);
                    NN_LOG("  Rx rate [>= %lf] : %lf  success? %d\n",
                           1.0f * m_MinRxRate,
                           tps,
                           (m_MinRxRate <= tps));
                }

                // max Rx PLR
                if( m_MaxPlr >= 0.0f )
                {
                    bool result = (m_MaxPlr > rxDetailedStats[iter->first]->Errors.GetErrorRate());
                    success &= result;
                    m_PlrResult &= ToTestResult(result);
                    NN_LOG("  PLR [< %lf] : %lf  success? %d\n",
                           m_MaxPlr,
                           rxDetailedStats[iter->first]->Errors.GetErrorRate(),
                           (m_MaxPlr > rxDetailedStats[iter->first]->Errors.GetErrorRate()));
                }

                // max Latency(Max)
                if( m_MaxLatencyMax != 0 )
                {
                    bool result = (m_MaxLatencyMax > rxDetailedStats[iter->first]->Latency.GetMaxLatency());
                    success &= result;
                    m_MaxLatencyResult &= ToTestResult(result);
                    NN_LOG("  Latency(Max) [< %u] : %u  success? %d\n",
                           m_MaxLatencyMax,
                           rxDetailedStats[iter->first]->Latency.GetMaxLatency(),
                           (m_MaxLatencyMax > rxDetailedStats[iter->first]->Latency.GetMaxLatency()));
                }

                // max Latency(Avg)
                if( m_MaxLatencyAvg != 0 )
                {
                    bool result = (1.0f * m_MaxLatencyAvg > rxDetailedStats[iter->first]->Latency.GetAvgLatency());
                    success &= result;
                    m_AvgLatencyResult &= ToTestResult(result);
                    NN_LOG("  Latency(Avg) [< %lf] : %lf  success? %d\n",
                           1.0f * m_MaxLatencyAvg,
                           rxDetailedStats[iter->first]->Latency.GetAvgLatency(),
                           (1.0f * m_MaxLatencyAvg > rxDetailedStats[iter->first]->Latency.GetAvgLatency()));
                }

                iter++;
            }

            if( success )
            {
                m_Result = RESULT_SUCCESS;
            }
            else
            {
                m_Result = RESULT_FAILURE;
                m_PerformanceTestSuccess = false;
            }
        }
        else
        {
            m_PerformanceTestSuccess = false;
            m_Result = RESULT_FAILURE;
        }

        return true;
    }

public:

    LcsMasterPerformanceTester()
    {
    }
    virtual ~LcsMasterPerformanceTester() {}

    bool SetTxParam()
    {
        m_TxController.Set();
        return true;
    }
    bool SetTxParam(const DataGeneratorParam& param)
    {
        m_TxController.Set(param);
        return true;
    }

    virtual void SetTestParam(const TestParam& param)
    {
        LocalMasterConnectivityTester::SetTestParam(param);

        SetMinTxRate(param.minTxRate);
        SetMinRxRate(param.minRxRate);
        SetMaxPlr(param.maxPlr);
        SetMaxLatencyMax(param.maxLatencyMax);
        SetMaxLatencyAvg(param.maxLatencyAvg);
    }

    Page& GetGeneratorControllerPage() { return m_TxController; }
    void SetGenerator(FixedSizeDataTransmitter& d)
    {
        m_pGenerator = &d;
        m_pGenerator->SetTarget(*target);
        m_TxController.SetTarget(*m_pGenerator);
    }
    void SetSink(MultiDataSink& m)
    {
        m_pSink = &m;
    }

    void SetMinTxRate(uint32_t v) { m_MinTxRate = v; }
    void SetMinRxRate(uint32_t v) { m_MinRxRate = v; }
    void SetMaxPlr(double v) { m_MaxPlr = v; }
    void SetMaxLatencyMax(uint32_t v) { m_MaxLatencyMax = v; }
    void SetMaxLatencyAvg(uint32_t v) { m_MaxLatencyAvg = v; }

    uint32_t GetMinTxRate() { return m_MinTxRate; }
    uint32_t GetMinRxRate() { return m_MinRxRate; }
    double GetMaxPlr() { return m_MaxPlr; }
    uint32_t GetMaxLatencyMax() { return m_MaxLatencyMax; }
    uint32_t GetMaxLatencyAvg() { return m_MaxLatencyAvg; }

    TestResult GetTxRateResult() { return m_TxRateResult; }
    TestResult GetRxRateResult() { return m_RxRateResult; }
    TestResult GetPlrResult() { return m_PlrResult; }
    TestResult GetMaxLatencyResult() { return m_MaxLatencyResult; }
    TestResult GetAvgLatencyResult() { return m_AvgLatencyResult; }

    virtual NodeStatistics& GetTxStatistics()
    {
        NN_ASSERT(target);
        return target->GetTxStatistics();
    }

    virtual ReceiverStatistics& GetRxStatistics()
    {
        NN_ASSERT(target);
        return target->GetRxStatistics();
    }

    virtual map<uint64_t, MultiDataSink::Statistics*>& GetRxDetailedStatistics()
    {
        NN_ASSERT(m_pSink);
        return m_pSink->GetStatistics();
    }

    map<uint64_t, string>& GetSources()
    {
        return m_pSink->GetSources();
    }

    bool GetPerformanceTestSuccess(){ return m_PerformanceTestSuccess; }
};


/*---------------------------------------------------------------------------
           LcsClientPerformanceTester
---------------------------------------------------------------------------*/
class LcsClientPerformanceTester : public LocalClientConnectivityTester
{
protected:

    FixedSizeDataTransmitterController m_TxController;
    bool m_PerformanceTestSuccess;

    uint32_t m_MinTxRate;
    uint32_t m_MinRxRate;
    double   m_MaxPlr;
    uint32_t m_MaxLatencyMax;
    uint32_t m_MaxLatencyAvg;
    nn::TimeSpan m_TxStartDelay;

    TestResult m_TxRateResult;
    TestResult m_RxRateResult;
    TestResult m_PlrResult;
    TestResult m_MaxLatencyResult;
    TestResult m_AvgLatencyResult;

protected:

    FixedSizeDataTransmitter* m_pGenerator;
    MultiDataSink* m_pSink;

    virtual bool Initialize()
    {
        LocalClientConnectivityTester::Initialize();
        m_PerformanceTestSuccess = true;
        m_TxRateResult = RESULT_NOT_COMPLETED;
        m_RxRateResult = RESULT_NOT_COMPLETED;
        m_PlrResult = RESULT_NOT_COMPLETED;
        m_MaxLatencyResult = RESULT_NOT_COMPLETED;
        m_AvgLatencyResult = RESULT_NOT_COMPLETED;
        SetTestTime(InfinityTime);

        // LCS モードでは Local4 固定
        SetWlanMode(nn::btm::WlanMode_Local4);

        return true;
    }

    virtual bool Open()
    {
        bool opened = LocalClientConnectivityTester::Open();
        if(opened)
        {
            //m_TxController.Set();
        }

        return opened;
    }

    virtual bool Ready()
    {
        bool ready = false;

        if(LocalClientConnectivityTester::Ready() )
        {
            ready = (m_pSink->GetSources().size() > 0 && Pad::GetInstance().IsTrigger(Button::A));
        }

        return ready;
    }

    virtual bool Start()
    {
        if(LocalClientConnectivityTester::Start())
        {
            Sleep(m_TxStartDelay);
            m_TxController.Start();

            return true;
        }

        return false;
    }

    virtual bool TestImpl()
    {
        return LocalClientConnectivityTester::TestImpl();
    }

    virtual bool Close()
    {
        m_TxController.Stop();

        nn::os::Tick start = nn::os::GetSystemTick();
        while(!m_TxController.IsIdle())
        {
            Sleep(nn::TimeSpan::FromMilliSeconds(10));
            NN_ASSERT((nn::os::GetSystemTick() - start).ToTimeSpan().GetMilliSeconds() <= 2000);
        }

        return LocalClientConnectivityTester::Close();
    }

    virtual bool Evaluate()
    {
        LocalClientConnectivityTester::Evaluate();

        if( m_Result == RESULT_SUCCESS )
        {
            bool success = true;
            NodeStatistics& txStats = GetTxStatistics();
            // bit
            double txSize = txStats.SendSize * 8.0f;
            // msec
            uint64_t txTime = (txStats.LastSendTime - txStats.FirstSendTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
            double txTps = 1.0f * txSize / txTime;

            // min Tx rate
            if( m_MinTxRate != 0 )
            {
                bool result = (1.0f * m_MinTxRate <= txTps);
                success &= result;
                m_TxRateResult = ToTestResult(result);
                NN_LOG("  Tx rate [>= %lf] : %lf  success? %d\n",
                       1.0f * m_MinTxRate,
                       txTps,
                       (1.0f * m_MinRxRate <= txTps));
            }

            map<uint64_t, MultiDataSink::Statistics*>& rxDetailedStats = GetRxDetailedStatistics();
            map<uint64_t, string>& sources = GetSources();
            map<uint64_t, string>::iterator iter = sources.begin();
            while( iter != sources.end() )
            {
                // min Rx rate
                if( m_MinRxRate != 0 )
                {
                    double tps = rxDetailedStats[iter->first]->Throughput.GetThroughput() * 1024.0f * 1024.0f;
                    bool result = (1.0f * m_MinRxRate <= tps);
                    success &= result;
                    m_RxRateResult &= ToTestResult(result);
                    NN_LOG("  Rx rate [>= %lf] : %lf  success? %d\n",
                           1.0f * m_MinRxRate,
                           tps,
                           (m_MinRxRate <= tps));
                }

                // max Rx PLR
                if( m_MaxPlr >= 0.0f )
                {
                    bool result = (m_MaxPlr > rxDetailedStats[iter->first]->Errors.GetErrorRate());
                    success &= result;
                    m_PlrResult &= ToTestResult(result);
                    NN_LOG("  PLR [<= %lf] : %lf  success? %d\n",
                           m_MaxPlr,
                           rxDetailedStats[iter->first]->Errors.GetErrorRate(),
                           (m_MaxPlr > rxDetailedStats[iter->first]->Errors.GetErrorRate()));
                }

                // max Latency(Max)
                if( m_MaxLatencyMax != 0 )
                {
                    bool result = (m_MaxLatencyMax > rxDetailedStats[iter->first]->Latency.GetMaxLatency());
                    success &= result;
                    m_MaxLatencyResult &= ToTestResult(result);
                    NN_LOG("  Latency(Max) [< %u] : %u  success? %d\n",
                           m_MaxLatencyMax,
                           rxDetailedStats[iter->first]->Latency.GetMaxLatency(),
                           (m_MaxLatencyMax > rxDetailedStats[iter->first]->Latency.GetMaxLatency()));
                }

                // max Latency(Avg)
                if( m_MaxLatencyAvg != 0 )
                {
                    bool result = (1.0f * m_MaxLatencyAvg > rxDetailedStats[iter->first]->Latency.GetAvgLatency());
                    success &= result;
                    m_AvgLatencyResult &= ToTestResult(result);
                    NN_LOG("  Latency(Avg) [< %lf] : %lf  success? %d\n",
                           1.0f * m_MaxLatencyAvg,
                           rxDetailedStats[iter->first]->Latency.GetAvgLatency(),
                           (1.0f * m_MaxLatencyAvg > rxDetailedStats[iter->first]->Latency.GetAvgLatency()));
                }

                iter++;
            }

            if( success )
            {
                m_Result = RESULT_SUCCESS;
            }
            else
            {
                m_Result = RESULT_FAILURE;
                m_PerformanceTestSuccess = false;
            }
        }
        else
        {
            m_PerformanceTestSuccess = false;
            m_Result = RESULT_FAILURE;
        }

        return true;
    }

public:

    LcsClientPerformanceTester()
    {
    }

    virtual ~LcsClientPerformanceTester() {}

    bool SetTxParam()
    {
        m_TxController.Set();
        return true;
    }
    bool SetTxParam(const DataGeneratorParam& param)
    {
        m_TxController.Set(param);
        return true;
    }

    virtual void SetTestParam(const TestParam& param)
    {
        LocalClientConnectivityTester::SetTestParam(param);

        SetTxStartDelay(param.txStartDelay);

        SetMinTxRate(param.minTxRate);
        SetMinRxRate(param.minRxRate);
        SetMaxPlr(param.maxPlr);
        SetMaxLatencyMax(param.maxLatencyMax);
        SetMaxLatencyAvg(param.maxLatencyAvg);
    }

    Page& GetGeneratorControllerPage() { return m_TxController; }
    void SetGenerator(FixedSizeDataTransmitter& d)
    {
        m_pGenerator = &d;
        m_pGenerator->SetTarget(*target);
        m_TxController.SetTarget(*m_pGenerator);
    }
    void SetSink(MultiDataSink& m)
    {
        m_pSink = &m;
    }

    void SetMinTxRate(uint32_t v) { m_MinTxRate = v; }
    void SetMinRxRate(uint32_t v) { m_MinRxRate = v; }
    void SetMaxPlr(double v) { m_MaxPlr = v; }
    void SetMaxLatencyMax(uint32_t v) { m_MaxLatencyMax = v; }
    void SetMaxLatencyAvg(uint32_t v) { m_MaxLatencyAvg = v; }
    void SetTxStartDelay(nn::TimeSpan v) { m_TxStartDelay = v; }

    uint32_t GetMinTxRate() { return m_MinTxRate; }
    uint32_t GetMinRxRate() { return m_MinRxRate; }
    double GetMaxPlr() { return m_MaxPlr; }
    uint32_t GetMaxLatencyMax() { return m_MaxLatencyMax; }
    uint32_t GetMaxLatencyAvg() { return m_MaxLatencyAvg; }
    nn::TimeSpan GetTxStartDelay() { return m_TxStartDelay; }

    TestResult GetTxRateResult() { return m_TxRateResult; }
    TestResult GetRxRateResult() { return m_RxRateResult; }
    TestResult GetPlrResult() { return m_PlrResult; }
    TestResult GetMaxLatencyResult() { return m_MaxLatencyResult; }
    TestResult GetAvgLatencyResult() { return m_AvgLatencyResult; }

    virtual NodeStatistics& GetTxStatistics()
    {
        NN_ASSERT(target);
        return target->GetTxStatistics();
    }

    virtual ReceiverStatistics& GetRxStatistics()
    {
        NN_ASSERT(target);
        return target->GetRxStatistics();
    }

    virtual map<uint64_t, MultiDataSink::Statistics*>& GetRxDetailedStatistics()
    {
        NN_ASSERT(m_pSink);
        return m_pSink->GetStatistics();
    }

    map<uint64_t, string>& GetSources()
    {
        return m_pSink->GetSources();
    }

    bool GetPerformanceTestSuccess(){ return m_PerformanceTestSuccess; }
};


/*---------------------------------------------------------------------------
           SocketPerformanceTester
---------------------------------------------------------------------------*/
class SocketPerformanceTester : public SocketConnectivityTester
{
protected:

    DataGeneratorController m_TxController;
    bool m_PerformanceTestSuccess;

    uint32_t m_MinTxRate;
    uint32_t m_MinRxRate;
    double   m_MaxPlr;
    nn::TimeSpan m_TxStartDelay;

    TestResult m_TxRateResult;
    TestResult m_RxRateResult;
    TestResult m_PlrResult;

protected:

    DataGenerator* m_pGenerator;
    MultiDataSink* m_pSink;

    virtual bool Initialize()
    {
        SocketConnectivityTester::Initialize();
        m_PerformanceTestSuccess = true;
        m_TxRateResult = RESULT_NOT_COMPLETED;
        m_RxRateResult = RESULT_NOT_COMPLETED;
        m_PlrResult = RESULT_NOT_COMPLETED;
        return true;
    }

    virtual bool Open()
    {
        bool opened = SocketConnectivityTester::Open();
        if(opened)
        {
            //m_TxController.Set();
        }

        return opened;
    }

    virtual bool Ready()
    {
        return SocketConnectivityTester::Ready();
    }

    virtual bool Start()
    {
        if(SocketConnectivityTester::Start())
        {
            Sleep(m_TxStartDelay);
            m_TxController.Start();

            return true;
        }

        return false;
    }

    virtual bool TestImpl()
    {
        return SocketConnectivityTester::TestImpl();
    }

    virtual bool Close()
    {
        m_TxController.Stop();

        nn::os::Tick start = nn::os::GetSystemTick();
        while(!m_TxController.IsIdle())
        {
            Sleep(nn::TimeSpan::FromMilliSeconds(10));
            NN_ASSERT((nn::os::GetSystemTick() - start).ToTimeSpan().GetMilliSeconds() <= 2000);
        }

        return SocketConnectivityTester::Close();
    }

    virtual bool Evaluate()
    {
        SocketConnectivityTester::Evaluate();

        if(m_Result == RESULT_SUCCESS)
        {
            bool success = true;
            NodeStatistics& txStats = GetTxStatistics();
            // bit
            double txSize = txStats.SendSize * 8.0f;
            // msec
            uint64_t txTime = (txStats.LastSendTime - txStats.FirstSendTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
            double txTps = 1.0f * txSize / txTime;

            // min Tx rate
            if(m_MinTxRate != 0)
            {
                bool result = (1.0f * m_MinTxRate <= txTps);
                success &= result;
                m_TxRateResult = ToTestResult(result);
                NN_LOG("  Tx rate [>= %lf] : %lf  success? %d\n",
                       1.0f * m_MinTxRate,
                       txTps,
                       (1.0f * m_MinRxRate <= txTps));
            }

            // bit
            double rxSize = txStats.ReceiveSize * 8.0f;
            // msec
            uint64_t rxTime = (txStats.LastReceiveTime - txStats.FirstReceiveTime).ToTimeSpan().GetMilliSeconds() / 1000.0f;
            double rxTps = 1.0f * rxSize / rxTime;
            double plr = GetError().GetErrorRate();

            // min Rx rate
            if(m_MinRxRate != 0)
            {
                bool result = (1.0f * m_MinRxRate <= rxTps);
                success &= result;
                m_RxRateResult &= ToTestResult(result);
                NN_LOG("  Rx rate [>= %lf] : %lf  success? %d\n",
                       1.0f * m_MinRxRate,
                       rxTps,
                       (1.0f * m_MinRxRate <= rxTps));
            }

            // max Rx PLR
            if(m_MaxPlr >= 0.0f)
            {
                bool result = (m_MaxPlr >= plr);
                success &= result;
                m_PlrResult &= ToTestResult(result);
                NN_LOG("  PLR [<= %lf] : %lf  success? %d\n",
                       m_MaxPlr,
                       plr,
                       (m_MaxPlr > plr));
            }

            if(success)
            {
                m_Result = RESULT_SUCCESS;
            }
            else
            {
                m_Result = RESULT_FAILURE;
                m_PerformanceTestSuccess = false;
            }
        }
        else
        {
            m_PerformanceTestSuccess = false;
            m_Result = RESULT_FAILURE;
        }

        return true;
    }

public:

    SocketPerformanceTester()
    {
        m_TxController.SelectType(PID_PAYLOAD_ONLY);
    }

    virtual ~SocketPerformanceTester() {}

    bool SetTxParam()
    {
        m_TxController.Set();
        return true;
    }
    bool SetTxParam(const DataGeneratorParam& param)
    {
        m_TxController.Set(param);
        return true;
    }

    virtual void SetTestParam(const TestParam& param)
    {
        SocketConnectivityTester::SetTestParam(param);

        SetTxStartDelay(param.txStartDelay);

        SetMinTxRate(param.minTxRate);
        SetMinRxRate(param.minRxRate);
        SetMaxPlr(param.maxPlr);
    }

    Page& GetGeneratorControllerPage() { return m_TxController; }
    void SetGenerator(DataGenerator& d)
    {
        m_pGenerator = &d;
        m_pGenerator->SetTarget(*target);
        m_TxController.SetTarget(*m_pGenerator);
    }
    void SetSink(MultiDataSink& m)
    {
        m_pSink = &m;
    }

    void SetMinTxRate(uint32_t v) { m_MinTxRate = v; }
    void SetMinRxRate(uint32_t v) { m_MinRxRate = v; }
    void SetMaxPlr(double v) { m_MaxPlr = v; }
    void SetTxStartDelay(nn::TimeSpan v) { m_TxStartDelay = v; }

    uint32_t GetMinTxRate() { return m_MinTxRate; }
    uint32_t GetMinRxRate() { return m_MinRxRate; }
    double GetMaxPlr() { return m_MaxPlr; }
    nn::TimeSpan GetTxStartDelay() { return m_TxStartDelay; }

    TestResult GetTxRateResult() { return m_TxRateResult; }
    TestResult GetRxRateResult() { return m_RxRateResult; }
    TestResult GetPlrResult() { return m_PlrResult; }

    virtual NodeStatistics& GetTxStatistics()
    {
        NN_ASSERT(target);
        return target->GetTxStatistics();
    }

    virtual ReceiverStatistics& GetRxStatistics()
    {
        NN_ASSERT(target);
        return target->GetRxStatistics();
    }

    virtual map<uint64_t, MultiDataSink::Statistics*>& GetRxDetailedStatistics()
    {
        NN_ASSERT(m_pSink);
        return m_pSink->GetStatistics();
    }

    map<uint64_t, string>& GetSources()
    {
        return m_pSink->GetSources();
    }

    CyclicSequenceNumberCounter GetError()
    {
        NN_ASSERT(target);
        return target->GetError();
    }

    bool GetPerformanceTestSuccess(){ return m_PerformanceTestSuccess; }
};

/*---------------------------------------------------------------------------
           HostDrivenDetectorPerformanceTester
---------------------------------------------------------------------------*/
class HostDrivenDetectorPerformanceTester : public AgingTester<Detector>
{

protected:

    bool m_IsTxReady;

    DetectorController m_DetectorController;
    DataGeneratorController m_TxController;
    bool m_PerformanceTestSuccess;

    uint32_t m_MinTxRate;
    uint32_t m_MinRxRate;
    double   m_MaxPlr;
    uint32_t m_MaxLatencyMax;
    uint32_t m_MaxLatencyAvg;

    TestResult m_TxRateResult;
    TestResult m_RxRateResult;
    TestResult m_PlrResult;
    TestResult m_MaxLatencyResult;
    TestResult m_AvgLatencyResult;

protected:

    DataGenerator* m_pGenerator;
    MultiDataSink* m_pSink;

    virtual bool Initialize()
    {
        AgingTester<Detector>::Initialize();
        m_IsTxReady = false;
        m_PerformanceTestSuccess = true;
        m_TxRateResult = RESULT_NOT_COMPLETED;
        m_RxRateResult = RESULT_NOT_COMPLETED;
        m_PlrResult = RESULT_NOT_COMPLETED;
        m_MaxLatencyResult = RESULT_NOT_COMPLETED;
        m_AvgLatencyResult = RESULT_NOT_COMPLETED;

        // Dynamic にすれば User8 も厳しい条件でテストできるため Dynamic にする
        SetWlanMode(nn::btm::WlanMode_None);

        m_DetectorController.Set();

        return true;
    }

    virtual bool Open()
    {
        AgingTester<Detector>::Open();
        nn::Result result;

        result = target->Initialize();
        NN_ASSERT(result.IsSuccess());

        return true;
    }

    virtual bool Ready()
    {
        bool isButtonPressed = false;

        if( !AgingTester<Detector>::Ready() )
        {
            return false;
        }

        if( Pad::GetInstance().IsTrigger(Button::A) )
        {
            nn::Result result;
            result = target->Open();
            NN_ASSERT(result.IsSuccess());
            isButtonPressed = true;
        }

        m_IsTxReady = isButtonPressed;

        return isButtonPressed;
    }

    virtual bool Start()
    {
        if( AgingTester<Detector>::Start() )
        {
            m_DetectorController.Start();

            if( m_DetectorController.IsPeriodicEnabled() )
            {
                m_DetectorController.StartPeriodicActionFrame();
            }

            if( m_DetectorController.IsOneShotEnabled() )
            {
                m_TxController.SetType(PID_PAYLOAD_ONLY);
                m_TxController.SetPayloadSize(target->GetPayloadSize());
                m_TxController.SetInterval(nn::TimeSpan::FromMilliSeconds(target->GetTxInterval()));
                m_TxController.SetTxCount(0);
                m_TxController.Start();
            }

            return true;
        }

        return false;
    }

    virtual bool TestImpl()
    {
        return AgingTester<Detector>::TestImpl();
    }

    virtual bool Close()
    {
        if( m_DetectorController.IsPeriodicEnabled() )
        {
            m_DetectorController.CancelPeriodicActionFrame();
        }

        if( m_DetectorController.IsOneShotEnabled() )
        {
            m_TxController.Stop();
        }

        m_DetectorController.Stop();

        nn::Result result;
        result = target->Close();
        NN_ASSERT(result.IsSuccess());

        result = target->Finalize();
        NN_ASSERT(result.IsSuccess());

        return AgingTester<Detector>::Close();
    }

    virtual bool Evaluate()
    {
        AgingTester<Detector>::Evaluate();
        m_Result = RESULT_SUCCESS;

        return true;
    }

public:

    HostDrivenDetectorPerformanceTester()
    {
    }
    virtual ~HostDrivenDetectorPerformanceTester()
    {
    }

    bool SetTxParam()
    {
        m_TxController.Set();
        return true;
    }
    bool SetTxParam(const DataGeneratorParam& param)
    {
        m_TxController.Set(param);
        return true;
    }

    virtual void SetTestParam(const TestParam& param)
    {
        AgingTester<Detector>::SetTestParam(param);

        SetMinTxRate(param.minTxRate);
        SetMinRxRate(param.minRxRate);
        SetMaxPlr(param.maxPlr);
        SetMaxLatencyMax(param.maxLatencyMax);
        SetMaxLatencyAvg(param.maxLatencyAvg);
    }

    Page& GetGeneratorControllerPage() { return m_TxController; }
    void SetGenerator(DataGenerator& d)
    {
        m_pGenerator = &d;
        m_pGenerator->SetTarget(*target);
        m_TxController.SetTarget(*m_pGenerator);
    }
    void SetSink(MultiDataSink& m)
    {
        m_pSink = &m;
    }
    void SetTarget(Detector& detector)
    {
        ModelController<Detector>::SetTarget(detector);
        m_DetectorController.SetTarget(detector);
    }

    void SetMinTxRate(uint32_t v) { m_MinTxRate = v; }
    void SetMinRxRate(uint32_t v) { m_MinRxRate = v; }
    void SetMaxPlr(double v) { m_MaxPlr = v; }
    void SetMaxLatencyMax(uint32_t v) { m_MaxLatencyMax = v; }
    void SetMaxLatencyAvg(uint32_t v) { m_MaxLatencyAvg = v; }

    uint32_t GetMinTxRate() { return m_MinTxRate; }
    uint32_t GetMinRxRate() { return m_MinRxRate; }
    double GetMaxPlr() { return m_MaxPlr; }
    uint32_t GetMaxLatencyMax() { return m_MaxLatencyMax; }
    uint32_t GetMaxLatencyAvg() { return m_MaxLatencyAvg; }

    TestResult GetTxRateResult() { return m_TxRateResult; }
    TestResult GetRxRateResult() { return m_RxRateResult; }
    TestResult GetPlrResult() { return m_PlrResult; }
    TestResult GetMaxLatencyResult() { return m_MaxLatencyResult; }
    TestResult GetAvgLatencyResult() { return m_AvgLatencyResult; }

    virtual NodeStatistics& GetTxStatistics()
    {
        NN_ASSERT(target);
        return target->GetTxStatistics();
    }

    virtual ReceiverStatistics& GetRxStatistics()
    {
        NN_ASSERT(target);
        return target->GetRxStatistics();
    }

    virtual map<uint64_t, MultiDataSink::Statistics*>& GetRxDetailedStatistics()
    {
        NN_ASSERT(m_pSink);
        return m_pSink->GetStatistics();
    }

    Page& GetNodeControllerPage() { return m_DetectorController; }

    map<uint64_t, string>& GetSources()
    {
        return m_pSink->GetSources();
    }

    bool GetPerformanceTestSuccess(){ return m_PerformanceTestSuccess; }
    bool IsTxReady(){ return m_IsTxReady; }
};




} // WlanTest
