﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdio>
#include <nn/os.h>

#include "Tester.h"
#include "TesterController.h"
#include "TesterViewer.h"
#include "Display.h"
#include "SocketNode.h"
#include "LcsMaster.h"
#include "LcsClient.h"
#include "Scanner.h"
#include "Log.h"
#include "Util.h"

namespace WlanTest {

enum NodeType
{
    NODE_NONE = 0x0,
    NODE_MASTER,
    NODE_CLIENT,
    NODE_AUDIENCE
};

class LogParam
{
public :
    enum ItemId
    {
        ITEM_ID_UNDEFINED = 0,

        ITEM_ID_LOGGING_INTERVAL
    };

    static const char* ITEM_STR_LOGGING_INTERVAL;

    nn::TimeSpan loggingInterval;

    LogParam()
    {
        loggingInterval = nn::TimeSpan::FromMilliSeconds(1000);
    }

    void Print()
    {
        NN_LOG("  Logging interval         : %s\n", ToString(loggingInterval).c_str());
    }

    ItemId GetItemId(const string& item)
    {
        if( strcmp(item.c_str(), ITEM_STR_LOGGING_INTERVAL) == 0 ) return ITEM_ID_LOGGING_INTERVAL;

        NN_LOG("  - failed : Unrecognized item name (%s)\n", item.c_str());

        return ITEM_ID_UNDEFINED;
    }

    bool SetParam(const string& item, const string& value)
    {
        ItemId id = GetItemId(item);
        if( id == ITEM_ID_UNDEFINED )
        {
            return false;
        }

        switch(id)
        {
        case ITEM_ID_LOGGING_INTERVAL : return SetLoggingInterval(value);
        default : return false;
        }

        return false;
    }

    bool SetLoggingInterval(const string& valueStr)
    {
        int64_t value;
        char unit;

        if( sscanf(valueStr.c_str(), "%lld %c", &value, &unit) == 2 )
        {
            GenerateTimeSpan(loggingInterval, value, unit);
            return true;
        }
        else
        {
            return false;
        }
    }

};

class Test
{
private:

protected:

    enum TestState
    {
        STATE_READY,
        STATE_SETTING_PARAM,
        STATE_TESTING,
        STATE_RESULT,
        STATE_DONE
    };


    TestState        m_State;
    bool             m_IsAutoTest;
    LogParam         m_LogParam;

    // page to display/change params
    Page                    m_ParamPage;
    SwitchablePage          m_ParamSwitcher;

    // page to test
    Page                    m_TestPage;
    SwitchablePage          m_TestSwitcher;

    // page to display result
    Page                    m_ResultPage;
    Label                   m_ResultContent;

    virtual void InitializeComponent(){}
    virtual void ProcessTest(){}
    virtual bool IsParamSet(){ return Pad::GetInstance().IsTrigger(Button::A) || m_IsAutoTest; }
    virtual bool IsTestEnded() = 0;
    virtual bool IsCanceled(){ return false; }
    virtual bool IsResultEnded(){ return Pad::GetInstance().IsTrigger(Button::A) || m_IsAutoTest; }

    virtual void SetParam() = 0;
    virtual void GenerateResult() = 0;

public:

    Test()
    {
        m_State = STATE_READY;
        m_IsAutoTest = false;
    }
    virtual ~Test(){}

    virtual void Execute()
    {
        static bool showLog = false;
        static nn::os::Tick lastTick = nn::os::GetSystemTick();

        switch(m_State)
        {
        case STATE_READY:
            {
                InitializeComponent();
                m_State = STATE_SETTING_PARAM;
            }
            break;

        case STATE_SETTING_PARAM:
            {
                if( !m_IsAutoTest )
                {
                    m_ParamPage.Show(Display::GetInstance());
                }

                if( IsParamSet() )
                {
                    Pad::GetInstance().Clear();
                    DBG_VLOG("Set Param");
                    // Start test thread
                    SetParam();
                    m_State = STATE_TESTING;
                    DBG_VLOG("Test Start");
                    lastTick = nn::os::GetSystemTick();
                    showLog = true;
                }
            }
            break;

        case STATE_TESTING:
            {
                if( (nn::os::GetSystemTick() - lastTick).ToTimeSpan() > m_LogParam.loggingInterval )
                {
                    showLog = true;
                    lastTick = nn::os::GetSystemTick();
                }

                if( showLog )
                {
                    Display::GetInstance().EnableConsoleOutput();
                }

                ProcessTest();
                m_TestPage.Show(Display::GetInstance());

                showLog = false;

                if( IsTestEnded() || IsCanceled() )
                {
                    DBG_VLOG("Test End");
                    GenerateResult();
                    showLog = true;
                    m_State = STATE_RESULT;
                }
            }
            break;

        case STATE_RESULT:
            {
                if( showLog )
                {
                    Display::GetInstance().EnableConsoleOutput();
                    showLog = false;
                }

                if( m_IsAutoTest )
                {
                    m_ResultPage.Show(Display::GetInstance());
                }
                else
                {
                    m_TestPage.Show(Display::GetInstance());
                }
                PrintResult();

                m_State = STATE_DONE;
            }
            break;

        case STATE_DONE:
            {
                if( showLog )
                {
                    Display::GetInstance().EnableConsoleOutput();
                    showLog = false;
                }

                if( m_IsAutoTest )
                {
                    m_ResultPage.Show(Display::GetInstance());
                }
                else
                {
                    m_TestPage.Show(Display::GetInstance());
                }

                m_State = STATE_DONE;
            }
            break;
        default:
            break;
        }
    }

    virtual bool IsDone()
    {
        return m_State == STATE_DONE;
    }

    virtual void Exit()
    {
        PacketLogger& logger = PacketLogger::GetInstance();
        if( logger.IsEnabled() )
        {
            logger.Finalize();
        }

        m_IsAutoTest = false;
        m_State = STATE_READY;

        m_ParamPage.Clear();
        m_ParamSwitcher.Clear();
        m_TestPage.Clear();
        m_TestSwitcher.Clear();
        m_ResultPage.Clear();
    }

    void SetLoggingInterval(nn::TimeSpan logginInterval)
    {
        m_LogParam.loggingInterval = logginInterval;
    }

    void SetLogParam(const LogParam& logParam)
    {
        SetLoggingInterval(logParam.loggingInterval);
    }

    virtual void PrintResult(){};

};

/*---------------------------------------------------------------------------
           LocalTest
---------------------------------------------------------------------------*/
class LocalTest : public Test
{
private:

protected:

    static void UpdateThreadFunction(void* arg)
    {
        DBG_VLOG("Thread start - UpdateThread");
        reinterpret_cast<LocalTest*>(arg)->UpdateStats();
        DBG_VLOG("Thread stop - UpdateThread");
    }

    virtual void UpdateStats(){}

public:

    LocalTest()
    {
    }

    virtual ~LocalTest()
    {
    }
};


/*---------------------------------------------------------------------------
           LocalMasterTest
---------------------------------------------------------------------------*/
class LocalMasterTest : public LocalTest
{
private:

    Master                  m_Master;
    DataGenerator           m_Generator;
    MultiDataSink           m_Sink;
    MultiAfSink             m_AfSink;
    AfTxScanner             m_AfTxScanner;

    TestParam               m_TestParam;
    MasterParam             m_MasterParam;
    DataGeneratorParam      m_TxParam;

    TestController                         m_TestController;
    LocalMasterPerformanceTesterController m_MasterTesterController;

    MasterViewer            m_NodeViewer;
    MultiDataSinkSwitchableViewer  m_SinkViewer;
    MultiAfSinkSwitchableViewer    m_AfSinkViewer;
    PacketLatencySwitchableViewer  m_LatencyViewer;
    SearchStatisticsViewer  m_SearchViewer;
    LocalMasterTesterViewer m_MasterTesterViewer;
    NPadViewer              m_RcViewer;
    BtPlrViewer             m_BtPlrViewer;
    ReceiveEventHandler(MultiDataSink) * m_pHandler;
    ReceiveEventHandler(MultiAfSink) * m_pAfHandler;

    LocalMasterResultViewer m_MasterResultViewer;

    TimerViewer<Master>     m_TimerViewer;
    Page                    m_HeaderPage;
    Label                   m_SeparatorLine;
    Label                   m_BlankLine;

    RssiStatistics          m_RssiStatistics;
    RssiViewer              m_RssiViewer;
    RssiController          m_RssiController;

    bool                    m_IsTerminationRequested;
    bool                    m_IsThreadRunning;
    nn::os::ThreadType      m_UpdateThread;

    // function to test
    LocalMasterPerformanceTester m_MasterTester;

protected:

    virtual bool IsTestEnded()
    {
        return !m_MasterTester.IsTesting();
    }

    virtual void InitializeComponent()
    {
        m_ParamSwitcher.Width = Display::GetInstance().GetWidth();
        m_TestSwitcher.Width = Display::GetInstance().GetWidth();

        m_TestController.SetTarget(*this);

        m_NodeViewer.SetTarget(m_Master);
        m_SearchViewer.SetTarget(m_AfTxScanner);

        m_RssiController.SetNode(&m_Master);
        m_RssiController.SetTarget(m_RssiStatistics);
        m_RssiViewer.SetTarget(m_RssiStatistics);

        m_MasterTester.SetGenerator(m_Generator);
        m_MasterTester.SetSink(m_Sink);
        m_MasterTester.SetScanner(m_AfTxScanner);
        m_MasterTesterViewer.SetTarget(m_MasterTester);
        m_MasterResultViewer.SetTarget(m_MasterTester);
        m_TimerViewer.SetTarget(m_MasterTester);

        m_ParamSwitcher.AddPage("Test Params", m_MasterTesterController);
        m_ParamSwitcher.AddPage("Master Params", m_MasterTester.GetNodeControllerPage());
        m_ParamSwitcher.AddPage("Search Params", m_MasterTester.GetSearchControllerPage());
        m_ParamSwitcher.AddPage("Tx Params", m_MasterTester.GetGeneratorControllerPage());
        m_ParamSwitcher.AddPage("Log Params", m_TestController);

        m_ParamPage.Add(m_ParamSwitcher);
        m_ParamPage.Add(m_BlankLine);

        // ##? for displaing on the console
        Display::GetInstance().EnableConsoleOutput();
        m_ParamPage.Show(Display::GetInstance());
    }

    virtual void SetParam()
    {
        if(m_IsAutoTest)
        {
            m_MasterTesterController.Set(m_TestParam, m_MasterParam, m_TxParam);

            m_TestPage.Add(m_HeaderPage);
            m_TestPage.Add(m_MasterTesterViewer);
            m_TestPage.Add(m_NodeViewer);
            m_TestPage.Add(m_SinkViewer);
            m_TestPage.Add(m_BlankLine);
        }
        else
        {
            m_MasterTesterController.Set();
            m_TestController.Set();
            m_RssiController.Set();

            m_TestPage.Add(m_HeaderPage);
            m_TestPage.Add(m_TestSwitcher);
            m_TestSwitcher.AddPage("Test Summary", m_MasterTesterViewer);
            m_TestSwitcher.AddPage("WLAN Summary", m_NodeViewer);
            m_TestSwitcher.AddPage("WLAN Rx (Data)", m_SinkViewer);
            m_TestSwitcher.AddPage("WLAN Latency", m_LatencyViewer);
            m_TestSwitcher.AddPage("WLAN Rx (Action Frame)", m_AfSinkViewer);
            m_TestSwitcher.AddPage("WLAN Search Stats", m_SearchViewer);
            m_TestSwitcher.AddPage("WLAN Rssi", m_RssiViewer);
            m_TestSwitcher.AddPage("Remote Controller", m_RcViewer);
            m_TestSwitcher.AddPage("Remote Controller PLR", m_BtPlrViewer);
            m_RcViewer.ErasePadText(Button::X);

            m_TestPage.Add(m_TimerViewer);
        }

        m_BtPlrViewer.Clear();
        m_RssiViewer.Clear();

        PacketLogger& logger = PacketLogger::GetInstance();
        if( logger.IsEnabled() )
        {
            // ログ書き出しのための初期化
            string fileName = GetMacAddressStr("") + "_" + GetTodayStr() + "_LocalMasterTest";
            logger.Initialize("wit", fileName.c_str());
        }

        // start update thread
        NN_ASSERT(m_IsThreadRunning == false);
        NN_ASSERT(m_IsTerminationRequested == false);
        static NN_ALIGNAS(4096) char s_UpdateThreadStack[4096 * 2];
        nn::Result result = nn::os::CreateThread(&m_UpdateThread, UpdateThreadFunction, this, s_UpdateThreadStack, sizeof(s_UpdateThreadStack), nn::os::DefaultThreadPriority - 1);
        NN_ASSERT( result.IsSuccess(), "UpdateThread can not run.");
        m_IsThreadRunning = true;
        nn::os::SetThreadNamePointer(&m_UpdateThread, "WitUpdateThread");
        nn::os::StartThread(&m_UpdateThread);

        // start test
        m_MasterTester.Execute();

        // to wait for tester to set params and start master/client
        Sleep(nn::TimeSpan::FromMilliSeconds(100));

        m_RssiController.Start();
    }

    virtual void ProcessTest()
    {
        if( !m_IsAutoTest )
        {
            if( m_MasterTester.IsTxReady() )
            {
                m_MasterTesterViewer.SetPadText(Button::A, "Start Tx");
                m_NodeViewer.SetPadText(Button::A, "Start Tx");
                m_SinkViewer.SetPadText(Button::A, "Start Tx");
                m_AfSinkViewer.SetPadText(Button::A, "Start Tx");
                m_LatencyViewer.SetPadText(Button::A, "Start Tx");
                m_RcViewer.SetPadText(Button::A, "Start Tx");
                m_BtPlrViewer.SetPadText(Button::A, "Start Tx");
                m_RssiViewer.SetPadText(Button::A, "Start Tx");
            }
            else
            {
                m_MasterTesterViewer.ErasePadText(Button::A);
                m_NodeViewer.ErasePadText(Button::A);
                m_SinkViewer.ErasePadText(Button::A);
                m_AfSinkViewer.ErasePadText(Button::A);
                m_LatencyViewer.ErasePadText(Button::A);
                m_RcViewer.ErasePadText(Button::A);
                m_BtPlrViewer.ErasePadText(Button::A);
                m_RssiViewer.ErasePadText(Button::A);
            }
        }
    }

    virtual void GenerateResult()
    {
        if(m_IsAutoTest)
        {
            m_ResultPage.Add(m_HeaderPage);
            m_ResultPage.Add(m_MasterResultViewer);
            m_ResultPage.Add(m_BlankLine);
            m_ResultPage.Add(m_ResultContent);
            m_ResultPage.Add(m_BlankLine);

            m_ResultContent.Text = " TOTAL RESULT : ";
            switch(m_MasterTester.GetResult())
            {
            case RESULT_SUCCESS : m_ResultContent.Text += "  SUCCESS"; break;
            case RESULT_FAILURE : m_ResultContent.Text += "  FAILURE"; break;
            case RESULT_NOT_COMPLETED : m_ResultContent.Text += "  CANCELED"; break;
            default: break;
            }
            m_ResultContent.X = DISPLAY_CONTENT_START_X;
            m_ResultContent.Y = 600;
            m_ResultContent.FitSize();
        }
    }

    virtual void PrintResult()
    {
        string key, value;
        string tmpKey, tmpValue;
        m_MasterResultViewer.PrintResult(tmpKey, tmpValue);
        key += tmpKey;
        value += tmpValue;

        key += ", Final result\n";
        if(m_MasterTester.GetResult() == RESULT_SUCCESS)
        {
            value += ", success\n";
        }
        else
        {
            value += ", failure\n";
        }

        if( m_IsAutoTest )
        {
            NN_LOG(key.c_str());
            NN_LOG(value.c_str());
        }
        else
        {

#ifdef HID_SHELL_MODE
            NN_LOG(key.c_str());
            NN_LOG(value.c_str());
#endif

            PacketLogger& logger = PacketLogger::GetInstance();
            if( logger.IsEnabled() )
            {
                logger.Write("\n");
                logger.Write(key);
                logger.Write(value);
            }
        }
    }

    virtual void UpdateStats()
    {
        BluetoothPlrStatistics &stats = BluetoothPlrStatistics::GetInstance();
        while( !m_IsTerminationRequested )
        {
            stats.Update();
            if( m_State == STATE_TESTING )
            {
                m_Sink.UpdateLastStatistics();
                m_AfSink.UpdateLastStatistics();
            }
            Sleep(nn::TimeSpan::FromMilliSeconds(10));
        }
    }

public:

    LocalMasterTest()
    {
        Initialize();
    }

    LocalMasterTest(const TestParam& testParam, const MasterParam& masterParam, const DataGeneratorParam& txParam)
    {
        Initialize();

        m_IsAutoTest = true;
        m_TestParam = testParam;
        m_MasterParam = masterParam;
        m_TxParam = txParam;
    }

    virtual ~LocalMasterTest()
    {
        delete m_pHandler;
        delete m_pAfHandler;
    }

    virtual void Initialize()
    {
        Clear();
        m_IsThreadRunning = false;
        m_IsTerminationRequested = false;

        m_IsAutoTest = false;
        m_pHandler = new ReceiveEventHandler(MultiDataSink)(m_Sink, &MultiDataSink::Sink);
        m_pAfHandler = new ReceiveEventHandler(MultiAfSink)(m_AfSink, &MultiAfSink::Sink);

        m_HeaderPage.Add(m_BlankLine);
        m_BlankLine.Text = " ";

        m_HeaderPage.Add(m_SeparatorLine);
        m_SeparatorLine.Text = DISPLAY_SEPARATOR;
        m_SeparatorLine.Y = 1000; // 表示させない
        m_SeparatorLine.Width = 400;

        m_Master.SetReceiveEventHandler(*m_pHandler);
        m_Master.SetAfReceiveEventHandler(*m_pAfHandler);
        m_MasterTester.SetTarget(m_Master);
        //##? ここで設定すると Master が送信できない
        //m_MasterTester.SetGenerator(m_Generator);
        m_MasterTesterController.SetTarget(m_MasterTester);

        m_SinkViewer.SetTarget(m_Sink);
        m_AfSinkViewer.SetTarget(m_AfSink);
        m_LatencyViewer.SetTarget(m_Sink);
    }

    virtual void Exit()
    {
        Clear();

        if( m_IsThreadRunning )
        {
            m_IsTerminationRequested = true;
            nn::os::DestroyThread(&m_UpdateThread);
            m_IsThreadRunning = false;
            m_IsTerminationRequested = false;
        }

        Test::Exit();
    }

    virtual void Clear()
    {
        m_MasterTester.Exit();
        m_RssiController.Stop();
        m_Master.ClearStatistics();
        m_BtPlrViewer.Clear();
        m_RssiViewer.Clear();

        m_Sink.Clear();
        m_AfSink.Clear();
    }

};

/*---------------------------------------------------------------------------
           LocalClientTest
---------------------------------------------------------------------------*/
class LocalClientTest : public LocalTest
{
private:

    Client                  m_Client;
    DataGenerator           m_Generator;
    MultiDataSink           m_Sink;
    MultiAfSink             m_AfSink;
    AfTxScanner             m_AfTxScanner;

    TestParam               m_TestParam;
    ClientParam             m_ClientParam;
    DataGeneratorParam      m_TxParam;

    TestController                         m_TestController;
    LocalClientPerformanceTesterController m_ClientTesterController;

    ClientViewer            m_NodeViewer;
    MultiDataSinkSwitchableViewer  m_SinkViewer;
    MultiAfSinkSwitchableViewer    m_AfSinkViewer;
    PacketLatencySwitchableViewer  m_LatencyViewer;
    SearchStatisticsViewer  m_SearchViewer;
    LoalClientTesterViewer  m_ClientTesterViewer;
    NPadViewer              m_RcViewer;
    BtPlrViewer             m_BtPlrViewer;
    ReceiveEventHandler(MultiDataSink) * m_pHandler;
    ReceiveEventHandler(MultiAfSink) * m_pAfHandler;

    LocalClientResultViewer m_ClientResultViewer;

    TimerViewer<Client>     m_TimerViewer;
    Page                    m_HeaderPage;
    Label                   m_SeparatorLine;
    Label                   m_BlankLine;

    RssiStatistics          m_RssiStatistics;
    RssiViewer              m_RssiViewer;
    RssiController          m_RssiController;

    bool                    m_IsTerminationRequested;
    bool                    m_IsThreadRunning;
    nn::os::ThreadType      m_UpdateThread;

    // function to test
    LocalClientPerformanceTester m_ClientTester;

protected:

    virtual bool IsTestEnded()
    {
        return !m_ClientTester.IsTesting();
    }

    virtual void InitializeComponent()
    {
        m_ParamSwitcher.Width = Display::GetInstance().GetWidth();
        m_TestSwitcher.Width = Display::GetInstance().GetWidth();

        m_TestController.SetTarget(*this);

        m_NodeViewer.SetTarget(m_Client);
        m_SearchViewer.SetTarget(m_AfTxScanner);

        m_RssiController.SetNode(&m_Client);
        m_RssiController.SetTarget(m_RssiStatistics);
        m_RssiViewer.SetTarget(m_RssiStatistics);

        m_ClientTester.SetGenerator(m_Generator);
        m_ClientTester.SetSink(m_Sink);
        m_ClientTester.SetScanner(m_AfTxScanner);
        m_ClientTesterViewer.SetTarget(m_ClientTester);
        m_ClientResultViewer.SetTarget(m_ClientTester);
        m_TimerViewer.SetTarget(m_ClientTester);

        m_ParamSwitcher.AddPage("Test Params", m_ClientTesterController);
        m_ParamSwitcher.AddPage("Client Params", m_ClientTester.GetNodeControllerPage());
        m_ParamSwitcher.AddPage("Search Params", m_ClientTester.GetSearchControllerPage());
        m_ParamSwitcher.AddPage("Tx Params", m_ClientTester.GetGeneratorControllerPage());
        m_ParamSwitcher.AddPage("Log Params", m_TestController);

        m_ParamPage.Add(m_ParamSwitcher);
        m_ParamPage.Add(m_BlankLine);

        // ##? for displaing on the console
        Display::GetInstance().EnableConsoleOutput();
        m_ParamPage.Show(Display::GetInstance());
    }

    virtual void SetParam()
    {
        if(m_IsAutoTest)
        {
            m_ClientTesterController.Set(m_TestParam, m_ClientParam, m_TxParam);

            m_TestPage.Add(m_HeaderPage);
            m_TestPage.Add(m_ClientTesterViewer);
            m_TestPage.Add(m_NodeViewer);
            m_TestPage.Add(m_SinkViewer);
            m_TestPage.Add(m_BlankLine);
        }
        else
        {
            m_ClientTesterController.Set();
            m_TestController.Set();
            m_RssiController.Set();

            m_TestPage.Add(m_HeaderPage);
            m_TestPage.Add(m_TestSwitcher);
            m_TestSwitcher.AddPage("Test Summary", m_ClientTesterViewer);
            m_TestSwitcher.AddPage("WLAN Summary", m_NodeViewer);
            m_TestSwitcher.AddPage("WLAN Rx (Data)", m_SinkViewer);
            m_TestSwitcher.AddPage("WLAN Latency", m_LatencyViewer);
            m_TestSwitcher.AddPage("WLAN Rx (Action Frame)", m_AfSinkViewer);
            m_TestSwitcher.AddPage("WLAN Search Stats", m_SearchViewer);
            m_TestSwitcher.AddPage("WLAN Rssi", m_RssiViewer);
            m_TestSwitcher.AddPage("Remote Controller", m_RcViewer);
            m_TestSwitcher.AddPage("Remote Controller PLR", m_BtPlrViewer);
            m_RcViewer.ErasePadText(Button::X);

            m_TestPage.Add(m_TimerViewer);
        }

        m_BtPlrViewer.Clear();
        m_RssiViewer.Clear();

        PacketLogger& logger = PacketLogger::GetInstance();
        if( logger.IsEnabled() )
        {
            // ログ書き出しのための初期化
            string fileName = GetMacAddressStr("") + "_" + GetTodayStr() + "_LocalClientTest";
            logger.Initialize("wit", fileName.c_str());
        }

        // start update thread
        NN_ASSERT(m_IsThreadRunning == false);
        NN_ASSERT(m_IsTerminationRequested == false);
        static NN_ALIGNAS(4096) char s_UpdateThreadStack[4096 * 2];
        nn::Result result = nn::os::CreateThread(&m_UpdateThread, UpdateThreadFunction, this, s_UpdateThreadStack, sizeof(s_UpdateThreadStack), nn::os::DefaultThreadPriority - 1);
        NN_ASSERT( result.IsSuccess(), "UpdateThread can not run.");
        m_IsThreadRunning = true;
        nn::os::SetThreadNamePointer(&m_UpdateThread, "WitUpdateThread");
        nn::os::StartThread(&m_UpdateThread);

        // start test
        m_ClientTester.Execute();

        // to wait for tester to set params and start master/client
        Sleep(nn::TimeSpan::FromMilliSeconds(100));

        m_RssiController.Start();
    }

    virtual void GenerateResult()
    {
        if( m_IsAutoTest )
        {
            m_ResultPage.Add(m_HeaderPage);
            m_ResultPage.Add(m_ClientResultViewer);
            m_ResultPage.Add(m_BlankLine);
            m_ResultPage.Add(m_ResultContent);
            m_ResultPage.Add(m_BlankLine);

            m_ResultContent.Text = " TOTAL RESULT : ";
            switch(m_ClientTester.GetResult())
            {
            case RESULT_SUCCESS : m_ResultContent.Text += "  SUCCESS"; break;
            case RESULT_FAILURE : m_ResultContent.Text += "  FAILURE"; break;
            case RESULT_NOT_COMPLETED : m_ResultContent.Text += "  CANCELED"; break;
            default: break;
            }
            m_ResultContent.X = DISPLAY_CONTENT_START_X;
            m_ResultContent.Y = 600;
            m_ResultContent.FitSize();
        }
    }

    virtual void PrintResult()
    {
        string key, value;
        string tmpKey, tmpValue;
        m_ClientResultViewer.PrintResult(tmpKey, tmpValue);
        key += tmpKey;
        value += tmpValue;

        key += ", Final result\n";
        if(m_ClientTester.GetResult() == RESULT_SUCCESS)
        {
            value += ", success\n";
        }
        else
        {
            value += ", failure\n";
        }

        if( m_IsAutoTest )
        {
            NN_LOG(key.c_str());
            NN_LOG(value.c_str());
        }
        else
        {
#ifdef HID_SHELL_MODE
            NN_LOG(key.c_str());
            NN_LOG(value.c_str());
#endif

            PacketLogger& logger = PacketLogger::GetInstance();
            if( logger.IsEnabled() )
            {
                logger.Write("\n");
                logger.Write(key);
                logger.Write(value);
            }
        }
    }

    virtual void UpdateStats()
    {
        BluetoothPlrStatistics &stats = BluetoothPlrStatistics::GetInstance();
        while( !m_IsTerminationRequested )
        {
            stats.Update();
            if( m_State == STATE_TESTING )
            {
                m_Sink.UpdateLastStatistics();
                m_AfSink.UpdateLastStatistics();
            }
            Sleep(nn::TimeSpan::FromMilliSeconds(10));
        }
    }

public:

    LocalClientTest()
    {
        Initialize();
    }

    LocalClientTest(const TestParam& testParam, const ClientParam& clientParam, const DataGeneratorParam& txParam)
    {
        Initialize();

        m_IsAutoTest = true;
        m_TestParam = testParam;
        m_ClientParam = clientParam;
        m_TxParam = txParam;
    }

    virtual ~LocalClientTest()
    {
        delete m_pHandler;
        delete m_pAfHandler;
    }

    virtual void Initialize()
    {
        Clear();
        m_IsThreadRunning = false;
        m_IsTerminationRequested = false;

        m_IsAutoTest = false;
        m_pHandler = new ReceiveEventHandler(MultiDataSink)(m_Sink, &MultiDataSink::Sink);
        m_pAfHandler = new ReceiveEventHandler(MultiAfSink)(m_AfSink, &MultiAfSink::Sink);

        m_HeaderPage.Add(m_BlankLine);
        m_BlankLine.Text = " ";

        m_HeaderPage.Add(m_SeparatorLine);
        m_SeparatorLine.Text = DISPLAY_SEPARATOR;
        m_SeparatorLine.Y = 1000; // 表示させない
        m_SeparatorLine.Width = 400;

        m_Client.SetReceiveEventHandler(*m_pHandler);
        m_Client.SetAfReceiveEventHandler(*m_pAfHandler);
        m_ClientTester.SetTarget(m_Client);
        //##? ここで設定すると Client が送信できない
        //m_ClientTester.SetGenerator(m_Generator);
        m_ClientTesterController.SetTarget(m_ClientTester);

        m_SinkViewer.SetTarget(m_Sink);
        m_AfSinkViewer.SetTarget(m_AfSink);
        m_LatencyViewer.SetTarget(m_Sink);
    }

    virtual void Exit()
    {
        Clear();

        if( m_IsThreadRunning )
        {
            m_IsTerminationRequested = true;
            nn::os::DestroyThread(&m_UpdateThread);
            m_IsThreadRunning = false;
            m_IsTerminationRequested = false;
        }

        Test::Exit();
    }

    virtual void Clear()
    {
        m_ClientTester.Exit();
        m_RssiController.Stop();
        m_Client.ClearStatistics();
        m_BtPlrViewer.Clear();

        m_Sink.Clear();
        m_AfSink.Clear();
    }

};


/*---------------------------------------------------------------------------
           InfraTest
---------------------------------------------------------------------------*/
class InfraTest : public Test
{
private:

    SocketNode*             m_pNode;
    DataGenerator           m_Generator;

    TestParam               m_TestParam;
    SocketParam             m_SocketParam;
    DataGeneratorParam      m_TxParam;

    TestController                    m_TestController;
    SocketPerformanceTesterController m_SocketTesterController;

    SocketNodeViewer        m_NodeViewer;
    NPadViewer              m_RcViewer;
    BtPlrViewer             m_BtPlrViewer;
    SocketTesterViewer      m_SocketTesterViewer;

    SocketResultViewer      m_SocketResultViewer;

    TimerViewer<SocketNode> m_TimerViewer;
    Page                    m_HeaderPage;
    Label                   m_SeparatorLine;
    Label                   m_BlankLine;

    RssiStatistics          m_RssiStatistics;
    RssiViewer              m_RssiViewer;
    RssiController          m_RssiController;

    bool                    m_IsTerminationRequested;
    bool                    m_IsThreadRunning;
    nn::os::ThreadType      m_UpdateThread;

    SocketPerformanceTester m_SocketTester;

protected:

    static void UpdateThreadFunction(void* arg)
    {
        DBG_VLOG("Thread start - UpdateThread");
        reinterpret_cast<InfraTest*>(arg)->UpdateStats();
        DBG_VLOG("Thread stop - UpdateThread");
    }

    virtual bool IsTestEnded()
    {
        return !m_SocketTester.IsTesting();
    }

    virtual void InitializeComponent()
    {
        m_ParamSwitcher.Width = Display::GetInstance().GetWidth();
        m_TestSwitcher.Width = Display::GetInstance().GetWidth();

        m_TestController.SetTarget(*this);

        m_ParamSwitcher.AddPage("Test Params", m_SocketTesterController);
        m_ParamSwitcher.AddPage("Socket Params", m_SocketTester.GetNodeControllerPage());
        m_ParamSwitcher.AddPage("Tx Params", m_SocketTester.GetGeneratorControllerPage());
        m_ParamSwitcher.AddPage("Log Params", m_TestController);

        m_ParamPage.Add(m_ParamSwitcher);
        m_ParamPage.Add(m_BlankLine);

        // ##? for displaing on the console
        Display::GetInstance().EnableConsoleOutput();
        m_ParamPage.Show(Display::GetInstance());
    }

    virtual void SetParam()
    {
        SocketType type = m_SocketTester.GetSelectingSocketType();
        if(type == SOCKET_TYPE_UDP)
        {
            m_pNode = new UdpNode();
        }
        else
        {
            NN_ASSERT(false);
        }

        m_SocketTester.SetTarget(*m_pNode);
        m_NodeViewer.SetTarget(*m_pNode);
        m_SocketTester.SetGenerator(m_Generator);
        m_SocketTesterViewer.SetTarget(m_SocketTester);
        m_SocketResultViewer.SetTarget(m_SocketTester);
        m_TimerViewer.SetTarget(m_SocketTester);

        m_RssiController.SetNode(m_pNode);
        m_RssiController.SetTarget(m_RssiStatistics);
        m_RssiViewer.SetTarget(m_RssiStatistics);

        if(m_IsAutoTest)
        {
            m_SocketTesterController.Set(m_TestParam, m_SocketParam, m_TxParam);

            m_TestPage.Add(m_HeaderPage);
            m_TestPage.Add(m_SocketTesterViewer);
            m_TestPage.Add(m_NodeViewer);
            m_TestPage.Add(m_BlankLine);
        }
        else
        {
            m_SocketTesterController.Set();
            m_TestController.Set();
            m_RssiController.Set();

            m_TestPage.Add(m_HeaderPage);
            m_TestPage.Add(m_TestSwitcher);
            m_TestSwitcher.AddPage("Test Summary", m_SocketTesterViewer);
            m_TestSwitcher.AddPage("WLAN Summary", m_NodeViewer);
            m_TestSwitcher.AddPage("WLAN Rssi", m_RssiViewer);
            m_TestSwitcher.AddPage("Remote Controller", m_RcViewer);
            m_TestSwitcher.AddPage("Remote Controller PLR", m_BtPlrViewer);
            m_SocketTesterViewer.SetPadText(Button::X, "Switch SlotMode(Ukyo only)");
            m_NodeViewer.SetPadText(Button::X, "Switch SlotMode(Ukyo only)");
            m_RcViewer.SetPadText(Button::X, "Switch SlotMode(Ukyo only)");
            m_BtPlrViewer.SetPadText(Button::X, "Switch SlotMode(Ukyo only)");

            m_TestPage.Add(m_TimerViewer);
        }

        m_BtPlrViewer.Clear();
        m_RssiViewer.Clear();

        PacketLogger& logger = PacketLogger::GetInstance();
        if( logger.IsEnabled() )
        {
            // ログ書き出しのための初期化
            string fileName = GetMacAddressStr("") + "_" + GetTodayStr() + "_InfraTest";
            logger.Initialize("wit", fileName.c_str());
        }

        // start update thread
        NN_ASSERT(m_IsThreadRunning == false);
        NN_ASSERT(m_IsTerminationRequested == false);
        static NN_ALIGNAS(4096) char s_UpdateThreadStack[4096 * 2];
        nn::Result result = nn::os::CreateThread(&m_UpdateThread, UpdateThreadFunction, this, s_UpdateThreadStack, sizeof(s_UpdateThreadStack), nn::os::DefaultThreadPriority - 1);
        NN_ASSERT( result.IsSuccess(), "UpdateThread can not run.");
        m_IsThreadRunning = true;
        nn::os::SetThreadNamePointer(&m_UpdateThread, "WitUpdateThread");
        nn::os::StartThread(&m_UpdateThread);

        // start test
        m_SocketTester.Execute();

        // to wait for tester to set params and start master/client
        Sleep(nn::TimeSpan::FromMilliSeconds(100));

        m_RssiController.Start();
    }

    virtual void ProcessTest()
    {
        if( !m_IsAutoTest )
        {
            if( m_SocketTester.IsTxReady() )
            {
                m_SocketTesterViewer.SetPadText(Button::A, "Start Tx");
                m_NodeViewer.SetPadText(Button::A, "Start Tx");
                m_RcViewer.SetPadText(Button::A, "Start Tx");
                m_BtPlrViewer.SetPadText(Button::A, "Start Tx");
                m_RssiViewer.SetPadText(Button::A, "Start Tx");
            }
            else
            {
                m_SocketTesterViewer.ErasePadText(Button::A);
                m_NodeViewer.ErasePadText(Button::A);
                m_RcViewer.ErasePadText(Button::A);
                m_BtPlrViewer.ErasePadText(Button::A);
                m_RssiViewer.ErasePadText(Button::A);
            }
        }
    }

    virtual void GenerateResult()
    {
        if( m_IsAutoTest )
        {
            m_ResultPage.Add(m_HeaderPage);
            m_ResultPage.Add(m_SocketResultViewer);
            m_ResultPage.Add(m_BlankLine);
            m_ResultPage.Add(m_ResultContent);
            m_ResultPage.Add(m_BlankLine);

            m_ResultContent.Text = " TOTAL RESULT : ";
            switch(m_SocketTester.GetResult())
            {
            case RESULT_SUCCESS : m_ResultContent.Text += "  SUCCESS"; break;
            case RESULT_FAILURE : m_ResultContent.Text += "  FAILURE"; break;
            case RESULT_NOT_COMPLETED : m_ResultContent.Text += "  CANCELED"; break;
            default: break;
            }
            m_ResultContent.X = DISPLAY_CONTENT_START_X;
            m_ResultContent.Y = 600;
            m_ResultContent.FitSize();
        }
    }

    virtual void PrintResult()
    {
        string key, value;
        string tmpKey, tmpValue;

        if( m_pNode == nullptr )
        {
            // return if parameters are not set
            return;
        }

        m_SocketResultViewer.PrintResult(tmpKey, tmpValue);
        key += tmpKey;
        value += tmpValue;

        key += ", Final result\n";
        if(m_SocketTester.GetResult() == RESULT_SUCCESS)
        {
            value += ", success\n";
        }
        else
        {
            value += ", failure\n";
        }

        if( m_IsAutoTest )
        {
            NN_LOG(key.c_str());
            NN_LOG(value.c_str());
        }
        else
        {
#ifdef HID_SHELL_MODE
            NN_LOG(key.c_str());
            NN_LOG(value.c_str());
#endif

            PacketLogger& logger = PacketLogger::GetInstance();
            if( logger.IsEnabled() )
            {
                logger.Write("\n");
                logger.Write(key);
                logger.Write(value);
            }
        }
    }

    virtual void UpdateStats()
    {
        BluetoothPlrStatistics &stats = BluetoothPlrStatistics::GetInstance();
        while( !m_IsTerminationRequested )
        {
            stats.Update();
            Sleep(nn::TimeSpan::FromMilliSeconds(10));
        }
    }

public:

    InfraTest()
    {
        Initialize();
    }

    InfraTest(const TestParam& testParam, const SocketParam& socketParam, const DataGeneratorParam& txParam)
    {
        Initialize();

        m_IsAutoTest = true;
        m_TestParam = testParam;
        m_SocketParam = socketParam;
        m_TxParam = txParam;
    }

    virtual ~InfraTest()
    {
    }

    virtual void Initialize()
    {
        m_pNode = nullptr;
        Clear();
        m_IsThreadRunning = false;
        m_IsTerminationRequested = false;

        m_IsAutoTest = false;

        m_HeaderPage.Add(m_BlankLine);
        m_BlankLine.Text = " ";

        m_HeaderPage.Add(m_SeparatorLine);
        m_SeparatorLine.Text = DISPLAY_SEPARATOR;
        m_SeparatorLine.Y = 1000; // 表示させない
        m_SeparatorLine.Width = 400;

        m_SocketTesterController.SetTarget(m_SocketTester);
    }

    virtual void Exit()
    {
        Clear();

        if( m_IsThreadRunning )
        {
            m_IsTerminationRequested = true;
            nn::os::DestroyThread(&m_UpdateThread);
            m_IsThreadRunning = false;
            m_IsTerminationRequested = false;
        }

        Test::Exit();
    }

    virtual void Clear()
    {
        m_BtPlrViewer.Clear();
        m_SocketTester.Exit();
        m_RssiController.Stop();

        if(m_pNode)
        {
            m_pNode->ClearStatistics();
            delete m_pNode;
            m_pNode = nullptr;

            // SocketTester で target が nullptr であるかの判定を使うためここで設定する
            m_SocketTester.SetTarget(*m_pNode);
        }
    }
};

/*---------------------------------------------------------------------------
           LcsMasterTest
---------------------------------------------------------------------------*/
class LcsMasterTest : public LocalTest
{
private:

    LcsMaster                m_Master;
    FixedSizeDataTransmitter m_Generator;
    MultiDataSink            m_Sink;

    TestParam                m_TestParam;
    MasterParam              m_MasterParam;
    DataGeneratorParam       m_TxParam;

    LcsSocketController      m_SocketController;
    TestController           m_TestController;

    LcsMasterViewer          m_NodeViewer;
    SocketTxViewer           m_SocketTxViewer;
    SocketRxViewer           m_SocketRxViewer;
    NPadViewer               m_RcViewer;
    BtPlrViewer              m_BtPlrViewer;
    ReceiveEventHandler(MultiDataSink) * m_pHandler;

    Page                     m_HeaderPage;
    Label                    m_SeparatorLine;
    Label                    m_BlankLine;

    RssiStatistics           m_RssiStatistics;
    RssiViewer               m_RssiViewer;
    RssiController           m_RssiController;

    bool                     m_IsTerminationRequested;
    bool                     m_IsThreadRunning;
    nn::os::ThreadType       m_UpdateThread;

    // function to test
    LcsMasterPerformanceTester m_MasterTester;

protected:

    virtual bool IsTestEnded()
    {
        return !m_MasterTester.IsTesting();
    }

    virtual void InitializeComponent()
    {
        m_ParamSwitcher.Width = Display::GetInstance().GetWidth();
        m_TestSwitcher.Width = Display::GetInstance().GetWidth();

        m_TestController.SetTarget(*this);
        m_SocketController.SetTarget(*m_Master.GetSocket());

        m_NodeViewer.SetTarget(m_Master);
        m_SocketTxViewer.SetTarget(*m_Master.GetSocket());
        m_SocketTxViewer.SetDataGenerator(&m_Generator);
        m_SocketRxViewer.SetTarget(*m_Master.GetSocket());

        m_RssiController.SetNode(&m_Master);
        m_RssiController.SetTarget(m_RssiStatistics);
        m_RssiViewer.SetTarget(m_RssiStatistics);

        m_MasterTester.SetGenerator(m_Generator);
        m_MasterTester.SetSink(m_Sink);

        m_ParamSwitcher.AddPage("Master Params", m_MasterTester.GetNodeControllerPage());
        m_ParamSwitcher.AddPage("Socket Params", m_SocketController);
        m_ParamSwitcher.AddPage("Tx Params", m_MasterTester.GetGeneratorControllerPage());
        m_ParamSwitcher.AddPage("Log Params", m_TestController);

        m_ParamPage.Add(m_ParamSwitcher);
        m_ParamPage.Add(m_BlankLine);

        // ##? for displaing on the console
        Display::GetInstance().EnableConsoleOutput();
        m_ParamPage.Show(Display::GetInstance());
    }

    virtual void SetParam()
    {
        if(m_IsAutoTest)
        {
            m_MasterTester.SetTargetParam();
            m_MasterTester.SetTxParam();

            m_TestPage.Add(m_HeaderPage);
            m_TestPage.Add(m_NodeViewer);
            m_TestPage.Add(m_BlankLine);
        }
        else
        {
            m_MasterTester.SetTargetParam();
            m_MasterTester.SetTxParam();

            m_SocketController.Set();
            m_TestController.Set();
            m_RssiController.Set();

            m_TestPage.Add(m_HeaderPage);
            m_TestPage.Add(m_TestSwitcher);
            m_TestSwitcher.AddPage("WLAN Summary", m_NodeViewer);
            m_TestSwitcher.AddPage("WLAN Tx", m_SocketTxViewer);
            m_TestSwitcher.AddPage("WLAN Rx", m_SocketRxViewer);
            m_TestSwitcher.AddPage("WLAN Rssi", m_RssiViewer);
            m_TestSwitcher.AddPage("Remote Controller", m_RcViewer);
            m_TestSwitcher.AddPage("Remote Controller PLR", m_BtPlrViewer);
            m_RcViewer.ErasePadText(Button::X);
        }

        m_BtPlrViewer.Clear();
        m_RssiViewer.Clear();

        PacketLogger& logger = PacketLogger::GetInstance();
        if( logger.IsEnabled() )
        {
            // ログ書き出しのための初期化
            string fileName = GetMacAddressStr("") + "_" + GetTodayStr() + "_LcsMasterTest";
            logger.Initialize("wit", fileName.c_str());
        }

        // start update thread
        NN_ASSERT(m_IsThreadRunning == false);
        NN_ASSERT(m_IsTerminationRequested == false);
        static NN_ALIGNAS(4096) char s_UpdateThreadStack[4096 * 2];
        nn::Result result = nn::os::CreateThread(&m_UpdateThread, UpdateThreadFunction, this, s_UpdateThreadStack, sizeof(s_UpdateThreadStack), nn::os::DefaultThreadPriority - 1);
        NN_ASSERT( result.IsSuccess(), "UpdateThread can not run.");
        m_IsThreadRunning = true;
        nn::os::SetThreadNamePointer(&m_UpdateThread, "WitUpdateThread");
        nn::os::StartThread(&m_UpdateThread);

        // start test
        m_MasterTester.Execute();

        // to wait for tester to set params and start master/client
        Sleep(nn::TimeSpan::FromMilliSeconds(100));

        m_RssiController.Start();
    }

    virtual void ProcessTest()
    {
        if( Pad::GetInstance().IsTrigger(Button::X) )
        {
            if( Display::IsEnabledDisplayOutput() )
            {
                Display::DisableDisplayOutput();
            }
            else
            {
                Display::EnableDisplayOutput();
            }
        }
    }

    virtual void GenerateResult()
    {
        if(m_IsAutoTest)
        {
            m_ResultPage.Add(m_HeaderPage);
            m_ResultPage.Add(m_BlankLine);
            m_ResultPage.Add(m_ResultContent);
            m_ResultPage.Add(m_BlankLine);

            m_ResultContent.Text = " TOTAL RESULT : ";
            switch(m_MasterTester.GetResult())
            {
            case RESULT_SUCCESS : m_ResultContent.Text += "  SUCCESS"; break;
            case RESULT_FAILURE : m_ResultContent.Text += "  FAILURE"; break;
            case RESULT_NOT_COMPLETED : m_ResultContent.Text += "  CANCELED"; break;
            default: break;
            }
            m_ResultContent.X = DISPLAY_CONTENT_START_X;
            m_ResultContent.Y = 600;
            m_ResultContent.FitSize();
        }
    }

    virtual void PrintResult()
    {
    }

    virtual void UpdateStats()
    {
        BluetoothPlrStatistics &stats = BluetoothPlrStatistics::GetInstance();
        while( !m_IsTerminationRequested )
        {
            stats.Update();
            if( m_State == STATE_TESTING )
            {
                m_Sink.UpdateLastStatistics();
                m_Master.UpdateLastTxStats();
                m_Master.UpdateLastRxStats();
            }
            Sleep(nn::TimeSpan::FromMilliSeconds(10));
        }
    }

public:

    LcsMasterTest()
    {
        Initialize();
    }

    LcsMasterTest(const TestParam& testParam, const MasterParam& masterParam, const DataGeneratorParam& txParam)
    {
        Initialize();

        m_IsAutoTest = true;
        m_TestParam = testParam;
        m_MasterParam = masterParam;
        m_TxParam = txParam;
    }

    virtual ~LcsMasterTest()
    {
        delete m_pHandler;
    }

    virtual void Initialize()
    {
        Clear();
        m_IsThreadRunning = false;
        m_IsTerminationRequested = false;

        m_IsAutoTest = false;
        m_pHandler = new ReceiveEventHandler(MultiDataSink)(m_Sink, &MultiDataSink::Sink);

        m_HeaderPage.Add(m_BlankLine);
        m_BlankLine.Text = " ";

        m_HeaderPage.Add(m_SeparatorLine);
        m_SeparatorLine.Text = DISPLAY_SEPARATOR;
        m_SeparatorLine.Y = 1000; // 表示させない
        m_SeparatorLine.Width = 400;

        m_Master.SetReceiveEventHandler(*m_pHandler);
        m_MasterTester.SetTarget(m_Master);
        //##? ここで設定すると Master が送信できない
        //m_MasterTester.SetGenerator(m_Generator);
    }

    virtual void Exit()
    {
        Clear();

        if( m_IsThreadRunning )
        {
            m_IsTerminationRequested = true;
            nn::os::DestroyThread(&m_UpdateThread);
            m_IsThreadRunning = false;
            m_IsTerminationRequested = false;
        }

        if( !Display::IsEnabledDisplayOutput() )
        {
            Display::EnableDisplayOutput();
        }

        Test::Exit();
    }

    virtual void Clear()
    {
        m_MasterTester.Exit();
        m_RssiController.Stop();
        m_Master.ClearStatistics();
        m_BtPlrViewer.Clear();
        m_RssiViewer.Clear();

        m_Sink.Clear();
    }

};

/*---------------------------------------------------------------------------
           LcsClientTest
---------------------------------------------------------------------------*/
class LcsClientTest : public LocalTest
{
private:

    LcsClient                m_Client;
    FixedSizeDataTransmitter m_Generator;
    MultiDataSink            m_Sink;

    TestParam                m_TestParam;
    ClientParam              m_ClientParam;
    DataGeneratorParam       m_TxParam;

    LcsSocketController      m_SocketController;
    TestController           m_TestController;

    LcsClientViewer          m_NodeViewer;
    SocketTxViewer           m_SocketTxViewer;
    SocketRxViewer           m_SocketRxViewer;
    NPadViewer               m_RcViewer;
    BtPlrViewer              m_BtPlrViewer;
    ReceiveEventHandler(MultiDataSink) * m_pHandler;

    Page                     m_HeaderPage;
    Label                    m_SeparatorLine;
    Label                    m_BlankLine;

    RssiStatistics           m_RssiStatistics;
    RssiViewer               m_RssiViewer;
    RssiController           m_RssiController;

    bool                     m_IsTerminationRequested;
    bool                     m_IsThreadRunning;
    nn::os::ThreadType       m_UpdateThread;

    // function to test
    LcsClientPerformanceTester m_ClientTester;

protected:

    virtual bool IsTestEnded()
    {
        return !m_ClientTester.IsTesting();
    }

    virtual void InitializeComponent()
    {
        m_ParamSwitcher.Width = Display::GetInstance().GetWidth();
        m_TestSwitcher.Width = Display::GetInstance().GetWidth();

        m_TestController.SetTarget(*this);
        m_SocketController.SetTarget(*m_Client.GetSocket());

        m_NodeViewer.SetTarget(m_Client);
        m_SocketTxViewer.SetTarget(*m_Client.GetSocket());
        m_SocketTxViewer.SetDataGenerator(&m_Generator);
        m_SocketRxViewer.SetTarget(*m_Client.GetSocket());

        m_RssiController.SetNode(&m_Client);
        m_RssiController.SetTarget(m_RssiStatistics);
        m_RssiViewer.SetTarget(m_RssiStatistics);

        m_ClientTester.SetGenerator(m_Generator);
        m_ClientTester.SetSink(m_Sink);

        m_ParamSwitcher.AddPage("Client Params", m_ClientTester.GetNodeControllerPage());
        m_ParamSwitcher.AddPage("Socket Params", m_SocketController);
        m_ParamSwitcher.AddPage("Tx Params", m_ClientTester.GetGeneratorControllerPage());
        m_ParamSwitcher.AddPage("Log Params", m_TestController);

        m_ParamPage.Add(m_ParamSwitcher);
        m_ParamPage.Add(m_BlankLine);

        // ##? for displaing on the console
        Display::GetInstance().EnableConsoleOutput();
        m_ParamPage.Show(Display::GetInstance());
    }

    virtual void SetParam()
    {
        if(m_IsAutoTest)
        {
            m_ClientTester.SetTargetParam();
            m_ClientTester.SetTxParam();

            m_TestPage.Add(m_HeaderPage);
            m_TestPage.Add(m_NodeViewer);
            m_TestPage.Add(m_BlankLine);
        }
        else
        {
            m_ClientTester.SetTargetParam();
            m_ClientTester.SetTxParam();

            m_SocketController.Set();
            m_TestController.Set();
            m_RssiController.Set();

            m_TestPage.Add(m_HeaderPage);
            m_TestPage.Add(m_TestSwitcher);
            m_TestSwitcher.AddPage("WLAN Summary", m_NodeViewer);
            m_TestSwitcher.AddPage("WLAN Tx", m_SocketTxViewer);
            m_TestSwitcher.AddPage("WLAN Rx", m_SocketRxViewer);
            m_TestSwitcher.AddPage("WLAN Rssi", m_RssiViewer);
            m_TestSwitcher.AddPage("Remote Controller", m_RcViewer);
            m_TestSwitcher.AddPage("Remote Controller PLR", m_BtPlrViewer);
            m_RcViewer.ErasePadText(Button::X);
        }

        m_BtPlrViewer.Clear();
        m_RssiViewer.Clear();

        PacketLogger& logger = PacketLogger::GetInstance();
        if( logger.IsEnabled() )
        {
            // ログ書き出しのための初期化
            string fileName = GetMacAddressStr("") + "_" + GetTodayStr() + "_LcsClientTest";
            logger.Initialize("wit", fileName.c_str());
        }

        // start update thread
        NN_ASSERT(m_IsThreadRunning == false);
        NN_ASSERT(m_IsTerminationRequested == false);
        static NN_ALIGNAS(4096) char s_UpdateThreadStack[4096 * 2];
        nn::Result result = nn::os::CreateThread(&m_UpdateThread, UpdateThreadFunction, this, s_UpdateThreadStack, sizeof(s_UpdateThreadStack), nn::os::DefaultThreadPriority - 1);
        NN_ASSERT( result.IsSuccess(), "UpdateThread can not run.");
        m_IsThreadRunning = true;
        nn::os::SetThreadNamePointer(&m_UpdateThread, "WitUpdateThread");
        nn::os::StartThread(&m_UpdateThread);

        // start test
        m_ClientTester.Execute();

        // to wait for tester to set params and start master/client
        Sleep(nn::TimeSpan::FromMilliSeconds(100));

        m_RssiController.Start();
    }

    virtual void ProcessTest()
    {
        if( Pad::GetInstance().IsTrigger(Button::X) )
        {
            if( Display::IsEnabledDisplayOutput() )
            {
                Display::DisableDisplayOutput();
            }
            else
            {
                Display::EnableDisplayOutput();
            }
        }
    }

    virtual void GenerateResult()
    {
        if( m_IsAutoTest )
        {
            m_ResultPage.Add(m_HeaderPage);
            m_ResultPage.Add(m_BlankLine);
            m_ResultPage.Add(m_ResultContent);
            m_ResultPage.Add(m_BlankLine);

            m_ResultContent.Text = " TOTAL RESULT : ";
            switch(m_ClientTester.GetResult())
            {
            case RESULT_SUCCESS : m_ResultContent.Text += "  SUCCESS"; break;
            case RESULT_FAILURE : m_ResultContent.Text += "  FAILURE"; break;
            case RESULT_NOT_COMPLETED : m_ResultContent.Text += "  CANCELED"; break;
            default: break;
            }
            m_ResultContent.X = DISPLAY_CONTENT_START_X;
            m_ResultContent.Y = 600;
            m_ResultContent.FitSize();
        }
    }

    virtual void PrintResult()
    {
    }

    virtual void UpdateStats()
    {
        BluetoothPlrStatistics &stats = BluetoothPlrStatistics::GetInstance();
        while( !m_IsTerminationRequested )
        {
            stats.Update();
            if( m_State == STATE_TESTING )
            {
                m_Sink.UpdateLastStatistics();
                m_Client.UpdateLastTxStats();
                m_Client.UpdateLastRxStats();
            }
            Sleep(nn::TimeSpan::FromMilliSeconds(10));
        }
    }

public:

    LcsClientTest()
    {
        Initialize();
    }

    LcsClientTest(const TestParam& testParam, const ClientParam& clientParam, const DataGeneratorParam& txParam)
    {
        Initialize();

        m_IsAutoTest = true;
        m_TestParam = testParam;
        m_ClientParam = clientParam;
        m_TxParam = txParam;
    }

    virtual ~LcsClientTest()
    {
        delete m_pHandler;
    }

    virtual void Initialize()
    {
        Clear();
        m_IsThreadRunning = false;
        m_IsTerminationRequested = false;

        m_IsAutoTest = false;
        m_pHandler = new ReceiveEventHandler(MultiDataSink)(m_Sink, &MultiDataSink::Sink);

        m_HeaderPage.Add(m_BlankLine);
        m_BlankLine.Text = " ";

        m_HeaderPage.Add(m_SeparatorLine);
        m_SeparatorLine.Text = DISPLAY_SEPARATOR;
        m_SeparatorLine.Y = 1000; // 表示させない
        m_SeparatorLine.Width = 400;

        m_Client.SetReceiveEventHandler(*m_pHandler);
        m_ClientTester.SetTarget(m_Client);
        //##? ここで設定すると Client が送信できない
        //m_ClientTester.SetGenerator(m_Generator);
    }

    virtual void Exit()
    {
        Clear();

        if( m_IsThreadRunning )
        {
            m_IsTerminationRequested = true;
            nn::os::DestroyThread(&m_UpdateThread);
            m_IsThreadRunning = false;
            m_IsTerminationRequested = false;
        }

        if( !Display::IsEnabledDisplayOutput() )
        {
            Display::EnableDisplayOutput();
        }

        Test::Exit();
    }

    virtual void Clear()
    {
        m_ClientTester.Exit();
        m_RssiController.Stop();
        m_Client.ClearStatistics();
        m_BtPlrViewer.Clear();

        m_Sink.Clear();
    }

};


/*---------------------------------------------------------------------------
           ScanTest
---------------------------------------------------------------------------*/
class ScanTest : public Test
{
protected:

    Scanner*           m_pScanner;
    ScannerController  m_ScannerController;
    ScannerViewer      m_ScannerViewer;

protected:

    virtual void InitializeComponent()
    {
        NN_ASSERT(m_pScanner != nullptr);
        m_ParamSwitcher.Width = Display::GetInstance().GetWidth();
        m_TestSwitcher.Width = Display::GetInstance().GetWidth();

        m_ScannerController.SetTarget(*m_pScanner);
        m_ScannerViewer.SetTarget(*m_pScanner);

        m_TestSwitcher.AddPage("Scan Params", m_ScannerController);
        m_TestSwitcher.AddPage("Scan Result", m_ScannerViewer);
    }

    virtual void Execute()
    {
        switch(m_State)
        {
        case STATE_READY:
            {
                InitializeComponent();
                m_State = STATE_TESTING;
            }
            break;

        case STATE_TESTING:
            {
                m_TestSwitcher.Show(Display::GetInstance());
            }
            break;

        default:
            {
                NN_ASSERT(false);
            }
            break;
        }
    }

public:

    ScanTest()
    {
        m_pScanner = nullptr;
    }

    virtual ~ScanTest()
    {
    }

    virtual void Exit()
    {
        m_ScannerController.Stop();
        m_ScannerController.Finalize();
        Test::Exit();
    }

    virtual bool IsTestEnded(){ return false; }
    virtual void SetParam(){}
    virtual void GenerateResult(){}
};

/*---------------------------------------------------------------------------
           LocalScanTest
---------------------------------------------------------------------------*/
class LocalScanTest : public ScanTest
{
private:
protected:
public:

    LocalScanTest()
    {
        m_pScanner = new LocalScanner();
    }

    virtual ~LocalScanTest()
    {
        delete m_pScanner;
    }
};

/*---------------------------------------------------------------------------
           InfraScanTest
---------------------------------------------------------------------------*/
class InfraScanTest : public ScanTest
{
private:
protected:
public:

    InfraScanTest()
    {
        m_pScanner = new InfraScanner();
    }

    virtual ~InfraScanTest()
    {
        delete m_pScanner;
    }
};

/*---------------------------------------------------------------------------
           RcViewer
---------------------------------------------------------------------------*/
class RcViewer : public Test
{
private:

    static const nn::TimeSpan LoggingInterval;

    NPadViewer              m_NpadViewer;
    BtPlrViewer             m_BtPlrViewer;

    bool                    m_IsTerminationRequested;
    bool                    m_IsThreadRunning;
    nn::os::ThreadType      m_UpdateThread;

    nn::os::Tick            m_StartTick;
    nn::os::Tick            m_LastUpdateTick;

protected:

    static void UpdateThreadFunction(void* arg)
    {
        DBG_VLOG("Thread start - UpdateThread");
        reinterpret_cast<RcViewer*>(arg)->UpdateStats();
        DBG_VLOG("Thread stop - UpdateThread");
    }

    virtual void InitializeComponent()
    {
        m_ParamSwitcher.Width = Display::GetInstance().GetWidth();
        m_TestSwitcher.Width = Display::GetInstance().GetWidth();

        m_TestSwitcher.AddPage("RC Summary", m_NpadViewer);
        m_TestSwitcher.AddPage("RC PLR", m_BtPlrViewer);
        m_BtPlrViewer.SetPadText(Button::X, "Switch SlotMode(Ukyo only)");

        NPad::GetInstance().EnableIrConfiguration(true);
    }

    virtual void Execute()
    {
        switch(m_State)
        {
        case STATE_READY:
            {
                InitializeComponent();
                m_BtPlrViewer.Clear();
                m_State = STATE_TESTING;
                m_StartTick = nn::os::GetSystemTick();
                m_LastUpdateTick = nn::os::Tick(0);

                // start update thread
                NN_ASSERT(m_IsThreadRunning == false);
                NN_ASSERT(m_IsTerminationRequested == false);
                static NN_ALIGNAS(4096) char s_UpdateThreadStack[4096 * 2];
                nn::Result result = nn::os::CreateThread(&m_UpdateThread, UpdateThreadFunction, this, s_UpdateThreadStack, sizeof(s_UpdateThreadStack), nn::os::DefaultThreadPriority - 1);
                NN_ASSERT( result.IsSuccess(), "UpdateThread can not run.");
                m_IsThreadRunning = true;
                nn::os::SetThreadNamePointer(&m_UpdateThread, "WitUpdateThread");
                nn::os::StartThread(&m_UpdateThread);
            }
            break;

        case STATE_TESTING:
            {
                nn::os::Tick tick = nn::os::GetSystemTick();
                if( (tick - m_LastUpdateTick).ToTimeSpan() >= LoggingInterval )
                {
                    Display::GetInstance().EnableConsoleOutput();
                    m_LastUpdateTick = tick;
                }

                m_TestSwitcher.Show(Display::GetInstance());

                Display::GetInstance().DisableConsoleOutput();
            }
            break;

        default:
            {
                NN_ASSERT(false);
            }
            break;
        }
    }

    virtual void UpdateStats()
    {
        BluetoothPlrStatistics &stats = BluetoothPlrStatistics::GetInstance();
        while( !m_IsTerminationRequested )
        {
            stats.Update();
            Sleep(nn::TimeSpan::FromMilliSeconds(10));
        }
    }

public:

    RcViewer()
    {
        m_IsThreadRunning = false;
        m_IsTerminationRequested = false;

        m_StartTick = nn::os::Tick(0);
        m_LastUpdateTick = nn::os::Tick(0);
    }

    virtual ~RcViewer()
    {
    }

    virtual void Exit()
    {
        if( m_IsThreadRunning )
        {
            m_IsTerminationRequested = true;
            nn::os::DestroyThread(&m_UpdateThread);
            m_IsThreadRunning = false;
            m_IsTerminationRequested = false;
        }

        NPad::GetInstance().EnableIrConfiguration(false);
        Test::Exit();
    }

    virtual bool IsTestEnded(){ return false; }
    virtual void SetParam(){}
    virtual void GenerateResult(){}
};


/*---------------------------------------------------------------------------
           HostDrivenDetectorTest
---------------------------------------------------------------------------*/
class HostDrivenDetectorTest : public LocalTest
{
private:

    Detector                m_Detector;
    DataGenerator           m_Generator;
    MultiNdhpSink           m_NdhpSink;

    TestController          m_TestController;
    HdDetectorPerformanceTesterController m_DetectorTesterController;

    DetectorViewer          m_NodeViewer;
    MultiNdhpSinkSwitchableViewer    m_NdhpSinkViewer;
    NPadViewer              m_RcViewer;
    BtPlrViewer             m_BtPlrViewer;
    ReceiveEventHandler(MultiNdhpSink) * m_pAfHandler;

    TimerViewer<Detector>   m_TimerViewer;
    Page                    m_HeaderPage;
    Label                   m_SeparatorLine;
    Label                   m_BlankLine;

    RssiStatistics          m_RssiStatistics;
    RssiViewer              m_RssiViewer;
    RssiController          m_RssiController;

    bool                    m_IsTerminationRequested;
    bool                    m_IsThreadRunning;
    nn::os::ThreadType      m_UpdateThread;

    // function to test
    HostDrivenDetectorPerformanceTester m_DetectorTester;

protected:

    virtual bool IsTestEnded()
    {
        return !m_DetectorTester.IsTesting();
    }

    virtual void InitializeComponent()
    {
        m_ParamSwitcher.Width = Display::GetInstance().GetWidth();
        m_TestSwitcher.Width = Display::GetInstance().GetWidth();

        m_TestController.SetTarget(*this);

        m_NodeViewer.SetTarget(m_Detector);

        m_DetectorTester.SetGenerator(m_Generator);
        m_TimerViewer.SetTarget(m_DetectorTester);

        m_ParamSwitcher.AddPage("Test Params", m_DetectorTesterController);
        m_ParamSwitcher.AddPage("Aloe Params", m_DetectorTester.GetNodeControllerPage());
        m_ParamSwitcher.AddPage("Log Params", m_TestController);

        m_ParamPage.Add(m_ParamSwitcher);
        m_ParamPage.Add(m_BlankLine);

        NPad::GetInstance().EnableIrConfiguration(true);

        // ##? for displaing on the console
        Display::GetInstance().EnableConsoleOutput();
        m_ParamPage.Show(Display::GetInstance());
    }

    virtual void SetParam()
    {
        if( m_IsAutoTest )
        {
            m_TestPage.Add(m_HeaderPage);
            m_TestPage.Add(m_NodeViewer);
            m_TestPage.Add(m_NdhpSinkViewer);
            m_TestPage.Add(m_BlankLine);
        }
        else
        {
            m_DetectorTesterController.Set();
            m_TestController.Set();

            m_TestPage.Add(m_HeaderPage);
            m_TestPage.Add(m_TestSwitcher);
            m_TestSwitcher.AddPage("WLAN Summary", m_NodeViewer);
            m_TestSwitcher.AddPage("WLAN Rx (Action Frame)", m_NdhpSinkViewer);
            m_TestSwitcher.AddPage("Remote Controller", m_RcViewer);
            m_TestSwitcher.AddPage("Remote Controller PLR", m_BtPlrViewer);

            m_TestPage.Add(m_TimerViewer);
        }

        m_BtPlrViewer.Clear();
        m_NdhpSinkViewer.Clear();

        // start update thread
        NN_ASSERT(m_IsThreadRunning == false);
        NN_ASSERT(m_IsTerminationRequested == false);
        static NN_ALIGNAS(4096) char s_UpdateThreadStack[4096 * 2];
        nn::Result result = nn::os::CreateThread(&m_UpdateThread, UpdateThreadFunction, this, s_UpdateThreadStack, sizeof(s_UpdateThreadStack), nn::os::DefaultThreadPriority - 1);
        NN_ASSERT( result.IsSuccess(), "UpdateThread can not run.");
        m_IsThreadRunning = true;
        nn::os::SetThreadNamePointer(&m_UpdateThread, "WitUpdateThread");
        nn::os::StartThread(&m_UpdateThread);

        // start test
        m_DetectorTester.Execute();

        // to wait for tester to set params and start master/client
        Sleep(nn::TimeSpan::FromMilliSeconds(100));
    }

    virtual void ProcessTest()
    {
        if( !m_IsAutoTest )
        {
            if( !m_DetectorTester.IsTxReady() )
            {
                m_NodeViewer.SetPadText(Button::A, "Start HDA");
                m_NdhpSinkViewer.SetPadText(Button::A, "Start HDA");
                m_RcViewer.SetPadText(Button::A, "Start HDA");
                m_BtPlrViewer.SetPadText(Button::A, "Start HDA");
            }
            else
            {
                m_NodeViewer.ErasePadText(Button::A);
                m_NdhpSinkViewer.ErasePadText(Button::A);
                m_RcViewer.ErasePadText(Button::A);
                m_BtPlrViewer.ErasePadText(Button::A);
            }
        }
    }

    virtual void GenerateResult()
    {
    }

    virtual void PrintResult()
    {
    }

    virtual void UpdateStats()
    {
        BluetoothPlrStatistics &stats = BluetoothPlrStatistics::GetInstance();
        while( !m_IsTerminationRequested )
        {
            stats.Update();
            if( m_State == STATE_TESTING )
            {
                m_NdhpSink.UpdateLastStatistics();
            }
            Sleep(nn::TimeSpan::FromMilliSeconds(10));
        }
    }

public:

    HostDrivenDetectorTest()
    {
        Initialize();
    }

    virtual ~HostDrivenDetectorTest()
    {
        delete m_pAfHandler;
    }

    virtual void Initialize()
    {
        Clear();
        m_IsThreadRunning = false;
        m_IsTerminationRequested = false;

        m_IsAutoTest = false;
        m_pAfHandler = new ReceiveEventHandler(MultiNdhpSink)(m_NdhpSink, &MultiNdhpSink::Sink);

        m_HeaderPage.Add(m_BlankLine);
        m_BlankLine.Text = " ";

        m_HeaderPage.Add(m_SeparatorLine);
        m_SeparatorLine.Text = DISPLAY_SEPARATOR;
        m_SeparatorLine.Y = 1000; // 表示させない
        m_SeparatorLine.Width = 400;

        m_DetectorTesterController.SetTarget(m_DetectorTester);
        m_Detector.SetAfReceiveEventHandler(*m_pAfHandler);
        m_DetectorTester.SetTarget(m_Detector);

        m_NdhpSinkViewer.SetTarget(m_NdhpSink);
    }

    virtual void Exit()
    {
        Clear();

        if( m_IsThreadRunning )
        {
            m_IsTerminationRequested = true;
            nn::os::DestroyThread(&m_UpdateThread);
            m_IsThreadRunning = false;
            m_IsTerminationRequested = false;
        }

        Test::Exit();
    }

    virtual void Clear()
    {
        m_DetectorTester.Exit();
        m_Detector.ClearStatistics();
        m_BtPlrViewer.Clear();

        m_NdhpSink.Clear();
    }

};

Test* GenerateAutoTest(int testId, int roleId, const char* filePath);
bool ParseLocalTestFile(const char* filePath, TestParam* testParam, MasterParam* masterParam, ClientParam* clientParam, DataGeneratorParam* dataGeneratorParam);

} // WlanTest
