﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <cmath>
#include <sstream>
#include <iomanip>

#include "Pad.h"
#include "UIControl.h"
#include "Value.h"
#include "Util.h"

namespace WlanTest {

/*!--------------------------------------------------------------------------*
  @brief        選択器

 *---------------------------------------------------------------------------*/
template<typename T>
class Selector : public UIControl{
public:
    Selector()
    {
        current_item = 0;
        cursor_position = 0;
        SetCursor('>');
        CursorSmoothness = 1.1;
        LineHeight = 32;
        FontSize = 24;
        Offset = 10;
        NextKey = Button::DOWN;
        BackKey = Button::UP;
        m_IsSideIndicatorEnabled = true;
        TextColor = ToColor(WHITE);
    }

    virtual ~Selector()
    {
        // ##? Selector 内で new していないので delete もしない
        // while(!items.empty())
        // {
        //     delete &(items.back());
        //     items.pop_back();
        // }
    }

    virtual void Register(string name, T* item)
    {
        names.push_back(name);
        items.push_back(item);
    }

    virtual T* GetSelectingItem()
    {
        if( items.size() == 0 )
        {
            return nullptr;
        }
        return items[current_item];
    }

    int Offset;
    int LineHeight;
    int FontSize;
    virtual void SetCursor(char cursor)
    {
        this->cursor[0] = cursor;
        this->cursor[1] = '\0';
    }

    void SetTextColor(Color c)
    {
        TextColor = c;
    }

    bool m_IsSideIndicatorEnabled;
    void SetSideIndicator(bool isEnabled)
    {
        m_IsSideIndicatorEnabled = isEnabled;
    }

    double CursorSmoothness;
    Button NextKey;
    Button BackKey;

    virtual void ShowImpl(Display& display)
    {
        display.SetColor(TextColor);
        int available = Height / LineHeight;
        static int start = 0;

        if( (start + available - 1) < current_item)
        {
            start += current_item - (start + available - 1);
        }
        if( start > current_item )
        {
            start -= start - current_item;
        }

        if(start > 0 && m_IsSideIndicatorEnabled)
        {
            display.DrawText(GetX(), GetY() - FontSize / 2, "^");
        }
        if(items.size() > start + available && m_IsSideIndicatorEnabled)
        {
            display.DrawText(GetX(), GetY() + ((available - 1) * LineHeight) + FontSize / 2, "v");
        }

        for(int i=start; i<items.size() && i<start + available; i++)
        {
            int x = GetX();
            int y = GetY() - start * LineHeight;

            //##?
            /*
              if(i == current_item)
              {
              double difference = abs(current_item * LineHeight - cursor_position);
              int sign = current_item * LineHeight < cursor_position ? -1 : 1;
              cursor_position += sign * ceil(difference / CursorSmoothness);
              display.SetFontSize(FontSize);
              display.DrawText(x, y + cursor_position, cursor);    // とりあえず、
              }

              display.DrawText(x + Offset, y + (i * LineHeight), GetItemName(i) );
            */
            // ##?
            string cursor = (i == current_item ? ">" : " ");
            display.DrawText(x + Offset, y + (i * LineHeight), cursor + " " + GetItemName(i) );
        }
    }

    virtual void InputPad(Pad& key)
    {
        if( key.IsTrigger(BackKey) )
        {
            if(current_item == 0)
            {
                current_item = items.size() - 1;
            }
            else
            {
                current_item--;
            }
        }
        else if( key.IsHold(BackKey) )
        {
            if(current_item == 0)
            {
            }
            else
            {
                current_item--;
            }
        }
        else if( key.IsTrigger(NextKey) )
        {
            if(current_item == items.size() - 1)
            {
                current_item = 0;
            }
            else
            {
                current_item++;
            }
        }
        else if( key.IsHold(NextKey) )
        {
            if(current_item == items.size() - 1)
            {
            }
            else
            {
                current_item++;
            }
        }
    }

    virtual string GetItemName()
    {
        return GetItemName(current_item);
    }

    virtual string GetItemName(int index)
    {
        return names[index];
    }

    size_t GetSize()
    {
        return names.size();
    }

    virtual void Clear()
    {
        items.clear();
        names.clear();
        current_item = 0;
        cursor_position = 0;
    }

    void SetCursorIndex(const int index)
    {
        if( 0 <= index && index < items.size() )
        {
            current_item = index;
        }
    }

    int GetCursorIndex()
    {
        return current_item;
    }

protected:

    vector<T*> items;
    vector<string> names;
    int current_item;
    int cursor_position;
    char cursor[2];
    Color TextColor;

};


/*!--------------------------------------------------------------------------*
  @brief        値選択器

 *---------------------------------------------------------------------------*/
class ValueSelector : public Selector<ISelectableValue>
{
public:

    ValueSelector()
    {
        max_name_length = 0;
        NextValueKey = Button::RIGHT;
        BackValueKey = Button::LEFT;
    }

    virtual ~ValueSelector()
    {
    }

    virtual void Register(string name, ISelectableValue* value)
    {
        selector::Register(name, value);

        if(max_name_length < name.length())
        {
            max_name_length = name.length();
        }
    }

    virtual void ShowImpl(Display& display)
    {
        int available = Height / LineHeight;
        static int start = 0;

        if( (start + available - 1) < current_item)
        {
            start += current_item - (start + available - 1);
        }
        if( start > current_item )
        {
            start -= start - current_item;
        }

        if(start > 0 && m_IsSideIndicatorEnabled)
        {
            display.DrawText(GetX(), GetY() - FontSize / 2, "^");
        }
        if(items.size() > start + available && m_IsSideIndicatorEnabled)
        {
            display.DrawText(GetX(), GetY() + ((available - 1) * LineHeight) + FontSize / 2, "v");
        }

        for(int i=start; i<items.size() && i<start + available; ++i)
        {
            int x = GetX();
            int y = GetY() - start * LineHeight;

            //##?
            /*
              if(i == current_item)
              {
              double difference = abs(current_item * LineHeight - cursor_position);
              int sign = current_item * LineHeight < cursor_position ? -1 : 1;
              cursor_position += sign * ceil(difference / CursorSmoothness);
              display.SetFontSize(FontSize);
              display.DrawText(x, y + cursor_position, cursor);    // とりあえず、
              }

              display.DrawText(x + Offset, y + (i * LineHeight), GetItemName(i) );
            */
            // ##?
            string cursor = (i == current_item ? ">" : " ");
            display.DrawText(x + Offset, y + (i * LineHeight), cursor + " " + GetItemName(i) );
        }
    }

    virtual void InputPad(Pad& key)
    {
        selector::InputPad(key);

        if( key.IsTrigger(NextValueKey) )
        {
            selector::items[selector::current_item]->Next(true);
        }
        else if( key.IsHold(NextValueKey) )
        {
            selector::items[selector::current_item]->Next();
        }
        else if( key.IsTrigger(BackValueKey) )
        {
            selector::items[selector::current_item]->Back(true);
        }
        else if( key.IsHold(BackValueKey) )
        {
            selector::items[selector::current_item]->Back();
        }
    }

    typedef Selector<ISelectableValue> selector;
    Button NextValueKey;
    Button BackValueKey;

protected:

    int max_name_length;
    virtual string GetItemName(int index)
    {
        ostringstream oss;
        oss << left << setw(max_name_length) << selector::names[index] << " : " << selector::items[index];

        return oss.str();
    }

};

/*!--------------------------------------------------------------------------*
  @brief        複数値選択器
 *---------------------------------------------------------------------------*/
class MultiValueSelector : public ValueSelector
{
public:

    MultiValueSelector()
    {
        addActionTime = -1;
        removeActionTime = -1;
        AddValueKey = Button::Y;
        RemoveValueKey = Button::B;
    }

    virtual ~MultiValueSelector()
    {
    }

    virtual void Register(string name, ISelectableValue* value)
    {
        ValueSelector::Register(name, value);
    }

    int CountLines(const string& str)
    {
        return count(str.begin(), str.end(), '\n');
    }

    string GenerateSelectedItemsString(const int index, const int availableLineCount)
    {
        int width = (Display::GetInstance().GetWidth() - GetX()) / FontSize;
        if( width <= 0 )
        {
            // 表示領域が狭すぎてアイテムひとつでもはみ出してしまうが、表示するため 1 を設定する
            width = 1;
        }

        const string indent = "    ";
        const string newLine = "\n" + indent;
        string from = items[index]->ToStringSelectingItem();
        string to = indent;
        size_t line = 0;
        uint16_t l = 0;
        // 表示領域内に表示可能な分だけコンマ区切りで表示する
        while( l < from.size() && line < availableLineCount )
        {
            ++line;

            // 表示領域に収まれば終了
            if( from.size() - l <= width )
            {
                to += Trim(from.substr(l)) + newLine;
                break;
            }

            // 表示領域内の最後のコンマを探索
            size_t idx = from.find_last_of(",", l + width);
            if( idx == string::npos )
            {
                // ひとつのアイテムで表示領域を超えるものはそのまま表示する
                idx = from.find(",", l + width);
                if( idx == string::npos )
                {
                    // 一つの場合
                    to = from;
                    l = from.size();
                }
            }
            else
            {
                to += Trim(from.substr(l, idx - l + 1)) + newLine;
                l = idx + 1;
            }
        }
        return to;
    }

    virtual void ShowImpl(Display& display)
    {
        static int start = 0;
        static int last = 0;
        int available = Height / LineHeight;
        bool isInverse = false;

        if( current_item < start )
        {
            start = current_item;
        }
        else if( current_item > last )
        {
            // 開始アイテムのインデックスの検索
            int line = 0;
            for(int i=current_item; i>=0 && line < available; --i)
            {
                if( items[i]->GetSelectingItemCountMax() > 1 )
                {
                    // available に収まるかを確認するため行制限なしで生成する
                    string itemLineStr = GenerateSelectedItemsString(i, 1000);

                    // 項目名とアイテムの行数
                    int newLine = 1 + CountLines(itemLineStr);
                    if( line + newLine > available )
                    {
                        break;
                    }
                    line += newLine;
                }
                else
                {
                    ++line;
                }
                start = i;
            }
        }

        int itemLine = 0;
        for(int i=start; i<items.size() && i<start + available - itemLine; ++i)
        {
            last = i;
            int x = GetX();
            int y = GetY() - start * LineHeight;

            string cursor = (i == current_item ? ">" : " ");
            display.DrawText(x + Offset, y + ((i + itemLine) * LineHeight), cursor + " " + GetItemName(i) );

            // 複数選択可能なアイテムは別途表示領域を設ける
            if( items[i]->GetSelectingItemCountMax() > 1 )
            {
                string itemLineStr = GenerateSelectedItemsString(i, available - (i - start + 1) - itemLine);
                display.DrawText(x + Offset, y + ((i + itemLine + 1) * LineHeight), itemLineStr);
                itemLine += CountLines(itemLineStr);
            }
        }

        if( start > 0 && m_IsSideIndicatorEnabled )
        {
            display.DrawText(GetX(), GetY() - FontSize / 2, "^");
        }
        if( last < items.size() - 1 && m_IsSideIndicatorEnabled )
        {
            display.DrawText(GetX(), GetY() + ((available - 1) * LineHeight) + FontSize / 2, "v");
        }
    }

    virtual void InputPad(Pad& key)
    {
        ValueSelector::InputPad(key);

        if( key.IsTrigger(AddValueKey) )
        {
            ISelectableValue* item = items[current_item];
            if( item->Select(item->GetCurrentIndex()) )
            {
            }
        }
        else if( key.IsTrigger(RemoveValueKey) || key.IsHold(RemoveValueKey) )
        {
            ISelectableValue* item = items[current_item];
            if( item->Deselect(item->GetSelectingItemCount() - 1) )
            {
            }
        }
    }

    virtual int FindItem(const ISelectableValue* value)
    {
        for(int i=0; i<selector::items.size(); ++i)
        {
            if( selector::items[i] == value )
            {
                return i;
            }
        }

        return -1;
    }

    virtual int AddItem(const ISelectableValue* value, const string& name, const int index)
    {
        if( !FindItem(value) )
        {
            return false;
        }

        //selectingItemList[index].push_back(make_pair(value, name));
        return true;
    }

    virtual int RemoveItem(const int index)
    {
        // if( selectingItemList.size() == 0 )
        // {
        //     return false;
        // }

        //selectingItemList[index].pop_back();
        return true;
    }

    Button AddValueKey;
    Button RemoveValueKey;
    //vector<vector<item> > selectingItemsList;

    int addActionTime;
    int removeActionTime;

protected:

    virtual string GetItemName(int index)
    {
        ostringstream oss;
        oss << left << setw(max_name_length) << selector::names[index] << " : " << selector::items[index]->ToStringSelectableItem();

        return oss.str();
    }

private:

};

/*!--------------------------------------------------------------------------*
  @brief        実行可能値選択器

 *---------------------------------------------------------------------------*/
class CommandSelector : public ValueSelector
{
public:

    CommandSelector()
    {
    }

    virtual ~CommandSelector()
    {
    }

    template<class T>
    void Register(string name, SelectableConfigCommand<T>* value)
    {
        ValueSelector::Register(name, value);
        m_CommandList.push_back(value);
    }

    virtual int Execute()
    {
        return Execute(current_item);
    }

    virtual int Execute(int index)
    {
        if( m_CommandList.size() < index )
        {
            return -1;
        }
        return m_CommandList[index]->Execute();
    }

    virtual int Reset()
    {
        return Reset(current_item);
    }

    virtual int Reset(int index)
    {
        if( m_CommandList.size() < index )
        {
            return -1;
        }
        return m_CommandList[index]->Reset();
    }

    virtual int Cancel()
    {
        return Cancel(current_item);
    }

    virtual int Cancel(int index)
    {
        if( m_CommandList.size() < index )
        {
            return -1;
        }
        return m_CommandList[index]->Cancel();
    }

protected:

    vector<IConfigCommand*> m_CommandList;

};

}    // WlanTest
