﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <sstream>
#include <iomanip>
#include <set>
#include "UIControl.h"
#include "DataGenerator.h"
#include "Node.h"
#include "LocalNode.h"
#include "SocketNode.h"
#include "LcsSocket.h"
#include "Detector.h"
#include "Selector.h"
#include "Pad.h"
#include "Rssi.h"

namespace WlanTest {

//class IModelViewer : public Page
//{
//public:
//    virtual void Show(Display& display)
//    {
//        Page::Show(display);
//    };
//};

string GenerateTitle(const string& title, const uint32_t size, const char sep = ':');
void SetUnitConfiguration(Label* pLabel, const string& text, const uint32_t& x, const uint32_t& y);
void SetUnitConfiguration(Label* pLabel, const string& text);
void SetUnitConfiguration(Label* pLabel);

template<typename T>
class ModelViewer : public Page //public IModelViewer
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
    T* target;
private:


/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    ModelViewer() : target(nullptr)
    {}

    virtual ~ModelViewer()
    {}

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    virtual void Show(Display& display)
    {
        Page::Show(display);
    }

    virtual void SetTarget(T& target)
    {
        this->target = &target;
    }

    virtual void UnsetTarget()
    {
        this->target = nullptr;
    }
};

template<typename T>
class SceneViewer : public ModelViewer<T>
{
private:
protected:

    map<Button, string> padTexts;
    Label padLabel;

public:

    SceneViewer()
    {
        SetPadText(Button::START, "Home");
        ModelViewer<T>::Add(padLabel);
        padLabel.X = DISPLAY_PAD_START_X;
        padLabel.Y = DISPLAY_PAD_START_Y;
        padLabel.Text = "";
        padLabel.Width = 2000;
    }

    virtual ~SceneViewer()
    {}

    virtual void ShowImpl(Display& display)
    {
        string text;

        map<Button, string>::iterator it = padTexts.begin();
        while( it != padTexts.end() )
        {
            text += "<" + ToString(it->first) + "> " + it->second + PAD_TEXT_SEPARATOR;
            ++it;
        }
        padLabel.Text = text;

        ModelViewer<T>::ShowImpl(display);
    }

    void SetPadText(const Button& button, const string& text)
    {
        padTexts[button] = text;
    }

    void ErasePadText(const Button& button)
    {
        padTexts.erase(button);
    }

};


class DataLine : public LabelBar
{
public:
    Color BackgroundColor;
    Color TextColor;

    string Source;
    int Aid;
    int Rssi;
    int Data;
    int Error;
    double Now;
    double All;
    double Throughput;
private:
    Label sourceLabel;
    Label aidLabel;
    Label rssiLabel;
    Label dataLabel;
    Label errorLabel;
    Label nowLabel;
    Label allLabel;
    Label thrptLabel;

public:
    DataLine()
    {
        Height = 12;

        Aid   = 0;
        Rssi  = 0;
        Data  = 0;
        Error = 0;
        Now   = 0;
        All   = 0;
        Throughput = 0;

        AddLabel(sourceLabel);
        //AddLabel(aidLabel);
        AddLabel(rssiLabel);
        AddLabel(dataLabel);
        AddLabel(errorLabel);
        AddLabel(nowLabel);
        AddLabel(allLabel);
        AddLabel(thrptLabel);

        sourceLabel.Alignment = MiddleRight;
        sourceLabel.Width = 136;
        sourceLabel.Height = Height;
        aidLabel.Alignment = MiddleRight;
        aidLabel.Width = 26;
        aidLabel.Height = Height;
        rssiLabel.Alignment = MiddleRight;
        rssiLabel.Width = 16;
        rssiLabel.Height = Height;
        dataLabel.Alignment = MiddleRight;
        dataLabel.Width = 64;
        dataLabel.Height = Height;
        errorLabel.Alignment = MiddleRight;
        errorLabel.Width = 56;
        errorLabel.Height = Height;
        nowLabel.Alignment = MiddleRight;
        nowLabel.Width = 40;
        nowLabel.Height = Height;
        allLabel.Alignment = MiddleRight;
        allLabel.Width = 40;
        allLabel.Height = Height;
        thrptLabel.Alignment = MiddleRight;
        thrptLabel.Width = 40;
        thrptLabel.Height = Height;
    }

    virtual ~DataLine(){}

    virtual void ShowImpl(Display& display)
    {
        ostringstream oss;

        sourceLabel.Text = Source;

        oss << Aid;
        aidLabel.Text = oss.str();
        oss.str("");

        oss << Rssi;
        rssiLabel.Text = oss.str();
        oss.str("");

        oss << Data;
        dataLabel.Text = oss.str();
        oss.str("");

        oss << Error;
        errorLabel.Text = oss.str();
        oss.str("");

        oss << setprecision(2) << setw(nowLabel.Width / nowLabel.FontWidth) << Now;
        nowLabel.Text = oss.str();
        oss.str("");

        oss << setprecision(2) << setw(allLabel.Width / allLabel.FontWidth) << All;
        allLabel.Text = oss.str();
        oss.str("");

        oss << setprecision(3) << setw(thrptLabel.Width / thrptLabel.FontWidth) << Throughput;
        thrptLabel.Text = oss.str();
        oss.str("");

        LabelBar::ShowImpl(display);
    }

    virtual void SetColor(Color c)
    {
        sourceLabel.BackgroundColor = c;
        dataLabel.BackgroundColor = c;
        nowLabel.BackgroundColor = c;
        thrptLabel.BackgroundColor = c;
        c.R /= 2;
        c.G /= 2;
        c.B /= 2;
        rssiLabel.BackgroundColor = c;
        errorLabel.BackgroundColor = c;
        allLabel.BackgroundColor = c;

        Color n = ToColor(BLACK);

        sourceLabel.TextColor = n;
        dataLabel.TextColor = n;
        nowLabel.TextColor = n;
        thrptLabel.TextColor = n;
    }
};

class AccessPointViewer : public SceneViewer<AccessPoint>
{
public :
private :

    static const uint32_t TITLE_SIZE_MAX = 10;

    Label chLabel;
    Label rssiLabel;
    Label linkLevelLabel;
    Label ssidLabel;
    Label ssidValueLabel;
    Label bssidLabel;
    Label wepLabel;
    Label wpaLabel;
    Label wpa2Label;
    Label wpsLabel;
    Label beaconIntervalLabel;
    Label capabilityLabel;

    Label infoLabel;
    uint32_t ssidStrWidth;

public :

    AccessPointViewer()
    {
        ssidStrWidth = 16;

        uint32_t w = Display::GetInstance().GetFixedWidth();
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(infoLabel);
        infoLabel.Text = " Not Selected";
        infoLabel.X = x;
        infoLabel.Y = y;
        infoLabel.Height = h;
        infoLabel.FitSize();

        Add(chLabel);
        chLabel.Text = GenerateTitle("Ch", TITLE_SIZE_MAX) + "000 ";
        chLabel.X = x;
        chLabel.Y = (y += h);
        chLabel.Height = h;
        chLabel.FitSize();

        Add(rssiLabel);
        rssiLabel.Text = GenerateTitle("RSSI", TITLE_SIZE_MAX) + "-0000 dBm ";
        rssiLabel.X = x;
        rssiLabel.Y = (y += h);
        rssiLabel.Height = h;
        rssiLabel.FitSize();

        Add(linkLevelLabel);
        linkLevelLabel.Text = GenerateTitle("Link Level", TITLE_SIZE_MAX) + "0 ";
        linkLevelLabel.X = x;
        linkLevelLabel.Y = (y += h);
        linkLevelLabel.Height = h;
        linkLevelLabel.FitSize();

        Add(ssidLabel);
        ssidLabel.Text = GenerateTitle("SSID", TITLE_SIZE_MAX);
        ssidLabel.X = x;
        ssidLabel.Y = (y += h);
        ssidLabel.FitSize();

        Add(ssidValueLabel);
        ssidValueLabel.X = x + TITLE_SIZE_MAX * DEFAULT_FONT_WIDTH;
        ssidValueLabel.Y = y;
        ssidValueLabel.Width = 500 * w;

        Add(bssidLabel);
        bssidLabel.Text = GenerateTitle("BSSID", TITLE_SIZE_MAX) + "XX:XX:XX:XX:XX:XX ";
        bssidLabel.X = x;
        bssidLabel.Y = (y += 3 * h);
        bssidLabel.Height = h;
        bssidLabel.FitSize();

        Add(wepLabel);
        wepLabel.Text = GenerateTitle("WEP", TITLE_SIZE_MAX) + "Unkown ";
        wepLabel.X = x;
        wepLabel.Y = (y += h);
        wepLabel.Height = h;
        wepLabel.FitSize();

        Add(wpaLabel);
        wpaLabel.Text = GenerateTitle("WPA", TITLE_SIZE_MAX) + "Unkown ";
        wpaLabel.X = x;
        wpaLabel.Y = (y += h);
        wpaLabel.Height = h;
        wpaLabel.FitSize();

        Add(wpa2Label);
        wpa2Label.Text = GenerateTitle("WPA2", TITLE_SIZE_MAX) + "Unknown ";
        wpa2Label.X = x;
        wpa2Label.Y = (y += h);
        wpa2Label.Height = h;
        wpa2Label.FitSize();

        Add(wpsLabel);
        wpsLabel.Text = GenerateTitle("WPS", TITLE_SIZE_MAX) + "Unknown ";
        wpsLabel.X = x;
        wpsLabel.Y = (y += h);
        wpsLabel.Height = h;
        wpsLabel.FitSize();

        Add(beaconIntervalLabel);
        beaconIntervalLabel.Text = GenerateTitle("Interval", TITLE_SIZE_MAX) + "00000000 ms ";
        beaconIntervalLabel.X = x;
        beaconIntervalLabel.Y = y;
        beaconIntervalLabel.Height = h;
        beaconIntervalLabel.FitSize();

        Add(capabilityLabel);
        capabilityLabel.Text = GenerateTitle("Capability", TITLE_SIZE_MAX) + "0x0000";
        capabilityLabel.X = x;
        capabilityLabel.Y = y;
        capabilityLabel.Height = h;
        capabilityLabel.FitSize();
    }

    virtual ~AccessPointViewer(){}

    virtual void ShowImpl(Display& display)
    {
        if( target )
        {
            uint32_t h = Display::GetInstance().GetLineHeight();
            ostringstream oss;

            infoLabel.Text = "";

            oss << GenerateTitle("Ch", TITLE_SIZE_MAX) << static_cast<int32_t>(target->GetChannel());
            chLabel.Text = oss.str();
            oss.str("");

            oss << GenerateTitle("RSSI", TITLE_SIZE_MAX) << target->GetRssi() << " dBm";
            rssiLabel.Text = oss.str();
            oss.str("");

            oss << GenerateTitle("Link Level", TITLE_SIZE_MAX) << target->GetLinkLevel();
            linkLevelLabel.Text = oss.str();
            oss.str("");

            ssidLabel.Text = GenerateTitle("SSID", TITLE_SIZE_MAX);

            char buf[nn::wlan::Ssid::SsidHexStringLengthMax] = {0};
            string ssid = target->GetSsid().GetHexString(buf);
            string linedSsid = PackString(ssid, ssidStrWidth, 5);
            ssidValueLabel.Text = DuplicateEscapeSequence(linedSsid, "%").c_str();

            uint32_t y = ssidValueLabel.Y + (Count(linedSsid, '\n')) * (h + 4);
            string bssid = target->GetBssid().GetString(buf);
            bssidLabel.Y = y;
            bssidLabel.Text = GenerateTitle("BSSID", TITLE_SIZE_MAX) + bssid;

            SecurityConfig config = target->GetSecurityConfig();
            wepLabel.Y = (y += h);
            wepLabel.Text = GenerateTitle("WEP", TITLE_SIZE_MAX) + "Unknow";

            wpaLabel.Y = (y += h);
            wpaLabel.Text = GenerateTitle("WPA", TITLE_SIZE_MAX) + ToStringYesNo(config.isWpaSupported, CASE_OPTION_PROPER);

            wpa2Label.Y = (y += h);
            wpa2Label.Text = GenerateTitle("WPA2", TITLE_SIZE_MAX) + ToStringYesNo(config.isWpa2Supported, CASE_OPTION_PROPER);

            wpsLabel.Y = (y += h);
            wpsLabel.Text = GenerateTitle("WPS", TITLE_SIZE_MAX) + ToStringYesNo(config.isWpsSupported, CASE_OPTION_PROPER);

            oss << GenerateTitle("Interval", TITLE_SIZE_MAX) << static_cast<uint16_t>(target->GetBeaconInterval()) << " ms";
            beaconIntervalLabel.Y = (y += h);
            beaconIntervalLabel.Text = oss.str();
            oss.str("");

            oss << GenerateTitle("Capability", TITLE_SIZE_MAX) << "0x"
                << hex << setw(4) << setfill('0') << static_cast<uint16_t>(target->GetCapability());
            capabilityLabel.Y = (y += h);
            capabilityLabel.Text = oss.str();
            oss.str("");
        }
        else
        {
            infoLabel.Text = " Not Selected";
            chLabel.Text = "";
            rssiLabel.Text = "";
            linkLevelLabel.Text = "";
            ssidLabel.Text = "";
            ssidValueLabel.Text = "";
            bssidLabel.Text = "";
            wepLabel.Text = "";
            wpaLabel.Text = "";
            wpa2Label.Text = "";
            wpsLabel.Text = "";
            beaconIntervalLabel.Text = "";
            capabilityLabel.Text = "";
        }

        SceneViewer<AccessPoint>::ShowImpl(display);
    }
};

class ScannerViewer : public SceneViewer<Scanner>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:

    static const uint8_t SSID_SIZE_MAX = 20;

    AccessPointViewer m_ApViewer;

    Selector<AccessPoint> m_Selector;
    Label m_Title;
    Label m_Ch;
    Label m_Rssi;
    Label m_Ssid;
    Label m_Count;
    Label m_Pad;
    Label m_Clear;
    set<uint64_t> m_ApSet;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    virtual void ShowImpl(Display& display)
    {
        uint32_t count = target->GetApCount();
        ostringstream oss;
        oss << count << " APs Found";
        m_Count.Text = oss.str();
        oss.str("");

        AccessPoint* ap = target->GetFirstAccessPoint();
        for(int i = 0; i < count; ++i)
        {
            uint64_t addr = 0;
            memcpy(&addr, ap->GetBssid().GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);

            // 新たな AP であれば追加する
            if( m_ApSet.find(addr) == m_ApSet.end() )
            {
                m_ApSet.insert(addr);

                char buf[nn::wlan::Ssid::SsidHexStringLengthMax] = {0};
                string ssid = ap->GetSsid().GetHexString(buf);

                oss << setw(3) << ap->GetChannel() << ' '
                    << setw(3) << ap->GetRssi() << ' '
                    << DuplicateEscapeSequence(ssid.substr(0, SSID_SIZE_MAX), "%");
                m_Selector.Register(oss.str(), ap);
                oss.str("");
            }

            ap = target->GetNextAccessPoint();
        }

        if( count > 0 )
        {
            SetPadText(Button::A, "Select");
            SetPadText(Button::B, "Clear");
        }
        else
        {
            ErasePadText(Button::A);
            ErasePadText(Button::B);
        }

        SceneViewer<Scanner>::ShowImpl(display);
    }

    ScannerViewer()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t w = Display::GetInstance().GetFixedWidth();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add( m_Title );
        m_Title.X = DISPLAY_TITLE_START_X;
        m_Title.Y = DISPLAY_TITLE_START_Y;
        m_Title.Text = "Scanner Viewer  ";
        m_Title.FitSize();

        Add(m_Count);
        m_Count.Text = "0 APs Found";
        m_Count.X = x + w;
        m_Count.Y = y;
        m_Count.Width = 100 * w;

        x += 2 * w;

        Add( m_Ch );
        m_Ch.X = x + 0.5f * w;
        m_Ch.Y = (y += 1.5f * h);
        m_Ch.Text = "Ch";
        m_Ch.FitSize();
        x += 3 * w;

        Add( m_Rssi );
        m_Rssi.X = x;
        m_Rssi.Y = y;
        m_Rssi.Text = "RSSI";
        m_Rssi.FitSize();
        x += 5 * w;

        Add( m_Ssid );
        m_Ssid.X = x - 0.5f * w;
        m_Ssid.Y = y;
        m_Ssid.Text = "SSID";
        m_Ssid.FitSize();

        Add( m_Selector );
        m_Selector.X = DISPLAY_CONTENT_START_X - w;
        m_Selector.Y = (y += h) + 10;
        m_Selector.Height = DISPLAY_CONTENT_HEIGHT - (2 * h);

        Add( m_ApViewer );
        m_ApViewer.X = Display::GetInstance().GetWidth() / 2;
        m_ApViewer.Y = m_Ch.Y - DISPLAY_CONTENT_START_Y;
        m_ApViewer.Height = DISPLAY_CONTENT_HEIGHT;

        SetPadText(Button::A, "Select");
        SetPadText(Button::B, "Clear");
    }

    virtual void InputPad(Pad& pad)
    {
        nn::Result result;

        if( pad.IsTrigger(Button::A) )
        {
            AccessPoint *ap = m_Selector.GetSelectingItem();
            if( ap )
            {
                m_ApViewer.SetTarget(*ap);
            }
        }
        else if( pad.IsTrigger(Button::B) )
        {
            target->Clear();
            m_ApSet.clear();
            m_Selector.Clear();
            m_ApViewer.UnsetTarget();
        }
    }

    virtual void SetTarget(Scanner& target)
    {
        this->target = &target;
    }
};

class NodeViewer : public SceneViewer<Node>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    string m_ChannelCaption;
    string m_RssiCaption;
    string m_AddressCaption;
    string m_BssidCaption;
    string m_ConnectSuccessCaption;
    string m_ConnectFailureCaption;
    string m_DisconnectedCaption;
    string m_SendSizeCaption;
    string m_ReceiveSizeCaption;
    string m_SendCountCaption;
    string m_SendErrorCountCaption;
    string m_ReceiveCountCaption;
    string m_ReceiveErrorCountCaption;
    string m_SendThroughputCaption;
    string m_ReceiveThroughputCaption;
    string m_ClientCountCaption;
    string m_ConnectionTimeCaption;

    Label m_Title;
    Label m_Address;
    Label m_Bssid;
    Label m_Channel;
    Label m_Rssi;
    Label m_ConnectSuccess;
    Label m_ConnectFailure;
    Label m_Disconnected;
    Label m_SendSize;
    Label m_ReceiveSize;
    Label m_SendCount;
    Label m_SendErrorCount;
    Label m_ReceiveCount;
    Label m_ReceiveErrorCount;
    Label m_SendThroughput;
    Label m_ReceiveThroughput;
    Label m_ClientCount;
    Label m_ConnectionTime;

    nn::os::Mutex m_Mutex;
    nn::os::Tick m_LastRefreshTime;
    uint32_t m_NowRxSize;
    uint32_t m_NowTxSize;
    uint32_t m_LastRxSize;
    uint32_t m_LastTxSize;

private:

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void ShowImpl(Display& display)
    {
        UpdateText();
        SceneViewer<Node>::ShowImpl(display);
    }

    virtual void InputPad(Pad& pad)
    {
        if( pad.IsTrigger(Button::B) )
        {
            target->ClearStatistics();
        }
    }

    virtual void Clear()
    {
        m_Mutex.Lock();
        {
            m_NowRxSize = 0;
            m_NowTxSize = 0;
            m_LastRxSize = 0;
            m_LastTxSize = 0;
            m_LastRefreshTime = nn::os::Tick(0);
        }
        m_Mutex.Unlock();
    }

    virtual void UpdateText()
    {
        m_Channel.Text = GetChannelText();
        m_Rssi.Text = GetRssiText();
        m_ClientCount.Text = GetClientCountText();
        m_ConnectionTime.Text = GetConnectionTimeText();
        m_Address.Text = GetMacAddressText();
        m_Bssid.Text = GetBssidText();
        m_ConnectSuccess.Text = GetConnectCountText();
        m_ConnectFailure.Text = GetConnectErrorCountText();
        m_Disconnected.Text = GetDisconnectedCountText();
        m_SendSize.Text = GetSendSizeText();
        m_SendCount.Text = GetSendCountText();
        m_SendErrorCount.Text = GetSendErrorCountText();
        m_SendThroughput.Text = GetSendThroughputText();
        m_ReceiveSize.Text = GetReceiveSizeText();
        m_ReceiveCount.Text = GetReceiveCountText();
        m_ReceiveErrorCount.Text = GetReceiveErrorCountText();
        m_ReceiveThroughput.Text = GetReceiveThroughputText();
    }

    virtual string GetChannelText()
    {
        ostringstream oss;
        int16_t ch = target->GetChannel();
        oss << m_ChannelCaption;
        if( ch <= 0 )
        {
            oss << "-";
        }
        else
        {
            oss << ch;
        }
        return oss.str();
    }

    virtual string GetRssiText()
    {
        ostringstream oss;
        vector<Rssi> rssiList;
        target->GetRssi(&rssiList);

        int32_t rssi = 0;
        nn::wlan::LinkLevel linkLevel = nn::wlan::LinkLevel_0;
        if( rssiList.size() > 0 )
        {
            rssi = rssiList[0].value;
            linkLevel = nn::wlan::Local::ConvertRssiToLinkLevel(rssi);
        }

        oss << m_RssiCaption;
        if( rssi <= RSSI_MIN )
        {
            oss << "-";
        }
        else
        {
            oss << rssi << " dBm (" << linkLevel << ")";
        }
        return oss.str();
    }

    virtual string GetClientCountText()
    {
        ostringstream oss;
        uint32_t connectableCount = target->GetConnectableCount();
        oss << m_ClientCountCaption << target->GetConnectedCount() << "/";
        if( connectableCount == ConnectableClientCount_Any)
        {
            oss << "Any";
        }
        else
        {
            oss << connectableCount;
        }
        return oss.str();
    }

    virtual string GetConnectionTimeText()
    {
        ostringstream oss;
        oss << m_ConnectionTimeCaption << target->GetConnectionTime() << " ms";
        return oss.str();
    }

    virtual string GetMacAddressText()
    {
        ostringstream oss;
        char adr[nn::wlan::MacAddress::MacStringSize];
        oss << m_AddressCaption << target->GetMacAddress().GetString(adr);
        return oss.str();
    }

    virtual string GetBssidText()
    {
        ostringstream oss;
        char adr[nn::wlan::MacAddress::MacStringSize];
        oss << m_BssidCaption << target->GetApAddress().GetString(adr) << "\n";
        return oss.str();
    }

    virtual string GetConnectCountText()
    {
        ostringstream oss;
        oss << m_ConnectSuccessCaption << target->GetTxStatistics().ConnectCount;
        return oss.str();
    }

    virtual string GetConnectErrorCountText()
    {
        ostringstream oss;
        oss << m_ConnectFailureCaption << target->GetTxStatistics().ConnectErrorCount;
        return oss.str();
    }

    virtual string GetDisconnectedCountText()
    {
        ostringstream oss;
        oss << m_DisconnectedCaption << target->GetTxStatistics().Disconnected << "\n";
        return oss.str();
    }

    virtual string GetSendSizeText()
    {
        ostringstream oss;
        oss << m_SendSizeCaption << target->GetTxStatistics().SendSize << " byte";
        return oss.str();
    }

    virtual string GetSendCountText()
    {
        ostringstream oss;
        oss << m_SendCountCaption << target->GetTxStatistics().SendCount;
        return oss.str();
    }

    virtual string GetSendErrorCountText()
    {
        ostringstream oss;
        oss << m_SendErrorCountCaption << target->GetTxStatistics().SendErrorCount;
        return oss.str();
    }

    virtual string GetSendThroughputText()
    {
        ostringstream oss;
        oss << m_SendThroughputCaption << fixed << setprecision(3)
            << CalculateThroughput(target->GetTxStatistics().SendSize,
                                   target->GetTxStatistics().FirstSendTime,
                                   target->GetTxStatistics().LastSendTime)
            << " Mbps\n";
        return oss.str();
    }

    virtual string GetReceiveSizeText()
    {
        ostringstream oss;
        oss << m_ReceiveSizeCaption << target->GetTxStatistics().ReceiveSize << " byte";
        return oss.str();
    }

    virtual string GetReceiveCountText()
    {
        ostringstream oss;
        oss << m_ReceiveCountCaption << target->GetTxStatistics().ReceiveCount;
        return oss.str();
    }

    virtual string GetReceiveErrorCountText()
    {
        ostringstream oss;
        oss << m_ReceiveErrorCountCaption << target->GetTxStatistics().ReceiveErrorCount;
        return oss.str();
    }

    virtual string GetReceiveThroughputText()
    {
        ostringstream oss;
        oss << m_ReceiveThroughputCaption << fixed << setprecision(3)
            << CalculateThroughput(target->GetTxStatistics().ReceiveSize,
                                   target->GetTxStatistics().FirstReceiveTime,
                                   target->GetTxStatistics().LastReceiveTime)
            << " Mbps\n";
        return oss.str();
    }

/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:
    NodeViewer() : m_Mutex(false, 0)
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        m_ChannelCaption             = "Channel         : ";
        m_RssiCaption                = "RSSI(LinkLevel) : ";
        m_ClientCountCaption         = "Client Count    : ";
        m_ConnectionTimeCaption      = "Create() Time   : ";
        m_AddressCaption             = "Mac Address     : ";
        m_BssidCaption               = "BSSID           : ";

        m_ConnectSuccessCaption      = "Connect Count : ";
        m_ConnectFailureCaption      = "Connect Error : ";
        m_DisconnectedCaption        = "Disconnected  : ";

        m_SendSizeCaption            = "Tx Size     : ";
        m_SendCountCaption           = "Tx Count    : ";
        m_SendErrorCountCaption      = "Tx Error    : ";
        m_SendThroughputCaption      = "Tx DataRate : ";

        m_ReceiveSizeCaption         = "Rx Size     : ";
        m_ReceiveCountCaption        = "Rx Count    : ";
        m_ReceiveErrorCountCaption   = "Rx Error    : ";
        m_ReceiveThroughputCaption   = "Rx DataRate : ";

        Add( m_Title );
        m_Title.X = DISPLAY_TITLE_START_X;
        m_Title.Y = DISPLAY_TITLE_START_Y;
        m_Title.Text = "WLAN Summary\n";
        m_Title.FitSize();

        Add(m_Channel);
        m_Channel.X = x;
        m_Channel.Y = y;
        m_Channel.Text = m_ChannelCaption + "000 ";
        m_Channel.FitSize();
        m_Channel.Width += 400;

        Add(m_Rssi);
        m_Rssi.X = x;
        m_Rssi.Y = 0;
        m_Rssi.Text = m_RssiCaption + "-00 dBm (X)";
        m_Rssi.FitSize();
        m_Rssi.Width += 400;

        Add(m_ClientCount);
        m_ClientCount.X = x;
        m_ClientCount.Y = (y += h);
        m_ClientCount.Text = m_ClientCountCaption + "0/0";
        m_ClientCount.FitSize();
        m_ClientCount.Width += 400;

        Add(m_ConnectionTime);
        m_ConnectionTime.X = x;
        m_ConnectionTime.Y = (y += h);
        m_ConnectionTime.Text = m_ConnectionTimeCaption + "0000 ms";
        m_ConnectionTime.FitSize();
        m_ConnectionTime.Width += 400;

        Add( m_Address );
        m_Address.Text = m_AddressCaption + "00:00:00:00:00:00";
        m_Address.X = x;
        m_Address.Y = (y += h);
        m_Address.FitSize();
        m_Address.Width += 400;

        Add( m_Bssid );
        m_Bssid.Text = m_BssidCaption + "00:00:00:00:00:00";
        m_Bssid.X = x;
        m_Bssid.Y = (y += h);
        m_Bssid.FitSize();
        m_Bssid.Width += 400;

        Add( m_ConnectSuccess );
        m_ConnectSuccess.Text = m_ConnectSuccessCaption + "0000000";
        m_ConnectSuccess.X = x;
        m_ConnectSuccess.Y = (y += 2 * h);
        m_ConnectSuccess.FitSize();

        Add( m_ConnectFailure );
        m_ConnectFailure.Text = m_ConnectFailureCaption + "0000000";
        m_ConnectFailure.X = x;
        m_ConnectFailure.Y = (y += h);
        m_ConnectFailure.FitSize();

        Add( m_Disconnected );
        m_Disconnected.Text = m_DisconnectedCaption + "0000000";
        m_Disconnected.X = x;
        m_Disconnected.Y = (y += h);
        m_Disconnected.FitSize();

        Add( m_SendSize );
        m_SendSize.Text = m_SendSizeCaption + "000000000 byte";
        m_SendSize.X = x;
        m_SendSize.Y = (y += 2 * h);
        m_SendSize.FitSize();
        m_SendSize.Width += 400;

        Add( m_SendCount );
        m_SendCount.Text = m_SendCountCaption + "00000000";
        m_SendCount.X = x;
        m_SendCount.Y = (y += h);
        m_SendCount.FitSize();
        m_SendCount.Width += 400;

        Add( m_SendErrorCount );
        m_SendErrorCount.Text = m_SendErrorCountCaption + "0000000";
        m_SendErrorCount.X = x;
        m_SendErrorCount.Y = (y += h);
        m_SendErrorCount.FitSize();
        m_SendErrorCount.Width += 400;

        Add( m_SendThroughput );
        m_SendThroughput.Text = m_SendThroughputCaption + "000.0000 Mbps";
        m_SendThroughput.X = x;
        m_SendThroughput.Y = (y += h);
        m_SendThroughput.FitSize();
        m_SendThroughput.Width += 400;

        Add( m_ReceiveSize );
        m_ReceiveSize.Text = m_ReceiveSizeCaption + "000000000 byte";
        m_ReceiveSize.X = x;
        m_ReceiveSize.Y = (y += 2 * h);
        m_ReceiveSize.FitSize();
        m_ReceiveSize.Width += 400;

        Add( m_ReceiveCount );
        m_ReceiveCount.Text = m_ReceiveCountCaption + "00000000";
        m_ReceiveCount.X = x;
        m_ReceiveCount.Y = (y += h);
        m_ReceiveCount.FitSize();
        m_ReceiveCount.Width += 400;

        Add( m_ReceiveErrorCount );
        m_ReceiveErrorCount.Text = m_ReceiveErrorCountCaption + "00000000";
        m_ReceiveErrorCount.X = x;
        m_ReceiveErrorCount.Y = (y += h);
        m_ReceiveErrorCount.FitSize();
        m_ReceiveErrorCount.Width += 400;

        Add( m_ReceiveThroughput );
        m_ReceiveThroughput.Text = m_ReceiveThroughputCaption + "000.0000 Mbps";
        m_ReceiveThroughput.X = x;
        m_ReceiveThroughput.Y = (y += h);
        m_ReceiveThroughput.FitSize();
        m_ReceiveThroughput.Width += 400;

        SetPadText(Button::B, "Clear");

        Clear();
    } //NOLINT(impl/function_size)
};

class MasterViewer : public NodeViewer
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:
    MasterViewer()
    {
        // キャプションの生成・更新
        m_ChannelCaption             = "Channel       : ";
        m_ClientCountCaption         = "Client Count  : ";
        m_ConnectionTimeCaption      = "Create() Time : ";
        m_AddressCaption             = "Mac Address   : ";
        m_BssidCaption               = "BSSID         : ";

        // 削除するラベル
        Remove(m_Rssi);

        // 位置の調整
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t y = DISPLAY_CONTENT_START_Y;
        m_Channel.Y = y;
        m_ClientCount.Y = (y += h);
        m_ConnectionTime.Y = (y += h);
        m_Address.Y = (y += h);
        m_Bssid.Y = (y += h);

        m_ConnectSuccess.Y = (y += 2 * h);
        m_ConnectFailure.Y = (y += h);
        m_Disconnected.Y = (y += h);

        m_SendSize.Y = (y += 2 * h);
        m_SendCount.Y = (y += h);
        m_SendErrorCount.Y = (y += h);
        m_SendThroughput.Y = (y += h);

        m_ReceiveSize.Y = (y += 2 * h);
        m_ReceiveCount.Y = (y += h);
        m_ReceiveErrorCount.Y = (y += h);
        m_ReceiveThroughput.Y = (y += h);
    }
};

class ClientViewer : public NodeViewer
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:
    ClientViewer()
    {
        // キャプションの生成・更新
        m_ChannelCaption             = "Channel         : ";
        m_ClientCountCaption         = "Client Count    : ";
        m_ConnectionTimeCaption      = "Connect() Time  : ";
        m_AddressCaption             = "Mac Address     : ";
        m_BssidCaption               = "BSSID           : ";

        // 削除するラベル
        Remove(m_ClientCount);
        Remove(m_Rssi);

        // 位置の調整
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t y = DISPLAY_CONTENT_START_Y;
        m_Channel.Y = y;
        m_ConnectionTime.Y = (y += h);
        m_Address.Y = (y += h);
        m_Bssid.Y = (y += h);

        m_ConnectSuccess.Y = (y += 2 * h);
        m_ConnectFailure.Y = (y += h);
        m_Disconnected.Y = (y += h);

        m_SendSize.Y = (y += 2 * h);
        m_SendCount.Y = (y += h);
        m_SendErrorCount.Y = (y += h);
        m_SendThroughput.Y = (y += h);

        m_ReceiveSize.Y = (y += 2 * h);
        m_ReceiveCount.Y = (y += h);
        m_ReceiveErrorCount.Y = (y += h);
        m_ReceiveThroughput.Y = (y += h);
    }
};

class SocketNodeViewer : public NodeViewer
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    string m_IpAddressCaption;
    string m_DestIpAddressCaption;
    string m_PortCaption;
    string m_ReceiveLostCountCaption;
    string m_ReceivePlrCaption;

    Label m_IpAddress;
    Label m_DestIpAddress;
    Label m_Port;
    Label m_ReceiveLostCount;
    Label m_ReceivePlr;

private:

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void UpdateText()
    {
        // NodeViewer::UpdateText();
        m_Channel.Text = GetChannelText();
        m_Rssi.Text = GetRssiText();
        // m_ClientCount.Text = GetClientCountText();
        // m_ConnectionTime.Text = GetConnectionTimeText();
        m_Address.Text = GetMacAddressText();
        // m_Bssid.Text = GetBssidText();
        m_ConnectSuccess.Text = GetConnectCountText();
        m_ConnectFailure.Text = GetConnectErrorCountText();
        m_Disconnected.Text = GetDisconnectedCountText();
        m_SendSize.Text = GetSendSizeText();
        m_SendCount.Text = GetSendCountText();
        m_SendErrorCount.Text = GetSendErrorCountText();
        m_SendThroughput.Text = GetSendThroughputText();
        m_ReceiveSize.Text = GetReceiveSizeText();
        m_ReceiveCount.Text = GetReceiveCountText();
        m_ReceiveErrorCount.Text = GetReceiveErrorCountText();
        m_ReceiveThroughput.Text = GetReceiveThroughputText();

        ostringstream oss;
        SocketNode* socketNode = reinterpret_cast<SocketNode*>(target);
        oss << m_IpAddressCaption;
        if( socketNode->IsConnected() )
        {
            oss << socketNode->GetIpAddress();
        }
        else
        {
            oss << "-";
        }
        m_IpAddress.Text = oss.str();
        oss.str("");

        oss << m_DestIpAddressCaption << socketNode->GetRemoteIpAddress();
        m_DestIpAddress.Text = oss.str();
        oss.str("");

        oss << m_PortCaption << socketNode->GetPort() << "\n";
        m_Port.Text = oss.str();
        oss.str("");

        oss << m_ReceiveLostCountCaption << socketNode->GetError().GetErrorCount();
        m_ReceiveLostCount.Text = oss.str();
        oss.str("");

        oss << m_ReceivePlrCaption << socketNode->GetError().GetErrorRate() * 100.f << " %%\n";
        m_ReceivePlr.Text = oss.str();
        oss.str("");
    }

/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:

    SocketNodeViewer()
    {
        // キャプションの生成・更新
        m_ChannelCaption         = "Channel         : ";
        m_AddressCaption         = "Mac Address     : ";
        m_IpAddressCaption       = "IP Address      : ";
        m_DestIpAddressCaption   = "Destination     : ";
        m_PortCaption            = "Port            : ";

        m_ReceiveLostCountCaption = "Rx Lost     : ";
        m_ReceivePlrCaption       = "Rx PLR      : ";

        // 削除するラベル
        Remove(m_ConnectionTime);
        Remove(m_Rssi);
        Remove(m_ClientCount);
        Remove(m_Bssid);

        // 追加するラベル
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;
        Add( m_IpAddress );
        m_IpAddress.Text = m_IpAddressCaption + "000.000.000.000";
        m_IpAddress.X = x;
        m_IpAddress.FitSize();
        m_IpAddress.Width += 400;

        Add( m_DestIpAddress );
        m_DestIpAddress.Text = m_DestIpAddressCaption + "000.000.000.000";
        m_DestIpAddress.X = x;
        m_DestIpAddress.FitSize();
        m_DestIpAddress.Width += 400;

        Add( m_Port );
        m_Port.Text = m_PortCaption + "00000";
        m_Port.X = x;
        m_Port.FitSize();
        m_Port.Width += 400;

        Add( m_ReceiveLostCount );
        m_ReceiveLostCount.Text = m_ReceiveLostCountCaption + "00000000";
        m_ReceiveLostCount.X = x;
        m_ReceiveLostCount.FitSize();
        m_ReceiveLostCount.Width += 400;

        Add( m_ReceivePlr );
        m_ReceivePlr.Text = m_ReceivePlrCaption + "000.00";
        m_ReceivePlr.X = x;
        m_ReceivePlr.FitSize();
        m_ReceivePlr.Width += 400;

        // 位置の調整
        m_Channel.Y = y;
        m_Address.Y = (y += h);
        m_IpAddress.Y = (y += h);
        m_DestIpAddress.Y = (y += h);
        m_Port.Y = (y += h);

        m_ConnectSuccess.Y = (y += 2 * h);
        m_ConnectFailure.Y = (y += h);
        m_Disconnected.Y = (y += h);

        uint32_t ty = y;

        m_SendSize.Y = (y += 2 * h);
        m_SendCount.Y = (y += h);
        m_SendErrorCount.Y = (y += h);
        m_SendThroughput.Y = (y += h);

        y = ty;
        x = 0.5 * Display::GetInstance().GetWidth() + DISPLAY_CONTENT_START_X;

        m_ReceiveSize.X = x;
        m_ReceiveSize.Y = (y += 2 * h);
        m_ReceiveCount.X = x;
        m_ReceiveCount.Y = (y += h);
        m_ReceiveLostCount.X = x;
        m_ReceiveLostCount.Y = (y += h);
        m_ReceiveErrorCount.X = x;
        m_ReceiveErrorCount.Y = (y += h);
        m_ReceiveThroughput.X = x;
        m_ReceiveThroughput.Y = (y += h);
        m_ReceivePlr.X = x;
        m_ReceivePlr.Y = (y += h);
    }
};

class LcsMasterViewer : public MasterViewer
{
/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:
    LcsMasterViewer()
    {
        // 削除するラベル
        Remove(m_SendSize);
        Remove(m_SendCount);
        Remove(m_SendErrorCount);
        Remove(m_SendThroughput);

        Remove(m_ReceiveSize);
        Remove(m_ReceiveCount);
        Remove(m_ReceiveErrorCount);
        Remove(m_ReceiveThroughput);
    }
};

class LcsClientViewer : public ClientViewer
{
/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:
    LcsClientViewer()
    {
        // 削除するラベル
        Remove(m_SendSize);
        Remove(m_SendCount);
        Remove(m_SendErrorCount);
        Remove(m_SendThroughput);

        Remove(m_ReceiveSize);
        Remove(m_ReceiveCount);
        Remove(m_ReceiveErrorCount);
        Remove(m_ReceiveThroughput);
    }
};

class DetectorViewer : public NodeViewer
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    string m_TxModeCaption;
    string m_PayloadSizeCaption;
    string m_TxIntervalCaption;
    string m_HashCaption;

    Label m_TxMode;
    Label m_PayloadSize;
    Label m_TxInterval;
    Label m_Hash;

private:

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void ShowImpl(Display& display)
    {
        Detector* detector = reinterpret_cast<Detector*>(target);
        ostringstream oss;

        UpdateText();

        string mode = "Disabled";
        if( detector->IsOneShotEnabled() )
        {
            mode = "OneShot API (w/ PLR)";
        }
        else
        {
            if( detector->IsPeriodicEnabled() )
            {
                mode = "Periodic API (w/o PLR)";
            }
            else
            {
                mode = "No Tx";
            }
            m_SendSize.Text = m_SendSizeCaption + "-";
            m_SendCount.Text = m_SendCountCaption + "-";
            m_SendErrorCount.Text = m_SendErrorCountCaption + "-";
            m_SendThroughput.Text = m_SendThroughputCaption + "-";
        }

        m_TxMode.Text = m_TxModeCaption + mode;

        oss << detector->GetPayloadSize() << " byte";
        m_PayloadSize.Text = m_PayloadSizeCaption + oss.str();
        oss.str("");

        oss << detector->GetTxInterval() << " ms";
        m_TxInterval.Text = m_TxIntervalCaption + oss.str();
        oss.str("");

        oss << "0x" << ToString(detector->GetHash().GetHash());
        m_Hash.Text = m_HashCaption + oss.str();
        oss.str("");

        SceneViewer<Node>::ShowImpl(display);
    }

/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:

    DetectorViewer()
    {
        // キャプションの生成・更新
        m_ChannelCaption             = "Channel      : ";
        m_AddressCaption             = "Mac Address  : ";
        m_TxModeCaption              = "Tx Mode      : ";
        m_PayloadSizeCaption         = "Payload Size : ";
        m_TxIntervalCaption          = "Tx Interval  : ";
        m_HashCaption                = "Hash         : ";
        m_SendSizeCaption            = "Tx Size      : ";
        m_SendCountCaption           = "Tx Count     : ";
        m_SendErrorCountCaption      = "Tx Error     : ";
        m_SendThroughputCaption      = "Tx DataRate  : ";

        // 削除するラベル
        Remove(m_Rssi);
        Remove(m_ClientCount);
        Remove(m_ConnectionTime);
        Remove(m_Bssid);

        Remove(m_ConnectSuccess);
        Remove(m_ConnectFailure);
        Remove(m_Disconnected);

        Remove(m_ReceiveSize);
        Remove(m_ReceiveCount);
        Remove(m_ReceiveErrorCount);
        Remove(m_ReceiveThroughput);

        // 追加するラベル
        uint32_t x = DISPLAY_CONTENT_START_X;
        Add(m_TxMode);
        m_TxMode.Text = "Tx mode-";
        m_TxMode.X = x;
        m_TxMode.Width = 3000;

        Add(m_PayloadSize);
        m_PayloadSize.Text = "1300 byte";
        m_PayloadSize.X = x;
        m_PayloadSize.Width = 3000;

        Add(m_TxInterval);
        m_TxInterval.Text = "100 ms";
        m_TxInterval.X = x;
        m_TxInterval.Width = 3000;

        Add(m_Hash);
        m_Hash.Text = "0x0123 4567 89ab cdef";
        m_Hash.X = x;
        m_Hash.Width = 3000;

        // 位置の調整
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t y = DISPLAY_CONTENT_START_Y;
        m_Channel.Y = y;
        m_Address.Y = (y += h);

        m_TxMode.Y = (y += 2 * h);
        m_TxInterval.Y = (y += h);
        m_PayloadSize.Y = (y += h);
        m_Hash.Y = (y += h);

        m_SendSize.Y = (y += 2 * h);
        m_SendCount.Y = (y += h);
        m_SendErrorCount.Y = (y += h);
        m_SendThroughput.Y = (y += h);
    }
};

class MultiDataSinkViewer : public SceneViewer<MultiDataSink>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:
    LabelBar bar;
    Label title;
    Label captionLabel;
    Label infoLabel;
    Label sourceLabel;
    Label aidLabel;
    Label rssiLabel;
    Label dataLabel;
    Label errorLabel;
    Label nowLabel;
    Label allLabel;
    Label throughputLabel;
    map<uint64_t, DataLine*> dataLines;
    Label m_Clear;

    Label plrUnitLabel;
    Label thruputUnitLabel;
    Label avgLatencyUnitLabel;
    Label maxLatencyUnitLabel;

    LabelBar ieBar;
    Label ieSourceLabel;
    map<uint64_t, DataLine*> ieLines;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void ShowImpl(Display& display)
    {
        ostringstream oss;
        map<uint64_t, string>& sources = target->GetSources();
        map<uint64_t, string>::iterator p = sources.begin();

        while(p != sources.end())
        {
            MultiDataSink::Statistics& statistics = *(target->GetStatistics()[p->first]);

            // [MacAddress] SeqNo ErrPacket ErrRate DataRate
            // マルチデータシンクから値をとってくる
            oss << p->second.c_str() << " "
                << setw(8) << statistics.Errors.GetReceiveCount() << " "
                << setw(8) << statistics.Errors.GetErrorCount() << " "
                << fixed << setprecision(1) << setw(5) << statistics.Errors.GetErrorRate() * 100.f << " "
                << fixed << setprecision(2) << setw(6) << statistics.Throughput.GetThroughput() << " "
                << fixed << setprecision(0) << setw(4) << statistics.Latency.GetAvgLatency() << " "
                << setw(4) << statistics.Latency.GetMaxLatency() << " "
                << endl;
            p++;
        }

        infoLabel.Text = oss.str();
        oss.str("");

        SceneViewer<MultiDataSink>::ShowImpl(display);

#if 0
        map<uint64_t, string>& sources = target->GetSources();
        map<uint64_t, string>::iterator p = sources.begin();

        while(p != sources.end())
        {
            ReceiverInfo receiver = target->GetReceiverInfos()[p->first];
            map<uint64_t, DataLine*> &lines = (receiver == PROTOCOL_RECEIVER ? dataLines : ieLines);
            uint8_t barY = (receiver == PROTOCOL_RECEIVER ? bar.Y : ieBar.Y);

            // 見つからない、つまり、新しい列ならば
            if( lines.find(p->first) == lines.end() )
            {
                DataLine* dl = new DataLine();

                lines[p->first] = dl;
                this->Add(*dl);
                dl->X = 0;
                dl->Y = lines.size() * 12 + barY;
                dl->SetColor( GenerateColor((void*)p->second.c_str(), strlen(p->second.c_str())));
            }

            // マルチデータシンクから値をとってくる
            (*lines[p->first]).Source = p->second.c_str();
            MultiDataSink::Statistics& statistics = *(target->GetStatistics()[p->first]);
            (*lines[p->first]).Rssi = statistics.Rssi.GetValue();
            (*lines[p->first]).Data = statistics.LastSequenceNo;
            (*lines[p->first]).Error = statistics.Errors.GetErrorCount();
            (*lines[p->first]).Now = statistics.LastErrors.GetErrorRate();
            (*lines[p->first]).All = statistics.Errors.GetErrorRate();
            (*lines[p->first]).Throughput = statistics.Throughput.GetThroughput();
            p++;
        }

        SceneViewer<MultiDataSink>::ShowImpl(display);
#endif
    }

    virtual void InputPad(Pad& pad)
    {
        if( pad.IsTrigger(Button::B) )
        {
            target->Clear();
        }
    }
protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    MultiDataSinkViewer()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;
        uint32_t blankLine = 27;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "WLAN Rx (Data)\n";
        title.FitSize();

        // this->Add(bar);
        // bar.X = 0;
        // bar.Y = 20;
        // bar.Height = 12;

        Add(captionLabel);
        captionLabel.Text =
            //     "----------------- -------- -------- ----- ------ ---- ----"
            string("                                                  Avg  Max\n") +
            string("      MAC Address  RxCount   RxLost   PLR   Thpt Lat. Lat.");
        captionLabel.X = x;
        captionLabel.Y = y;
        captionLabel.Width = 3000;

        uint32_t unitY = y + 2 * h + 3;

        Add(plrUnitLabel);
        SetUnitConfiguration(&plrUnitLabel, "(%%)", x + 789, unitY);

        Add(thruputUnitLabel);
        SetUnitConfiguration(&thruputUnitLabel, "(Mbps)", x + 890, unitY);

        Add(avgLatencyUnitLabel);
        SetUnitConfiguration(&avgLatencyUnitLabel, "(ms)", x + 1011, unitY);

        Add(maxLatencyUnitLabel);
        SetUnitConfiguration(&maxLatencyUnitLabel, "(ms)", x + 1110, unitY);

        Add(infoLabel);
        infoLabel.Text = "info";
        infoLabel.X = x;
        infoLabel.Y = (y += 2 * h + blankLine);
        infoLabel.Width = 7 * 4000;

#if 0
        bar.AddLabel(sourceLabel);
        //bar.AddLabel(rssiLabel);
        bar.AddLabel(dataLabel);
        bar.AddLabel(errorLabel);
        //bar.AddLabel(nowLabel);
        bar.AddLabel(allLabel);
        bar.AddLabel(throughputLabel);

        sourceLabel.Alignment = MiddleCenter;
        sourceLabel.Width = 136;
        sourceLabel.Height = 12;
        sourceLabel.Text = "Source(Data)";
        // rssiLabel.Alignment = MiddleCenter;
        // rssiLabel.Width = 16;
        // rssiLabel.Height = 12;
        // rssiLabel.Text = "RS";
        dataLabel.Alignment = MiddleCenter;
        dataLabel.Width = 64;
        dataLabel.Height = 12;
        dataLabel.Text = "SEQ";
        errorLabel.Alignment = MiddleCenter;
        errorLabel.Width = 56;
        errorLabel.Height = 12;
        errorLabel.Text = "ERR";
        // nowLabel.Alignment = MiddleCenter;
        // nowLabel.Width = 40;
        // nowLabel.Height = 12;
        // nowLabel.Text = "NOW";
        allLabel.Alignment = MiddleCenter;
        allLabel.Width = 40;
        allLabel.Height = 12;
        allLabel.Text = "ALL";
        throughputLabel.Alignment = MiddleCenter;
        throughputLabel.Width = 40;
        throughputLabel.Height = 12;
        throughputLabel.Text = "THPT";

        // this->Add(ieBar);
        // ieBar.X = 0;
        // ieBar.Y = 120;
        // ieBar.Height = 12;

        // ieBar.AddLabel(ieSourceLabel);
        // ieBar.AddLabel(rssiLabel);
        // ieBar.AddLabel(dataLabel);
        // ieBar.AddLabel(errorLabel);
        // ieBar.AddLabel(nowLabel);
        // ieBar.AddLabel(allLabel);
        // ieBar.AddLabel(throughputLabel);

        // ieSourceLabel = sourceLabel;
        // ieSourceLabel.Text = "Source(IE)";
#endif

        SetPadText(Button::B, "Clear");
    }

private:

};

class SocketTxViewer : public SceneViewer<LcsSocket>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:

    FixedSizeDataTransmitter* m_pDataGenerator;

    enum DisplayMode
    {
        DisplayMode_Total = 0,
        DisplayMode_Interval
    };

    int m_Index;
    int m_TxSocket;
    DisplayMode m_DisplayMode;

    LabelBar bar;
    Label title;

    Label ipAddressLabel;
    Label captionLabel;
    Label infoLabel;
    Label sizeUnitLabel;
    Label avgSizeUnitLabel;
    Label thruputUnitLabel;
    Label progressUnitLabel;

    ActiveIndicator activeIndicator;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void ShowImpl(Display& display)
    {
        ostringstream oss;
        const DisplayMode& mode = m_DisplayMode;
        const auto pSocketInfoList = target->GetSocketInfo();
        auto p = pSocketInfoList->begin();

        UpdateIndex();
        UpdateIndicator();

        ipAddressLabel.Text = "   IP Address : " + target->GetSrcIpAddress();

        int i = 0;
        bool isReady = false;
        while( p != pSocketInfoList->end() )
        {
            const string& dstIpAddress = p->m_IpAddressStr;
            if( target->IsDataSocketAccepted(i) )
            {
                const auto& stats = p->m_TxStats;
                auto time = (stats.LastTxTime - stats.FirstTxTime).ToTimeSpan();

                const auto& txSize = (mode == DisplayMode_Total ? stats.TxDataSize : stats.LastTxDataSize);
                const auto& throughput = (mode == DisplayMode_Total ? stats.Throughput.GetThroughput() : stats.LastThroughput.GetThroughput());
                const auto& txSuccess = (mode == DisplayMode_Total ? stats.TxSuccess : stats.LastTxSuccess);
                const auto& txError = (mode == DisplayMode_Total ? stats.TxError : stats.LastTxError);

                oss << (m_pDataGenerator->IsIdle() && m_Index == i ? ">" : " ") << " "
                    << setw(11) << dstIpAddress << " "
                    << setw(8) << ToString(time, TIME_UNIT_MINUTES, TimeFormat_Colon) << "  "
                    << setw(7) << ToStringForBinaryPrefix(txSize, "", 1) << "  "
                    << setw(5) << ToStringForBinaryPrefix(static_cast<uint64_t>(1.0f * txSize / txSuccess), "", 0) << "  "
                    << setw(5) << fixed << setprecision(1) << throughput << "  "
                    //<< setw(3) << static_cast<int>(100.0f * txSize / stats.TotalDataSize) << "  "
                    << setw(6) << txSuccess << "  "
                    << setw(6) << txError
                    << endl;

                isReady = true;
            }

            ++p;
            ++i;
        }

        infoLabel.Text = oss.str();
        oss.str("");

        UpdateTransmissionText(isReady);

        SceneViewer<LcsSocket>::ShowImpl(display);
    }

    virtual void InputPad(Pad& pad)
    {
        if( pad.IsTrigger(Button::B) )
        {
            if( target->SetDataSocketId(m_Index) )
            {
                target->ResetTxStatistics();
            }
        }
        else if( pad.IsTrigger(Button::Y) )
        {
            SwitchDisplayMode();
            ErasePadText(Button::Y);
            SetPadText(Button::Y, GetDisplayModeText());
        }
        else if( Pad::GetInstance().IsTrigger(Button::A) )
        {
            if( m_pDataGenerator->IsIdle() )
            {
                if( target->SetDataSocketId(m_Index) && target->IsDataSocketAccepted(m_Index) )
                {
                    StartTransmission();
                }
            }
            else
            {
                TerminateTransmission();
            }
        }

        int size = target->GetSocketInfo()->size();
        if( m_pDataGenerator->IsIdle() && size > 0 && m_Index >= 0 )
        {
            int hoge = m_Index;
            if( pad.IsTrigger(Button::UP) )
            {
                --m_Index;
                if( m_Index < 0 )
                {
                    m_Index += size;
                }
            }
            else if( pad.IsTrigger(Button::DOWN) )
            {
                ++m_Index;
                if( m_Index >= size )
                {
                    m_Index = 0;
                }
            }
        }
    }

    void StartTransmission()
    {
        m_TxSocket = target->GetDataSocket();
        activeIndicator.Activate();

        target->ResetTxStatistics();
        target->SetDataSize(m_pDataGenerator->GetDataSize());
        m_pDataGenerator->ResetCount();
        m_pDataGenerator->Generate(0);
        UpdateIndicator();
    }

    void TerminateTransmission()
    {
        m_TxSocket = -1;
        activeIndicator.Inactivate();
        m_pDataGenerator->Cancel();
    }

    int FindSocket(const int& socket)
    {
        int i = 0;
        for(const auto& pSocketInfo : *target->GetSocketInfo())
        {
            if( socket == pSocketInfo.m_Socket )
            {
                return i;
            }
            ++i;
        }
        return -1;
    }

    void UpdateIndex()
    {
        int newIndex = FindSocket(m_TxSocket);
        if( !m_pDataGenerator->IsIdle() )
        {
            if( newIndex == -1 )
            {
                // 送信中に対象がいなくなっていたら送信をとめる
                TerminateTransmission();
                --m_Index;
            }
            else
            {
                // 送信中に対象外のクライアントの接続・切断で順序が変わる可能性があるため更新する
                m_Index = newIndex;
                target->SetDataSocketId(m_Index);
            }
        }

        int size = target->GetSocketInfo()->size();
        if( size == 0 )
        {
            m_Index = -1;
        }
        else if( m_Index < 0 )
        {
            m_Index = 0;
        }
        else if( m_Index >= size )
        {
            m_Index = size - 1;
        }
    }

    void UpdateIndicator()
    {
        if( m_pDataGenerator->IsIdle() )
        {
            // 送信終了した際に非アクティブにする
            activeIndicator.Inactivate();
        }
        else
        {
            uint32_t h = Display::GetInstance().GetLineHeight();
            activeIndicator.Y = infoLabel.Y + m_Index * h;
        }
    }

    void UpdateTransmissionText(const bool isReady)
    {
        ErasePadText(Button::A);
        if( isReady )
        {
            if( m_pDataGenerator->IsIdle() )
            {
                SetPadText(Button::A, "Start");
            }
            else
            {
                SetPadText(Button::A, "Cancel");
            }
        }
    }

    string GetDisplayModeText()
    {
        string text = "";
        if( m_DisplayMode == DisplayMode_Total )
        {
            text = "Interval Stats";
        }
        else if( m_DisplayMode == DisplayMode_Interval )
        {
            text = "Total Stats";
        }
        return text;
    }

    void SwitchDisplayMode()
    {
        switch(m_DisplayMode)
        {
        case DisplayMode_Total :
            {
                m_DisplayMode = DisplayMode_Interval;
            }
            break;

        case DisplayMode_Interval :
            {
                m_DisplayMode = DisplayMode_Total;
            }
            break;

        default :
            {}
            break;
        }
    }

    void SetDataGenerator(FixedSizeDataTransmitter* pDataGenerator)
    {
        m_pDataGenerator = pDataGenerator;
    }

protected:
private:

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    SocketTxViewer()
    {
        m_Index = -1;
        m_TxSocket = -1;
        m_DisplayMode = DisplayMode_Total;

        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;
        uint32_t blankLine = 27;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Socket Tx";
        title.FitSize();

        Add(ipAddressLabel);
        ipAddressLabel.Text = " IP Address : 111.111.1.1";
        ipAddressLabel.X = x;
        ipAddressLabel.Y = (y += h);
        ipAddressLabel.Width = 1000;

        Add(captionLabel);
        captionLabel.Text =
            string("                                  Avg.                       \n") +
            string("  Destination     Time   TxSize TxSize   Thpt Success   Error");
          //string("> 111.111.1.1 00:00:00  1024.1K  1024K  111.1  123456  123456

          //string("  Destination      Time   TxSize   Thpt Prog  Success    Error");
          //string("> 111.111.1.1 000:00:00  1024.1K  111.1  100  1234567  1234567"
        captionLabel.X = x;
        captionLabel.Y = (y += 2 * h);
        captionLabel.Width = 3000;

        uint32_t unitY = y + 2 * h + 3;

        Add(sizeUnitLabel);
        SetUnitConfiguration(&sizeUnitLabel, "(byte)", x + 548, unitY);

        Add(avgSizeUnitLabel);
        SetUnitConfiguration(&avgSizeUnitLabel, "(byte)", x + 687, unitY);

        Add(thruputUnitLabel);
        //SetUnitConfiguration(&thruputUnitLabel, "(Mbps)", x + 708, unitY);
        SetUnitConfiguration(&thruputUnitLabel, "(Mbps)", x + 827, unitY);

        // Add(progressUnitLabel);
        // SetUnitConfiguration(&progressUnitLabel, "(%%)", x + 846, unitY);

        Add(infoLabel);
        infoLabel.Text = "info";
        infoLabel.X = x;
        infoLabel.Y = (y += 2 * h + blankLine);
        infoLabel.Width = 7 * 4000;

        Add(activeIndicator);
        activeIndicator.X = x;
        activeIndicator.Y = y;
        activeIndicator.Inactivate();

        SetPadText(Button::B, "Clear");
        SetPadText(Button::Y, GetDisplayModeText());
    }

private:

};

class SocketRxViewer : public SceneViewer<LcsSocket>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:

    enum DisplayMode
    {
        DisplayMode_Total = 0,
        DisplayMode_Interval
    };

    DisplayMode m_DisplayMode;

    LabelBar bar;
    Label title;

    Label ipAddressLabel;
    Label captionLabel;
    Label infoLabel;
    Label sizeUnitLabel;
    Label avgSizeUnitLabel;
    Label thruputUnitLabel;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void ShowImpl(Display& display)
    {
        ostringstream oss;
        const DisplayMode& mode = m_DisplayMode;
        const auto pSocketInfoList = target->GetSocketInfo();
        auto p = pSocketInfoList->begin();

        ipAddressLabel.Text = "   IP Address : " + target->GetSrcIpAddress();

        int i = 0;
        while( p != pSocketInfoList->end() )
        {
            if( target->IsDataSocketAccepted(i) )
            {
                const auto& stats = p->m_RxStats;
                auto time = (stats.LastRxTime - stats.FirstRxTime).ToTimeSpan();

                const auto& rxSize = (mode == DisplayMode_Total ? stats.RxDataSize : stats.LastRxDataSize);
                const auto& throughput = (mode == DisplayMode_Total ? stats.Throughput.GetThroughput() : stats.LastThroughput.GetThroughput());
                const auto& rxSuccess = (mode == DisplayMode_Total ? stats.RxSuccess : stats.LastRxSuccess);
                const auto& rxError = (mode == DisplayMode_Total ? stats.RxError : stats.LastRxError);

                oss << "  "
                    << setw(11) << p->m_IpAddressStr << " "
                    << setw(8) << ToString(time, TIME_UNIT_MINUTES, TimeFormat_Colon) << "  "
                    << setw(7) << ToStringForBinaryPrefix(rxSize, "", 1) << "  "
                    << setw(5) << ToStringForBinaryPrefix(static_cast<uint64_t>(1.0f * rxSize / rxSuccess), "", 0) << "  "
                    << setw(5) << fixed << setprecision(1) << throughput << "  "
                    << setw(6) << rxSuccess << "  "
                    << setw(6) << rxError
                    << endl;
            }

            ++p;
            ++i;
        }

        infoLabel.Text = oss.str();
        oss.str("");

        SceneViewer<LcsSocket>::ShowImpl(display);
    }

    virtual void InputPad(Pad& pad)
    {
        if( pad.IsTrigger(Button::B) )
        {
            target->ResetAllRxStatistics();
        }
        else if( pad.IsTrigger(Button::Y) )
        {
            SwitchDisplayMode();
            ErasePadText(Button::Y);
            SetPadText(Button::Y, GetDisplayModeText());
        }
    }

protected:
private:

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    SocketRxViewer()
    {
        m_DisplayMode = DisplayMode_Total;

        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;
        uint32_t blankLine = 27;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "Socket Rx";
        title.FitSize();

        Add(ipAddressLabel);
        ipAddressLabel.Text = "  IP Address : 111.111.1.1";
        ipAddressLabel.X = x;
        ipAddressLabel.Y = (y += h);
        ipAddressLabel.Width = 1000;

        Add(captionLabel);
        captionLabel.Text =
            string("                                  Avg.                       \n") +
            string("       Source     Time   RxSize RxSize   Thpt Success   Error");
          //string("> 111.111.1.1 00:00:00  1024.1K  1024K  111.1  123456  123456"

          //string("       Source      Time   RxSize   Thpt Success   Error");
          //string("> 111.111.1.1 000:00:00  1024.1K  111.1 1234567 1234567"
        captionLabel.X = x;
        captionLabel.Y = (y += 2 * h);
        captionLabel.Width = 3000;

        uint32_t unitY = y + 2 * h + 3;

        Add(sizeUnitLabel);
        SetUnitConfiguration(&sizeUnitLabel, "(byte)", x + 548, unitY);

        Add(avgSizeUnitLabel);
        SetUnitConfiguration(&avgSizeUnitLabel, "(byte)", x + 689, unitY);

        Add(thruputUnitLabel);
        SetUnitConfiguration(&thruputUnitLabel, "(Mbps)", x + 828, unitY);

        Add(infoLabel);
        infoLabel.Text = "info";
        infoLabel.X = x;
        infoLabel.Y = (y += 2 * h + blankLine);
        infoLabel.Width = 7 * 4000;

        SetPadText(Button::B, "Clear");
        SetPadText(Button::Y, GetDisplayModeText());
    }

    string GetDisplayModeText()
    {
        string text = "";
        if( m_DisplayMode == DisplayMode_Total )
        {
            text = "Interval Stats";
        }
        else if( m_DisplayMode == DisplayMode_Interval )
        {
            text = "Total Stats";
        }
        return text;
    }

    void SwitchDisplayMode()
    {
        switch(m_DisplayMode)
        {
        case DisplayMode_Total :
            {
                m_DisplayMode = DisplayMode_Interval;
            }
            break;

        case DisplayMode_Interval :
            {
                m_DisplayMode = DisplayMode_Total;
            }
            break;

        default :
            {}
            break;
        }
    }

private:

};

class MultiDataSinkSwitchableViewer : public SceneViewer<MultiDataSink>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:

    enum DisplayMode
    {
        DisplayMode_Total = 0,
        DisplayMode_Interval
    };

    enum PageName
    {
        PageName_RxCount = 0,
        PageName_Interval,

        PageName_Count
    };

    static const uint32_t CaptionLength = 200;
    static const char CaptionText[PageName_Count][CaptionLength];

    DisplayMode m_DisplayMode;

    Label title;
    Label macCaptionLabel;
    Label macInfoLabel;
    Label captionLabel[PageName_Count];
    Label dataLabel[PageName_Count];

    Label sourceLabel;
    Label aidLabel;
    Label rssiLabel;
    //Label dataLabel;
    Label errorLabel;
    Label nowLabel;
    Label allLabel;
    Label throughputLabel;
    map<uint64_t, DataLine*> dataLines;
    Label m_Clear;

    Label plrUnitLabel;
    Label ttlPlrUnitLabel;
    Label thruputUnitLabel;
    Label avgRxIntervalUnitLabel;
    Label maxRxIntervalUnitLabel;

    ScrollableTextPage::TextPage textPage[PageName_Count];
    ScrollableTextPage scrollableTextPage;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void ShowImpl(Display& display)
    {
        ostringstream oss[PageName_Count];
        const DisplayMode& mode = m_DisplayMode;
        macInfoLabel.Text = "";
        for(auto& source : target->GetSources())
        {
            macInfoLabel.Text += source.second + "\n";

            MultiDataSink::Statistics& stats = *(target->GetStatistics()[source.first]);
            const LatencyCounter& latency                 = (mode == DisplayMode_Total) ? stats.Latency : stats.LastLatency;
            const SequenceNumberCounter& errors           = (mode == DisplayMode_Total) ? stats.Errors : stats.LastErrors;
            const SequenceNumberCounter& errorsForLatency = (mode == DisplayMode_Total) ? stats.ErrorsForLatency : stats.LastErrorsForLatency;
            const ThroughputMeter& throughput             = (mode == DisplayMode_Total) ? stats.Throughput : stats.LastThroughput;
            const Average& rxInterval                     = (mode == DisplayMode_Total) ? stats.RxInterval : stats.LastRxInterval;
            uint64_t lost  = errorsForLatency.GetErrorCount() + latency.GetDeadPackets();
            uint64_t total = latency.GetBurstLatencyCount(0, latency.GetLastIndex() - 1);
            double ttlPlr = 0.0f;
            if( total > 0 )
            {
                // Latency のカウントが受信開始の 1.5s 後であるため
                // stats.Errors の受信数と stats.Latency の受信数は異なる
                ttlPlr = std::min(100.0, static_cast<double>(lost) / total);
                ttlPlr = std::max(0.0, ttlPlr);
            }

            oss[PageName_RxCount]
                << setw(8) << errors.GetReceiveCount() << " "
                << setw(8) << errors.GetErrorCount() << " "
                << fixed << setprecision(1) << setw(5) << errors.GetErrorRate() * 100.f << " "
                << fixed << setprecision(1) << setw(5) << ttlPlr * 100.f << " "
                << fixed << setprecision(2) << setw(6) << throughput.GetThroughput()
                << endl;

            oss[PageName_Interval]
                << fixed << setprecision(1) << setw(7) << rxInterval.GetValue() << " "
                << fixed << setprecision(0) << setw(5) << rxInterval.GetMaxValue()
                << endl;
        }

        for(int i=0; i<PageName_Count; ++i)
        {
            dataLabel[i].Text = oss[i].str();
            oss[i].str("");
        }

        SceneViewer<MultiDataSink>::ShowImpl(display);
    }

    virtual void InputPad(Pad& pad)
    {
        if( pad.IsTrigger(Button::B) )
        {
            for(auto& stats : target->GetStatistics())
            {
                stats.second->ClearRxStats();
            }
        }
        else if( pad.IsTrigger(Button::Y) )
        {
            SwitchDisplayMode();
            ErasePadText(Button::Y);
            SetPadText(Button::Y, GetDisplayModeText());
        }
    }

    string GetDisplayModeText()
    {
        string text = "";
        if( m_DisplayMode == DisplayMode_Total )
        {
            text = "Interval Stats";
        }
        else if( m_DisplayMode == DisplayMode_Interval )
        {
            text = "Total Stats";
        }
        return text;
    }

    void SwitchDisplayMode()
    {
        switch(m_DisplayMode)
        {
        case DisplayMode_Total :
            {
                m_DisplayMode = DisplayMode_Interval;
            }
            break;

        case DisplayMode_Interval :
            {
                m_DisplayMode = DisplayMode_Total;
            }
            break;

        default :
            {}
            break;
        }
    }

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    MultiDataSinkSwitchableViewer()
    {
        m_DisplayMode = DisplayMode_Total;

        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;
        uint32_t blankLine = 27;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "WLAN Rx (Data)\n";
        title.FitSize();

        Add(macCaptionLabel);
        macCaptionLabel.Text = "      MAC Address";
        macCaptionLabel.X = x;
        macCaptionLabel.Y = (y += h);
        macCaptionLabel.Width = 3000;

        Add(macInfoLabel);
        macInfoLabel.Text = "";
        macInfoLabel.X = x;
        macInfoLabel.Y = y + h + blankLine;
        macInfoLabel.Width = 7 * 600;

        size_t captionLength = 0;
        for(auto caption : CaptionText)
        {
            captionLength = std::max(captionLength, strlen(caption));
        }

        float fontSize = Display::GetInstance().GetFixedWidth();
        uint32_t offset = (strlen(macCaptionLabel.Text.c_str()) + 1) * fontSize;

        Add(scrollableTextPage);
        scrollableTextPage.X = x + offset;
        scrollableTextPage.Y = y;
        scrollableTextPage.Width = (0.5 * captionLength) * fontSize;
        scrollableTextPage.SetDisplayRange(x + offset, x + offset + scrollableTextPage.Width);
        scrollableTextPage.SetSmoothness(3.3f);
        scrollableTextPage.SetLoop(false);
        for(int i=0; i<PageName_Count; ++i)
        {
            captionLabel[i].Text = string(CaptionText[i]);
            captionLabel[i].Y = -1.0f * h;
            captionLabel[i].Width = 3000;

            dataLabel[i].Y = h + blankLine;
            dataLabel[i].Width = 7 * 4000;

            textPage[i].push_back(&captionLabel[i]);
            textPage[i].push_back(&dataLabel[i]);
            scrollableTextPage.AddPage(textPage[i]);
        }

        uint32_t unitY = h + 3;
        textPage[PageName_RxCount].push_back(&plrUnitLabel);
        SetUnitConfiguration(&plrUnitLabel, "(%%)", 430, unitY);

        textPage[PageName_RxCount].push_back(&ttlPlrUnitLabel);
        SetUnitConfiguration(&ttlPlrUnitLabel, "(%%)", 550, unitY);

        textPage[PageName_RxCount].push_back(&thruputUnitLabel);
        SetUnitConfiguration(&thruputUnitLabel, "(Mbps)", 650, unitY);

        textPage[PageName_Interval].push_back(&avgRxIntervalUnitLabel);
        SetUnitConfiguration(&avgRxIntervalUnitLabel, "(ms)", 93, unitY);

        textPage[PageName_Interval].push_back(&maxRxIntervalUnitLabel);
        SetUnitConfiguration(&maxRxIntervalUnitLabel, "(ms)", 212, unitY);

        SetPadText(Button::B, "Clear");
        SetPadText(Button::Y, GetDisplayModeText());
    }

private:

};


class MultiAfSinkSwitchableViewer : public SceneViewer<MultiAfSink>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:

    enum DisplayMode
    {
        DisplayMode_Total = 0,
        DisplayMode_Interval
    };

    enum PageName
    {
        PageName_RxCount = 0,
        PageName_Interval,

        PageName_Count
    };

    static const uint32_t CaptionLength = 200;
    static const char CaptionText[PageName_Count][CaptionLength];

    DisplayMode m_DisplayMode;

    Label title;
    Label macCaptionLabel;
    Label macInfoLabel;
    Label captionLabel[PageName_Count];
    Label dataLabel[PageName_Count];

    Label sourceLabel;
    Label aidLabel;
    Label rssiLabel;
    //Label dataLabel;
    Label errorLabel;
    Label nowLabel;
    Label allLabel;
    Label throughputLabel;
    map<uint64_t, DataLine*> dataLines;
    Label m_Clear;

    Label plrUnitLabel;
    Label ttlPlrUnitLabel;
    Label thruputUnitLabel;
    Label avgRxIntervalUnitLabel;
    Label maxRxIntervalUnitLabel;

    ScrollableTextPage::TextPage textPage[PageName_Count];
    ScrollableTextPage scrollableTextPage;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void ShowImpl(Display& display)
    {
        ostringstream oss[PageName_Count];
        const DisplayMode& mode = m_DisplayMode;
        macInfoLabel.Text = "";
        for(auto& source : target->GetSources())
        {
            for(auto& chStats : target->GetStatistics()[source.first])
            {
                int16_t channel = chStats.first;
                string channelStr = "  -";
                if( channel != 0 )
                {
                    ostringstream ossCh;
                    ossCh << setw(3) << static_cast<int16_t>(channel);
                    channelStr = ossCh.str();
                }
                macInfoLabel.Text += source.second + " " + channelStr + "\n";

                const MultiAfSink::Statistics& stats = *chStats.second;
                const LatencyCounter& latency       = (mode == DisplayMode_Total) ? stats.Latency : stats.LastLatency;
                const SequenceNumberCounter& errors = (mode == DisplayMode_Total) ? stats.Errors : stats.LastErrors;
                const ThroughputMeter& throughput   = (mode == DisplayMode_Total) ? stats.Throughput : stats.LastThroughput;
                const Average& rxInterval           = (mode == DisplayMode_Total) ? stats.RxInterval : stats.LastRxInterval;
                uint64_t lost = errors.GetErrorCount() + latency.GetDeadPackets();
                uint64_t total = latency.GetBurstLatencyCount(0, latency.GetLastIndex() - 1);
                double ttlPlr = 0.0f;
                if( total > 0 )
                {
                    // Latency のカウントが受信開始の 1.5s 後であるため
                    // stats.Errors の受信数と stats.Latency の受信数は異なる
                    ttlPlr = std::min(100.0, static_cast<double>(lost) / total);
                    ttlPlr = std::max(0.0, ttlPlr);
                }

                oss[PageName_RxCount] << setw(8) << errors.GetReceiveCount() << " ";

                if( channel == 0 )
                {
                    oss[PageName_RxCount]
                        << setw(8) << "-" << " "
                        << setw(5) << "-" << " "
                        << setw(6) << "-"
                        << endl;

                    oss[PageName_Interval]
                        << setw(7) << "-" << " "
                        << setw(5) << "-"
                        << endl;
                }
                else
                {
                    oss[PageName_RxCount]
                        << setw(8) << errors.GetErrorCount() << " "
                        << fixed << setprecision(1) << setw(5) << errors.GetErrorRate() * 100.f << " "
                        << fixed << setprecision(2) << setw(6) << throughput.GetThroughput()
                        << endl;

                    oss[PageName_Interval]
                        << fixed << setprecision(1) << setw(7) << rxInterval.GetValue() << " "
                        << fixed << setprecision(0) << setw(5) << rxInterval.GetMaxValue()
                        << endl;
                }
            }
        }

        for(int i=0; i<PageName_Count; ++i)
        {
            dataLabel[i].Text = oss[i].str();
            oss[i].str("");
        }

        SceneViewer<MultiAfSink>::ShowImpl(display);
    }

    virtual void InputPad(Pad& pad)
    {
        if( pad.IsTrigger(Button::B) )
        {
            target->Clear();
        }
        else if( pad.IsTrigger(Button::Y) )
        {
            SwitchDisplayMode();
            ErasePadText(Button::Y);
            SetPadText(Button::Y, GetDisplayModeText());
        }
    }

    string GetDisplayModeText()
    {
        string text = "";
        if( m_DisplayMode == DisplayMode_Total )
        {
            text = "Interval Stats";
        }
        else if( m_DisplayMode == DisplayMode_Interval )
        {
            text = "Total Stats";
        }
        return text;
    }

    void SwitchDisplayMode()
    {
        switch(m_DisplayMode)
        {
        case DisplayMode_Total :
            {
                m_DisplayMode = DisplayMode_Interval;
            }
            break;

        case DisplayMode_Interval :
            {
                m_DisplayMode = DisplayMode_Total;
            }
            break;

        default :
            {}
            break;
        }
    }

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    MultiAfSinkSwitchableViewer()
    {
        m_DisplayMode = DisplayMode_Total;

        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;
        uint32_t blankLine = 27;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "WLAN Rx (Action Frame)\n";
        title.FitSize();

        Add(macCaptionLabel);
        macCaptionLabel.Text = "      MAC Address  Ch";
        macCaptionLabel.X = x;
        macCaptionLabel.Y = (y += h);
        macCaptionLabel.Width = 3000;

        Add(macInfoLabel);
        macInfoLabel.Text = "";
        macInfoLabel.X = x;
        macInfoLabel.Y = y + h + blankLine;
        macInfoLabel.Width = 20 * 600;

        size_t captionLength = 0;
        for(auto caption : CaptionText)
        {
            captionLength = std::max(captionLength, strlen(caption));
        }

        float fontSize = Display::GetInstance().GetFixedWidth();
        uint32_t offset = (strlen(macCaptionLabel.Text.c_str()) + 1) * fontSize;

        Add(scrollableTextPage);
        scrollableTextPage.X = x + offset;
        scrollableTextPage.Y = y;
        scrollableTextPage.Width = (0.5 * captionLength) * fontSize;
        scrollableTextPage.SetDisplayRange(x + offset, x + offset + scrollableTextPage.Width);
        scrollableTextPage.SetSmoothness(3.3f);
        scrollableTextPage.SetLoop(false);
        for(int i=0; i<PageName_Count; ++i)
        {
            captionLabel[i].Text = string(CaptionText[i]);
            captionLabel[i].Y = -1.0f * h;
            captionLabel[i].Width = 3000;

            dataLabel[i].Y = h + blankLine;
            dataLabel[i].Width = 7 * 4000;

            textPage[i].push_back(&captionLabel[i]);
            textPage[i].push_back(&dataLabel[i]);
            scrollableTextPage.AddPage(textPage[i]);
        }

        uint32_t unitY = h + 3;
        textPage[PageName_RxCount].push_back(&plrUnitLabel);
        SetUnitConfiguration(&plrUnitLabel, "(%%)", 430, unitY);

        // textPage[PageName_RxCount].push_back(&ttlPlrUnitLabel);
        // SetUnitConfiguration(&ttlPlrUnitLabel, "(%%)", 550, unitY);

        textPage[PageName_RxCount].push_back(&thruputUnitLabel);
        SetUnitConfiguration(&thruputUnitLabel, "(Mbps)", 525, unitY);

        textPage[PageName_Interval].push_back(&avgRxIntervalUnitLabel);
        SetUnitConfiguration(&avgRxIntervalUnitLabel, "(ms)", 93, unitY);

        textPage[PageName_Interval].push_back(&maxRxIntervalUnitLabel);
        SetUnitConfiguration(&maxRxIntervalUnitLabel, "(ms)", 212, unitY);

        SetPadText(Button::B, "Clear");
        SetPadText(Button::Y, GetDisplayModeText());
    }

private:

};


class MultiNdhpSinkSwitchableViewer : public SceneViewer<MultiNdhpSink>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:

    enum DisplayMode
    {
        DisplayMode_Total = 0,
        DisplayMode_Interval
    };

    enum PageName
    {
        PageName_RxCount = 0,
        PageName_Interval,

        PageName_Count
    };

    static const uint32_t CaptionLength = 200;
    static const char CaptionText[PageName_Count][CaptionLength];

    DisplayMode m_DisplayMode;

    uint8_t displayLineCount;
    uint8_t topDataIndex;

    Label title;
    Label macCaptionLabel;
    Label macInfoLabel;
    Label captionLabel[PageName_Count];
    Label dataLabel[PageName_Count];

    Label plrUnitLabel;
    Label ttlPlrUnitLabel;
    Label thruputUnitLabel;
    Label avgRxIntervalUnitLabel;
    Label maxRxIntervalUnitLabel;
    Label avgRssiUnitLabel;

    ScrollableTextPage::TextPage textPage[PageName_Count];
    ScrollableTextPage scrollableTextPage;

    ArrowIndicator arrowIndicator[2];

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void ShowImpl(Display& display)
    {
        ostringstream oss[PageName_Count];
        const DisplayMode& mode = m_DisplayMode;
        macInfoLabel.Text = "";
        uint32_t dataCount = 0;
        int i = 0;
        for(auto& source : target->GetSources())
        {
            for(auto& hashStats : target->GetStatistics()[source.first])
            {
                ++dataCount;
                if( topDataIndex <= i && i < topDataIndex + displayLineCount )
                {
                    macInfoLabel.Text += source.second + "\n";

                    bool isValid = hashStats.second->IsWitFormat;
                    uint64_t hash = hashStats.first;
                    string hashStr = string(15, ' ') + "-";

                    ostringstream ossHash;
                    ossHash << hex << setw(16) << setfill('0') << static_cast<uint64_t>(hash);
                    hashStr = ossHash.str();

                    const MultiNdhpSink::Statistics& stats = *hashStats.second;
                    const LatencyCounter& latency       = (mode == DisplayMode_Total) ? stats.Latency : stats.LastLatency;
                    const SequenceNumberCounter& errors = (mode == DisplayMode_Total) ? stats.Errors : stats.LastErrors;
                    const ThroughputMeter& throughput   = (mode == DisplayMode_Total) ? stats.Throughput : stats.LastThroughput;
                    const Average& rxInterval           = (mode == DisplayMode_Total) ? stats.RxInterval : stats.LastRxInterval;
                    const Average& rssi                 = (mode == DisplayMode_Total) ? stats.Rssi : stats.LastRssi;
                    uint64_t lost = errors.GetErrorCount() + latency.GetDeadPackets();
                    uint64_t total = latency.GetBurstLatencyCount(0, latency.GetLastIndex() - 1);
                    double ttlPlr = 0.0f;
                    if( total > 0 )
                    {
                        // Latency のカウントが受信開始の 1.5s 後であるため
                        // stats.Errors の受信数と stats.Latency の受信数は異なる
                        ttlPlr = std::min(100.0, static_cast<double>(lost) / total);
                        ttlPlr = std::max(0.0, ttlPlr);
                    }

                    oss[PageName_RxCount] << setw(8) << errors.GetReceiveCount() << " ";

                    if( isValid )
                    {
                        oss[PageName_RxCount]
                            << setw(8) << errors.GetErrorCount() << " "
                            << fixed << setprecision(1) << setw(5) << errors.GetErrorRate() * 100.f << " "
                            << fixed << setprecision(2) << setw(6) << throughput.GetThroughput() << " "
                            << fixed << setw(4) << static_cast<int16_t>(rssi.GetValue())
                            << endl;

                        oss[PageName_Interval]
                            << fixed << setprecision(1) << setw(7) << rxInterval.GetValue() << " "
                            << fixed << setprecision(0) << setw(5) << rxInterval.GetMaxValue() << " "
                            << hashStr
                            << endl;
                    }
                    else
                    {
                        oss[PageName_RxCount]
                            << setw(8) << "-" << " "
                            << setw(5) << "-" << " "
                            << setw(6) << "-" << " "
                            << fixed << setw(4) << static_cast<int16_t>(rssi.GetValue())
                            << endl;

                        oss[PageName_Interval]
                            << setw(7) << "-" << " "
                            << setw(5) << "-" << " "
                            << hashStr
                            << endl;
                    }
                }
                ++i;
            }
        }

        for(int i=0; i<PageName_Count; ++i)
        {
            dataLabel[i].Text = oss[i].str();
            oss[i].str("");
        }

        if( topDataIndex != 0 )
        {
            arrowIndicator[0].Activate();
        }
        else
        {
            arrowIndicator[0].Inactivate();
        }

        if( topDataIndex + displayLineCount < dataCount )
        {
            arrowIndicator[1].Activate();
        }
        else
        {
            arrowIndicator[1].Inactivate();
        }

        SceneViewer<MultiNdhpSink>::ShowImpl(display);
    }

    virtual void InputPad(Pad& pad)
    {
        if( pad.IsTrigger(Button::B) )
        {
            target->Clear();
            Clear();
        }
        else if( pad.IsTrigger(Button::Y) )
        {
            SwitchDisplayMode();
            ErasePadText(Button::Y);
            SetPadText(Button::Y, GetDisplayModeText());
        }
        else if( pad.IsTrigger(Button::UP) )
        {
            if( topDataIndex > 0 )
            {
                --topDataIndex;
            }
        }
        else if( pad.IsTrigger(Button::DOWN) )
        {
            auto dataCount = 0;
            for(auto& source : target->GetSources())
            {
                dataCount += target->GetStatistics()[source.first].size();
            }

            if( topDataIndex + displayLineCount + 1 <= dataCount )
            {
                ++topDataIndex;
            }
        }
    }

    string GetDisplayModeText()
    {
        string text = "";
        if( m_DisplayMode == DisplayMode_Total )
        {
            text = "Interval Stats";
        }
        else if( m_DisplayMode == DisplayMode_Interval )
        {
            text = "Total Stats";
        }
        return text;
    }

    void SwitchDisplayMode()
    {
        switch(m_DisplayMode)
        {
        case DisplayMode_Total :
            {
                m_DisplayMode = DisplayMode_Interval;
            }
            break;

        case DisplayMode_Interval :
            {
                m_DisplayMode = DisplayMode_Total;
            }
            break;

        default :
            {}
            break;
        }
    }

    virtual void Clear()
    {
        topDataIndex = 0;
    }

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    MultiNdhpSinkSwitchableViewer()
    {
        topDataIndex = 0;
        displayLineCount = 15;
        m_DisplayMode = DisplayMode_Total;

        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;
        uint32_t blankLine = 27;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "WLAN Rx (Action Frame)\n";
        title.FitSize();

        Add(macCaptionLabel);
        macCaptionLabel.Text = "      MAC Address";
        macCaptionLabel.X = x;
        macCaptionLabel.Y = (y += h);
        macCaptionLabel.Width = 3000;

        Add(macInfoLabel);
        macInfoLabel.Text = "";
        macInfoLabel.X = x;
        macInfoLabel.Y = y + h + blankLine;
        macInfoLabel.Width = 20 * 600;

        size_t captionLength = 0;
        for(auto caption : CaptionText)
        {
            captionLength = std::max(captionLength, strlen(caption));
        }

        float fontSize = Display::GetInstance().GetFixedWidth();
        uint32_t offset = (strlen(macCaptionLabel.Text.c_str()) + 1) * fontSize;

        Add(scrollableTextPage);
        scrollableTextPage.X = x + offset;
        scrollableTextPage.Y = y;
        scrollableTextPage.Width = (0.5 * captionLength) * fontSize;
        scrollableTextPage.SetDisplayRange(x + offset, x + offset + scrollableTextPage.Width);
        scrollableTextPage.SetSmoothness(3.3f);
        scrollableTextPage.SetLoop(false);
        for(int i=0; i<PageName_Count; ++i)
        {
            captionLabel[i].Text = string(CaptionText[i]);
            captionLabel[i].Y = -1.0f * h;
            captionLabel[i].Width = 3000;

            dataLabel[i].Y = h + blankLine;
            dataLabel[i].Width = 7 * 4000;

            textPage[i].push_back(&captionLabel[i]);
            textPage[i].push_back(&dataLabel[i]);
            scrollableTextPage.AddPage(textPage[i]);
        }

        uint32_t unitY = h + 3;
        textPage[PageName_RxCount].push_back(&plrUnitLabel);
        SetUnitConfiguration(&plrUnitLabel, "(%%)", 430, unitY);

        // textPage[PageName_RxCount].push_back(&ttlPlrUnitLabel);
        // SetUnitConfiguration(&ttlPlrUnitLabel, "(%%)", 550, unitY);

        textPage[PageName_RxCount].push_back(&thruputUnitLabel);
        SetUnitConfiguration(&thruputUnitLabel, "(Mbps)", 525, unitY);

        textPage[PageName_Interval].push_back(&avgRxIntervalUnitLabel);
        SetUnitConfiguration(&avgRxIntervalUnitLabel, "(ms)", 93, unitY);

        textPage[PageName_Interval].push_back(&maxRxIntervalUnitLabel);
        SetUnitConfiguration(&maxRxIntervalUnitLabel, "(ms)", 212, unitY);

        auto dataY = y + h + blankLine;
        Add(arrowIndicator[0]);
        arrowIndicator[0].X = DISPLAY_CONTENT_START_X - 20;
        arrowIndicator[0].Y = dataY - 20;
        arrowIndicator[0].Activate();
        arrowIndicator[0].SetIndicator('^');
        float lux[3] = {0, 0,  0};
        float luy[3] = {5, 0, -5};
        arrowIndicator[0].SetPosition(lux, luy, 3);

        Add(arrowIndicator[1]);
        arrowIndicator[1].X = DISPLAY_CONTENT_START_X - 20;
        arrowIndicator[1].Y = dataY + (displayLineCount * h) + 20;
        arrowIndicator[1].Activate();
        arrowIndicator[1].SetIndicator('v');
        float ldx[3] = {0, 0,  0};
        float ldy[3] = {-5, 0, 5};
        arrowIndicator[1].SetPosition(ldx, ldy, 3);

        SetPadText(Button::B, "Clear");
        SetPadText(Button::Y, GetDisplayModeText());
    }

private:

};


class PacketLatencyViewer : public SceneViewer<MultiDataSink>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:

    LabelBar bar;
    Label title;
    Label captionLabel;
    Label infoLabel;
    map<uint64_t, DataLine*> dataLines;
    Label m_Clear;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void ShowImpl(Display& display)
    {

        ostringstream oss;
        map<uint64_t, string>& sources = target->GetSources();
        map<uint64_t, string>::iterator p = sources.begin();

      //oss << "-----------------  ----  ----  ----  ----  ----  ---- -----"
        oss << "      MAC Address  <1ms  <2ms  <3ms  <4ms  <5ms  <6ms >=6ms";
        captionLabel.Text = oss.str();
        oss.str("");

        while(p != sources.end())
        {
            MultiDataSink::Statistics& statistics = *(target->GetStatistics()[p->first]);

            // [MacAddress] SeqNo ErrPacket ErrRate DataRate
            // マルチデータシンクから値をとってくる
            oss << p->second << " ";

            LatencyCounter& latencyCounter = statistics.Latency;
            uint32_t sum = 0;
            for(int i=0; i<latencyCounter.GetLastIndex(); ++i)
            {
                uint64_t latency = latencyCounter.GetBurstLatencyCount(i);
                if( i < 6 )
                {
                    oss << setw(5) << ToStringForDecimalPrefix(latency, "", 0) << " ";
                }
                else
                {
                    sum += latency;
                }
            }
            oss << setw(5) << ToStringForDecimalPrefix(sum, "", 0) << "\n";

            p++;
        }

        infoLabel.Text = oss.str();
        oss.str("");

        SceneViewer<MultiDataSink>::ShowImpl(display);
    }

    virtual void InputPad(Pad& pad)
    {
    }

protected:
private:

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    PacketLatencyViewer()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "WLAN Latency\n";
        title.FitSize();

        Add(captionLabel);
        captionLabel.Text = "      MAC Address   <1ms  <2ms  <3ms  <4ms  <5ms  <6ms  >=6ms";
        captionLabel.X = x;
        captionLabel.Y = (y += h);
        captionLabel.Width = 3000;

        Add(infoLabel);
        infoLabel.Text = "info";
        infoLabel.X = x;
        infoLabel.Y = (y += h + 20);
        infoLabel.Width = 7 * 4000;
    }

private:

};


class PacketLatencySwitchableViewer : public SceneViewer<MultiDataSink>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:

    // <1ms, <2ms, ... , <49ms, <50ms, >=50ms
    static const int8_t LATENCY_MAX = 50;
    // 1ページに表示する要素数
    static const int8_t ELEMENT_COUNT = 5;
    // ページ数 ceil((LATENCY_MAX + 1) / ELEMENT_COUNT);
    static const int8_t PAGE_COUNT = ((LATENCY_MAX + 1) / ELEMENT_COUNT) + (((LATENCY_MAX + 1) % ELEMENT_COUNT) == 0 ? 0 : 1);

    Label title;
    Label macCaptionLabel;
    Label macInfoLabel;
    Label captionLabel[PAGE_COUNT];
    Label dataLabel[PAGE_COUNT];

    Label summaryCaptionLabel;
    Label avgLatencyUnitLabel;
    Label maxLatencyUnitLabel;
    Label summaryDataLabel;

    ScrollableTextPage::TextPage textPage[PAGE_COUNT];
    ScrollableTextPage scrollableTextPage;
    map<uint64_t, DataLine*> dataLines;
    Label m_Clear;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void ShowImpl(Display& display)
    {
        // MAC address, Avg/Max Latency
        ostringstream oss;
        macInfoLabel.Text = "";
        for(auto& source : target->GetSources())
        {
            macInfoLabel.Text += source.second + "\n";

            MultiDataSink::Statistics& stats = *(target->GetStatistics()[source.first]);
            oss << fixed << setprecision(0) << setw(4) << stats.Latency.GetAvgLatency() << " "
                << setw(4) << stats.Latency.GetMaxLatency() << " "
                << endl;
        }
        summaryDataLabel.Text = oss.str();
        oss.str("");

        // Histogram
        for(int i=0; i<PAGE_COUNT; ++i)
        {
            GenerateHistogram(&captionLabel[i].Text, &dataLabel[i].Text, i * ELEMENT_COUNT);
        }

        SceneViewer<MultiDataSink>::ShowImpl(display);
    }

    virtual void InputPad(Pad& pad)
    {
        if( pad.IsTrigger(Button::B) )
        {
            for(auto& stats : target->GetStatistics())
            {
                stats.second->ClearLatency();
            }
        }
    }

private:

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    PacketLatencySwitchableViewer()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;
        uint32_t blankLine = 27;

        Add(title);
        title.X = DISPLAY_TITLE_START_X;
        title.Y = DISPLAY_TITLE_START_Y;
        title.Text = "WLAN Latency\n";
        title.FitSize();

        Add(macCaptionLabel);
        macCaptionLabel.Text = "      MAC Address";
        macCaptionLabel.X = x;
        macCaptionLabel.Y = (y += h);
        macCaptionLabel.Width = 3000;

        Add(macInfoLabel);
        macInfoLabel.Text = "";
        macInfoLabel.X = x;
        macInfoLabel.Y = y + h + blankLine;
        macInfoLabel.Width = 7 * 600;

        float fontSize = Display::GetInstance().GetFixedWidth();
        uint32_t offset = (strlen(macCaptionLabel.Text.c_str()) + 1) * fontSize;

        Add(scrollableTextPage);
        scrollableTextPage.X = x + offset;
        scrollableTextPage.Y = y;
        scrollableTextPage.Width = (7 * ELEMENT_COUNT) * fontSize;
        scrollableTextPage.SetDisplayRange(x + offset, x + offset + scrollableTextPage.Width);
        scrollableTextPage.SetSmoothness(3.3f);
        scrollableTextPage.SetLoop(false);
        for(int i=0; i<PAGE_COUNT; ++i)
        {
            captionLabel[i].Width = 3000;

            dataLabel[i].Y = h + blankLine;
            dataLabel[i].Width = 7 * 4000;

            textPage[i].push_back(&captionLabel[i]);
            textPage[i].push_back(&dataLabel[i]);
            scrollableTextPage.AddPage(textPage[i]);
        }

        Add(summaryCaptionLabel);
        summaryCaptionLabel.Text =
            //     "--:--:--:--:--:-- ------ ------ ------ ------ ------ ____ ____
            string("                                                      Avg  Max");
        summaryCaptionLabel.X = x + 10;
        summaryCaptionLabel.Y = y;
        summaryCaptionLabel.Width = 3000;

        Add(avgLatencyUnitLabel);
        avgLatencyUnitLabel.Text = "(ms)";
        avgLatencyUnitLabel.X = x + 1106;
        avgLatencyUnitLabel.Y = y + h + 3;
        avgLatencyUnitLabel.FontWidth *= 0.6;
        avgLatencyUnitLabel.FontHeight *= 0.6;
        avgLatencyUnitLabel.FixedWidth *= 0.6;
        avgLatencyUnitLabel.FitSize();

        Add(maxLatencyUnitLabel);
        maxLatencyUnitLabel.Text = "(ms)";
        maxLatencyUnitLabel.X = x + 1205;
        maxLatencyUnitLabel.Y = y + h + 3;
        maxLatencyUnitLabel.FontWidth *= 0.6;
        maxLatencyUnitLabel.FontHeight *= 0.6;
        maxLatencyUnitLabel.FixedWidth *= 0.6;
        maxLatencyUnitLabel.FitSize();

        Add(summaryDataLabel);
        summaryDataLabel.Text = "data";
        summaryDataLabel.X = x + offset + scrollableTextPage.Width + 10;
        summaryDataLabel.Y = y + h + blankLine;
        summaryDataLabel.Width = 7 * 4000;

        SetPadText(Button::B, "Clear");
    }

private:

    void GenerateHistogram(string* caption, string* info, const int8_t id)
    {
        ostringstream oss;

        // caption
        for(int i=id; i<id + ELEMENT_COUNT; ++i)
        {
            if( i < LATENCY_MAX )
            {
                oss << "  <" << setw(2) << i + 1 << "ms";
            }
            else
            {
                oss << "  >=" << setw(2) << static_cast<int32_t>(LATENCY_MAX) << "ms";
                break;
            }
        }
        *caption = oss.str();
        oss.str("");

        // histogram
        for(auto& source : target->GetSources())
        {
            MultiDataSink::Statistics& statistics = *(target->GetStatistics()[source.first]);

            // マルチデータシンクから値をとってくる
            LatencyCounter& latencyCounter = statistics.Latency;
            uint32_t sum = 0;
            for(int i=id; i<latencyCounter.GetLastIndex(); ++i)
            {
                uint64_t latency = latencyCounter.GetBurstLatencyCount(i);
                if( i < id + ELEMENT_COUNT && i < LATENCY_MAX )
                {
                    oss << " " << setw(6) << ToStringForDecimalPrefix(latency, "", 0);
                }
                else
                {
                    sum += latency;
                }
            }

            if( LATENCY_MAX < id + ELEMENT_COUNT )
            {
                oss << "  " << setw(6) << ToStringForDecimalPrefix(sum, "", 0);
            }
            oss << "\n";
        }
        *info = oss.str();
        oss.str("");
    }

};


class NPadViewer : public SceneViewer<NPad>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    Label m_Title;
    Label m_Count;
    Label m_WlanMode;
    Label m_BtMode;
    Label m_Header;
    Label m_Controller[REMOTE_COUNT_MAX];

private:

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void ShowImpl(Display& display)
    {
        // 現在の DeviceCondition 情報を取得
        nn::btm::DeviceConditionList list;
        nn::btm::GetConnectedDeviceCondition(&list);

        ostringstream oss;

        oss << "Num of RCs     : " << static_cast<uint32_t>(list.deviceCount);
        m_Count.Text = oss.str();
        oss.str("");

        oss << "WLAN Mode      : " << ToString(list.wlanMode);
        m_WlanMode.Text = oss.str();
        oss.str("");

        oss << "Bluetooth Mode : " << ToString(list.bluetoothMode, list.wlanMode);
        m_BtMode.Text = oss.str();
        oss.str("");

        for(int i=0; i<REMOTE_COUNT_MAX; ++i)
        {
            if( i < list.deviceCount )
            {
                nn::btm::HidDeviceCondition &hdc = list.device[i].hidDeviceCondition;
                oss << ToString(list.device[i].bdAddress) << "    "
                    << setw(9) << ToString(hdc.sniffMode) << "    "
                    << setw(8) << ToString(hdc.slotMode);
            }

            m_Controller[i].Text = oss.str();
            oss.str("");
        }

        SceneViewer<NPad>::ShowImpl(display);
    }

    virtual void InputPad(Pad& pad)
    {
    }

    virtual void Clear()
    {
    }

/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:
    NPadViewer()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add( m_Title );
        m_Title.X = DISPLAY_TITLE_START_X;
        m_Title.Y = DISPLAY_TITLE_START_Y;
        m_Title.Text = "Remote Controller Summary\n";
        m_Title.FitSize();

        Add( m_Count );
        m_Count.Text = "Num of RCs    : ";
        m_Count.X = x;
        m_Count.Y = y;
        m_Count.FitSize();
        m_Count.Width += 200;

        Add( m_WlanMode );
        m_WlanMode.Text    = "WLAN mode     : ";
        m_WlanMode.X = x;
        m_WlanMode.Y = (y += h);
        m_WlanMode.FitSize();
        m_WlanMode.Width += 200;

        Add( m_BtMode );
        m_BtMode.Text    = "Bluetooth mode : ";
        m_BtMode.X = x;
        m_BtMode.Y = (y += h);
        m_BtMode.FitSize();
        m_BtMode.Width += 200;

        y += h;

        Add( m_Header );
        m_Header.Text    = "Bluttooth Address    SniffMode    SlotMode";
        m_Header.X = x;
        m_Header.Y = (y += h);
        m_Header.FitSize();
        m_Header.Width += 200;
        for(int i=0; i<REMOTE_COUNT_MAX; ++i)
        {
            Add( m_Controller[i] );
            m_Controller[i].Text = "";
            m_Controller[i].X = x;
            m_Controller[i].Y = (y += h);
            m_Controller[i].FitSize();
            m_Controller[i].Width += 16 * 3000;
        }

        SetPadText(Button::X, "Switch SlotMode(Ukyo only)");
    }
};

class NPadPlrViewer : public SceneViewer<NPad>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    static const uint16_t EXPECTED_SAMPLING_COUNT_PER_SEC = 200;

    int16_t m_PlotCount;
    int64_t m_DisplayRange; //us
    nn::os::Tick m_StartTick;
    nn::os::Tick m_EndTick;

    class PlrView
    {
    public :

        Label m_Indicator;
        Label m_CurrentPlr;
        Label m_AvgPlr;
        Label m_MinMaxPlr;
        Label m_Bar;

        void Enable(const bool& isEnabled)
        {
            if( isEnabled )
            {
                m_Indicator.TextColor = ToColor(LAWN_GREEN);
                m_CurrentPlr.TextColor = ToColor(WHITE);
                m_AvgPlr.TextColor = ToColor(WHITE);
                m_MinMaxPlr.TextColor = ToColor(WHITE);
            }
            else
            {
                static Color c = ToColor(0x333333);
                m_Indicator.TextColor = c;
                m_CurrentPlr.Text = "-";
                m_CurrentPlr.TextColor = c;
                m_AvgPlr.TextColor = c;
                m_MinMaxPlr.TextColor = c;
            }
        }

        void SetLedPattern(const uint8_t ledPattern)
        {
            uint8_t bitPattern = ledPattern;
            string strPattern = "";
            for(int i=0; i<4; ++i)
            {
                if( bitPattern & 0x1 )
                {
                    strPattern += "●\n";
                }
                else
                {
                    strPattern += "○\n";
                }
                bitPattern >>= 1;
            }
            m_Indicator.Text = strPattern;
        }
    };
    PlrView m_PlrView[REMOTE_COUNT_MAX];

    uint32_t m_ChartLeft;
    uint32_t m_ChartRight;
    uint32_t m_ChartTop;
    uint32_t m_ChartBottom;

    uint32_t m_MinValue;
    uint32_t m_MaxValue;

    Label* m_Plot[REMOTE_COUNT_MAX];
    Color m_Color[REMOTE_COUNT_MAX];

    Label m_Title;
    Label m_AxisX;
    Label m_AxisY;

    double m_ResolutionX;
    double m_ResolutionY;

    Label m_AuxLine[10];

private:

    double GetPlr(const uint32_t& count)
    {
        double plr = 100.0f * (1.0f * EXPECTED_SAMPLING_COUNT_PER_SEC - count) / EXPECTED_SAMPLING_COUNT_PER_SEC;
        if( plr < 0.0f )
        {
            plr = 0.0f;
        }

        return plr;
    }

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void ShowImpl(Display& display)
    {
        // 表示範囲の更新
        m_EndTick = nn::os::GetSystemTick();
        m_StartTick = m_EndTick - nn::os::Tick(nn::TimeSpan::FromMicroSeconds(m_DisplayRange));

        uint8_t padCount = 0;
        const NPad::PadInfo* info[REMOTE_COUNT_MAX];
        NPad::GetPadInfo(info, &padCount);
        for(int i=0; i<REMOTE_COUNT_MAX; ++i)
        {
            // RC の LED パターンの表示
            uint32_t pattern = nn::hid::GetPlayerLedPattern(info[i]->npadId);
            m_PlrView[i].SetLedPattern(pattern);

            const std::deque<SamplingCountRecorder::TimedCount>& tc = info[i]->recorder.GetUpdatedCounts();

            // データがない or 最新のデータが3秒以内出なければ切断しているとする
            bool isConnected = true;
            if( tc.size() == 0 )
            {
                isConnected = false;
            }
            else
            {
                isConnected = (m_EndTick - tc.back().tick).ToTimeSpan().GetMilliSeconds() < 3000;
            }
            m_PlrView[i].Enable(isConnected);

            // パケロス率のグラフ表示
            uint16_t sumCnt = 0;
            double sumPlr = 0.0f;
            double minPlr = 100.0f;
            double maxPlr = 0.0f;
            for(int j=0; j<m_PlotCount; ++j)
            {
                Label& plot = m_Plot[i][j];

                if( j < tc.size() )
                {
                    double plr = GetPlr(tc[j].count);
                    if( (m_EndTick - tc[j].tick).ToTimeSpan().GetMilliSeconds() < 60 * 1000 )
                    {
                        sumPlr += plr;
                        ++sumCnt;
                        minPlr = std::min(minPlr, plr);
                        maxPlr = std::max(maxPlr, plr);
                    }

                    uint8_t charDiff = 30; // "-" と "." の高さのずれの補正
                    plot.X = m_ChartLeft + (tc[j].tick - m_StartTick).ToTimeSpan().GetMicroSeconds() * m_ResolutionX;
                    plot.Y = m_ChartBottom - (plr - m_MinValue) * m_ResolutionY;
                    plot.Y -= charDiff;
                    if( plot.X > m_ChartLeft + 5 && m_ChartTop <= plot.Y + charDiff && plot.Y + charDiff<= m_ChartBottom )
                    {
                        plot.Text = ".";
                    }
                    else
                    {
                        plot.Text = " ";
                    }
                }
                else
                {
                    plot.Text = " ";
                    plot.X = -100;
                    plot.Y = -100;
                }
            }

            // パケロス数の数値表示
            if( isConnected )
            {
                ostringstream oss;
                oss << fixed << setw(5) << setprecision(1) << GetPlr(tc.back().count) << "%";
                m_PlrView[i].m_CurrentPlr.Text = oss.str();
                oss.str("");

                double avgPlr = sumPlr / sumCnt;
                oss << fixed << setw(5) << setprecision(1) << avgPlr << "%";
                m_PlrView[i].m_AvgPlr.Text = oss.str();
                oss.str("");

                oss << fixed << setw(5) << setprecision(1) << minPlr << "%/"
                    << fixed << setw(5) << setprecision(1) << maxPlr << "%";
                m_PlrView[i].m_MinMaxPlr.Text = oss.str();
                oss.str("");
            }
        }

        // 無駄なものが多いのでコンソール表示しない
        bool isEnabled = false;
        if( Display::GetInstance().IsEnabledConsoleOutput() )
        {
            Display::GetInstance().DisableConsoleOutput();
            isEnabled = true;
        }

        SceneViewer<NPad>::ShowImpl(display);

        if( isEnabled )
        {
            Display::GetInstance().EnableConsoleOutput();
        }
    }

    virtual void InputPad(Pad& pad)
    {
        if( pad.IsTrigger(Button::B) )
        {
            NPad::ClearStatistics();
        }
    }

    virtual void Clear()
    {
    }

/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:
    NPadPlrViewer()
    {
        m_DisplayRange = 180 * 1000 * 1000; // us

        nn::os::Tick tick = nn::os::GetSystemTick();
        m_StartTick = tick;
        m_EndTick = tick;

        m_ChartLeft = 10;
        m_ChartRight = 1210;
        m_ChartTop = 230;
        m_ChartBottom = 600;

        m_MaxValue = 100;
        m_MinValue = 0;

        m_ResolutionX = 1.0f * (m_ChartRight - m_ChartLeft) / m_DisplayRange; // dot / ms
        m_ResolutionY = 1.0f * (m_ChartBottom - m_ChartTop) / (m_MaxValue - m_MinValue); // dot / score

        uint32_t c[REMOTE_COUNT_MAX] = {RED, ORANGE_RED1, YELLOW2, GREEN, CYAN, BLUE3, DARK_SLATE_BLUE, HOT_PINK};
        for(int i=0; i<REMOTE_COUNT_MAX; ++i)
        {
            m_Color[i] = ToColor(c[i]);
        }

        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y;

        Add( m_Title );
        m_Title.X = DISPLAY_TITLE_START_X;
        m_Title.Y = DISPLAY_TITLE_START_Y;
        m_Title.Text = "Remote Controller PLR\n";
        m_Title.FitSize();

        y += h;
        for(int i=0; i<REMOTE_COUNT_MAX; ++i)
        {
            uint32_t dx = 160 * i + 5;
            PlrView &pv = m_PlrView[i];
            Add( pv.m_Indicator );
            pv.m_Indicator.Text = "○\n○\n○\n○";
            pv.m_Indicator.X = dx;
            pv.m_Indicator.Y = y;
            pv.m_Indicator.TextColor = ToColor(LAWN_GREEN);
            pv.m_Indicator.FontWidth *= 0.5;
            pv.m_Indicator.FontHeight *= 0.5;
            pv.m_Indicator.FitSize();

            Add( pv.m_CurrentPlr );
            pv.m_CurrentPlr.Text = "-";
            pv.m_CurrentPlr.X = dx + 20;
            pv.m_CurrentPlr.Y = y - 5;
            pv.m_CurrentPlr.Width = 500;

            Add( pv.m_AvgPlr );
            pv.m_AvgPlr.Text = "-";
            pv.m_AvgPlr.X = dx + 20;
            pv.m_AvgPlr.Y = y + 23;
            pv.m_AvgPlr.FontWidth *= 0.5;
            pv.m_AvgPlr.FontHeight *= 0.5;
            pv.m_AvgPlr.FixedWidth *= 0.5;
            pv.m_AvgPlr.Width = 500;

            Add( pv.m_MinMaxPlr );
            pv.m_MinMaxPlr.Text = "-";
            pv.m_MinMaxPlr.X = dx + 20;
            pv.m_MinMaxPlr.Y = y + 38;
            pv.m_MinMaxPlr.FontWidth *= 0.5;
            pv.m_MinMaxPlr.FontHeight *= 0.5;
            pv.m_MinMaxPlr.FixedWidth *= 0.5;
            pv.m_MinMaxPlr.Width = 500;

            Add( pv.m_Bar );
            pv.m_Bar.Text = "------";
            pv.m_Bar.X = dx + 20;
            pv.m_Bar.Y = y + 38;
            pv.m_Bar.TextColor = m_Color[i];
            pv.m_Bar.FitSize();
        }

        Add( m_AxisY );
        m_AxisY.Text = "^\n";
        for(int i=0; i<13; ++i)
        {
            m_AxisY.Text += "|\n";
        }
        m_AxisY.X = m_ChartLeft;
        m_AxisY.Y = m_ChartTop - 30;
        m_AxisY.FitSize();

        Add( m_AxisX );
        m_AxisX.Text = string(60, '-') + ">";
        m_AxisX.X = m_ChartLeft;
        m_AxisX.Y = m_ChartBottom;
        m_AxisX.FitSize();

        for(int i=0; i<10; ++i)
        {
            Add( m_AuxLine[i] );
            m_AuxLine[i].Text = string(59, '-');
            m_AuxLine[i].X = m_ChartLeft + 10;
            m_AuxLine[i].Y = m_ChartBottom - (i + 1) * (m_ChartBottom - m_ChartTop) / 10;

            if( i == 4 )
            {
                m_AuxLine[i].Text += "50%";
                m_AuxLine[i].TextColor = ToColor(0x555555);
            }
            else if( i == 9 )
            {
                m_AuxLine[i].Text += "100%";
                m_AuxLine[i].TextColor = ToColor(0x555555);
            }
            else
            {
                m_AuxLine[i].TextColor = ToColor(0x111111);
            }
            m_AuxLine[i].FitSize();
        }

        uint8_t padCount = 0;
        const NPad::PadInfo* info[REMOTE_COUNT_MAX];
        NPad::GetInstance().GetPadInfo(info, &padCount);
        for(int i=0; i<padCount; ++i)
        {
            m_PlotCount = info[i]->recorder.GetRecordCount();
            m_Plot[i] = new Label[m_PlotCount];

            for(int j=0; j<m_PlotCount; ++j)
            {
                m_Plot[i][j].Text = " ";
                m_Plot[i][j].X = -10;
                m_Plot[i][j].Y = -10;
                m_Plot[i][j].FontWidth *= 2;
                m_Plot[i][j].FontHeight *= 2;
                m_Plot[i][j].FitSize();
                m_Plot[i][j].TextColor = m_Color[i];
                Add( m_Plot[i][j] );
            }
        }

        SetPadText(Button::B, "Clear");
    } //NOLINT(impl/function_size)

    virtual ~NPadPlrViewer()
    {
        for(int i=0; i<REMOTE_COUNT_MAX; ++i)
        {
            if( m_Plot[i] )
            {
                delete[] m_Plot[i];
            }
        }
    }
};


class BtPlrViewer : public SceneViewer<NPad>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    static const uint16_t EXPECTED_SAMPLING_COUNT_PER_SEC = 200;
    static const uint16_t PLOT_COUNT = 120 + 30; // for 3 mins (1.5sec * 120plots) + alpha

    int64_t m_DisplayRange; //us
    nn::os::Tick m_StartTick;
    nn::os::Tick m_EndTick;
    nn::os::Tick m_ClearTick;

    class PlrView
    {
    public :

        Label m_Address;
        Label m_Indicator;
        Label m_CurrentPlr;
        Label m_AvgPlr;
        Label m_MinMaxPlr;
        Label m_Bar;

        void Enable(const bool& isEnabled)
        {
            if( isEnabled )
            {
                m_Address.TextColor = ToColor(WHITE);
                m_Indicator.TextColor = ToColor(LAWN_GREEN);
                m_CurrentPlr.TextColor = ToColor(WHITE);
                m_AvgPlr.TextColor = ToColor(WHITE);
                m_MinMaxPlr.TextColor = ToColor(WHITE);
            }
            else
            {
                static Color c = ToColor(0x333333);
                m_Address.TextColor = c;
                m_Indicator.TextColor = c;
                m_CurrentPlr.Text = "-";
                m_CurrentPlr.TextColor = c;
                m_AvgPlr.TextColor = c;
                m_MinMaxPlr.TextColor = c;
            }
        }

        void SetLedPattern(const uint8_t ledPattern)
        {
            uint8_t bitPattern = ledPattern;
            string strPattern = "";
            for(int i=0; i<4; ++i)
            {
                if( bitPattern & 0x1 )
                {
                    strPattern += "●\n";
                }
                else
                {
                    strPattern += "○\n";
                }
                bitPattern >>= 1;
            }
            m_Indicator.Text = strPattern;
        }
    };
    PlrView m_PlrView[REMOTE_COUNT_MAX];
    Label   m_ConsolePlrLog;

    uint32_t m_ChartLeft;
    uint32_t m_ChartRight;
    uint32_t m_ChartTop;
    uint32_t m_ChartBottom;

    uint32_t m_MinValue;
    uint32_t m_MaxValue;

    struct PlrLabel
    {
        BluetoothPlr plr;
        Label    label;
    };
    std::deque<PlrLabel> m_Plot[REMOTE_COUNT_MAX];
    Color m_Color[REMOTE_COUNT_MAX];

    Label m_Title;
    Label m_AxisX;
    Label m_AxisY;

    double m_ResolutionX;
    double m_ResolutionY;

    Label m_AuxLine[10];

    Label m_ChMap0Ind;
    Label m_ChMap40Ind;
    Label m_ChMap41Ind;
    Label m_ChMap80Ind;
    Label m_ChMap[2];

    Label m_ChMapCountInd;
    Label m_ChMapCount;

private:

    uint32_t ToPointX(nn::os::Tick tick)
    {
        return m_ChartLeft + (tick - m_StartTick).ToTimeSpan().GetMicroSeconds() * m_ResolutionX;
    }

    uint32_t ToPointY(int32_t plr)
    {
        return m_ChartBottom - (plr - m_MinValue) * m_ResolutionY;
    }

    void UpdatePlr()
    {
        const BluetoothPlrStatistics::StatisticsList& stats = BluetoothPlrStatistics::GetInstance().GetStatistics();
        for(int i=0; i<REMOTE_COUNT_MAX; ++i)
        {
            m_Plot[i].clear();
            auto it = stats[i].rbegin();
            while(it != stats[i].rend())
            {
                PlrLabel pl;
                pl.plr = *it;
                pl.label.FontWidth *= 2;
                pl.label.FontHeight *= 2;

                if( it->isConnected )
                {
                    m_Plot[i].push_front(pl);
                }
                ++it;
            }
        }
    }

    void UpdateChannelMap()
    {
        m_ChMap[0].Text = "";
        m_ChMap[1].Text = "";

        const uint8_t* channelMap = BluetoothChannelMap::GetInstance().GetChannelMap();

        for(int i = 0; i < 10; ++i)
        {
            for(int j = 0; j < 8; ++j)
            {
                if(((channelMap[i] >> j) & 0x01) != 0)
                {
                    m_ChMap[i / 5].Text += '+';
                }
                else
                {
                    m_ChMap[i / 5].Text += '-';
                }
            }
        }

        int channelCount = BluetoothChannelMap::GetInstance().GetChannelCount();

        ostringstream oss;
        oss.setf(std::ios::right);
        oss.fill('0');
        oss.width(2);
        oss << channelCount;
        m_ChMapCount.Text = oss.str();
    }

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void ShowImpl(Display& display)
    {
        // 表示範囲の更新
        m_EndTick = nn::os::GetSystemTick();
        m_StartTick = m_EndTick - nn::os::Tick(nn::TimeSpan::FromMicroSeconds(m_DisplayRange));

        UpdatePlr();
        UpdateChannelMap();

        m_ConsolePlrLog.Text = "";
        for(int i=0; i<REMOTE_COUNT_MAX; ++i)
        {
            // パケロス率のグラフ表示
            double sumCnt = 0;
            double sumPlr = 0;
            double minPlr = 100;
            double maxPlr = 0;
            for(int j=0; j<m_Plot[i].size(); ++j)
            {
                PlrLabel& plot = m_Plot[i][j];
                double plr = plot.plr.plr;
                uint8_t charDiff = 30; // "-" と "." の高さのずれの補正
                plot.label.X = ToPointX(plot.plr.updateTime);
                plot.label.Y = ToPointY(plr);
                plot.label.Y -= charDiff;
                plot.label.TextColor = m_Color[i];

                nn::os::Tick m_DisplayRangeLeftTick = std::max(m_ClearTick, m_StartTick);
                if( plot.plr.isConnected && m_DisplayRangeLeftTick < plot.plr.updateTime &&
                    m_ChartTop <= plot.label.Y + charDiff && plot.label.Y + charDiff <= m_ChartBottom )
                {
                    // 直近 1 分 かつ ActiveMode 中の PLR でなければ計算する
                    if( (m_EndTick - plot.plr.updateTime).ToTimeSpan().GetMilliSeconds() < 60 * 1000 &&
                        plr != BluetoothPlrStatistics::INVALID_PLR )
                    {
                        sumPlr += plr;
                        ++sumCnt;
                        minPlr = std::min(minPlr, plr);
                        maxPlr = std::max(maxPlr, plr);
                    }

                    if( plr == BluetoothPlrStatistics::INVALID_PLR )
                    {
                        // ActiveMode 時のものはグラフに表示しない
                        plot.label.Text = " ";
                    }
                    else
                    {
                        plot.label.Text = ".";
                    }
                    Add(plot.label);
                }
            }

            // RC の LED パターンの表示
            m_PlrView[i].SetLedPattern(0xf);

            // データがない or 最新のデータが3秒以内出なければ切断しているとする
            bool isConnected = true;
            if( m_Plot[i].size() == 0 )
            {
                isConnected = false;
            }
            else
            {
                isConnected = (m_EndTick - m_Plot[i].back().plr.updateTime).ToTimeSpan().GetMilliSeconds() < 10000;
            }
            m_PlrView[i].Enable(isConnected);

            // パケロス数の数値表示
            if( isConnected )
            {
                m_PlrView[i].m_Address.Text = ToString(m_Plot[i].back().plr.address);

                ostringstream oss;
                double plr = m_Plot[i].back().plr.plr;
                if( plr == BluetoothPlrStatistics::INVALID_PLR )
                {
                    oss << "Active";
                }
                else
                {
                    oss << fixed << setw(5) << setprecision(1) << plr << "%";
                }
                m_PlrView[i].m_CurrentPlr.Text = oss.str();
                oss.str("");

                double avgPlr = sumPlr / sumCnt;
                oss << fixed << setw(5) << setprecision(1) << avgPlr << "%";
                m_PlrView[i].m_AvgPlr.Text = oss.str();
                oss.str("");

                oss << fixed << setw(5) << setprecision(1) << minPlr << "%/"
                    << fixed << setw(5) << setprecision(1) << maxPlr << "%";
                m_PlrView[i].m_MinMaxPlr.Text = oss.str();
                oss.str("");

                oss << ToString(m_Plot[i].back().plr.address) << ", "
                    << "cur:" << fixed << setprecision(1) << plr << "%, "
                    << "avg:" << fixed << setprecision(1) << avgPlr << "%, "
                    << "min:" << fixed << setprecision(1) << minPlr << "%, "
                    << "max:" << fixed << setprecision(1) << maxPlr << "%"
                    << endl;
                m_ConsolePlrLog.Text += oss.str();
                oss.str("");
            }
        }

        // コンソール出力
        m_ConsolePlrLog.ShowImpl(display);

        // 無駄なものが多いのでコンソール表示しない
        bool isEnabled = false;
        if( Display::GetInstance().IsEnabledConsoleOutput() )
        {
            Display::GetInstance().DisableConsoleOutput();
            isEnabled = true;
        }

        SceneViewer<NPad>::ShowImpl(display);

        if( isEnabled )
        {
            Display::GetInstance().EnableConsoleOutput();
        }

        // 全プロットを削除する
        for(int i=0; i<REMOTE_COUNT_MAX; ++i)
        {
            for(int j=0; j<m_Plot[i].size(); ++j)
            {
                Remove(m_Plot[i][j].label);
            }
            m_Plot[i].clear();
        }
    } //NOLINT(impl/function_size)

    virtual void InputPad(Pad& pad)
    {
        if( pad.IsTrigger(Button::B) )
        {
            Clear();
        }
    }

    virtual void Clear()
    {
        m_ClearTick = nn::os::GetSystemTick();
        for(auto& pv :  m_PlrView)
        {
            pv.m_CurrentPlr.Text = "-";
            pv.m_AvgPlr.Text = "-";
            pv.m_MinMaxPlr.Text = "-";
            pv.m_Address.Text = "--:--:--:--:--:--";
        }
    }

/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:
    BtPlrViewer()
    {
        m_DisplayRange = 180 * 1000 * 1000; // us

        nn::os::Tick tick = nn::os::GetSystemTick();
        m_StartTick = tick;
        m_EndTick = tick;

        m_ChartLeft = 10;
        m_ChartRight = 1210;
        m_ChartTop = 230;
        m_ChartBottom = 600;

        m_MaxValue = 100;
        m_MinValue = 0;

        m_ResolutionX = 1.0f * (m_ChartRight - m_ChartLeft) / m_DisplayRange; // dot / ms
        m_ResolutionY = 1.0f * (m_ChartBottom - m_ChartTop) / (m_MaxValue - m_MinValue); // dot / score

        uint32_t c[REMOTE_COUNT_MAX] = {RED, ORANGE_RED1, YELLOW2, GREEN, CYAN, BLUE3, DARK_SLATE_BLUE, HOT_PINK};
        for(int i=0; i<REMOTE_COUNT_MAX; ++i)
        {
            m_Color[i] = ToColor(c[i]);
        }

        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y - 20;

        Add( m_Title );
        m_Title.X = DISPLAY_TITLE_START_X;
        m_Title.Y = DISPLAY_TITLE_START_Y;
        m_Title.Text = "Remote Controller PLR\n";
        m_Title.FitSize();

        y += h;
        for(int i=0; i<REMOTE_COUNT_MAX; ++i)
        {
            uint32_t dx = 160 * i + 5;
            PlrView &pv = m_PlrView[i];
            Add( pv.m_Indicator );
            pv.m_Indicator.Text = "○\n○\n○\n○\n";
            pv.m_Indicator.X = dx;
            pv.m_Indicator.Y = y;
            pv.m_Indicator.TextColor = ToColor(LAWN_GREEN);
            pv.m_Indicator.FontWidth *= 0.5;
            pv.m_Indicator.FontHeight *= 0.5;
            pv.m_Indicator.FitSize();
            pv.m_Indicator.Width *= 2; // FitSize() は全角に対応していないため大きめに確保する

            Add( pv.m_CurrentPlr );
            pv.m_CurrentPlr.Text = "-";
            pv.m_CurrentPlr.X = dx + 20;
            pv.m_CurrentPlr.Y = y - 5;
            pv.m_CurrentPlr.Width = 500;

            Add( pv.m_AvgPlr );
            pv.m_AvgPlr.Text = "-";
            pv.m_AvgPlr.X = dx + 20;
            pv.m_AvgPlr.Y = y + 23;
            pv.m_AvgPlr.FontWidth *= 0.5;
            pv.m_AvgPlr.FontHeight *= 0.5;
            pv.m_AvgPlr.FixedWidth *= 0.5;
            pv.m_AvgPlr.Width = 500;

            Add( pv.m_MinMaxPlr );
            pv.m_MinMaxPlr.Text = "-";
            pv.m_MinMaxPlr.X = dx + 20;
            pv.m_MinMaxPlr.Y = y + 38;
            pv.m_MinMaxPlr.FontWidth *= 0.5;
            pv.m_MinMaxPlr.FontHeight *= 0.5;
            pv.m_MinMaxPlr.FixedWidth *= 0.5;
            pv.m_MinMaxPlr.Width = 500;

            Add( pv.m_Bar );
            pv.m_Bar.Text = "-------";
            pv.m_Bar.X = dx + 20;
            pv.m_Bar.Y = y + 38;
            pv.m_Bar.TextColor = m_Color[i];
            pv.m_Bar.FitSize();

            Add( pv.m_Address );
            pv.m_Address.Text = "--:--:--:--:--:--";
            pv.m_Address.X = dx + 10;
            pv.m_Address.Y = y + 60;
            pv.m_Address.FontWidth *= 0.5;
            pv.m_Address.FontHeight *= 0.5;
            pv.m_Address.FixedWidth = 8;
            pv.m_Address.FitSize();
        }

        Add( m_AxisY );
        m_AxisY.Text = "^\n";
        for(int i=0; i<13; ++i)
        {
            m_AxisY.Text += "|\n";
        }
        m_AxisY.X = m_ChartLeft - 10;
        m_AxisY.Y = m_ChartTop - 30;
        m_AxisY.FitSize();

        Add( m_AxisX );
        m_AxisX.Text = string(60, '-') + ">";
        m_AxisX.X = m_ChartLeft;
        m_AxisX.Y = m_ChartBottom;
        m_AxisX.FitSize();

        uint32_t auxCount = sizeof(m_AuxLine) / sizeof(m_AuxLine[0]);
        for(int i=0; i<auxCount; ++i)
        {
            Add( m_AuxLine[i] );
            m_AuxLine[i].Text = string(59, '-');
            m_AuxLine[i].X = m_ChartLeft + 10;
            m_AuxLine[i].Y = m_ChartBottom - (i + 1) * (m_ChartBottom - m_ChartTop) / auxCount;

            if( i == 4 )
            {
                m_AuxLine[i].Text += "50%";
                m_AuxLine[i].TextColor = ToColor(0x555555);
            }
            else if( i == 9 )
            {
                m_AuxLine[i].Text += "100%";
                m_AuxLine[i].TextColor = ToColor(0x555555);
            }
            else
            {
                m_AuxLine[i].TextColor = ToColor(0x111111);
            }
            m_AuxLine[i].FitSize();
        }

        Add(m_ChMap0Ind);
        m_ChMap0Ind.Text = string(1, '1');
        m_ChMap0Ind.X = m_ChartLeft + 10;
        m_ChMap0Ind.Y = m_ChartBottom + 25;
        m_ChMap0Ind.TextColor = ToColor(0xFFFFFF);
        m_ChMap0Ind.FitSize();

        Add(m_ChMap40Ind);
        m_ChMap40Ind.Text = "40";
        m_ChMap40Ind.X = m_ChartLeft + 860;
        m_ChMap40Ind.Y = m_ChartBottom + 25;
        m_ChMap40Ind.TextColor = ToColor(0xFFFFFF);
        m_ChMap40Ind.FitSize();

        Add(m_ChMap41Ind);
        m_ChMap41Ind.Text = "41";
        m_ChMap41Ind.X = m_ChartLeft + 10;
        m_ChMap41Ind.Y = m_ChartBottom + 50;
        m_ChMap41Ind.TextColor = ToColor(0xFFFFFF);
        m_ChMap41Ind.FitSize();

        Add(m_ChMap80Ind);
        m_ChMap80Ind.Text = "80";
        m_ChMap80Ind.X = m_ChartLeft + 860;
        m_ChMap80Ind.Y = m_ChartBottom + 50;
        m_ChMap80Ind.TextColor = ToColor(0xFFFFFF);
        m_ChMap80Ind.FitSize();

        for(int i = 0; i < 2; ++i)
        {
            Add( m_ChMap[i] );
            m_ChMap[i].Text = string(40, '-');
            m_ChMap[i].X = m_ChartLeft + 50;
            m_ChMap[i].Y = m_ChartBottom + 25 * (i + 1);
            m_ChMap[i].TextColor = ToColor(0x555555);
            m_ChMap[i].FitSize();
        }

        Add( m_ChMapCountInd );
        m_ChMapCountInd.Text = "CH count:";
        m_ChMapCountInd.X = m_ChartLeft + 860 + 100;
        m_ChMapCountInd.Y = m_ChartBottom + 37;
        m_ChMapCountInd.TextColor = ToColor(0xFFFFFF);
        m_ChMapCountInd.FitSize();

        Add( m_ChMapCount );
        m_ChMapCount.Text = "00";
        m_ChMapCount.X = m_ChartLeft + 860 + 280;
        m_ChMapCount.Y = m_ChartBottom + 37;
        m_ChMapCount.TextColor = ToColor(0xFFFFFF);
        m_ChMapCount.FitSize();

        // コンソール出力のみにするため、画面外に表示する
        m_ConsolePlrLog.X = 5000;
        m_ConsolePlrLog.Y = 5000;
        m_ConsolePlrLog.Width = 5000;

        SetPadText(Button::B, "Clear");
    } //NOLINT(impl/function_size)

    virtual ~BtPlrViewer()
    {
    }
};

class RssiViewer : public SceneViewer<RssiStatistics>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    static const uint32_t ChartLeft     = 10;
    static const uint32_t ChartRight    = 1210;
    static const uint32_t ChartTop      = 230;
    static const uint32_t ChartBottom   = 600;
    static const int32_t  MinValue      = -90;
    static const int32_t  MaxValue      = -10;
    static const uint32_t PlotCount     = 180; // update every 1 sec, record for 180 sec
    static const int64_t  DisplayRange  = PlotCount * 1000 * 1000; //us
    double m_ResolutionX;
    double m_ResolutionY;

    nn::os::Tick m_StartTick;
    nn::os::Tick m_EndTick;
    nn::os::Tick m_ClearTick;

    class RssiView
    {
    public :

        Label m_Address;
        Label m_Indicator;
        Label m_CurrentRssi;
        Label m_AvgRssi;
        Label m_MinMaxRssi;
        Label m_Bar;

        void Enable(const bool& isEnabled)
        {
            if( isEnabled )
            {
                m_Address.TextColor = ToColor(WHITE);
                m_Indicator.TextColor = ToColor(ROYAL_BLUE);
                m_CurrentRssi.TextColor = ToColor(WHITE);
                m_AvgRssi.TextColor = ToColor(WHITE);
                m_MinMaxRssi.TextColor = ToColor(WHITE);
            }
            else
            {
                static Color c = ToColor(0x333333);
                m_Address.TextColor = c;
                m_Indicator.TextColor = c;
                m_CurrentRssi.Text = "-";
                m_CurrentRssi.TextColor = c;
                m_AvgRssi.TextColor = c;
                m_MinMaxRssi.TextColor = c;
            }
        }

        void SetLedPattern(const uint8_t ledPattern)
        {
            uint8_t bitPattern = ledPattern;
            string strPattern = "";
            for(int i=0; i<4; ++i)
            {
                if( bitPattern & 0x1 )
                {
                    strPattern += "●\n";
                }
                else
                {
                    strPattern += "○\n";
                }
                bitPattern >>= 1;
            }
            m_Indicator.Text = strPattern;
        }
    };
    RssiView m_RssiView[ConnectableNodeCountMax];
    Label   m_ConsoleRssiLog;

    struct RssiLabel
    {
        Rssi    rssi;
        Label   label;
    };

    RssiLabel m_Plot[ConnectableNodeCountMax][PlotCount];
    Color m_Color[ConnectableNodeCountMax];

    Label m_Title;
    Label m_AxisX;
    Label m_AxisY;

    Label m_AuxLine[8];
    Label m_AuxUnit[3];

private:

    uint32_t ToPointX(nn::os::Tick tick)
    {
        return ChartLeft + (tick - m_StartTick).ToTimeSpan().GetMicroSeconds() * m_ResolutionX;
    }

    uint32_t ToPointY(uint32_t rssi)
    {
        return ChartBottom - (rssi - MinValue) * m_ResolutionY;
    }

    void UpdateRssi()
    {
        const vector<RssiHistory>& stats = target->GetStatistics();
        for(int i=0; i<ConnectableNodeCountMax; ++i)
        {
            if( i < stats.size() )
            {
                auto jt = stats[i].rbegin();
                for(int j=0; j<PlotCount; ++j)
                {
                    RssiLabel& plot = m_Plot[i][PlotCount - j - 1];
                    plot.rssi.Clear();

                    if( jt != stats[i].rend() )
                    {
                        plot.rssi = *jt;
                        ++jt;
                    }
                }
            }
            else
            {
                for(int j=0; j<PlotCount; ++j)
                {
                    m_Plot[i][j].rssi.Clear();
                }
            }
        }
    }

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void ShowImpl(Display& display)
    {
        // 表示範囲の更新
        m_EndTick = nn::os::GetSystemTick();
        m_StartTick = m_EndTick - nn::os::Tick(nn::TimeSpan::FromMicroSeconds(DisplayRange));

        UpdateRssi();

        m_ConsoleRssiLog.Text = "";
        for(int i=0; i<ConnectableNodeCountMax; ++i)
        {
            // パケロス率のグラフ表示
            int32_t sumCnt = 0;
            int32_t sumRssi = 0;
            int32_t minRssi = RSSI_MAX;
            int32_t maxRssi = RSSI_MIN;
            for(int j=0; j<PlotCount; ++j)
            {
                RssiLabel& plot = m_Plot[i][j];
                int32_t rssi = plot.rssi.value;
                uint8_t charDiff = 30; // "-" と "." の高さのずれの補正
                plot.label.X = ToPointX(plot.rssi.recordTime);
                plot.label.Y = ToPointY(rssi);
                plot.label.Y -= charDiff;
                plot.label.TextColor = m_Color[i];

                nn::os::Tick m_DisplayRangeLeftTick = std::max(m_ClearTick, m_StartTick);
                if( m_DisplayRangeLeftTick < plot.rssi.recordTime && MinValue <= rssi && rssi <= MaxValue )
                {
                    // 直近 1 分の RSSI であれば計算する
                    if( (m_EndTick - plot.rssi.recordTime).ToTimeSpan().GetMilliSeconds() < 60 * 1000 )
                    {
                        ++sumCnt;
                        sumRssi += rssi;
                        minRssi = std::min(minRssi, rssi);
                        maxRssi = std::max(maxRssi, rssi);
                    }

                    plot.label.Text = ".";
                }
                else
                {
                    plot.label.Text = " ";
                }
            }

            // 最新のデータが3秒以内出なければ切断しているとする
            RssiLabel& plot = m_Plot[i][PlotCount - 1];
            bool isConnected = (m_EndTick - plot.rssi.recordTime).ToTimeSpan().GetMilliSeconds() < 3000;
            m_RssiView[i].Enable(isConnected);

            // RSSI の数値表示
            if( isConnected )
            {
                m_RssiView[i].m_Address.Text = ToString(plot.rssi.address);

                ostringstream oss;
                int32_t rssi = plot.rssi.value;
                if( rssi == RSSI_MAX )
                {
                    oss << " - dBm";
                }
                else
                {
                    oss << setw(3) << rssi << "dBm";
                }
                m_RssiView[i].m_CurrentRssi.Text = oss.str();
                oss.str("");

                double avgRssi = 1.0f * sumRssi / sumCnt;
                oss << fixed << setw(4) << setprecision(1) << avgRssi << "dBm";
                m_RssiView[i].m_AvgRssi.Text = oss.str();
                oss.str("");

                oss << setw(3) << minRssi << "dBm/" << setw(3) << maxRssi << "dBm";
                m_RssiView[i].m_MinMaxRssi.Text = oss.str();
                oss.str("");

                oss << ToString(m_Plot[i][PlotCount - 1].rssi.address) << ", "
                    << "cur:" << rssi << "dBm, "
                    << "avg:" << fixed << setprecision(1) << avgRssi << "dBm, "
                    << "min:" << minRssi << "dBm, "
                    << "max:" << maxRssi << "dBm"
                    << endl;
                m_ConsoleRssiLog.Text += oss.str();
                oss.str("");
            }
        }

        // コンソール出力
        m_ConsoleRssiLog.ShowImpl(display);

        // 無駄なものが多いのでコンソール表示しない
        bool isEnabled = false;
        if( Display::GetInstance().IsEnabledConsoleOutput() )
        {
            Display::GetInstance().DisableConsoleOutput();
            isEnabled = true;
        }

        SceneViewer<RssiStatistics>::ShowImpl(display);

        if( isEnabled )
        {
            Display::GetInstance().EnableConsoleOutput();
        }
    }

    virtual void InputPad(Pad& pad)
    {
        if( pad.IsTrigger(Button::B) )
        {
            Clear();
        }
    }

    virtual void Clear()
    {
        m_ClearTick = nn::os::GetSystemTick();
        for(auto& rv : m_RssiView)
        {
            rv.m_CurrentRssi.Text = "-";
            rv.m_AvgRssi.Text = "-";
            rv.m_MinMaxRssi.Text = "-";
            rv.m_Address.Text = "--:--:--:--:--:--";
        }

        for(auto& row : m_Plot)
        {
            for(auto& plot : row)
            {
                plot.rssi.Clear();
            }
        }
    }

/*---------------------------------------------------------------------------
　　　　　コンストラクタ
---------------------------------------------------------------------------*/
public:

    RssiViewer()
    {
        nn::os::Tick tick = nn::os::GetSystemTick();
        m_StartTick = tick;
        m_EndTick = tick;

        m_ResolutionX = 1.0f * (ChartRight - ChartLeft) / DisplayRange; // dot / ms
        m_ResolutionY = 1.0f * (ChartBottom - ChartTop) / (MaxValue - MinValue); // dot / score

        uint32_t c[ConnectableNodeCountMax] = {RED, ORANGE_RED1, YELLOW2, GREEN, CYAN, BLUE3, DARK_SLATE_BLUE};
        for(int i=0; i<ConnectableNodeCountMax; ++i)
        {
            m_Color[i] = ToColor(c[i]);
        }

        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t w = Display::GetInstance().GetFixedWidth();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t y = DISPLAY_CONTENT_START_Y - 20;

        Add( m_Title );
        m_Title.X = DISPLAY_TITLE_START_X;
        m_Title.Y = DISPLAY_TITLE_START_Y;
        m_Title.Text = "WLAN RSSI\n";
        m_Title.FitSize();

        y += h;
        for(int i=0; i<ConnectableNodeCountMax; ++i)
        {
            uint32_t dx = 180 * i + 15;
            RssiView &rv = m_RssiView[i];
            Add( rv.m_Indicator );
            rv.m_Indicator.Text = "○\n○\n○\n○\n";
            rv.m_Indicator.X = dx;
            rv.m_Indicator.Y = y;
            rv.m_Indicator.TextColor = ToColor(LAWN_GREEN);
            rv.m_Indicator.FontWidth *= 0.5;
            rv.m_Indicator.FontHeight *= 0.5;
            rv.m_Indicator.FitSize();
            rv.m_Indicator.Width *= 2; // FitSize() は全角に対応していないため大きめに確保する

            Add( rv.m_CurrentRssi );
            rv.m_CurrentRssi.Text = "-";
            rv.m_CurrentRssi.X = dx + 20;
            rv.m_CurrentRssi.Y = y - 5;
            rv.m_CurrentRssi.Width = 500;

            Add( rv.m_AvgRssi );
            rv.m_AvgRssi.Text = "-";
            rv.m_AvgRssi.X = dx + 20;
            rv.m_AvgRssi.Y = y + 23;
            rv.m_AvgRssi.FontWidth *= 0.5;
            rv.m_AvgRssi.FontHeight *= 0.5;
            rv.m_AvgRssi.FixedWidth *= 0.5;
            rv.m_AvgRssi.Width = 500;

            Add( rv.m_MinMaxRssi );
            rv.m_MinMaxRssi.Text = "-";
            rv.m_MinMaxRssi.X = dx + 20;
            rv.m_MinMaxRssi.Y = y + 38;
            rv.m_MinMaxRssi.FontWidth *= 0.5;
            rv.m_MinMaxRssi.FontHeight *= 0.5;
            rv.m_MinMaxRssi.FixedWidth *= 0.5;
            rv.m_MinMaxRssi.Width = 500;

            Add( rv.m_Bar );
            rv.m_Bar.Text = "-------";
            rv.m_Bar.X = dx + 20;
            rv.m_Bar.Y = y + 38;
            rv.m_Bar.TextColor = m_Color[i];
            rv.m_Bar.FitSize();

            Add( rv.m_Address );
            rv.m_Address.Text = "--:--:--:--:--:--";
            rv.m_Address.X = dx + 10;
            rv.m_Address.Y = y + 60;
            rv.m_Address.FontWidth *= 0.5;
            rv.m_Address.FontHeight *= 0.5;
            rv.m_Address.FixedWidth = 8;
            rv.m_Address.FitSize();

            rv.SetLedPattern(0xf);
        }

        Add( m_AxisY );
        m_AxisY.Text = "^\n";
        for(int i=0; i<13; ++i)
        {
            m_AxisY.Text += "|\n";
        }
        m_AxisY.X = ChartLeft - 10;
        m_AxisY.Y = ChartTop - 30;
        m_AxisY.FitSize();

        Add( m_AxisX );
        m_AxisX.Text = string(60, '-') + ">";
        m_AxisX.X = ChartLeft;
        m_AxisX.Y = ChartBottom;
        m_AxisX.FitSize();

        uint32_t auxLineCount = sizeof(m_AuxLine) / sizeof(m_AuxLine[0]);
        for(int i=0; i<auxLineCount; ++i)
        {
            Add( m_AuxLine[i] );
            m_AuxLine[i].Text = string(59, '-');
            m_AuxLine[i].X = ChartLeft + 10;
            m_AuxLine[i].Y = ChartBottom - (i + 1) * (ChartBottom - ChartTop) / auxLineCount;

            if( i == 3 || i == 7 )
            {
                m_AuxLine[i].TextColor = ToColor(0x555555);
            }
            else
            {
                m_AuxLine[i].TextColor = ToColor(0x111111);
            }
            m_AuxLine[i].FitSize();
        }

        uint32_t auxUnitCount = sizeof(m_AuxUnit) / sizeof(m_AuxUnit[0]);
        for(int i=0; i<auxUnitCount; ++i)
        {
            Add( m_AuxUnit[i] );
            ostringstream oss;
            if( i == 0 )
            {
                oss << MinValue;
                m_AuxUnit[i].Y = m_AxisX.Y;
            }
            else if( i == 1 )
            {
                oss << static_cast<int>((MaxValue - MinValue) / 2 + MinValue);
                m_AuxUnit[i].Y = m_AuxLine[auxLineCount / 2 - 1].Y;
            }
            else if( i == 2 )
            {
                oss << MaxValue;
                m_AuxUnit[i].Y = m_AuxLine[auxLineCount - 1].Y;
            }
            m_AuxUnit[i].Text = oss.str() + "\ndBm";
            m_AuxUnit[i].FontWidth *= 0.9;
            m_AuxUnit[i].FontHeight *= 0.9;
            m_AuxUnit[i].FixedWidth = 14;
            m_AuxUnit[i].X = (m_AuxLine[0].Text.size() + 2.5f) * w;
            m_AuxUnit[i].Y -= 0.5f * 0.8 * h;
            m_AuxUnit[i].TextColor = ToColor(0x555555);
        }

        for(int i=0; i<ConnectableNodeCountMax; ++i)
        {
            for(int j=0; j<PlotCount; ++j)
            {
                Add(m_Plot[i][j].label);
                m_Plot[i][j].label.FontWidth *= 2;
                m_Plot[i][j].label.FontHeight *= 2;
            }
        }

        // コンソール出力のみにするため、画面外に表示する
        m_ConsoleRssiLog.X = 5000;
        m_ConsoleRssiLog.Y = 5000;
        m_ConsoleRssiLog.Width = 10000;

        SetPadText(Button::B, "Clear");
    } //NOLINT(impl/function_size)

    virtual ~RssiViewer()
    {
    }
};


class SearchStatisticsViewer : public SceneViewer<AfTxScanner>
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:

    enum DisplayMode
    {
        DisplayMode_Total = 0,
        DisplayMode_Interval
    };

    Label m_Title;

    Label m_TxTitle;
    Label m_TxChannel;
    Label m_TxSize;
    Label m_TxCount;
    Label m_TxError;
    Label m_TxDataRate;
    Label m_TxExecutionTime;

    Label m_ScanTitle;
    Label m_ScanChannel;
    Label m_ScanCount;
    Label m_ScanExecutionTime;

    Label m_Clear;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void ShowImpl(Display& display)
    {
        ostringstream oss;
        auto stats = target->GetStatistics();
        auto afParam = target->GetAfParam();
        auto scanParam = target->GetScanParam();

        oss << "Tx Channel          : ";
        if( target->IsAfEnabled() )
        {
            oss << GetChannelListStr(afParam.channelList, afParam.channelCount);
        }
        else
        {
            oss << "-";
        }
        m_TxChannel.Text = oss.str();
        oss.str("");

        oss << "Tx Size             : " << stats.sendSize << " byte";
        m_TxSize.Text = oss.str();
        oss.str("");

        oss << "Tx Count            : " << stats.sendCount;
        m_TxCount.Text = oss.str();
        oss.str("");

        oss << "Tx Error            : " << stats.sendErrorCount;
        m_TxError.Text = oss.str();
        oss.str("");

        oss << "Tx DataRate         : " << fixed << setprecision(3)
            <<CalculateThroughput(stats.sendSize, stats.firstSendTime, stats.lastSendTime) << " Mbps";
        m_TxDataRate.Text = oss.str();
        oss.str("");

        oss << "Tx Execution Time   : " << fixed << setprecision(0) << stats.sendExecutionTime.GetValue() << " ms ("
            << stats.sendExecutionTime.GetMinValue() << " ms/ " << stats.sendExecutionTime.GetMaxValue() << " ms)";
        m_TxExecutionTime.Text = oss.str();
        oss.str("");


        oss << "Scan Channel        : ";
        if( target->IsScanEnabled() )
        {
            oss << GetChannelListStr(scanParam.channelList, scanParam.channelCount);
        }
        else
        {
            oss << "-";
        }
        m_ScanChannel.Text = oss.str();
        oss.str("");

        oss << "Scan Count          : " << stats.scanCount;
        m_ScanCount.Text = oss.str();
        oss.str("");

        oss << "Scan Execution Time : " << fixed << setprecision(0) << stats.scanExecutionTime.GetValue() << " ms ("
            << stats.scanExecutionTime.GetMinValue() << " ms/ " << stats.scanExecutionTime.GetMaxValue() << " ms)";
        m_ScanExecutionTime.Text = oss.str();
        oss.str("");

        SceneViewer<AfTxScanner>::ShowImpl(display);
    }

    virtual void InputPad(Pad& pad)
    {
        if( pad.IsTrigger(Button::B) )
        {
            target->ClearStatistics();
        }
    }

protected:

    string GetChannelListStr(int16_t chList[nn::wlan::WirelessChannelsCountMax], uint8_t chCount)
    {
        ostringstream oss;
        uint8_t threshold = 5;

        oss << chList[0];
        if( chCount <= threshold )
        {
            for(int i=1; i<chCount && i<threshold; ++i)
            {
                oss << ", " << chList[i];
            }
        }
        else
        {
            for(int i=1; i<chCount && i<threshold - 2; ++i)
            {
                oss << ", " << chList[i];
            }
            oss << ", ..., " << chList[chCount - 1];
        }
        oss << "ch";

        return oss.str();
    }

private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    SearchStatisticsViewer()
    {
        uint32_t h = Display::GetInstance().GetLineHeight();
        uint32_t x = DISPLAY_CONTENT_START_X;
        uint32_t sx = DISPLAY_CONTENT_START_X + 3 * h;
        uint32_t y = DISPLAY_CONTENT_START_Y;
        uint32_t blankLine = 27;

        Add(m_Title);
        m_Title.X = DISPLAY_TITLE_START_X;
        m_Title.Y = DISPLAY_TITLE_START_Y;
        m_Title.Text = "WLAN Search Stats\n";
        m_Title.FitSize();

        Add(m_TxTitle);
        m_TxTitle.X = x;
        m_TxTitle.Y = (y += h);
        m_TxTitle.Text = "[Action Frame]";
        m_TxTitle.Width += 400;

        y += h;

        Add(m_TxChannel);
        m_TxChannel.X = sx;
        m_TxChannel.Y = (y += h);
        m_TxChannel.Width += 400;

        Add(m_TxSize);
        m_TxSize.X = sx;
        m_TxSize.Y = (y += h);
        m_TxSize.Width += 400;

        Add(m_TxCount);
        m_TxCount.X = sx;
        m_TxCount.Y = (y += h);
        m_TxCount.Width += 400;

        Add(m_TxError);
        m_TxError.X = sx;
        m_TxError.Y = (y += h);
        m_TxError.Width += 400;

        Add(m_TxDataRate);
        m_TxDataRate.X = sx;
        m_TxDataRate.Y = (y += h);
        m_TxDataRate.Width += 400;

        Add(m_TxExecutionTime);
        m_TxExecutionTime.X = sx;
        m_TxExecutionTime.Y = (y += h);
        m_TxExecutionTime.Width += 400;

        y += 2 * h;

        Add(m_ScanTitle);
        m_ScanTitle.X = x;
        m_ScanTitle.Y = (y += h);
        m_ScanTitle.Text = "[Scan]";
        m_ScanTitle.Width += 400;

        y += h;

        Add(m_ScanChannel);
        m_ScanChannel.X = sx;
        m_ScanChannel.Y = (y += h);
        m_ScanChannel.Width += 400;

        Add(m_ScanCount);
        m_ScanCount.X = sx;
        m_ScanCount.Y = (y += h);
        m_ScanCount.Width += 400;

        Add(m_ScanExecutionTime);
        m_ScanExecutionTime.X = sx;
        m_ScanExecutionTime.Y = (y += h);
        m_ScanExecutionTime.Width += 400;

        SetPadText(Button::B, "Clear");
    }

private:

};

}    // WlanTest
