﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <cstring>
#include <string>
#include <vector>

#include <nn/os/os_Mutex.h>
#include <nn/socket/socket_Api.h>
#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/bsdsocket/cfg/cfg_ClientApi.h>

#include "Counter.h"
#include "Common.h"
#include "Util.h"

namespace WlanTest {

enum SocketMode
{
    SocketModeServer = 0x0,
    SocketModeClient = 0x1
};

/*!--------------------------------------------------------------------------*
  @brief        LcsSocketクラス
 *---------------------------------------------------------------------------*/

class LcsSocket
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:

    struct TxStatistics
    {
        // 更新間隔
        static nn::TimeSpan        UpdateInterval;
        static nn::os::Tick        LastUpdateTime;

        nn::os::Tick               FirstTxTime;
        nn::os::Tick               LastTxTime;

        // 総送信サイズ
        uint64_t                   TotalDataSize;

        // 送信開始後からの統計情報
        uint64_t                   TxDataSize;
        uint64_t                   TxSuccess;
        uint64_t                   TxError;
        ThroughputMeter            Throughput;

        // 一定間隔の統計情報
        uint64_t                   LastTxDataSize;
        uint64_t                   LastTxSuccess;
        uint64_t                   LastTxError;
        ThroughputMeter            LastThroughput;

        // 一定間隔の統計情報の処理変数
        uint64_t                   WorkTxDataSize;
        uint64_t                   WorkTxSuccess;
        uint64_t                   WorkTxError;
        ThroughputMeter            WorkThroughput;

        void Clear()
        {
            TotalDataSize = 0;
            ResetCount();
        }

        void ResetCount()
        {
            FirstTxTime = nn::os::Tick(0);
            LastTxTime = nn::os::Tick(0);

            TxDataSize = 0;
            TxSuccess = 0;
            TxError = 0;
            Throughput.Clear();

            LastTxDataSize = 0;
            LastTxSuccess = 0;
            LastTxError = 0;
            LastThroughput.Clear();

            WorkTxDataSize = 0;
            WorkTxSuccess = 0;
            WorkTxError = 0;
            WorkThroughput.Clear();
        }
    };

    struct RxStatistics
    {
        // 更新間隔
        static nn::TimeSpan        UpdateInterval;
        static nn::os::Tick        LastUpdateTime;

        nn::os::Tick               FirstRxTime;
        nn::os::Tick               LastRxTime;

        // 受信開始後からの統計情報
        uint64_t                   RxDataSize;
        uint64_t                   RxSuccess;
        uint64_t                   RxError;
        ThroughputMeter            Throughput;

        // 一定間隔の統計情報
        uint64_t                   LastRxDataSize;
        uint64_t                   LastRxSuccess;
        uint64_t                   LastRxError;
        ThroughputMeter            LastThroughput;

        // 一定間隔の統計情報の処理変数
        uint64_t                   WorkRxDataSize;
        uint64_t                   WorkRxSuccess;
        uint64_t                   WorkRxError;
        ThroughputMeter            WorkThroughput;

        void Clear()
        {
            ResetCount();
        }

        void ResetCount()
        {
            FirstRxTime = nn::os::Tick(0);
            LastRxTime = nn::os::Tick(0);

            RxDataSize = 0;
            RxSuccess = 0;
            RxError = 0;
            Throughput.Clear();

            LastRxDataSize = 0;
            LastRxSuccess = 0;
            LastRxError = 0;
            LastThroughput.Clear();

            WorkRxDataSize = 0;
            WorkRxSuccess = 0;
            WorkRxError = 0;
            WorkThroughput.Clear();
        }
    };

    class SocketInfo
    {
    public:
        bool m_IsAccepted;
        int32_t m_Socket;
        string m_IpAddressStr;
        TxStatistics m_TxStats;
        RxStatistics m_RxStats;

        SocketInfo()
        {
            m_IsAccepted = false;
            m_Socket = -1;
            m_IpAddressStr = "";
            m_TxStats.Clear();
            m_RxStats.Clear();
        }
    };

protected:

    fd_set m_ReadFds;
    string m_InterfaceName;

    int m_ListenSocket;
    vector<SocketInfo> m_DataSocketList;
    int m_DataSocketId;
    uint16_t m_Port;

    SocketMode m_SocketMode;
    SocketType m_SocketType;
    string m_SrcIpAddress;
    string m_DstIpAddress;
    string m_SubnetMask;
    string m_Dns1;
    string m_Dns2;
    string m_GwIpAddress;

    bool m_IsOpened;

    nn::os::Mutex m_TxStatsCs;
    nn::os::Mutex m_RxStatsCs;

    size_t m_TxBufferSize;
    size_t m_RxBufferSize;
    bool m_IsNegleEnabled;
    uint32_t m_StatsUpdateInterval;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    void Open();
    void Close();

    bool Accept();
    int Connect();
    bool Remove(const string& ipAddress);

    virtual nn::Result Send(const uint8_t data[], size_t size, size_t* pSentSize);
    virtual nn::Result Receive(uint8_t buf[], size_t* pRecvSize, const size_t bufSize);

    void UpdateLastTxStats();
    void UpdateLastRxStats();

    bool IsListIdValid();
    bool IsListIdValid(const int& id);

protected:

    void CreateSocket(int* pSocket);
    void ConfigureInterface();
    virtual bool WaitReceive(nn::TimeSpan timeSpan);

    void OpenSocket(int* pSocket);
    void CloseSocket(const int socket);
    nn::Result OpenServer(const int socket);
    nn::Result OpenClient(const int socket);

    void SetOption(const int socket);
    nn::Result SocketRecv(uint8_t pOutput[], size_t* pSize, sockaddr_in *pFromAddr, const size_t maxSize, int flags);

    bool RemoveByAddress(const string& address, SocketInfo* pSocketInfo);

    void UpdateTxStats(const int32_t& socket, const ssize_t& txSize);
    void UpdateRxStats(const int32_t& socket, const ssize_t& rxSize);

private:

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    LcsSocket();
    virtual ~LcsSocket();

private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

    static string GenerateIpAddress(uint8_t ip1, uint8_t ip2, uint8_t ip3, uint8_t ip4);
    static string GenerateIpAddress(uint8_t ip[4]);

    bool IsOpened() { return m_IsOpened; }

    void SetSrcIpAddress(const string& addr) { m_SrcIpAddress = addr; }
    string GetSrcIpAddress(){ return m_SrcIpAddress; }

    void SetDstIpAddress(const string& addr) { m_DstIpAddress = addr; }
    string GetDstIpAddress(){ return m_DstIpAddress; }

    void SetSubnetMask(const string& mask) { m_SubnetMask = mask; }
    string GetSubnetMask(){ return m_SubnetMask; }

    void SetGwIpAddress(const string& addr) { m_GwIpAddress = addr; }
    string GetGwIpAddress() { return m_GwIpAddress; }

    void SetDns1(const string& addr) { m_Dns1 = addr; }
    string GetDns1() { return m_Dns1; }

    void SetDns2(const string& addr) { m_Dns2 = addr; }
    string GetDns2() { return m_Dns2; }

    void SetPort(const uint16_t& port) { m_Port = port; }
    uint16_t GetPort() { return m_Port; }

    void SetSocketType(const SocketType& type) { m_SocketType = type; }
    SocketType GetSocketType() { return m_SocketType; }

    void SetSocketMode(const SocketMode& mode) { m_SocketMode = mode; }
    SocketMode GetSocketMode() { return m_SocketMode; }

    int GetDataSocketSize() { return m_DataSocketList.size(); }
    const vector<SocketInfo>* GetSocketInfo() { return &m_DataSocketList; }

    void SetTxBufferSize(const size_t& size) { m_TxBufferSize = size; }
    size_t GetTxBufferSize() { return m_TxBufferSize; }

    void SetRxBufferSize(const size_t& size) { m_RxBufferSize = size; }
    size_t GetRxBufferSize() { return m_RxBufferSize; }

    void SetNegleAlgorithm(const bool isEnabled) { m_IsNegleEnabled = isEnabled; }
    bool GetNegleAlgorithm() { return m_IsNegleEnabled; }

    void SetTxStatsUpdateInterval(const uint32_t interval) { TxStatistics::UpdateInterval = nn::TimeSpan::FromSeconds(interval); }
    uint32_t GetTxStatsUpdateInterval() { return TxStatistics::UpdateInterval.GetSeconds(); }

    void SetRxStatsUpdateInterval(const uint32_t interval) { RxStatistics::UpdateInterval = nn::TimeSpan::FromSeconds(interval); }
    uint32_t GetRxStatsUpdateInterval() { return RxStatistics::UpdateInterval.GetSeconds(); }

    void SetNextDataSocketId()
    {
        int size = GetDataSocketSize();
        if( size > 0 )
        {
            m_DataSocketId = (m_DataSocketId + 1) % size;
        }
    }
    void SetPreviousDataSocketId()
    {
        int size = GetDataSocketSize();
        if( size > 0 )
        {
            m_DataSocketId = (m_DataSocketId + size - 1) % size;
        }
    }
    bool SetDataSocketId(const int& id)
    {
        if( IsListIdValid(id) )
        {
            m_DataSocketId = id;
            return true;
        }
        return false;
    }
    int GetDataSocketId() { return m_DataSocketId; }

    void SetDataSize(const uint64_t& size)
    {
        SetDataSize(m_DataSocketId, size);
    }
    void SetDataSize(const int& id, const uint64_t& size)
    {
        if( IsListIdValid(id) )
        {
            m_DataSocketList[id].m_TxStats.TotalDataSize = size;
        }
    }
    uint64_t GetDataSize(const int& id)
    {
        if( IsListIdValid(id) )
        {
            return m_DataSocketList[id].m_TxStats.TotalDataSize;
        }
        else
        {
            return 0;
        }
    }
    uint64_t GetDataSize() { return GetDataSize(m_DataSocketId); }

    bool IsDataSocketAccepted(const int& id)
    {
        if( IsListIdValid(id) )
        {
            return m_DataSocketList[id].m_IsAccepted;
        }
        else
        {
            return false;
        }
    }
    bool IsDataSocketAccepted() { return IsDataSocketAccepted(m_DataSocketId); }

    int GetDataSocket(const int& id)
    {
        if( IsListIdValid(id) )
        {
            return m_DataSocketList[id].m_Socket;
        }
        else
        {
            return -1;
        }
    }
    int GetDataSocket() { return GetDataSocket(m_DataSocketId); }

    const TxStatistics* GetTxStatistics(const int& id)
    {
        if( IsListIdValid(id) )
        {
            return &m_DataSocketList[id].m_TxStats;
        }
        else
        {
            return nullptr;
        }
    }
    const TxStatistics* GetTxStatistics() { return GetTxStatistics(m_DataSocketId); }

    void ResetTxStatistics(const int& id)
    {
        if( IsListIdValid(id) )
        {
            m_TxStatsCs.Lock();
            m_DataSocketList[id].m_TxStats.ResetCount();
            m_TxStatsCs.Unlock();
        }
    }
    void ResetTxStatistics()
    {
        ResetTxStatistics(m_DataSocketId);
    }
    void ResetAllTxStatistics()
    {
        m_TxStatsCs.Lock();
        for(auto& ds : m_DataSocketList)
        {
            ds.m_TxStats.ResetCount();
        }
        m_TxStatsCs.Unlock();
    }

    const RxStatistics* GetRxStatistics(const int& id)
    {
        if( IsListIdValid(id) )
        {
            return &m_DataSocketList[id].m_RxStats;
        }
        else
        {
            return nullptr;
        }
    }
    const RxStatistics* GetRxStatistics() { return GetRxStatistics(m_DataSocketId); }

    void ResetRxStatistics(const int& id)
    {
        if( IsListIdValid(id) )
        {
            m_RxStatsCs.Lock();
            m_DataSocketList[id].m_RxStats.ResetCount();
            m_RxStatsCs.Unlock();
        }
    }
    void ResetRxStatistics()
    {
        ResetRxStatistics(m_DataSocketId);
    }
    void ResetAllRxStatistics()
    {
        m_RxStatsCs.Lock();
        for(auto& ds : m_DataSocketList)
        {
            ds.m_RxStats.ResetCount();
        }
        m_RxStatsCs.Unlock();
    }

};


}
