﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>

#include "LcsSocket.h"
#include "Client.h"
#include "Scanner.h"
#include "Util.h"

namespace WlanTest {

/*!--------------------------------------------------------------------------*
  @brief        LCS Clientクラス
 *---------------------------------------------------------------------------*/

class LcsClient : public Client
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    enum State
    {
        State_Disconnected = 0x00,
        State_L2Connected,
        State_L3Completed,
        State_L4Connecting,
        State_L4Connected,
        State_Disconnecting
    };

    bool      m_IsIpAddrAssigned;
    LcsSocket m_Socket;
    State     m_State;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual nn::Result Initialize();
    virtual nn::Result Finalize();
    virtual void MaintainConnection();
    virtual void StopMaintainConnection();
    virtual nn::Result WlanGetFrame( uint32_t rxId, uint8_t pOutput[], size_t* pSize, size_t maxSize, int8_t* pRssi = NULL );

    virtual nn::Result Send(uint8_t data[], size_t size, uint8_t ieInd, size_t* pSentSize);
    virtual nn::Result Receive(uint8_t pOutput[], size_t* pSize, const size_t maxSize);

    void UpdateLastTxStats() { m_Socket.UpdateLastTxStats(); }
    void UpdateLastRxStats() { m_Socket.UpdateLastRxStats(); }

    virtual nn::Result StartReceiveCommand(int32_t priority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread()))
    { return Node::StartReceiveCommand(priority); }
    virtual nn::Result StopReceiveCommand()
    { return Node::StopReceiveCommand(); }

protected:

    void TransitState();
    virtual nn::Result RunAssignSrcIpAddressCommand(const string& ipAddress);

private:

/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    LcsClient();
    virtual ~LcsClient();

private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

    virtual void GetMacAddressListInBss(nn::wlan::MacAddress list[], uint32_t& num)
    {
        if( num >= 1 )
        {
            // ##? wlan プロセスで BSSID のみ初期化されず、前回の BSSID が残っているため state で判断する
            //if( m_ConnectionStatus.bssid != nn::wlan::MacAddress::CreateZeroMacAddress() )
            if( m_ConnectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                list[0] = nn::wlan::MacAddress(m_ConnectionStatus.bssid);
                num = 1;
            }
            else
            {
                num = 0;
            }
        }
        return;
    }


    virtual uint32_t GetConnectableCount()
    {
        return 1;
    }

    virtual uint32_t GetConnectedCount()
    {
        return (IsConnected() ? 1 : 0);
    }

    virtual void ClearStatistics()
    {
        Node::ClearStatistics();
        //m_InformationElementReceiver->GetStatistics().Clear();
    }

    virtual bool IsConnected();

    LcsSocket* GetSocket() { return &m_Socket; }
};


}
