﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <vector>
#include <map>
#include <list>
#include <deque>

#include <nn/os/os_TickTypes.h>

#include "RingBuffer.h"

using namespace std;

namespace WlanTest {

class Counter
{
private:
    int64_t m_Count;

public:
    Counter();

    virtual void Count(int64_t val);
    virtual int64_t GetValue() const;
    virtual void Clear();
};


/*!--------------------------------------------------------------------------*
  @brief        シーケンス番号カウンター

 *---------------------------------------------------------------------------*/
class SequenceNumberCounter
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:
    uint64_t count;
    uint32_t burst[17]; // 0: 最大バーストエラー 1: 1フレーム抜け 2: 2フレーム抜け 16: 16フレーム以上抜け
    uint64_t max;
    uint64_t min;
    uint64_t last;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void Count(uint64_t seq);

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    SequenceNumberCounter();
    virtual ~SequenceNumberCounter();

private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    virtual double GetErrorRate() const;
    virtual double GetRate() const;
    virtual uint64_t GetReceiveCount() const;
    virtual uint64_t GetErrorCount() const;
    virtual uint64_t GetExpectantCount() const;
    virtual uint32_t GetBurstErrorCount(uint8_t index) const;
    virtual void Clear();
};


/*!--------------------------------------------------------------------------*
  @brief        直近エラーカウンター

 *---------------------------------------------------------------------------*/
class LastSequenceNumberCounter : public SequenceNumberCounter
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:
    size_t size;
    SequenceNumberCounter* counters;
    uint32_t index;
    uint64_t last;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void Count(uint64_t seq);

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    NN_IMPLICIT LastSequenceNumberCounter(size_t size = 10);
    virtual ~LastSequenceNumberCounter();

private:
    LastSequenceNumberCounter(const LastSequenceNumberCounter& lastSequenceNumberCounter);
    LastSequenceNumberCounter& operator=(const LastSequenceNumberCounter& lastSequenceNumberCounter);


/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    virtual double GetErrorRate() const;
    virtual double GetRate() const;
    virtual uint64_t GetErrorCount() const;
    virtual uint64_t GetExpectantCount() const;
    virtual void Clear();
    virtual void SetSize(size_t size);
};


/*!--------------------------------------------------------------------------*
  @brief        循環シーケンス番号カウンター

 *---------------------------------------------------------------------------*/
class CyclicSequenceNumberCounter
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:
    bool first;
    uint64_t startNo; // デフォルトでは 0 から始まることを想定
    uint64_t endNo;

    uint64_t count;
    uint64_t lost;
    uint32_t burst[17]; // 0: 最大バーストエラー 1: 1フレーム抜け 2: 2フレーム抜け 16: 16フレーム以上抜け
    uint64_t last;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void Count(uint64_t seq);

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    CyclicSequenceNumberCounter();
    virtual ~CyclicSequenceNumberCounter();

private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    virtual double GetErrorRate() const;
    virtual double GetRate() const;
    virtual uint64_t GetReceivedCount() const;
    virtual uint64_t GetErrorCount() const;
    virtual uint64_t GetExpectantCount() const;
    virtual uint32_t GetBurstErrorCount(uint8_t index) const;

    virtual void SetStartNo(uint64_t no){ startNo = no; }
    virtual void SetEndNo(uint64_t no){ endNo = no; }

    virtual void Clear();
};




/*!--------------------------------------------------------------------------*
  @brief        スループット測定器

 *---------------------------------------------------------------------------*/
class ThroughputMeter
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:
    nn::os::Tick begin;
    nn::os::Tick end;
    uint64_t     totalSize;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void Count(uint64_t size);
    virtual double GetThroughput() const;
    virtual void Clear();

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    ThroughputMeter();
    virtual ~ThroughputMeter();

private:
};


template class RingBuffer<uint64_t>;
template class RingBuffer<nn::os::Tick>;

/*!--------------------------------------------------------------------------*
  @brief        直近スループット測定器

 *---------------------------------------------------------------------------*/
class LastThroughputMeter : public ThroughputMeter
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:
private:
    uint64_t                       size;
    RingBuffer<nn::os::Tick>*      ticks;
    RingBuffer<uint64_t>*          sizes;
    uint64_t totalSize;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void Count(uint64_t bytes);
    virtual double GetThroughput() const;
    virtual void Clear();

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    NN_IMPLICIT LastThroughputMeter(uint64_t size = 1000);
    virtual ~LastThroughputMeter();
    void SetSize(uint32_t size);

private:
};






/*!--------------------------------------------------------------------------*
  @brief        平均器

 *---------------------------------------------------------------------------*/
class Average
{
public:

    static const int64_t MaxValue = 100000000;
    static const int64_t MinValue = -100000000;

    Average();
    virtual ~Average();
    virtual void Count(double value);
    virtual double GetValue() const;
    virtual double GetMinValue() const;
    virtual double GetMaxValue() const;
    virtual void Clear();

protected:
    double total;
    double minValue;
    double maxValue;
    int count;
};

/*!--------------------------------------------------------------------------*
  @brief        直近平均器

 *---------------------------------------------------------------------------*/
class LastAverage : public Average
{
public:
    NN_IMPLICIT LastAverage(int size = 32);
    virtual void Count(double value);
    virtual double GetMinValue() const;
    virtual double GetMaxValue() const;
    void SetSize(uint32_t s);

protected:
    int size;
    list<double> lastValues;
};

/*!--------------------------------------------------------------------------*
  @brief        レイテンシ―測定器

 *---------------------------------------------------------------------------*/
class LatencyCounter
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:

    static const int32_t Lifetime_Infinity;

protected:
private:

    static const uint8_t LastIndex = 101;
    static uint32_t      lifetime;

    int64_t min;
    int64_t max;
    Average  avg;
    // [0] : num of <1ms, [1] : num of <2ms, ... , [9] : num of <10ms, [100] : num of >= 100ms
    int64_t burst[LastIndex];

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:
    virtual void Count(int64_t time);

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
    LatencyCounter();
    virtual ~LatencyCounter();

private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:
    virtual int64_t GetMinLatency() const;
    virtual int64_t GetMaxLatency() const;
    virtual double GetAvgLatency() const;
    virtual int64_t GetBurstLatencyCount(uint8_t index) const;
    virtual int64_t GetBurstLatencyCount(uint8_t from, uint8_t to) const;
    virtual uint8_t GetLastIndex() const;
    virtual void Clear();

    virtual int64_t GetAlivePackets() const;
    virtual int64_t GetDeadPackets() const;

    static uint32_t GetLifetime();
    static void SetLifetime(uint32_t time);
};

/*!--------------------------------------------------------------------------*
  @brief        サンプリング数記録器

 *---------------------------------------------------------------------------*/
class SamplingCountRecorder
{
/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:

    struct TimedCount
    {
        int64_t count;
        nn::os::Tick tick;

        TimedCount(int64_t c, nn::os::Tick t) :
            count(c), tick(t) {}
    };

protected:
private:
    uint16_t recordInterval; // ms
    uint16_t recordCount;
    std::deque<TimedCount> updatedCount;
    int64_t min;
    int64_t max;

    int64_t currentUpdatedCount;
    int64_t lastSequenceNumber;
    nn::os::Tick lastUpdateTime;

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void Clear();
    // シーケンス番号の差から更新回数を割り出して記録する
    void RecordBySequenceNumber(const int64_t& number);
    // サンプリング回数を更新回数として記録する
    void RecordBySamplingCount(const int64_t& count = 1);

    void SetRecordInterval(const uint16_t& interval);
    void SetRecordCount(const uint16_t& count);
    uint16_t GetRecordInterval() const;
    uint16_t GetRecordCount() const;

protected:

    virtual void Record(const int64_t& updatedCount, bool* isRecorded);

private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:

    SamplingCountRecorder();
    virtual ~SamplingCountRecorder();

private:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

    virtual int64_t GetMinCount() const;
    virtual int64_t GetMaxCount() const;
    virtual const std::deque<SamplingCountRecorder::TimedCount>& GetUpdatedCounts() const;

};


}
