﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../common.h"

namespace {

uint8_t  g_txBuffer[1024];  // buffer for tx
uint8_t  g_rxBuffer[2048];  // buffer for rx
const size_t  buffSize = 100 * 1024; // 100KB
NN_ALIGNAS(4096) char  g_scanBuffer[ buffSize ];

}

void Demo()
{
    nn::Result result;

    // Event object for connection status change
    nn::os::SystemEventType connectionEvent;

    // Connection status storage
    nn::wlan::ConnectionStatus connectionStatus;

    // Initializes InfraManager of wlan
    nn::wlan::InitializeInfraManager();

    // Initializes SocketManager of wlan
    nn::wlan::InitializeSocketManager();

    // Requests to open infra mode
    nn::wlan::Infra::OpenMode();
    NN_LOG("OpenMode done\n");

    // Gets current mac address
    nn::wlan::MacAddress macAddr;
    nn::wlan::Infra::GetMacAddress(&macAddr);
    char macStr[nn::wlan::MacAddress::MacStringSize];
    NN_LOG("mac addr : %s\n", macAddr.GetString(macStr));

    // Attaches connectionEvent handle to the systen event of wlan process
    nn::wlan::Infra::GetConnectionEvent(&connectionEvent);
    NN_LOG("GetConnectionEvent\n");


    // SSID of AP
    nn::wlan::Ssid mySsid("INFRA_TEST_AP");

    // Security mode of AP
    nn::wlan::Security security = {
            nn::wlan::SecurityMode_Open,
            nn::wlan::SecurityMode_Open,
            0,
            "",
    };

    // Starts to connect
    nn::wlan::Infra::Connect(mySsid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1, security, false);

    nn::os::WaitSystemEvent(&connectionEvent);
    NN_LOG("Connection event signaled\n");

    // Gets connection status
    nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
    if( connectionStatus.state != nn::wlan::ConnectionState_Connected )
    {
        NN_LOG("Failed to connect AP\n");

        nn::wlan::Infra::CloseMode();
        nn::wlan::FinalizeInfraManager();
        nn::wlan::FinalizeSocketManager();
        return;
    }

    // Tx --------------------------------------------------------------------------------------------------------
    // Generates tx data
    for( int i = 0; i < 1024; i++ )
    {
        g_txBuffer[i] = i & 0xFF;
    }
    // First 6 bytes should be destination address
    std::memcpy(&g_txBuffer[0], connectionStatus.bssid.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
    // Next 6 bytes should be source address
    std::memcpy(&g_txBuffer[6], macAddr.GetMacAddressData(), nn::wlan::MacAddress::MacAddressSize);
    // Next 2 bytes should be ethertype
    g_txBuffer[12] = 0x08;
    g_txBuffer[13] = 0x00;

    // Sned tx data
    result = nn::wlan::Socket::PutFrameRaw(g_txBuffer, 1024);
    NN_ASSERT( result.IsSuccess() );
    NN_LOG("Sent tx data\n");
    // -----------------------------------------------------------------------------------------------------------

    // Rx --------------------------------------------------------------------------------------------------------
    // Creates rx entry
    uint32_t rxId;  // output variable for rxid
    uint16_t ethertypes[] = { 0x0800 };  // Ethertype (IPv4)
    result = nn::wlan::Socket::CreateRxEntry(&rxId, ethertypes, sizeof(ethertypes) / sizeof(uint16_t), 50);
    NN_ASSERT( result.IsSuccess() );
    NN_LOG("Created Rx Entry [%d]\n", rxId);

    size_t rxSize = 0;  // output variable for actual received data size
    NN_LOG("GetFrameRaw\n");
    result = nn::wlan::Socket::GetFrameRaw(g_rxBuffer, 2048, &rxSize, rxId);
    if( result.IsSuccess() )
    {
        NN_LOG("Gets rx data : %d[bytes]\n", rxSize);
    }

    // Deletes rx entry
    nn::wlan::Socket::DeleteRxEntry(rxId);
    // -----------------------------------------------------------------------------------------------------------

    // Disconnects from AP
    nn::wlan::Infra::Disconnect();

    nn::wlan::Infra::CloseMode();
    NN_LOG("CloseMode done\n");

    // Finalizes InfraManager of wlan
    nn::wlan::FinalizeInfraManager();

    // Finalizes SocketManager of wlan
    nn::wlan::FinalizeSocketManager();
}

void ScanOnConnectionDemo()
{
    nn::Result result;
    nn::os::SystemEventType connectionEvent;
    nn::wlan::ConnectionStatus connectionStatus;

    // Initializes InfraManager of wlan
    nn::wlan::InitializeInfraManager();

    // Requests to open infra mode
    nn::wlan::Infra::OpenMode();
    nn::wlan::Infra::GetConnectionEvent(&connectionEvent);

    for( int i = 0; i < 1000000; i++ )
    {
        NN_LOG("Loop count [%d]\n", i);

        nn::wlan::ScanParameters scanParam = {
                nn::wlan::ScanType_Passive,
                {1, 6, 11},
                0,  // all channel scan
                120,
                0,
                NULL,
                0,
                nn::wlan::MacAddress::CreateBroadcastMacAddress()
        };

        NN_LOG("Do scan\n");
        nn::wlan::Infra::StartScan(g_scanBuffer, buffSize, scanParam);
        NN_LOG("Scan finished.\n");
        nn::wlan::BeaconScanResultReader resultReader(g_scanBuffer);
        NN_LOG("Found %d APs\n", resultReader.GetCount());

        // SSID of AP
        nn::wlan::Ssid mySsid("WLAN_TEST_WPA2_AES");

        // Security mode of AP
        nn::wlan::Security security = {
                nn::wlan::SecurityMode_Wpa2Aes,
                nn::wlan::SecurityMode_Wpa2Aes,
                0,
                "012345678",
        };

        while( 1 )
        {
            // Starts to connect
            nn::wlan::Infra::Connect(mySsid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1, security, false);

            nn::os::WaitSystemEvent(&connectionEvent);
            NN_LOG("Connection event signaled\n");

            // Gets connection status
            nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
            WlanTest::PrintConnectionStatus(&connectionStatus);
            if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
            {
                NN_LOG("Connection success\n");
                break;
            }
            else
            {
                NN_LOG("Connection failed. Retry 3 seconds later.\n");
            }

            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(3));
        }

        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(3));

        NN_LOG("Do scan on connection\n");
        nn::wlan::Infra::StartScan(g_scanBuffer, buffSize, scanParam);
        NN_LOG("Scan finished.\n");
        NN_LOG("Found %d APs\n", resultReader.GetCount());

        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(3));

        // Disconnects from AP
        nn::wlan::Infra::Disconnect();
        NN_LOG("Disconnected.\n");
    }

    nn::wlan::Infra::CloseMode();

    // Finalizes InfraManager of wlan
    nn::wlan::FinalizeInfraManager();
}

extern "C" void nnMain()
{
    NN_LOG("\n\n Start SimpleInfraTxRx \n\n");
    WlanTest::SystemInitialize();
    ScanOnConnectionDemo();
    WlanTest::SystemFinalize();
    NN_LOG("\n\n End SimpleInfraTxRx \n\n");
}

