﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneTitle.h"

SceneTitle::SceneTitle(ISceneChanger* changer)
: BaseScene(changer), m_curMenu(Menu_MeasureAllChannels)
{
}

void SceneTitle::Initialize() NN_NOEXCEPT
{
}

void SceneTitle::Finalize() NN_NOEXCEPT
{
}

void SceneTitle::Update() NN_NOEXCEPT
{
    Npad npad = Npad::GetInstance();
    npad.UpdatePadState();

    if( npad.IsTrigger(Npad::PLUS) )
    {
        m_SceneChanger->ChangeScene(eScene_Exit);
    }
    else if (npad.IsTrigger(Npad::UP))
    {
        m_curMenu = (m_curMenu + Menu_End - 1 ) % Menu_End;
    }
    else if (npad.IsTrigger(Npad::DOWN))
    {
        m_curMenu = (m_curMenu + 1) % Menu_End;
    }
    else if (npad.IsTrigger(Npad::A))
    {
        if (m_curMenu == Menu_MeasureAllChannels)
        {
            m_SceneChanger->ChangeScene(eScene_MeasureAllChannels);
        }
        else if(m_curMenu == Menu_LoggingChannel)
        {
            m_SceneChanger->ChangeScene(eScene_LoggingChannel);
        }
    }

    // 時刻取得
    {
        nn::time::PosixTime posixTime;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime) );

        // PosixTime を CalendarTime へ変換します。
        // 計算に利用されるタイムゾーンはデバイスに設定されたものを利用します。
        nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&m_calendarTime, &calendarAdditionalInfo, posixTime) );
    }
}

void SceneTitle::Draw(GraphicTools* pTools) NN_NOEXCEPT
{
    pTools->pPrimitiveRenderer->Update(pTools->bufferIndex);

    // コマンド生成
    pTools->pGraphicsFramework->BeginFrame(pTools->bufferIndex);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = pTools->pGraphicsFramework->GetRootCommandBuffer(pTools->bufferIndex);

        nn::util::Vector3f translate;
        nn::gfx::ColorTargetView* target = pTools->pGraphicsFramework->GetColorTargetView();
        nn::gfx::DepthStencilView* depthStencil = pTools->pGraphicsFramework->GetDepthStencilView();
        rootCommandBuffer->ClearColor(target, 0.f, 0.f, 0.f, 1.0f, NULL);
        rootCommandBuffer->SetRenderTargets(1, &target, depthStencil);
        rootCommandBuffer->SetViewportScissorState(pTools->pGraphicsFramework->GetViewportScissorState());

        // スクリーン左上原点となる画面を設定
        pTools->pPrimitiveRenderer->SetDefaultParameters();
        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::Matrix4x3f modelMatrix;
        nn::util::MatrixIdentity( &viewMatrix );
        nn::util::MatrixIdentity( &projectionMatrix );
        nn::util::MatrixIdentity( &modelMatrix );
        pTools->pPrimitiveRenderer->SetViewMatrix( &viewMatrix );
        pTools->pPrimitiveRenderer->SetProjectionMatrix( &projectionMatrix );
        pTools->pPrimitiveRenderer->SetModelMatrix( &modelMatrix );

        {
            // ツールタイトル
            pTools->pWriter->SetFontSize(40.f);
            pTools->pWriter->SetLineHeight(44.f);
            pTools->pWriter->SetTextColor(nn::util::Color4u8::White());
            pTools->pWriter->SetCursor(pTools->pWriter->GetFontWidth(), 0);
            pTools->pWriter->Print("Measure Channel Tool");
            // 仕切り腺
            pTools->pPrimitiveRenderer->SetLineWidth(1.f);
            pTools->pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());
            pTools->pPrimitiveRenderer->Draw2DLine(rootCommandBuffer,
                    0.f, pTools->pWriter->GetLineHeight(),
                    pTools->pGraphicsFramework->GetDisplayWidth(), pTools->pWriter->GetLineHeight());
        }

        {
            float menuY = pTools->pWriter->GetLineHeight() * 2.f;
            // MeasureAllChannels
            {
                pTools->pWriter->SetTextColor(nn::util::Color4u8::White());
                pTools->pWriter->SetCursor(
                        pTools->pWriter->GetFontWidth() * 2.f,
                        menuY);
                pTools->pWriter->Print("Measure All Channels");
            }
            // LoggingChannel
            {
                pTools->pWriter->SetTextColor(nn::util::Color4u8::White());
                pTools->pWriter->SetCursor(
                        pTools->pWriter->GetFontWidth() * 2.f,
                        menuY + pTools->pWriter->GetLineHeight());
                pTools->pWriter->Print("Logging Channel");
            }

            // Cursor
            {
                pTools->pWriter->SetTextColor(nn::util::Color4u8::White());
                pTools->pWriter->SetCursor(
                        pTools->pWriter->GetFontWidth(),
                        menuY + m_curMenu * pTools->pWriter->GetLineHeight());
                pTools->pWriter->Print(">");
            }
        }

        {
            // 操作
            pTools->pWriter->SetFontSize(20.f);
            pTools->pWriter->SetTextColor(nn::util::Color4u8::White());
            pTools->pWriter->SetCursor(10, 700);
            pTools->pWriter->Print("[+]:Exit");
        }

        // Time
        {
            pTools->pWriter->SetFontSize(20.f);
            pTools->pWriter->SetCursor(1090, 700);
            pTools->pWriter->Print("%04d/%02d/%02d %02d:%02d:%02d\n",
                m_calendarTime.year, m_calendarTime.month,
                m_calendarTime.day, m_calendarTime.hour,
                m_calendarTime.minute, m_calendarTime.second);
        }

        // デバッグフォント用のコマンド生成
        pTools->pWriter->Draw(rootCommandBuffer);
    }
    pTools->pGraphicsFramework->EndFrame(pTools->bufferIndex);
}
