﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_LocalApi.h>
#include <nn/wlan/wlan_Result.h>

#include "SceneMeasureAllCh.h"

bool SceneMeasureAllCh::m_exitFlag;
nn::wlan::ScanParameters SceneMeasureAllCh::m_scanParam;
int SceneMeasureAllCh::m_scanInterval;
nn::wlan::ChannelStats SceneMeasureAllCh::m_stats[nn::wlan::WirelessChannelsCountMax + 1];
SceneMeasureAllCh::DisplayStats SceneMeasureAllCh::m_dispStats[nn::wlan::WirelessChannelsCountMax + 1];
nn::os::MutexType SceneMeasureAllCh::m_mutex;
int SceneMeasureAllCh::m_curPage;
NN_OS_ALIGNAS_THREAD_STACK char g_MeasureAllChThreadStack[ThreadStackSize];

const int16_t ChList2G[] = {1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13};
const int16_t ChListW52W53[] = {36, 40, 44, 48, 52, 56, 60, 64};

SceneMeasureAllCh::SceneMeasureAllCh(ISceneChanger* changer)
: BaseScene(changer)
{
}

void SceneMeasureAllCh::Initialize() NN_NOEXCEPT
{
    m_curMenu = Menu_ScanTime;
    m_curPage = Page_2gNoise;
    m_exitFlag = false;
    for( int i = 0; i < sizeof(m_stats) / sizeof(m_stats[0]); i++ )
    {
        memset(&m_stats[i], 0, sizeof(nn::wlan::ChannelStats));
        memset(&m_dispStats[i], 0, sizeof(DisplayStats));
    }
    m_scanParam.scanType = nn::wlan::ScanType_Passive;
    m_scanParam.channelCount = 0;
    m_scanParam.channelScanTime = 120;
    m_scanParam.homeChannelTime = 0;
    m_scanParam.ssidList = NULL;
    m_scanParam.ssidCount = 0;
    m_scanParam.bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();
    m_scanInterval = 1000;

    nn::os::InitializeMutex(&m_mutex, false, 0);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread(
            &m_thread, MeasureThreadFunc, NULL, g_MeasureAllChThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority)
    );
    nn::os::StartThread( &m_thread );
}

void SceneMeasureAllCh::Finalize() NN_NOEXCEPT
{
    m_exitFlag = true;
    nn::os::WaitThread(&m_thread);
    nn::os::DestroyThread(&m_thread);
    nn::os::FinalizeMutex(&m_mutex);
}

void SceneMeasureAllCh::Update() NN_NOEXCEPT
{
    Npad npad = Npad::GetInstance();
    npad.UpdatePadState();

    if( npad.IsTrigger(Npad::PLUS) )
    {
        m_SceneChanger->ChangeScene(eScene_Title);
    }
    else if (npad.IsTrigger(Npad::UP))
    {
        m_curMenu = (m_curMenu + Menu_End - 1 ) % Menu_End;
    }
    else if (npad.IsTrigger(Npad::DOWN))
    {
        m_curMenu = (m_curMenu + 1) % Menu_End;
    }
    else if (npad.IsTrigger(Npad::LEFT) || npad.IsHold(Npad::LEFT))
    {
        if (m_curMenu == Menu_ScanTime)
        {
            m_scanParam.channelScanTime -= 10;
            if (m_scanParam.channelScanTime < 10)
            {
                m_scanParam.channelScanTime = 10;
            }
        }
        else if (m_curMenu == Menu_ScanInterval)
        {
            m_scanInterval -= 100;
            if (m_scanInterval < 100)
            {
                m_scanInterval = 100;
            }
        }
    }
    else if (npad.IsTrigger(Npad::RIGHT) || npad.IsHold(Npad::RIGHT))
    {
        if (m_curMenu == Menu_ScanTime)
        {
            m_scanParam.channelScanTime += 10;
            if (m_scanParam.channelScanTime > 500)
            {
                m_scanParam.channelScanTime = 500;
            }
        }
        else if (m_curMenu == Menu_ScanInterval)
        {
            m_scanInterval += 100;
            if (m_scanInterval > 10000)
            {
                m_scanInterval = 10000;
            }
        }
    }
    // ページの切り替え
    else if (npad.IsTrigger(Npad::L))
    {
        if (m_curPage > Page_2gNoise && m_curPage <= Page_W52W53Txop)
        {
            m_curPage--;
        }
    }
    else if (npad.IsTrigger(Npad::R))
    {
        if (m_curPage < Page_W52W53Txop && m_curPage >= Page_2gNoise)
        {
            m_curPage++;
        }
    }
    // 平均値リセット
    else if (npad.IsTrigger(Npad::B))
    {
        nn::os::LockMutex(&m_mutex);
        for( int i = 0; i < sizeof(m_stats) / sizeof(m_stats[0]); i++ )
        {
            memset(&m_dispStats[i], 0, sizeof(DisplayStats));
        }
        nn::os::UnlockMutex(&m_mutex);
    }

    // 時刻取得
    {
        nn::time::PosixTime posixTime;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime) );

        // PosixTime を CalendarTime へ変換します。
        // 計算に利用されるタイムゾーンはデバイスに設定されたものを利用します。
        nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&m_calendarTime, &calendarAdditionalInfo, posixTime) );
    }
}

void SceneMeasureAllCh::Draw(GraphicTools* pTools) NN_NOEXCEPT
{
    pTools->pPrimitiveRenderer->Update(pTools->bufferIndex);

    // コマンド生成
    pTools->pGraphicsFramework->BeginFrame(pTools->bufferIndex);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = pTools->pGraphicsFramework->GetRootCommandBuffer(pTools->bufferIndex);

        nn::util::Vector3f translate;
        nn::gfx::ColorTargetView* target = pTools->pGraphicsFramework->GetColorTargetView();
        nn::gfx::DepthStencilView* depthStencil = pTools->pGraphicsFramework->GetDepthStencilView();
        rootCommandBuffer->ClearColor(target, 0.f, 0.f, 0.f, 1.0f, NULL);
        rootCommandBuffer->SetRenderTargets(1, &target, depthStencil);
        rootCommandBuffer->SetViewportScissorState(pTools->pGraphicsFramework->GetViewportScissorState());

        // スクリーン左上原点となる画面を設定
        pTools->pPrimitiveRenderer->SetDefaultParameters();
        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::Matrix4x3f modelMatrix;
        nn::util::MatrixIdentity( &viewMatrix );
        nn::util::MatrixIdentity( &projectionMatrix );
        nn::util::MatrixIdentity( &modelMatrix );
        pTools->pPrimitiveRenderer->SetViewMatrix( &viewMatrix );
        pTools->pPrimitiveRenderer->SetProjectionMatrix( &projectionMatrix );
        pTools->pPrimitiveRenderer->SetModelMatrix( &modelMatrix );

        {
            // 軸を描画
            float endX = pTools->pGraphicsFramework->GetDisplayWidth();
            pTools->pPrimitiveRenderer->SetLineWidth(1.f);
            pTools->pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());
            // 上軸
            pTools->pPrimitiveRenderer->Draw2DLine(rootCommandBuffer, 100.f, 39.f, endX - 100.f, 39.f);
            // 下軸
            pTools->pPrimitiveRenderer->Draw2DLine(rootCommandBuffer, 100.f, 541.f, endX - 100.f, 541.f);

            switch (m_curPage)
            {
            case Page_2gNoise:
                Draw2gNoise(pTools);
                break;
            case Page_2gTxop:
                Draw2gTxop(pTools);
                break;
            case Page_W52W53Noise:
                DrawW52W53Noise(pTools);
                break;
            case Page_W52W53Txop:
                DrawW52W53Txop(pTools);
                break;
            default:
                Draw2gNoise(pTools);
                break;
            }
        }

        {
            float menuY = 640.f;
            pTools->pWriter->SetFontSize(20.f);
            pTools->pWriter->SetLineHeight(24.f);
            {
                pTools->pWriter->SetTextColor(nn::util::Color4u8::White());
                pTools->pWriter->SetCursor(
                        pTools->pWriter->GetFontWidth() * 2.f,
                        menuY);
                pTools->pWriter->Print("Scan Time:%d[ms]", m_scanParam.channelScanTime);
            }
            {
                pTools->pWriter->SetTextColor(nn::util::Color4u8::White());
                pTools->pWriter->SetCursor(
                        pTools->pWriter->GetFontWidth() * 2.f,
                        menuY + pTools->pWriter->GetLineHeight());
                pTools->pWriter->Print("Scan Interval:%d[ms]", m_scanInterval);
            }

            // Cursor
            {
                pTools->pWriter->SetTextColor(nn::util::Color4u8::White());
                pTools->pWriter->SetCursor(
                        pTools->pWriter->GetFontWidth(),
                        menuY + m_curMenu * pTools->pWriter->GetLineHeight());
                pTools->pWriter->Print(">");
            }
        }

        // Time
        {
            pTools->pWriter->SetCursor(1090, 700);
            pTools->pWriter->Print("%04d/%02d/%02d %02d:%02d:%02d\n",
                m_calendarTime.year, m_calendarTime.month,
                m_calendarTime.day, m_calendarTime.hour,
                m_calendarTime.minute, m_calendarTime.second);
        }

        // デバッグフォント用のコマンド生成
        pTools->pWriter->Draw(rootCommandBuffer);
    }
    pTools->pGraphicsFramework->EndFrame(pTools->bufferIndex);
}

void SceneMeasureAllCh::Draw2gNoise(GraphicTools* pTools) NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pGraphicsFramework = pTools->pGraphicsFramework;
    int bufferIndex = pTools->bufferIndex;
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = pTools->pPrimitiveRenderer;
    nn::gfx::util::DebugFontTextWriter* pWriter = pTools->pWriter;
    nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

    // ラベル
    pTools->pWriter->SetFontSize(20.f);
    pWriter->SetTextColor(nn::util::Color4u8::White());
    pWriter->SetCursor(10, 0);
    pWriter->Print("2.4GHz Noise Level");
    pWriter->SetCursor(1000, 0);
    pWriter->Print("[R]:2.4GHz TXOP");

    pWriter->SetCursor(22, 28);
    pWriter->Print("0 dBm");
    pWriter->SetCursor(22, 530);
    pWriter->Print("-99 dBm");

    // メーター
    nn::util::Color4u8 mColor(0xff, 0xff, 0xff, 0xff);
    pPrimitiveRenderer->SetLineWidth(1.f);
    float meterW = 63.f;
    float meterH = 5.f;
    float meterX = 105.f;
    float meterY = 535.f;
    for( int j = 0; j < 13; j++ )
    {
        pWriter->SetCursor(110 + (84 * j), 550);
        pWriter->Print("%2d ch", m_stats[j].channel);
        pWriter->SetCursor(110 + (84 * j), 570);
        pWriter->Print("%4d", m_stats[j].noiseLevel);
        pWriter->SetCursor(110 + (84 * j), 590);
        pWriter->Print("%3d", m_stats[j].txop);
        int id = ConvertChannelToId(m_stats[j].channel);
        for( int i = 0; i < m_dispStats[id].currentMeterNoise; i++ )
        {
            int colorR = i * i;
            if( colorR > 0xff )
            {
                colorR = 0xff;
            }
            mColor.Set(static_cast<uint8_t>(colorR), static_cast<uint8_t>(0xdd - i * 2), 0, 0xff);
            pPrimitiveRenderer->SetColor(mColor);
            pPrimitiveRenderer->Draw2DRect(rootCommandBuffer,
                    meterX + 1.f + ((meterW + 20.f + 1.f) * j),
                    meterY + 1.f - (5.f * i),
                    meterW - 2.f,
                    meterH - 2.f
                    );
        }
        if( m_dispStats[id].currentMeterNoise == 100 + m_stats[j].noiseLevel )
        {
            m_dispStats[id].currentMeterNoise = 100 + m_stats[j].noiseLevel;
        }
        else if( m_dispStats[id].currentMeterNoise > 100 + m_stats[j].noiseLevel )
        {
            m_dispStats[id].currentMeterNoise--;
        }
        else
        {
            m_dispStats[id].currentMeterNoise++;
        }

        // 平均値を描画
        uint64_t avg = m_dispStats[id].noiseSum / m_dispStats[id].count;
        pPrimitiveRenderer->SetColor(nn::util::Color4u8::Cyan());
        pPrimitiveRenderer->Draw2DRect(rootCommandBuffer,
                meterX + 1.f + ((meterW + 20.f + 1.f) * j),
                meterY + 1.f - (5.f * (100 - avg)),
                meterW - 2.f,
                meterH - 2.f
                );
    }
}

void SceneMeasureAllCh::Draw2gTxop(GraphicTools* pTools) NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pGraphicsFramework = pTools->pGraphicsFramework;
    int bufferIndex = pTools->bufferIndex;
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = pTools->pPrimitiveRenderer;
    nn::gfx::util::DebugFontTextWriter* pWriter = pTools->pWriter;
    nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

    // ラベル
    pTools->pWriter->SetFontSize(20.f);
    pWriter->SetTextColor(nn::util::Color4u8::White());
    pWriter->SetCursor(10, 0);
    pWriter->Print("2.4GHz TXOP");
    pWriter->SetCursor(700, 0);
    pWriter->Print("[L]:2.4GHz Noise Level [R]:W52 W53 Noise Level");

    pWriter->SetCursor(33, 28);
    pWriter->Print("0");
    pWriter->SetCursor(33, 530);
    pWriter->Print("100");

    // メーター
    nn::util::Color4u8 mColor(0xff, 0xff, 0xff, 0xff);
    pPrimitiveRenderer->SetLineWidth(1.f);
    float meterW = 63.f;
    float meterH = 5.f;
    float meterX = 105.f;
    float meterY = 535.f;
    for( int j = 0; j < 13; j++ )
    {
        pWriter->SetCursor(110 + (84 * j), 550);
        pWriter->Print("%2d ch", m_stats[j].channel);
        pWriter->SetCursor(110 + (84 * j), 570);
        pWriter->Print("%4d", m_stats[j].noiseLevel);
        pWriter->SetCursor(110 + (84 * j), 590);
        pWriter->Print("%3d", m_stats[j].txop);
        int id = ConvertChannelToId(m_stats[j].channel);
        for( int i = 0; i < m_dispStats[id].currentMeterTxop; i++ )
        {
            int colorR = i * i;
            if( colorR > 0xff )
            {
                colorR = 0xff;
            }
            mColor.Set(static_cast<uint8_t>(colorR), static_cast<uint8_t>(0xdd - i * 2), 0, 0xff);
            pPrimitiveRenderer->SetColor(mColor);
            pPrimitiveRenderer->Draw2DRect(rootCommandBuffer,
                    meterX + 1.f + ((meterW + 20.f + 1.f) * j),
                    meterY + 1.f - (5.f * i),
                    meterW - 2.f,
                    meterH - 2.f
                    );
        }
        if( m_dispStats[id].currentMeterTxop == 100 - m_stats[j].txop )
        {
            m_dispStats[id].currentMeterTxop = 100 - m_stats[j].txop;
        }
        else if( m_dispStats[id].currentMeterTxop > 100 - m_stats[j].txop )
        {
            m_dispStats[id].currentMeterTxop--;
        }
        else
        {
            m_dispStats[id].currentMeterTxop++;
        }

        // 平均値を描画
        uint64_t avg = m_dispStats[id].txopSum / m_dispStats[id].count;
        pPrimitiveRenderer->SetColor(nn::util::Color4u8::Cyan());
        pPrimitiveRenderer->Draw2DRect(rootCommandBuffer,
                meterX + 1.f + ((meterW + 20.f + 1.f) * j),
                meterY + 1.f - (5.f * (100 - avg)),
                meterW - 2.f,
                meterH - 2.f
                );
    }
}

void SceneMeasureAllCh::DrawW52W53Noise(GraphicTools* pTools) NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pGraphicsFramework = pTools->pGraphicsFramework;
    int bufferIndex = pTools->bufferIndex;
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = pTools->pPrimitiveRenderer;
    nn::gfx::util::DebugFontTextWriter* pWriter = pTools->pWriter;
    nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

    // ラベル
    pTools->pWriter->SetFontSize(20.f);
    pWriter->SetTextColor(nn::util::Color4u8::White());
    pWriter->SetCursor(10, 0);
    pWriter->Print("W52 W53 Noise Level");
    pWriter->SetCursor(700, 0);
    pWriter->Print("[L]:2.4GHz TXOP [R]:W52 W53 TXOP");

    pWriter->SetCursor(22, 28);
    pWriter->Print("0 dBm");
    pWriter->SetCursor(22, 530);
    pWriter->Print("-99 dBm");

    // メーター
    nn::util::Color4u8 mColor(0xff, 0xff, 0xff, 0xff);
    pPrimitiveRenderer->SetLineWidth(1.f);
    float meterW = 63.f;
    float meterH = 5.f;
    float meterX = 105.f;
    float meterY = 535.f;
    for( int j = 0, k = ConvertChannelToId(36); k < ConvertChannelToId(64) + 1; j++, k++ )
    {
        pWriter->SetCursor(110 + (84 * j), 550);
        pWriter->Print("%2d ch", m_stats[k].channel);
        pWriter->SetCursor(110 + (84 * j), 570);
        pWriter->Print("%4d", m_stats[k].noiseLevel);
        pWriter->SetCursor(110 + (84 * j), 590);
        pWriter->Print("%3d", m_stats[k].txop);
        int id = ConvertChannelToId(m_stats[k].channel);
        for( int i = 0; i < m_dispStats[id].currentMeterNoise; i++ )
        {
            int colorR = i * i;
            if( colorR > 0xff )
            {
                colorR = 0xff;
            }
            mColor.Set(static_cast<uint8_t>(colorR), static_cast<uint8_t>(0xdd - i * 2), 0, 0xff);
            pPrimitiveRenderer->SetColor(mColor);
            pPrimitiveRenderer->Draw2DRect(rootCommandBuffer,
                    meterX + 1.f + ((meterW + 20.f + 1.f) * j),
                    meterY + 1.f - (5.f * i),
                    meterW - 2.f,
                    meterH - 2.f
                    );
        }
        if( m_dispStats[id].currentMeterNoise == 100 + m_stats[k].noiseLevel )
        {
            m_dispStats[id].currentMeterNoise = 100 + m_stats[k].noiseLevel;
        }
        else if( m_dispStats[id].currentMeterNoise > 100 + m_stats[k].noiseLevel )
        {
            m_dispStats[id].currentMeterNoise--;
        }
        else
        {
            m_dispStats[id].currentMeterNoise++;
        }

        // 平均値を描画
        uint64_t avg = m_dispStats[id].noiseSum / m_dispStats[id].count;
        pPrimitiveRenderer->SetColor(nn::util::Color4u8::Cyan());
        pPrimitiveRenderer->Draw2DRect(rootCommandBuffer,
                meterX + 1.f + ((meterW + 20.f + 1.f) * j),
                meterY + 1.f - (5.f * (100 - avg)),
                meterW - 2.f,
                meterH - 2.f
                );
    }
}

void SceneMeasureAllCh::DrawW52W53Txop(GraphicTools* pTools) NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pGraphicsFramework = pTools->pGraphicsFramework;
    int bufferIndex = pTools->bufferIndex;
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = pTools->pPrimitiveRenderer;
    nn::gfx::util::DebugFontTextWriter* pWriter = pTools->pWriter;
    nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

    // ラベル
    pTools->pWriter->SetFontSize(20.f);
    pWriter->SetTextColor(nn::util::Color4u8::White());
    pWriter->SetCursor(10, 0);
    pWriter->Print("W52 W53 TXOP");
    pWriter->SetCursor(900, 0);
    pWriter->Print("[L]:W52 W53 Noise Level");

    pWriter->SetCursor(33, 28);
    pWriter->Print("0");
    pWriter->SetCursor(33, 530);
    pWriter->Print("100");

    // メーター
    nn::util::Color4u8 mColor(0xff, 0xff, 0xff, 0xff);
    pPrimitiveRenderer->SetLineWidth(1.f);
    float meterW = 63.f;
    float meterH = 5.f;
    float meterX = 105.f;
    float meterY = 535.f;
    for( int j = 0, k = ConvertChannelToId(36); k < ConvertChannelToId(64) + 1; j++, k++ )
    {
        pWriter->SetCursor(110 + (84 * j), 550);
        pWriter->Print("%2d ch", m_stats[k].channel);
        pWriter->SetCursor(110 + (84 * j), 570);
        pWriter->Print("%4d", m_stats[k].noiseLevel);
        pWriter->SetCursor(110 + (84 * j), 590);
        pWriter->Print("%3d", m_stats[k].txop);
        int id = ConvertChannelToId(m_stats[k].channel);
        for( int i = 0; i < m_dispStats[id].currentMeterTxop; i++ )
        {
            int colorR = i * i;
            if( colorR > 0xff )
            {
                colorR = 0xff;
            }
            mColor.Set(static_cast<uint8_t>(colorR), static_cast<uint8_t>(0xdd - i * 2), 0, 0xff);
            pPrimitiveRenderer->SetColor(mColor);
            pPrimitiveRenderer->Draw2DRect(rootCommandBuffer,
                    meterX + 1.f + ((meterW + 20.f + 1.f) * j),
                    meterY + 1.f - (5.f * i),
                    meterW - 2.f,
                    meterH - 2.f
                    );
        }
        if( m_dispStats[id].currentMeterTxop == 100 - m_stats[k].txop )
        {
            m_dispStats[id].currentMeterTxop = 100 - m_stats[k].txop;
        }
        else if( m_dispStats[id].currentMeterTxop > 100 - m_stats[k].txop )
        {
            m_dispStats[id].currentMeterTxop--;
        }
        else
        {
            m_dispStats[id].currentMeterTxop++;
        }

        // 平均値を描画
        uint64_t avg = m_dispStats[id].txopSum / m_dispStats[id].count;
        pPrimitiveRenderer->SetColor(nn::util::Color4u8::Cyan());
        pPrimitiveRenderer->Draw2DRect(rootCommandBuffer,
                meterX + 1.f + ((meterW + 20.f + 1.f) * j),
                meterY + 1.f - (5.f * (100 - avg)),
                meterW - 2.f,
                meterH - 2.f
                );
    }
}

void SceneMeasureAllCh::MeasureThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    while( m_exitFlag != true )
    {
        if (m_curPage == Page_2gNoise || m_curPage == Page_2gTxop)
        {
            memcpy(m_scanParam.channelList, ChList2G, sizeof(ChList2G));
            m_scanParam.channelCount = sizeof(ChList2G) / sizeof(ChList2G[0]);
        }
        else
        {
            memcpy(m_scanParam.channelList, ChListW52W53, sizeof(ChListW52W53));
            m_scanParam.channelCount = sizeof(ChListW52W53) / sizeof(ChListW52W53[0]);
        }
        nn::os::Tick tick = nn::os::GetSystemTick();
        uint32_t count;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::GetChannelStats(m_stats, &count, m_scanParam));
        nn::os::LockMutex(&m_mutex);
        for( int i = 0; i < count; i++ )
        {
            m_dispStats[i].noiseSum -= m_stats[i].noiseLevel;
            m_dispStats[i].txopSum += m_stats[i].txop;
            m_dispStats[i].count++;
        }
        nn::os::UnlockMutex(&m_mutex);
        int64_t elapsed = (nn::os::GetSystemTick() - tick).ToTimeSpan().GetMilliSeconds();
        if(elapsed < m_scanInterval)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(m_scanInterval - elapsed));
        }
    }
}
