﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_LocalApi.h>
#include <nn/wlan/wlan_Result.h>

#include "SceneLoggingCh.h"

bool SceneLoggingCh::m_measureExitFlag;
SceneLoggingCh::loggingInfo SceneLoggingCh::m_logInfo;
nn::wlan::ChannelStats SceneLoggingCh::m_stats[nn::wlan::WirelessChannelsCountMax + 1];
nn::os::MutexType SceneLoggingCh::m_mutex;
SceneLoggingCh::unitTimeStats* SceneLoggingCh::m_pLastUTS;
NN_OS_ALIGNAS_THREAD_STACK char g_measureThreadStack[ThreadStackSize];
FsSdCard SceneLoggingCh::m_SdWriter;
bool SceneLoggingCh::m_saveExitFlag;
NN_OS_ALIGNAS_THREAD_STACK char g_saveThreadStack[ThreadStackSize];
SceneLoggingCh::unitTimeStats* SceneLoggingCh::m_pSaveUTS;
char SceneLoggingCh::m_saveStr[30 * 1024];

SceneLoggingCh::SceneLoggingCh(ISceneChanger* changer)
: BaseScene(changer)
{
}

void SceneLoggingCh::Initialize() NN_NOEXCEPT
{
    m_curMenu = Menu_Channel;
    m_curPage = Page_Setting;
    m_curSubMenu = SubMenu_MA1min;
    m_curChId = 0;
    m_measureExitFlag = false;
    m_saveExitFlag = false;
    for( int i = 0; i < sizeof(m_stats) / sizeof(m_stats[0]); i++ )
    {
        memset(&m_stats[i], 0, sizeof(nn::wlan::ChannelStats));
    }
    m_logInfo.scanParam.scanType = nn::wlan::ScanType_Passive;
    m_logInfo.scanParam.channelCount = 0;
    m_logInfo.scanParam.channelScanTime = 120;
    m_logInfo.scanParam.homeChannelTime = 0;
    m_logInfo.scanParam.ssidList = NULL;
    m_logInfo.scanParam.ssidCount = 0;
    m_logInfo.scanParam.bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();
    m_logInfo.measureInterval = 5000;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::GetAllowedChannels(m_allowedChannels, &m_allowedChannelCount));
    m_logInfo.scanParam.channelList[0] = m_allowedChannels[0];

    nn::os::InitializeMutex(&m_mutex, false, 0);
    m_IsLastUTSPointed = true;
    m_IsPointedUTSDisplayed = false;
    for( int i = 0; i < MAWindow_End; i++ )
    {
        m_IsMaDisplayed[i] = false;
    }
    m_IsSdInserted = false;
    m_IsSubMenuOpened = false;
}

void SceneLoggingCh::Finalize() NN_NOEXCEPT
{
    nn::os::FinalizeMutex(&m_mutex);
}

void SceneLoggingCh::Update() NN_NOEXCEPT
{
    Npad npad = Npad::GetInstance();
    npad.UpdatePadState();

    switch (m_curPage)
    {
    case Page_Setting:
        if( npad.IsTrigger(Npad::PLUS) )
        {
            m_SceneChanger->ChangeScene(eScene_Title);
        }
        else if( npad.IsTrigger(Npad::A) )
        {
            StartMeasure();
            m_curPage = Page_Noise;
        }
        else if (npad.IsTrigger(Npad::UP))
        {
            m_curMenu = (m_curMenu + Menu_End - 1 ) % Menu_End;
        }
        else if (npad.IsTrigger(Npad::DOWN))
        {
            m_curMenu = (m_curMenu + 1) % Menu_End;
        }
        else if (npad.IsTrigger(Npad::LEFT) || npad.IsHold(Npad::LEFT))
        {
            if (m_curMenu == Menu_Channel)
            {
                if (m_curChId > 0 && m_curChId <= m_allowedChannelCount - 1)
                {
                    m_logInfo.scanParam.channelList[0] = m_allowedChannels[--m_curChId];
                }
            }
            else if (m_curMenu == Menu_ScanTime)
            {
                m_logInfo.scanParam.channelScanTime -= 10;
                if (m_logInfo.scanParam.channelScanTime < 10)
                {
                    m_logInfo.scanParam.channelScanTime = 10;
                }
            }
            else if (m_curMenu == Menu_MeasureInterval)
            {
                m_logInfo.measureInterval -= 100;
                if (m_logInfo.measureInterval < 100)
                {
                    m_logInfo.measureInterval = 100;
                }
            }
        }
        else if (npad.IsTrigger(Npad::RIGHT) || npad.IsHold(Npad::RIGHT))
        {
            if (m_curMenu == Menu_Channel)
            {
                if (m_curChId >= 0 && m_curChId < m_allowedChannelCount - 1)
                {
                    m_logInfo.scanParam.channelList[0] = m_allowedChannels[++m_curChId];
                }
            }
            else if (m_curMenu == Menu_ScanTime)
            {
                m_logInfo.scanParam.channelScanTime += 10;
                if (m_logInfo.scanParam.channelScanTime > 500)
                {
                    m_logInfo.scanParam.channelScanTime = 500;
                }
            }
            else if (m_curMenu == Menu_MeasureInterval)
            {
                m_logInfo.measureInterval += 100;
                if (m_logInfo.measureInterval > 10000)
                {
                    m_logInfo.measureInterval = 10000;
                }
            }
        }
        break;
    case Page_Noise:
    case Page_Txop:
        if( m_IsSubMenuOpened == true )
        {
            if (npad.IsTrigger(Npad::PLUS))
            {
                StopMeasure();
                m_curPage = Page_Setting;
            }
            else if (npad.IsTrigger(Npad::X))
            {
                // 閉じる
                m_IsSubMenuOpened = false;
            }
            else if (npad.IsTrigger(Npad::UP))
            {
                m_curSubMenu = (m_curSubMenu + SubMenu_End - 1 ) % SubMenu_End;
            }
            else if (npad.IsTrigger(Npad::DOWN))
            {
                m_curSubMenu = (m_curSubMenu + 1 ) % SubMenu_End;
            }
            else if (npad.IsTrigger(Npad::LEFT))
            {
            }
            else if (npad.IsTrigger(Npad::RIGHT))
            {
            }
            else if (npad.IsTrigger(Npad::A))
            {
                switch (m_curSubMenu)
                {
                case SubMenu_MA1min:
                    m_IsMaDisplayed[MAWindow_1min] ^= 1;
                    break;
                case SubMenu_MA5min:
                    m_IsMaDisplayed[MAWindow_5min] ^= 1;
                    break;
                case SubMenu_MA10min:
                    m_IsMaDisplayed[MAWindow_10min] ^= 1;
                    break;
                case SubMenu_MA25min:
                    m_IsMaDisplayed[MAWindow_25min] ^= 1;
                    break;
                case SubMenu_TimeScale:
                    break;
                default:
                    break;
                }
            }
        }
        else
        {
            if( m_IsPointedUTSDisplayed == false )
            {
                // まだ詳細表示がONになっていない場合は、詳細表示ノードは描画最新ノードを指すようにしておく
                m_pPointedUTS = m_pDispLastUTS;
            }
            if( npad.IsTrigger(Npad::PLUS) )
            {
                StopMeasure();
                m_curPage = Page_Setting;
            }
            else if (npad.IsTrigger(Npad::LEFT) || npad.IsHold(Npad::LEFT))
            {
                m_IsPointedUTSDisplayed = true;
                if( m_IsLastUTSPointed == true && m_pPointedUTS->id < m_pDispFirstUTS->id )
                {
                    // 最新ノード追従状態だが、既に画面外のノードを指してしまっている場合（放置状態でなりうる）は、
                    // 画面表示用先頭ノードに移しておく
                    m_pPointedUTS->id = m_pDispFirstUTS->id;
                }
                // チャート移動
                if (npad.IsOn(Npad::Y))
                {
                    // Yとの同時押しなら移動量10倍
                    for( int i = 0; i < 10; i++ )
                    {
                        if( m_pPointedUTS != m_pTopUTS )
                        {
                            m_pPointedUTS = m_pPointedUTS->prev;
                        }
                        else
                        {
                            break;
                        }
                    }
                }
                else
                {
                    if( m_pPointedUTS != m_pTopUTS )
                    {
                        m_pPointedUTS = m_pPointedUTS->prev;
                    }
                }
                if( m_IsLastUTSPointed == true && m_pDispLastUTS->id > POINTABLE_COUNTS )
                {
                    if( m_pPointedUTS->id <= m_pDispLastUTS->id - POINTABLE_COUNTS )
                    {
                        // 最新ノード追従状態から外れる場合
                        m_IsLastUTSPointed = false;
                    }
                }
            }
            else if (npad.IsTrigger(Npad::RIGHT) || npad.IsHold(Npad::RIGHT))
            {
                m_IsPointedUTSDisplayed = true;
                if( m_IsLastUTSPointed == true && m_pPointedUTS->id < m_pDispFirstUTS->id )
                {
                    // 最新ノード追従状態だが、既に画面外のノードを指してしまっている場合（放置状態でなりうる）は、
                    // 画面表示用先頭ノードに移しておく
                    m_pPointedUTS->id = m_pDispFirstUTS->id;
                }
                // チャート移動
                if (npad.IsOn(Npad::Y))
                {
                    // Yとの同時押しなら移動量10倍
                    for( int i = 0; i < 10; i++ )
                    {
                        if( m_pPointedUTS != m_pLastUTS )
                        {
                            m_pPointedUTS = m_pPointedUTS->next;
                        }
                        else
                        {
                            break;
                        }
                    }
                }
                else
                {
                    if( m_pPointedUTS != m_pLastUTS )
                    {
                        m_pPointedUTS = m_pPointedUTS->next;
                    }
                }
                if( m_pPointedUTS == m_pLastUTS )
                {
                    // 最新ノードに表示を追従させる
                    m_IsLastUTSPointed = true;
                }
                else if( m_IsLastUTSPointed == true && m_pDispLastUTS->id > POINTABLE_COUNTS )
                {
                    if( m_pPointedUTS->id <= m_pDispLastUTS->id - POINTABLE_COUNTS )
                    {
                        // 最新ノード追従状態から外れる場合
                        m_IsLastUTSPointed = false;
                    }
                }
            }
            else if (npad.IsTrigger(Npad::R))
            {
                // ページ切り替え
                if (m_curPage == Page_Noise)
                {
                    m_curPage = Page_Txop;
                }
                else
                {
                    m_curPage = Page_Noise;
                }
            }
            else if (npad.IsTrigger(Npad::X))
            {
                // submenu開く
                m_IsSubMenuOpened = true;
            }
        }
        break;
    default:
        break;
    }

    // 時刻取得
    {
        nn::time::PosixTime posixTime;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime) );

        // PosixTime を CalendarTime へ変換します。
        // 計算に利用されるタイムゾーンはデバイスに設定されたものを利用します。
        nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&m_calendarTime, &calendarAdditionalInfo, posixTime) );
    }
} //NOLINT(impl/function_size)

void SceneLoggingCh::Draw(GraphicTools* pTools) NN_NOEXCEPT
{
    pTools->pPrimitiveRenderer->Update(pTools->bufferIndex);

    // コマンド生成
    pTools->pGraphicsFramework->BeginFrame(pTools->bufferIndex);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = pTools->pGraphicsFramework->GetRootCommandBuffer(pTools->bufferIndex);

        nn::util::Vector3f translate;
        nn::gfx::ColorTargetView* target = pTools->pGraphicsFramework->GetColorTargetView();
        nn::gfx::DepthStencilView* depthStencil = pTools->pGraphicsFramework->GetDepthStencilView();
        rootCommandBuffer->ClearColor(target, 0.f, 0.f, 0.f, 1.0f, NULL);
        rootCommandBuffer->SetRenderTargets(1, &target, depthStencil);
        rootCommandBuffer->SetViewportScissorState(pTools->pGraphicsFramework->GetViewportScissorState());

        // スクリーン左上原点となる画面を設定
        pTools->pPrimitiveRenderer->SetDefaultParameters();
        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::Matrix4x3f modelMatrix;
        nn::util::MatrixIdentity( &viewMatrix );
        nn::util::MatrixIdentity( &projectionMatrix );
        nn::util::MatrixIdentity( &modelMatrix );
        pTools->pPrimitiveRenderer->SetViewMatrix( &viewMatrix );
        pTools->pPrimitiveRenderer->SetProjectionMatrix( &projectionMatrix );
        pTools->pPrimitiveRenderer->SetModelMatrix( &modelMatrix );

        {
            switch (m_curPage)
            {
            case Page_Setting:
                DrawSetting(pTools);
                break;
            case Page_Noise:
            case Page_Txop:
                DrawLog(pTools);
                DrawMa(pTools);
                DrawSubmenu(pTools);
                break;
            default:
                break;
            }
        }

        // Time
        {
            pTools->pWriter->SetTextColor(nn::util::Color4u8::White());
            pTools->pWriter->SetFontSize(20.f);
            pTools->pWriter->SetCursor(1090, 700);
            pTools->pWriter->Print("%04d/%02d/%02d %02d:%02d:%02d\n",
                m_calendarTime.year, m_calendarTime.month,
                m_calendarTime.day, m_calendarTime.hour,
                m_calendarTime.minute, m_calendarTime.second);
        }

        // デバッグフォント用のコマンド生成
        pTools->pWriter->Draw(rootCommandBuffer);
    }
    pTools->pGraphicsFramework->EndFrame(pTools->bufferIndex);
}

void SceneLoggingCh::StartMeasure() NN_NOEXCEPT
{
    m_pTopUTS = new unitTimeStats;
    NN_ABORT_UNLESS_NOT_NULL(m_pTopUTS);
    m_pTopUTS->next = nullptr;
    m_pTopUTS->prev = nullptr;
    m_pTopUTS->noise = 100;
    m_pTopUTS->txop = 0;
    m_pTopUTS->id = 0;
    m_pLastUTS = m_pTopUTS;
    m_pDispLastUTS = m_pTopUTS;
    m_pDispFirstUTS = m_pTopUTS;
    m_pPointedUTS = m_pTopUTS;
    m_pSaveUTS = m_pTopUTS;

    // SDカードへ保存するファイル名は現在時刻から作成
    {
        nn::time::PosixTime posixTime;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime) );
        nn::time::CalendarTime cTime;
        nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&cTime, &calendarAdditionalInfo, posixTime) );
        char filePath[256];
        nn::util::SNPrintf(filePath, sizeof(filePath), "sd:/measure_ch%d_%04d%02d%02d_%02d%02d%02d.txt",
                m_logInfo.scanParam.channelList[0], cTime.year, cTime.month, cTime.day, cTime.hour, cTime.minute, cTime.second);
        bool ret = m_SdWriter.Initialize(filePath);
        if( ret == true )
        {
            NN_LOG("SD card is inserted. Will save results automatically.\n");
            m_IsSdInserted = true;
        }
        else
        {
            m_IsSdInserted = false;
        }
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread(
            &m_measureThread, MeasureThreadFunc, NULL, g_measureThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority)
    );
    nn::os::StartThread( &m_measureThread );

    if( m_IsSdInserted == true )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
                nn::os::CreateThread(
                &m_saveThread, SaveResultsThreadFunc, NULL, g_saveThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority)
        );
        nn::os::StartThread( &m_saveThread );
    }
}

void SceneLoggingCh::StopMeasure() NN_NOEXCEPT
{
    m_measureExitFlag = true;
    nn::os::WaitThread(&m_measureThread);
    nn::os::DestroyThread(&m_measureThread);

    if( m_IsSdInserted == true )
    {
        m_saveExitFlag = true;
        nn::os::WaitThread(&m_saveThread);
        nn::os::DestroyThread(&m_saveThread);
    }

    m_SdWriter.Finalize();

    // リストの消去
    unitTimeStats* ptr;
    for (ptr = m_pTopUTS; ptr != nullptr; )
    {
        unitTimeStats* pTmp = ptr;
        ptr = ptr->next;
        delete pTmp;
    }
}

void SceneLoggingCh::DrawSetting(GraphicTools* pTools) NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pGraphicsFramework = pTools->pGraphicsFramework;
    int bufferIndex = pTools->bufferIndex;
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = pTools->pPrimitiveRenderer;
    nn::gfx::util::DebugFontTextWriter* pWriter = pTools->pWriter;
    nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

    {
        //メニュータイトル
        pWriter->SetFontSize(40.f);
        pWriter->SetLineHeight(44.f);
        pWriter->SetTextColor(nn::util::Color4u8::White());
        pWriter->SetCursor(pWriter->GetFontWidth(), 0);
        pWriter->Print("Logging Channel");
        // 仕切り腺
        pPrimitiveRenderer->SetLineWidth(1.f);
        pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());
        pPrimitiveRenderer->Draw2DLine(rootCommandBuffer,
                0.f, pWriter->GetLineHeight(),
                pGraphicsFramework->GetDisplayWidth(), pWriter->GetLineHeight());
    }

    {
        // 操作
        pWriter->SetFontSize(20.f);
        pWriter->SetTextColor(nn::util::Color4u8::White());
        pWriter->SetCursor(10, 700);
        pWriter->Print("[+]:Back");
    }

    {
        // メニュー
        pWriter->SetFontSize(40.f);
        pWriter->SetLineHeight(44.f);
        float menuY = pWriter->GetLineHeight() * 2.f;
        // Channel
        {
            pWriter->SetTextColor(nn::util::Color4u8::White());
            pWriter->SetCursor(
                    pWriter->GetFontWidth() * 2.f,
                    menuY);
            pWriter->Print("Channel");
            pWriter->SetCursor(
                    pWriter->GetFontWidth() * 20.f,
                    menuY);
            pWriter->Print(":");
            if( m_curMenu == Menu_Channel )
            {
                pWriter->SetTextColor(nn::util::Color4u8::Yellow());
            }
            else
            {
                pWriter->SetTextColor(nn::util::Color4u8::White());
            }
            pWriter->SetCursor(
                    pWriter->GetFontWidth() * 21.f,
                    menuY);
            pWriter->Print("%d", m_logInfo.scanParam.channelList[0]);
        }
        // Scan Time
        {
            pWriter->SetTextColor(nn::util::Color4u8::White());
            pWriter->SetCursor(
                    pWriter->GetFontWidth() * 2.f,
                    menuY + pWriter->GetLineHeight());
            pWriter->Print("Scan Time [ms]");
            pWriter->SetCursor(
                    pWriter->GetFontWidth() * 20.f,
                    menuY + pWriter->GetLineHeight());
            pWriter->Print(":");
            if( m_curMenu == Menu_ScanTime )
            {
                pWriter->SetTextColor(nn::util::Color4u8::Yellow());
            }
            else
            {
                pWriter->SetTextColor(nn::util::Color4u8::White());
            }
            pWriter->SetCursor(
                    pWriter->GetFontWidth() * 21.f,
                    menuY + pWriter->GetLineHeight());
            pWriter->Print("%d", m_logInfo.scanParam.channelScanTime);
        }
        // MeasureInterval Time
        {
            pWriter->SetTextColor(nn::util::Color4u8::White());
            pWriter->SetCursor(
                    pWriter->GetFontWidth() * 2.f,
                    menuY + pWriter->GetLineHeight() * 2.f);
            pWriter->Print("Measure Interval [ms]");
            pWriter->SetCursor(
                    pWriter->GetFontWidth() * 20.f,
                    menuY + pWriter->GetLineHeight() * 2.f);
            pWriter->Print(":");
            if( m_curMenu == Menu_MeasureInterval )
            {
                pWriter->SetTextColor(nn::util::Color4u8::Yellow());
            }
            else
            {
                pWriter->SetTextColor(nn::util::Color4u8::White());
            }
            pWriter->SetCursor(
                    pWriter->GetFontWidth() * 21.f,
                    menuY + pWriter->GetLineHeight() * 2.f);
            pWriter->Print("%d", m_logInfo.measureInterval);
        }

        // Cursor
        {
            pWriter->SetTextColor(nn::util::Color4u8::White());
            pWriter->SetCursor(
                    pWriter->GetFontWidth(),
                    menuY + m_curMenu * pWriter->GetLineHeight());
            pWriter->Print(">");
        }

        // NOTE
        {
            pWriter->SetFontSize(20.f);
            pWriter->SetTextColor(nn::util::Color4u8::White());
            pWriter->SetCursor(
                    50.f,
                    340.f);
            pWriter->Print("SDカードを挿している場合、測定開始後10分おきに自動的に計測結果がSDカードに保存されます。");
            pWriter->SetCursor(
                    50.f,
                    340.f + pWriter->GetLineHeight());
            pWriter->Print("測定終了時にも自動的に保存されます。");
            pWriter->SetCursor(
                    50.f,
                    340.f + pWriter->GetLineHeight() * 2.f);
            pWriter->Print("SDカードは測定開始前に挿している必要があります。");
            pWriter->SetCursor(
                    50.f,
                    340.f + pWriter->GetLineHeight() * 3.f);
            pWriter->Print("If a SD card is inserted, measurement results will be automatically saved to the SD card every 10 minutes after starting measurement.");
            pWriter->SetCursor(
                    50.f,
                    340.f + pWriter->GetLineHeight() * 4.f);
            pWriter->Print("It will also be automatically saved when you quit measurement.");
            pWriter->SetCursor(
                    50.f,
                    340.f + pWriter->GetLineHeight() * 5.f);
            pWriter->Print("You need to insert a SD card before starting measurement.");
            // 枠
            pPrimitiveRenderer->SetLineWidth(2.f);
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());
            pPrimitiveRenderer->Draw2DFrame(
                    rootCommandBuffer,
                    40.f, 330.f,
                    pWriter->GetFontWidth() * 70.f, pWriter->GetLineHeight() * 6.f + 20.f);
        }
    }
} //NOLINT(impl/function_size)

void SceneLoggingCh::DrawLog(GraphicTools* pTools) NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pGraphicsFramework = pTools->pGraphicsFramework;
    int bufferIndex = pTools->bufferIndex;
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = pTools->pPrimitiveRenderer;
    nn::gfx::util::DebugFontTextWriter* pWriter = pTools->pWriter;
    nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

    // ラベル
    pWriter->SetFontSize(20.f);
    pWriter->SetTextColor(nn::util::Color4u8::White());
    pWriter->SetCursor(10, 0);
    pWriter->Print("Channel %d %s    Scan Time:%d    Measure Interval:%d",
            m_logInfo.scanParam.channelList[0],
            (m_curPage == Page_Noise) ? "Noise Level" : "TXOP",
            m_logInfo.scanParam.channelScanTime,
            m_logInfo.measureInterval);
    pWriter->SetCursor(10, 700);
    if( m_IsSubMenuOpened == true )
    {
        pWriter->Print("[+]:Back [X]:Close Submenu");
    }
    else
    {
        pWriter->Print("[+]:Back [X]:Open Submenu [R]:Display %s",
                (m_curPage == Page_Noise) ? "Noise Level" : "TXOP");
    }

    // 軸を描画
    pPrimitiveRenderer->SetLineWidth(1.f);
    pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());
    // 上軸
    pPrimitiveRenderer->Draw2DLine(
            rootCommandBuffer,
            DISPLAY_X, DISPLAY_Y,
            SCREEN_WIDTH - DISPLAY_X, DISPLAY_Y);
    // 下軸
    pPrimitiveRenderer->Draw2DLine(rootCommandBuffer,
            DISPLAY_X, DISPLAY_Y + DISPLAY_UNIT_H * 100,
            SCREEN_WIDTH - DISPLAY_X, DISPLAY_Y + DISPLAY_UNIT_H * 100);
    // 横枠
    pPrimitiveRenderer->Draw2DLine(
            rootCommandBuffer,
            DISPLAY_X, DISPLAY_Y,
            DISPLAY_X, DISPLAY_Y + DISPLAY_UNIT_H * 100);
    pPrimitiveRenderer->Draw2DLine(
            rootCommandBuffer,
            SCREEN_WIDTH - DISPLAY_X, DISPLAY_Y,
            SCREEN_WIDTH - DISPLAY_X, DISPLAY_Y + DISPLAY_UNIT_H * 100);
    // -20dBm, -40dBm, -60dBm, -80dBm線
    pPrimitiveRenderer->SetColor(static_cast<const nn::util::Color4u8&>(MY_GRAY));
    for( int i = 20; i < 100; i += 20)
    {
        pPrimitiveRenderer->Draw2DLine(rootCommandBuffer,
                DISPLAY_X, DISPLAY_Y + DISPLAY_UNIT_H * i,
                SCREEN_WIDTH - DISPLAY_X, DISPLAY_Y + DISPLAY_UNIT_H * i);
        pWriter->SetCursor(32, DISPLAY_Y + DISPLAY_UNIT_H * i - 16.f);
        pWriter->Print("%s%d", (m_curPage == Page_Noise) ? "-" : "", i);
    }
    pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());

    pWriter->SetCursor(32, DISPLAY_Y - 16.f);
    pWriter->Print("   0");
    pWriter->SetCursor(32, DISPLAY_Y + DISPLAY_UNIT_H * 100 - 16.f);
    pWriter->Print("%d", (m_curPage == Page_Noise) ? -100 : 100);

    // チャート表示
    if( m_IsLastUTSPointed == true )
    {
        m_pDispLastUTS = m_pLastUTS;
    }
    if( m_pDispLastUTS->id == 0 )
    {
        // まだ計測が1回もされていない
        return;
    }
    // 詳細表示ノード位置から実際の表示範囲を決定
    if( m_pPointedUTS->id < m_pDispFirstUTS->id && m_IsLastUTSPointed == false )
    {
        // 詳細表示ノードが古い方向に範囲外
        m_pDispFirstUTS = m_pPointedUTS;
        m_pDispLastUTS = m_pDispFirstUTS;
        for( int i = 0; i < POINTABLE_COUNTS - 1; i++ )
        {
            m_pDispLastUTS = m_pDispLastUTS->next;
        }
    }
    else if( m_pPointedUTS->id > m_pDispLastUTS->id )
    {
        // 詳細表示ノードが新しい方向に範囲外
        m_pDispLastUTS = m_pPointedUTS;
    }

    pPrimitiveRenderer->SetLineWidth(1.f);
    pPrimitiveRenderer->SetColor(nn::util::Color4u8::Cyan());
    unitTimeStats* pUTS = m_pDispLastUTS;
    for( int i = 0; i < POINTABLE_COUNTS; i++ )
    {
        int delta = (m_curPage == Page_Noise) ? -pUTS->noise : pUTS->txop;
        char infoStr[10];
        nn::util::SNPrintf(infoStr, sizeof(infoStr), "%d%s",
                (m_curPage == Page_Noise) ? pUTS->noise : pUTS->txop,
                (m_curPage == Page_Noise) ? " dBm" : "");
        // 点を打つ
        pPrimitiveRenderer->Draw2DRect(
                rootCommandBuffer,
                SCREEN_WIDTH - DISPLAY_X - (DISPLAY_UNIT_W * i) - 2.f,
                DISPLAY_Y + (delta * DISPLAY_UNIT_H) - 2.f,
                4.f, 4.f);
        // 表示されている先頭ノードの値を描画
        if( i == 0 )
        {
            pWriter->SetTextColor(nn::util::Color4u8::Cyan());
            pWriter->SetCursor(SCREEN_WIDTH - DISPLAY_X + 10.f, DISPLAY_Y + (delta * DISPLAY_UNIT_H) - 16.f);
            pWriter->Print("%s", infoStr);
        }

        // 詳細表示
        if( pUTS == m_pPointedUTS && m_IsPointedUTSDisplayed == true)
        {
            pPrimitiveRenderer->SetLineWidth(2.f);
            pPrimitiveRenderer->SetColor(static_cast<const nn::util::Color4u8&>(MY_PURPLE));
            // noise値表示用線
            pPrimitiveRenderer->Draw2DLine(
                    rootCommandBuffer,
                    SCREEN_WIDTH - DISPLAY_X - (DISPLAY_UNIT_W * i), DISPLAY_Y + (delta * DISPLAY_UNIT_H),
                    SCREEN_WIDTH - DISPLAY_X + 10.f, DISPLAY_Y + (delta * DISPLAY_UNIT_H));
            // 時刻表示用線
            pPrimitiveRenderer->Draw2DLine(
                    rootCommandBuffer,
                    SCREEN_WIDTH - DISPLAY_X - (DISPLAY_UNIT_W * i), DISPLAY_Y + (delta * DISPLAY_UNIT_H),
                    SCREEN_WIDTH - DISPLAY_X - (DISPLAY_UNIT_W * i), DISPLAY_Y + (100.f * DISPLAY_UNIT_H) + 15.f);
            // noise値表示用背景
            pPrimitiveRenderer->Draw2DRect(
                    rootCommandBuffer,
                    SCREEN_WIDTH - DISPLAY_X + 10.f - 4.f,
                    DISPLAY_Y + (delta * DISPLAY_UNIT_H) - 8.f,
                    90.f, pWriter->GetLineHeight());
            // 時刻表示用背景
            pPrimitiveRenderer->Draw2DRect(
                    rootCommandBuffer,
                    SCREEN_WIDTH - DISPLAY_X - (DISPLAY_UNIT_W * i) - 60.f,
                    DISPLAY_Y + (100.f * DISPLAY_UNIT_H) + 15.f - 5.f,
                    120.f, pWriter->GetLineHeight() * 2.f);
            // 値表示
            pWriter->SetTextColor(static_cast<const nn::util::Color4u8&>(MY_DARK_BLUE));
            pWriter->SetCursor(SCREEN_WIDTH - DISPLAY_X + 10.f + 3.f, DISPLAY_Y + (delta * DISPLAY_UNIT_H) - 8.f);
            pWriter->Print("%s", infoStr);
            pWriter->SetCursor(SCREEN_WIDTH - DISPLAY_X - (DISPLAY_UNIT_W * i) - 50.f, DISPLAY_Y + (100.f * DISPLAY_UNIT_H) + 15.f - 5.f);
            pWriter->Print("%04d/%02d/%02d\n%02d:%02d:%02d",
                            pUTS->time.year, pUTS->time.month,
                            pUTS->time.day, pUTS->time.hour,
                            pUTS->time.minute, pUTS->time.second);
            // カラーを戻す
            pWriter->SetTextColor(nn::util::Color4u8::White());
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::Cyan());
            pPrimitiveRenderer->SetLineWidth(1.f);
        }

        pUTS = pUTS->prev;
        if( pUTS == nullptr || i == POINTABLE_COUNTS - 1)
        {
            break;
        }
        else
        {
            delta = (m_curPage == Page_Noise) ? -pUTS->noise : pUTS->txop;
            int nextDelta = (m_curPage == Page_Noise) ? -pUTS->next->noise : pUTS->next->txop;
            pPrimitiveRenderer->Draw2DLine(
                    rootCommandBuffer,
                    SCREEN_WIDTH - DISPLAY_X - (DISPLAY_UNIT_W * i), DISPLAY_Y + (nextDelta * DISPLAY_UNIT_H),
                    SCREEN_WIDTH - DISPLAY_X - (DISPLAY_UNIT_W * (i + 1)), DISPLAY_Y + (delta * DISPLAY_UNIT_H));
        }
    }
    m_pDispFirstUTS = (pUTS == nullptr) ? m_pTopUTS : pUTS->next;
} //NOLINT(impl/function_size)

void SceneLoggingCh::DrawMa(GraphicTools* pTools) NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pGraphicsFramework = pTools->pGraphicsFramework;
    int bufferIndex = pTools->bufferIndex;
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = pTools->pPrimitiveRenderer;
    nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

    for( int i = 0; i < MAWindow_End; i++ )
    {
        if( m_IsMaDisplayed[i] == true )
        {
            pPrimitiveRenderer->SetLineWidth(3.f);
            MAWindow window;
            unitTimeStats* pUTS = m_pDispLastUTS;
            switch( i )
            {
            case MAWindow_1min:
                pPrimitiveRenderer->SetColor(nn::util::Color4u8::Yellow());
                window = MAWindow_1min;
                break;
            case MAWindow_5min:
                pPrimitiveRenderer->SetColor(nn::util::Color4u8::Green());
                window = MAWindow_5min;
                break;
            case MAWindow_10min:
                pPrimitiveRenderer->SetColor(nn::util::Color4u8::Red());
                window = MAWindow_10min;
                break;
            case MAWindow_25min:
                pPrimitiveRenderer->SetColor(nn::util::Color4u8::Blue());
                window = MAWindow_25min;
                break;
            default:
                continue;
                break;
            }
            for( int j = 0; j < POINTABLE_COUNTS; j++ )
            {
                int val;
                val = GetAverage(pUTS, window) * ((m_curPage == Page_Noise) ? -1 : 1);

                pUTS = pUTS->prev;
                if( pUTS == nullptr || j == POINTABLE_COUNTS - 1)
                {
                    break;
                }
                else
                {
                    int nextVal = GetAverage(pUTS, window) * ((m_curPage == Page_Noise) ? -1 : 1);
                    pPrimitiveRenderer->Draw2DLine(
                            rootCommandBuffer,
                            SCREEN_WIDTH - DISPLAY_X - (DISPLAY_UNIT_W * j), DISPLAY_Y + (val * DISPLAY_UNIT_H),
                            SCREEN_WIDTH - DISPLAY_X - (DISPLAY_UNIT_W * (j + 1)), DISPLAY_Y + (nextVal * DISPLAY_UNIT_H));
                }
            }
        }
    }
}

void SceneLoggingCh::DrawSubmenu(GraphicTools* pTools) NN_NOEXCEPT
{
    static int anmCnt = 0;

    nns::gfx::GraphicsFramework* pGraphicsFramework = pTools->pGraphicsFramework;
    int bufferIndex = pTools->bufferIndex;
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = pTools->pPrimitiveRenderer;
    nn::gfx::util::DebugFontTextWriter* pWriter = pTools->pWriter;
    nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

    pWriter->SetFontSize(30.f);
    float subMenuW = 300.f;
    float subMenuH = pWriter->GetLineHeight() * SubMenu_End + 20.f;
    float subMenuX = pGraphicsFramework->GetDisplayWidth() - subMenuW;
    float subMenuY = 0.f;

    pPrimitiveRenderer->SetLineWidth(4.f);
    if( m_IsSubMenuOpened == false )
    {
        switch( anmCnt )
        {
        case 0:
            // 何も表示なし
            break;
        case 1:
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::Black());  // 背景色
            pPrimitiveRenderer->Draw2DRect(
                    rootCommandBuffer,
                    subMenuX, subMenuY,
                    subMenuW, subMenuH / 10.f);
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());  // 枠色
            pPrimitiveRenderer->Draw2DFrame(
                    rootCommandBuffer,
                    subMenuX, subMenuY,
                    subMenuW, subMenuH / 10.f);
            anmCnt--;
            break;
        case 2:
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::Black());  // 背景色
            pPrimitiveRenderer->Draw2DRect(
                    rootCommandBuffer,
                    subMenuX, subMenuY,
                    subMenuW, subMenuH * 2.f / 10.f);
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());  // 枠色
            pPrimitiveRenderer->Draw2DFrame(
                    rootCommandBuffer,
                    subMenuX, subMenuY,
                    subMenuW, subMenuH * 2.f / 10.f);
            anmCnt--;
            break;
        case 3:
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::Black());  // 背景色
            pPrimitiveRenderer->Draw2DRect(
                    rootCommandBuffer,
                    subMenuX, subMenuY,
                    subMenuW, subMenuH * 5.f / 10.f);
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());  // 枠色
            pPrimitiveRenderer->Draw2DFrame(
                    rootCommandBuffer,
                    subMenuX, subMenuY,
                    subMenuW, subMenuH * 5.f / 10.f);
            anmCnt--;
            break;
        default:
            break;
        }
        return;
    }
    else
    {
        switch( anmCnt )
        {
        case 0:
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::Black());  // 背景色
            pPrimitiveRenderer->Draw2DRect(
                    rootCommandBuffer,
                    subMenuX, subMenuY,
                    subMenuW, subMenuH / 10.f);
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());  // 枠色
            pPrimitiveRenderer->Draw2DFrame(
                    rootCommandBuffer,
                    subMenuX, subMenuY,
                    subMenuW, subMenuH / 10.f);
            anmCnt++;
            break;
        case 1:
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::Black());  // 背景色
            pPrimitiveRenderer->Draw2DRect(
                    rootCommandBuffer,
                    subMenuX, subMenuY,
                    subMenuW, subMenuH * 2.f / 10.f);
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());  // 枠色
            pPrimitiveRenderer->Draw2DFrame(
                    rootCommandBuffer,
                    subMenuX, subMenuY,
                    subMenuW, subMenuH * 2.f / 10.f);
            anmCnt++;
            break;
        case 2:
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::Black());  // 背景色
            pPrimitiveRenderer->Draw2DRect(
                    rootCommandBuffer,
                    subMenuX, subMenuY,
                    subMenuW, subMenuH * 5.f / 10.f);
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());  // 枠色
            pPrimitiveRenderer->Draw2DFrame(
                    rootCommandBuffer,
                    subMenuX, subMenuY,
                    subMenuW, subMenuH * 5.f / 10.f);
            anmCnt++;
            break;
        case 3:
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::Black());  // 背景色
            pPrimitiveRenderer->Draw2DRect(
                    rootCommandBuffer,
                    subMenuX, subMenuY,
                    subMenuW, subMenuH);
            pPrimitiveRenderer->SetColor(nn::util::Color4u8::White());  // 枠色
            pPrimitiveRenderer->Draw2DFrame(
                    rootCommandBuffer,
                    subMenuX, subMenuY,
                    subMenuW, subMenuH);
            {
                // メニュー
                float subMenuTxtX = subMenuX + 10.f;
                float subMenuTxtY = subMenuY + 10.f;
                pWriter->SetTextColor(nn::util::Color4u8::White());
                {
                    pWriter->SetCursor(
                            subMenuTxtX + pWriter->GetFontWidth() * 2.f,
                            subMenuTxtY + pWriter->GetLineHeight() * SubMenu_MA1min);
                    pWriter->Print("MA 1 min");
                    pWriter->SetCursor(
                            subMenuTxtX + pWriter->GetFontWidth() * 9.f,
                            subMenuTxtY + pWriter->GetLineHeight() * SubMenu_MA1min);
                    pWriter->Print("%s", (m_IsMaDisplayed[MAWindow_1min] == true ? "■" : "□"));

                    pWriter->SetCursor(
                            subMenuTxtX + pWriter->GetFontWidth() * 2.f,
                            subMenuTxtY + pWriter->GetLineHeight() * SubMenu_MA5min);
                    pWriter->Print("MA 5 min");
                    pWriter->SetCursor(
                            subMenuTxtX + pWriter->GetFontWidth() * 9.f,
                            subMenuTxtY + pWriter->GetLineHeight() * SubMenu_MA5min);
                    pWriter->Print("%s", (m_IsMaDisplayed[MAWindow_5min] == true ? "■" : "□"));

                    pWriter->SetCursor(
                            subMenuTxtX + pWriter->GetFontWidth() * 2.f,
                            subMenuTxtY + pWriter->GetLineHeight() * SubMenu_MA10min);
                    pWriter->Print("MA 10 min");
                    pWriter->SetCursor(
                            subMenuTxtX + pWriter->GetFontWidth() * 9.f,
                            subMenuTxtY + pWriter->GetLineHeight() * SubMenu_MA10min);
                    pWriter->Print("%s", (m_IsMaDisplayed[MAWindow_10min] == true ? "■" : "□"));

                    pWriter->SetCursor(
                            subMenuTxtX + pWriter->GetFontWidth() * 2.f,
                            subMenuTxtY + pWriter->GetLineHeight() * SubMenu_MA25min);
                    pWriter->Print("MA 25 min");
                    pWriter->SetCursor(
                            subMenuTxtX + pWriter->GetFontWidth() * 9.f,
                            subMenuTxtY + pWriter->GetLineHeight() * SubMenu_MA25min);
                    pWriter->Print("%s", (m_IsMaDisplayed[MAWindow_25min] == true ? "■" : "□"));

                    pWriter->SetTextColor(nn::util::Color4u8::Gray());
                    pWriter->SetCursor(
                            subMenuTxtX + pWriter->GetFontWidth() * 2.f,
                            subMenuTxtY + pWriter->GetLineHeight() * SubMenu_TimeScale);
                    pWriter->Print("Time Scale");
                }
                // Cursor
                {
                    pWriter->SetTextColor(nn::util::Color4u8::White());
                    pWriter->SetCursor(
                            subMenuTxtX,
                            subMenuTxtY + pWriter->GetLineHeight() * m_curSubMenu);
                    pWriter->Print(">");
                }
            }
            break;
        default:
            break;
        }
    }
} //NOLINT(impl/function_size)

int SceneLoggingCh::GetAverage(unitTimeStats* pHead, MAWindow window) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pHead);
    nn::TimeSpan span;
    switch( window )
    {
    case MAWindow_1min:
        span = nn::TimeSpan::FromMinutes(1);
        break;
    case MAWindow_5min:
        span = nn::TimeSpan::FromMinutes(5);
        break;
    case MAWindow_10min:
        span = nn::TimeSpan::FromMinutes(10);
        break;
    case MAWindow_25min:
        span = nn::TimeSpan::FromMinutes(25);
        break;
    default:
        return 100;
    }
    unitTimeStats* ptr = pHead->prev;
    int sum = (m_curPage == Page_Noise) ? pHead->noise : pHead->txop;
    int count = 1;
    while( 1 )
    {
        if( ptr == nullptr )
        {
            break;
        }

        // 一つ前の測定結果がspan時間内に計測されたものかどうか
        if( pHead->time - ptr->time <= span )
        {
            sum += ((m_curPage == Page_Noise) ? ptr->noise : ptr->txop);
            count++;
        }
        else
        {
            break;
        }
        ptr = ptr->prev;
    }

    return static_cast<int>(sum / count);
}

void SceneLoggingCh::MeasureThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    while( m_measureExitFlag != true )
    {
        nn::os::Tick tick = nn::os::GetSystemTick();
        uint32_t count;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::GetChannelStats(m_stats, &count, m_logInfo.scanParam));

        int id = ConvertChannelToId(m_logInfo.scanParam.channelList[0]);
        unitTimeStats* ptr;
        if( m_pLastUTS->id == 0 )
        {
            NN_LOG("First measurement!\n");
            ptr = m_pLastUTS;
        }
        else
        {
            ptr = new unitTimeStats;
            NN_ABORT_UNLESS_NOT_NULL(ptr);  // TODO 古いノードを消す
            m_pLastUTS->next = ptr;
            ptr->prev = m_pLastUTS;
        }
        ptr->next = nullptr;
        ptr->noise = m_stats[id].noiseLevel;
        ptr->txop = m_stats[id].txop;
        ptr->id = m_pLastUTS->id + 1;
        // 時刻格納
        {
            nn::time::PosixTime posixTime;
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime) );
            nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&ptr->time, &calendarAdditionalInfo, posixTime) );
        }
        m_pLastUTS = ptr;

        nn::os::LockMutex(&m_mutex);
        m_logInfo.noiseSum -= m_stats[id].noiseLevel;
        m_logInfo.txopSum += m_stats[id].txop;
        m_logInfo.count++;
        nn::os::UnlockMutex(&m_mutex);

        // 測定間隔評価
        int64_t elapsed = (nn::os::GetSystemTick() - tick).ToTimeSpan().GetMilliSeconds();
        if(elapsed < m_logInfo.measureInterval)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(m_logInfo.measureInterval - elapsed));
        }
    }
}

void SceneLoggingCh::SaveResultsThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    nn::os::Tick tick = nn::os::GetSystemTick();
    char str[45];
    while( m_saveExitFlag != true )
    {
        if( (nn::os::GetSystemTick() - tick).ToTimeSpan().GetMinutes() > SAVE_INTERVAL || m_saveExitFlag == true )
        {
            memset(m_saveStr, 0, sizeof(m_saveStr));
            strcpy(m_saveStr, "");
            unitTimeStats* pLast = m_pLastUTS;
            while( 1 )
            {
                memset(str, 0, sizeof(str));
                nn::util::SNPrintf(str, sizeof(str),
                        "%04d/%02d/%02d %02d:%02d:%02d,noise,%d,txop,%d\n",
                        m_pSaveUTS->time.year, m_pSaveUTS->time.month, m_pSaveUTS->time.day,
                        m_pSaveUTS->time.hour, m_pSaveUTS->time.minute, m_pSaveUTS->time.second,
                        m_pSaveUTS->noise, m_pSaveUTS->txop);
                strcat(m_saveStr, str);
                if( m_pSaveUTS == pLast )
                {
                    m_SdWriter.Append(m_saveStr, strlen(m_saveStr));
                    break;
                }
                m_pSaveUTS = m_pSaveUTS->next;
            }
            tick = nn::os::GetSystemTick();
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(60));
    }
}
