﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneMasterSetting.h"

#include <cstring>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/swkbd/swkbd_Api.h>
#include <nn/swkbd/swkbd_Result.h>
#include <nn/util/util_FormatString.h>

SceneMasterSetting::SceneMasterSetting(
        ISceneChanger* changer,
        WirelessData* pDistributor
        )
: BaseScene(changer, pDistributor)
{
    m_curY = 0;
    memset(&m_bssParam, 0, sizeof(nn::wlan::MasterBssParameters));
    memset(&m_afParam, 0, sizeof(WirelessData::PeriodicActionFrameParam));
    m_pApplicationHeap = NULL;
    strcpy(m_pwStr, "");
    strcpy(m_commentStr, "");
}

void SceneMasterSetting::Initialize() NN_NOEXCEPT
{
    m_distributor->GetMasterParam(&m_bssParam, &m_afParam);
    memcpy(m_pwStr, m_bssParam.security.key, sizeof(m_pwStr));
}

void SceneMasterSetting::Finalize() NN_NOEXCEPT
{
}

void SceneMasterSetting::Update() NN_NOEXCEPT
{
    Npad npad = Npad::GetInstance();
    npad.UpdatePadState();

    if( npad.IsTrigger(Npad::B) )
    {
        m_SceneChanger->ChangeScene(eScene_Title);
    }
    else if( npad.IsTrigger(Npad::A) )
    {
        switch( m_curY )
        {
        case curItem_Ssid:
        {
            char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
            char outStr[nn::wlan::Ssid::SsidHexStringLengthMax] = "";
            InputBySwkbd(m_bssParam.ssid.GetHexString(ssidStr), outStr, InputItem_Ssid);
            m_bssParam.ssid.Set(outStr);
            break;
        }
        case curItem_SecurityPw:
        {
            char outStr[64] = "";
            InputBySwkbd(m_pwStr, outStr, InputItem_Pw);
            strcpy(m_pwStr, outStr);
            break;
        }
        case curItem_Go:
        {
            bool isValid = true;
            switch( m_bssParam.security.privacyMode )
            {
            case nn::wlan::SecurityMode_Open:
                break;
            case nn::wlan::SecurityMode_StaticAes:
                // 入力文字が16文字かチェック
                if( strlen(m_pwStr) != 16 )
                {
                    isValid = false;
                    strcpy(m_commentStr, "You need to input 13 characters for PW");
                }
                break;
            default:
                break;
            }
            if( m_bssParam.ssid.GetLength() == 0 )
            {
                strcpy(m_commentStr, "You need to input SSID");
                isValid = false;
            }
            if( isValid == false )
            {
                // 入力情報が不正なので接続させない
                break;
            }

            // 入力情報に不正がない場合
            if( m_bssParam.security.privacyMode == nn::wlan::SecurityMode_StaticAes )
            {
                strcpy(reinterpret_cast<char*>(m_bssParam.security.key), m_pwStr);
            }
            m_distributor->SetMasterParam(m_bssParam, m_afParam);
            m_SceneChanger->ChangeScene(eScene_MasterRunning);
            break;
        }
        default:
            break;
        }
    }
    else if( npad.IsTrigger(Npad::DOWN) || npad.IsHold(Npad::DOWN) )
    {
        m_curY++;
        if( m_curY > curItem_NumOfItem - 1 )
        {
            m_curY = 0;
        }
    }
    else if( npad.IsTrigger(Npad::UP) || npad.IsHold(Npad::UP) )
    {
        m_curY--;
        if( m_curY < 0 )
        {
            m_curY = curItem_NumOfItem - 1;
        }
    }
    else if( npad.IsTrigger(Npad::RIGHT) || npad.IsHold(Npad::RIGHT) )
    {
        switch( m_curY )
        {
        case curItem_Channel:
            m_bssParam.channel++;
            if( m_bssParam.channel > 13 )
            {
                m_bssParam.channel = -1;
            }
            else if( m_bssParam.channel == 0 )
            {
                m_bssParam.channel = 1;
            }
            break;
        case curItem_hiddenSsid:
            m_bssParam.hiddenSsid ^= 1;
            break;
        case curItem_inavtivePeriod:
            m_bssParam.inactivePeriod += 10;
            if( m_bssParam.inactivePeriod > 180 )
            {
                m_bssParam.inactivePeriod = 0;
            }
            break;
        case curItem_SecurityMode:
            if( m_bssParam.security.privacyMode == nn::wlan::SecurityMode_Open )
            {
                m_bssParam.security.privacyMode = nn::wlan::SecurityMode_StaticAes;
            }
            else
            {
                m_bssParam.security.privacyMode = nn::wlan::SecurityMode_Open;
            }
            break;
        case curItem_contentsChangeInterval:
            m_afParam.contentsChangeInterval += 500;
            if( m_afParam.contentsChangeInterval > 100000 )
            {
                m_afParam.contentsChangeInterval = 0;
            }
            break;
        default:
            break;
        }
    }
    else if( npad.IsTrigger(Npad::LEFT) || npad.IsHold(Npad::LEFT) )
    {
        switch( m_curY )
        {
        case curItem_Channel:
            m_bssParam.channel--;
            if( m_bssParam.channel < -1 )
            {
                m_bssParam.channel = 13;
            }
            else if( m_bssParam.channel == 0 )
            {
                m_bssParam.channel = -1;
            }
            break;
        case curItem_hiddenSsid:
            m_bssParam.hiddenSsid ^= 1;
            break;
        case curItem_inavtivePeriod:
            if( m_bssParam.inactivePeriod == 0 )
            {
                m_bssParam.inactivePeriod = 180;
            }
            else
            {
                m_bssParam.inactivePeriod -= 10;
            }
            break;
        case curItem_SecurityMode:
            if( m_bssParam.security.privacyMode == nn::wlan::SecurityMode_Open )
            {
                m_bssParam.security.privacyMode = nn::wlan::SecurityMode_StaticAes;
            }
            else
            {
                m_bssParam.security.privacyMode = nn::wlan::SecurityMode_Open;
            }
            break;
        case curItem_contentsChangeInterval:
            m_afParam.contentsChangeInterval -= 500;
            if( m_afParam.contentsChangeInterval < 0 )
            {
                m_afParam.contentsChangeInterval = 100000;
            }
            break;
        default:
            break;
        }
    }

    if( m_curY == curItem_contentsChangeInterval )
    {
        strcpy(m_commentStr, "0 means it will not be updated.");
    }
    else
    {
        strcpy(m_commentStr, "");
    }
} // NOLINT(impl/function_size)

void SceneMasterSetting::Draw(
        GraphicsSystem* pGraphicsSystem,
        FontSystem* pFontSystem
        ) NN_NOEXCEPT
{
    BaseScene::Draw(pGraphicsSystem, pFontSystem);

    nn::gfx::util::DebugFontTextWriter&
        textWriter = pFontSystem->GetDebugFontTextWriter();

    const nn::util::Unorm8x4& textColor = Color::White;
    const nn::util::Unorm8x4& textColorGray = Color::Gray;
    const nn::util::Unorm8x4& textColorRed = Color::Red;

    textWriter.SetTextColor( textColor );
    textWriter.SetFontSize(FONT_SIZE_X, FONT_SIZE_Y);

    textWriter.SetCursor( INITIAL_X, INITIAL_Y + (LINE_SPACE * 2) );
    textWriter.Print( "<<Master parameter setting>>" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 1), MY_INITIAL_Y + (LINE_SPACE * curItem_Channel) );
    textWriter.Print( "Channel" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 1), MY_INITIAL_Y + (LINE_SPACE * curItem_hiddenSsid) );
    textWriter.Print( "Hidden Ssid" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 1), MY_INITIAL_Y + (LINE_SPACE * curItem_Ssid) );
    textWriter.Print( "Ssid" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 1), MY_INITIAL_Y + (LINE_SPACE * curItem_SecurityMode) );
    textWriter.Print( "Security" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 1), MY_INITIAL_Y + (LINE_SPACE * curItem_SecurityPw) );
    textWriter.Print( "Password" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 1), MY_INITIAL_Y + (LINE_SPACE * curItem_inavtivePeriod) );
    textWriter.Print( "Inactive [ms]" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 1), MY_INITIAL_Y + (LINE_SPACE * curItem_contentsChangeInterval) );
    textWriter.Print( "BeaconAf update [ms]" );

    for( int i = 0; i < curItem_NumOfItem - 1; i++ )
    {
        textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 14), MY_INITIAL_Y + (LINE_SPACE * i) );
        textWriter.Print( ":" );
    }

    // SSID
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 15), MY_INITIAL_Y + (LINE_SPACE * curItem_Ssid) );
    char ssidStr[nn::wlan::Ssid::SsidHexStringLengthMax];
    textWriter.Print( "[ %s ]", m_bssParam.ssid.GetHexString(ssidStr) );
    // PW
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 15), MY_INITIAL_Y + (LINE_SPACE * curItem_SecurityPw) );
    if( m_bssParam.security.privacyMode == nn::wlan::SecurityMode_Open )
    {
        textWriter.SetTextColor( textColorGray );
        textWriter.Print( "-" );
    }
    else
    {
        if( strlen(m_pwStr) > 0 )
        {
            textWriter.Print( "[ %s ]", m_pwStr );
        }
        else
        {
            textWriter.SetTextColor( textColorGray );
            textWriter.Print( "(Please input password)" );
        }
    }
    textWriter.SetTextColor( textColor );
    // SECURITY
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 15), MY_INITIAL_Y + (LINE_SPACE * curItem_SecurityMode) );
    textWriter.Print( "%s", (m_bssParam.security.privacyMode == nn::wlan::SecurityMode_Open) ? "Open" : "StaticAes" );
    // Channel
    {
        textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 15), MY_INITIAL_Y + (LINE_SPACE * curItem_Channel) );
        char str[5];
        if( m_bssParam.channel == -1 )
        {
            strcpy(str, "Auto");
        }
        else
        {
            nn::util::SNPrintf(str, sizeof(str), "%d", m_bssParam.channel);
        }
        textWriter.Print( "%s", str );
    }
    // Hidden ssid
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 15), MY_INITIAL_Y + (LINE_SPACE * curItem_hiddenSsid) );
    textWriter.Print( "%s", (m_bssParam.hiddenSsid == true) ? "true" : "false" );
    // Inactive period
    {
        textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 15), MY_INITIAL_Y + (LINE_SPACE * curItem_inavtivePeriod) );
        char str[5];
        nn::util::SNPrintf(str, sizeof(str), "%d", m_bssParam.inactivePeriod);
        textWriter.Print( "%s", str );
    }
    // BeaconAf update interval
    {
        textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 15), MY_INITIAL_Y + (LINE_SPACE * curItem_contentsChangeInterval) );
        char str[10];
        nn::util::SNPrintf(str, sizeof(str), "%d", m_afParam.contentsChangeInterval);
        textWriter.Print( "%s", str );
    }


    // Go next
    textWriter.SetCursor( INITIAL_X + (FONT_SIZE_X * 16), INITIAL_Y + (LINE_SPACE * 15) );
    textWriter.Print( "Start" );

    // カーソル
    if( m_curY == curItem_Go )
    {
        textWriter.SetCursor( INITIAL_X + (FONT_SIZE_X * 15), INITIAL_Y + (LINE_SPACE * 15) );
    }
    else
    {
        textWriter.SetCursor( MY_INITIAL_X, MY_INITIAL_Y + (LINE_SPACE * m_curY) );
    }
    textWriter.Print( ">" );

    textWriter.SetCursor( INITIAL_X + FONT_SIZE_X, FOOTER_Y );
    switch( m_curY )
    {
    case curItem_Channel:
    case curItem_SecurityMode:
    case curItem_contentsChangeInterval:
    case curItem_hiddenSsid:
    case curItem_inavtivePeriod:
        textWriter.Print( "[B]:Back to Title [<-][->]:Select" );
        break;
    case curItem_SecurityPw:
    case curItem_Ssid:
        textWriter.Print( "[B]:Back to Title [A]:Edit" );
        break;
    case curItem_Go:
        textWriter.Print( "[B]:Back to Title [A]:Start" );
        break;
    default:
        break;
    }

    textWriter.SetFontSize(FONT_SIZE_XS, FONT_SIZE_YS);
    textWriter.SetTextColor( textColorRed );
    textWriter.SetCursor( INITIAL_X + (FONT_SIZE_X * 2), INITIAL_Y + (LINE_SPACE * 16) );
    textWriter.Print( "%s", m_commentStr );

    pGraphicsSystem->BeginDraw();
    pFontSystem->Draw();
    pGraphicsSystem->EndDraw();

    pGraphicsSystem->Synchronize(
        nn::TimeSpan::FromNanoSeconds( 1000 * 1000 * 1000 / FRAME_RATE ) );
} //NOLINT(impl/function_size)

void SceneMasterSetting::SetApplicationHeap(ApplicationHeap* pApplicationHeap) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pApplicationHeap);
    m_pApplicationHeap = pApplicationHeap;
}

void SceneMasterSetting::InputBySwkbd(char* initStr, char* pOutStr, InputItem item) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(initStr);
    NN_ABORT_UNLESS_NOT_NULL(pOutStr);
    nn::swkbd::ShowKeyboardArg showKeyboardArg;
    nn::swkbd::MakePreset( &( showKeyboardArg.keyboardConfig ), nn::swkbd::Preset_Default );

    // 入力文字数制限
    if( item == InputItem_Ssid )
    {
        showKeyboardArg.keyboardConfig.textMaxLength = 32;
        showKeyboardArg.keyboardConfig.inputFormMode = nn::swkbd::InputFormMode_OneLine;
        showKeyboardArg.keyboardConfig.isUseUtf8 = true;
    }
    else if( item == InputItem_Pw )
    {
        showKeyboardArg.keyboardConfig.textMaxLength = 63;
        showKeyboardArg.keyboardConfig.inputFormMode = nn::swkbd::InputFormMode_OneLine;
        showKeyboardArg.keyboardConfig.isUseUtf8 = true;
    }

    // 共有メモリ用バッファの割り当て
    size_t in_heap_size = nn::swkbd::GetRequiredWorkBufferSize( false );
    void* swkbd_work_buffer = m_pApplicationHeap->Allocate( in_heap_size, nn::os::MemoryPageSize );

    showKeyboardArg.workBuf = swkbd_work_buffer;
    showKeyboardArg.workBufSize = in_heap_size;

    // 終了パラメータの設定
    size_t out_heap_size = nn::swkbd::GetRequiredStringBufferSize();
    nn::swkbd::String output_string;
    output_string.ptr = reinterpret_cast<char*>(
        m_pApplicationHeap->Allocate( out_heap_size, nn::os::MemoryPageSize ) );
    output_string.bufSize = out_heap_size;
    // 初期入力文字列の設定
    if( strlen(initStr) > 0 )
    {
        nn::swkbd::SetInitialTextUtf8(&showKeyboardArg, initStr);
    }
    else
    {
        // ガイド文字列の設定
        char guide_string[32];
        if( item == InputItem_Ssid )
        {
            strcpy(guide_string, "Please input SSID");
        }
        else if( item == InputItem_Pw )
        {
            strcpy(guide_string, "Please input password");
        }
        else
        {
            strcpy(guide_string, "Please input word");
        }
        nn::swkbd::SetGuideTextUtf8( &showKeyboardArg.keyboardConfig, guide_string );
    }

    nn::Result result = nn::swkbd::ShowKeyboard( &output_string, showKeyboardArg );
    if( nn::swkbd::ResultCanceled::Includes( result ) )
    {
        NN_LOG( " -- cancel --\n" );
    }
    else if( result.IsSuccess() )
    {
        NN_LOG( " -- ok --\n" );
    }

    // 結果文字列を受け取る
    char* str_ptr = reinterpret_cast< char* >( output_string.ptr );
    strcpy(pOutStr, str_ptr);

    // 共有メモリ用バッファの解放
    m_pApplicationHeap->Deallocate( output_string.ptr );
    m_pApplicationHeap->Deallocate( swkbd_work_buffer );
}
