﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SceneMasterRunning.h"

#include <cstring>
#include <cstdlib>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_LocalApi.h>
#include <nn/wlan/wlan_Result.h>

namespace {
// スレッドスタック
const size_t ThreadStackSize = 8192;
NN_OS_ALIGNAS_THREAD_STACK char g_afThreadStack[ ThreadStackSize ];
NN_OS_ALIGNAS_THREAD_STACK char g_monThreadStack[ ThreadStackSize ];
NN_OS_ALIGNAS_THREAD_STACK char g_scanThreadStack[ ThreadStackSize ];
// 受信バッファ
const size_t  afBufSize = 2048;
NN_ALIGNAS(4096) char g_afRxBuffer[ afBufSize ];
// BeaconLikeActionFrame Txバッファ
NN_ALIGNAS(4096) char g_beaconAfTxBuf[ afBufSize ];
// スキャンバッファ
const size_t  scanBufSize = 100 * 1024; // 100KB
NN_ALIGNAS(4096) char  g_scanBuffer[ scanBufSize ];

static bool g_exitFlagAfRx = false;
static bool g_exitFlagMonitor = false;
static bool g_exitScan = false;
nn::os::SystemEventType g_connectionEvent;

}

SceneMasterRunning::ActionFrameSender SceneMasterRunning::m_afSenders[SceneMasterRunning::AF_SENDER_MAX];
nn::wlan::ClientStatus SceneMasterRunning::m_clientStatus[nn::wlan::ConnectableClientsCountMax];
nn::os::MutexType SceneMasterRunning::m_afmutex;
nn::os::EventType SceneMasterRunning::m_scanEvent;
nn::wlan::ScanParameters SceneMasterRunning::m_scanParam;
int SceneMasterRunning::m_recvAfCnt;
SceneMasterRunning::SceneMasterRunning(
        ISceneChanger* changer,
        WirelessData* pDistributor
        )
: BaseScene(changer, pDistributor)
{
    for( int i = 0; i < AF_SENDER_MAX; i++ )
    {
        memset(&m_afSenders[i], 0x0, sizeof(ActionFrameSender));
    }
    for( int i = 0; i < nn::wlan::ConnectableClientsCountMax; i++ )
    {
        memset(&m_clientStatus[i], 0x0, sizeof(nn::wlan::ClientStatus));
    }
    m_afRxId = 0;
    m_curPage = Page_Main;
    m_topidOfAfSender = 0;
    m_recvAfCnt = 0;
}

void SceneMasterRunning::ClearActionFrameStats() NN_NOEXCEPT
{
    nn::os::LockMutex(&m_afmutex);
    for( int i = 0; i < AF_SENDER_MAX; i++ )
    {
        memset(&m_afSenders[i], 0x0, sizeof(ActionFrameSender));
    }
    m_topidOfAfSender = 0;
    m_recvAfCnt = 0;
    nn::os::UnlockMutex(&m_afmutex);
}

void SceneMasterRunning::ReceiveActionFrameThreadFunc(void* arg) NN_NOEXCEPT
{
    nn::Result result;
    g_exitFlagAfRx = false;

    uint32_t* rxId = reinterpret_cast<uint32_t*>(arg);

    while( !g_exitFlagAfRx )
    {
        nn::wlan::MacAddress mac;
        size_t rxSize = 0;
        uint16_t channel;
        int16_t rssi;
        result = nn::wlan::Local::GetActionFrameEx(&mac, reinterpret_cast<uint8_t*>(g_afRxBuffer),
                sizeof(g_afRxBuffer), &rxSize, *rxId, &channel, &rssi); // ブロックされる
        if( result.IsSuccess() )
        {
            nn::os::LockMutex(&m_afmutex);
            int i;
            for( i = 0; i < AF_SENDER_MAX; i++ )
            {
                if( m_afSenders[i].srcMac == mac && m_afSenders[i].channel == channel )
                {
                    m_afSenders[i].recvCnt++;
                    m_afSenders[i].channel = channel;
                    m_afSenders[i].rssi = rssi;
                    break;
                }
            }
            if( i == AF_SENDER_MAX )
            {
                // 新規送信元
                for( i = 0; i < AF_SENDER_MAX; i++ )
                {
                    if( m_afSenders[i].recvCnt == 0 )
                    {
                        m_afSenders[i].srcMac = mac;
                        m_afSenders[i].recvCnt++;
                        m_afSenders[i].channel = channel;
                        m_afSenders[i].rssi = rssi;
                        m_recvAfCnt++;
                        break;
                    }
                }
            }
            nn::os::UnlockMutex(&m_afmutex);
        }
    }
}

void SceneMasterRunning::MonitorThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    nn::Result result;
    g_exitFlagMonitor = false;

    nn::wlan::Local::GetConnectionEvent(&g_connectionEvent);
    while( !g_exitFlagMonitor )
    {
        if( nn::os::TryWaitSystemEvent(&g_connectionEvent) == true )
        {
            nn::Bit32 bit;
            nn::wlan::Local::GetClientStatus(m_clientStatus, &bit);
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
}

void SceneMasterRunning::UpdateActionFrame() NN_NOEXCEPT
{
    memset(&g_beaconAfTxBuf[0], 0x0, sizeof(g_beaconAfTxBuf));
    struct ActionFrameHeader {
        uint8_t etherType;
        uint8_t oui[3];
        uint8_t subtype;
        uint8_t reserved;
        uint16_t length;
    };
    ActionFrameHeader head = {
            0x7F,
            {0x00, 0x22, 0xAA},
            nn::wlan::ActionFrameType_Beacon,
            0,
            1024,
    };
    memcpy(&g_beaconAfTxBuf[0], &head, sizeof(ActionFrameHeader));

    // Payloadをランダムデータで埋める
    char* pData = reinterpret_cast<char*>(&g_beaconAfTxBuf[sizeof(ActionFrameHeader)]);
    srand(nn::os::GetSystemTick().GetInt64Value());
    for( int i = 0; i < head.length; i++, pData++ )
    {
        *pData = static_cast<uint8_t>(rand() % 0xFF);
    }
    strcpy(reinterpret_cast<char*>(&g_beaconAfTxBuf[sizeof(ActionFrameHeader)]), "BeaconLikeActionFrame");

    // アクションフレームセット
    NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::wlan::Local::SetActionFrameWithBeacon(reinterpret_cast<uint8_t*>(g_beaconAfTxBuf), sizeof(ActionFrameHeader) + head.length));
}

void SceneMasterRunning::Initialize() NN_NOEXCEPT
{
    m_distributor->GetMasterParam(&m_bssParam, &m_afParam);
    m_elapsedTime = nn::TimeSpan::FromMilliSeconds(static_cast<int64_t>(m_afParam.contentsChangeInterval));

    nn::os::InitializeMutex(&m_afmutex, false, 0);
    nn::os::InitializeEvent(&m_scanEvent, false, nn::os::EventClearMode_AutoClear);

    nn::Result result;
    // Master開始
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::OpenMasterMode());
    UpdateActionFrame(); // Beacon like action frameのセット
    m_bssParam.autoKeepAlive = false;
    m_bssParam.basicRates = nn::wlan::RateSetLegacy_11gMask;
    m_bssParam.supportedRates = nn::wlan::RateSetLegacy_11gMask;
    m_bssParam.beaconInterval = 100;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::CreateBss(m_bssParam));
    m_afTick = nn::os::GetSystemTick();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::GetConnectionStatus(&m_conStatus));
    m_channel = m_conStatus.channel;
    m_conStatus.ssid.GetHexString(m_ssidStr);
    m_conStatus.bssid.GetString(m_bssidStr);
    if( m_bssParam.security.privacyMode == nn::wlan::SecurityMode_Open )
    {
        strcpy(m_security, "Open");
    }
    else
    {
        strcpy(m_security, "StaticAes");
    }

    // ActionFrame受信
    uint16_t afTypes[] = {   // 受信したいActionFrameType）
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Beacon),
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Local)
    };
    // ActionFrame用受信エントリ作成
    NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::wlan::Local::CreateRxEntryForActionFrame(&m_afRxId, afTypes, sizeof(afTypes) / sizeof(uint16_t), 8));

    // ActionFrame受信スレッド作成
    NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread( &m_afRecvThread, ReceiveActionFrameThreadFunc, &m_afRxId, g_afThreadStack, sizeof(g_afThreadStack), nn::os::DefaultThreadPriority ));
    nn::os::StartThread( &m_afRecvThread );

    // Monitorスレッド
    NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread( &m_monitorThread, MonitorThreadFunc, NULL, g_monThreadStack, sizeof(g_monThreadStack), nn::os::DefaultThreadPriority ));
    nn::os::StartThread( &m_monitorThread );

    m_scanParam.scanType = nn::wlan::ScanType_Passive;
    m_scanParam.channelList[0] = 1;
    m_scanParam.channelCount = 0;
    m_scanParam.channelScanTime = 120;
    m_scanParam.homeChannelTime = 0;
    m_scanParam.ssidCount = 0;
    m_scanParam.ssidList = NULL;
    m_scanParam.bssid = nn::wlan::MacAddress::CreateBroadcastMacAddress();
    // Scanスレッド
    NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread( &m_scanThread, ScanThreadFunc, NULL, g_scanThreadStack, sizeof(g_scanThreadStack), nn::os::DefaultThreadPriority ));
    nn::os::StartThread( &m_scanThread );
}

void SceneMasterRunning::Finalize() NN_NOEXCEPT
{
    g_exitFlagAfRx = true;
    nn::wlan::Local::CancelGetActionFrame(m_afRxId);
    nn::os::WaitThread( &m_afRecvThread );
    nn::os::DestroyThread( &m_afRecvThread );
    g_exitFlagMonitor = true;
    nn::os::WaitThread( &m_monitorThread );
    nn::os::DestroyThread( &m_monitorThread );
    g_exitScan = true;
    nn::os::WaitThread( &m_scanThread );
    nn::os::DestroyThread( &m_scanThread );

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::DeleteRxEntryForActionFrame(m_afRxId));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::DestroyBss());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::wlan::Local::CloseMasterMode());

    nn::os::FinalizeMutex(&m_afmutex);
    nn::os::FinalizeEvent(&m_scanEvent);
}

void SceneMasterRunning::Update() NN_NOEXCEPT
{
    Npad npad = Npad::GetInstance();
    npad.UpdatePadState();

    if( npad.IsTrigger(Npad::PLUS) )
    {
        m_SceneChanger->ChangeScene(eScene_MasterSetting);
    }
    else if( npad.IsTrigger(Npad::L) )
    {
        if( m_curPage == Page_Main )
        {
            m_curPage = Page_Num - 1;
        }
        else
        {
            m_curPage--;
        }
    }
    else if( npad.IsTrigger(Npad::R) )
    {
        m_curPage++;
        if( m_curPage == Page_Num )
        {
            m_curPage = Page_Main;
        }
    }
    else if( npad.IsTrigger(Npad::B) )
    {
        switch(m_curPage)
        {
        case Page_RecvActionFrame:
            ClearActionFrameStats();
            break;
        default:
            break;
        }
    }
    else if( npad.IsTrigger(Npad::X) )
    {
        switch(m_curPage)
        {
        case Page_RecvActionFrame:
        case Page_Main:
            StartScan();
            break;
        default:
            break;
        }
    }
    else if( npad.IsTrigger(Npad::UP) || npad.IsHold(Npad::UP) )
    {
        switch(m_curPage)
        {
        case Page_RecvActionFrame:
            if( m_topidOfAfSender > 0 )
            {
                m_topidOfAfSender--;
            }
            break;
        default:
            break;
        }
    }
    else if( npad.IsTrigger(Npad::DOWN) || npad.IsHold(Npad::DOWN) )
    {
        switch(m_curPage)
        {
        case Page_RecvActionFrame:
            if( m_recvAfCnt - NUM_OF_AF_SENDER_DISPLAYED > 0)
            {
                m_topidOfAfSender++;
                if( m_topidOfAfSender > m_recvAfCnt - NUM_OF_AF_SENDER_DISPLAYED )
                {
                    m_topidOfAfSender = m_recvAfCnt - NUM_OF_AF_SENDER_DISPLAYED;
                }
            }
            break;
        default:
            break;
        }
    }


    // 時間経過計算
    if( m_afParam.contentsChangeInterval != 0 )
    {
        m_elapsedTime = (nn::os::GetSystemTick() - m_afTick).ToTimeSpan();
        if( m_elapsedTime.GetMilliSeconds() > m_afParam.contentsChangeInterval )
        {
            // Beacon Action Frameの中身変更
            UpdateActionFrame();
            m_afTick = nn::os::GetSystemTick();
        }
    }
}

void SceneMasterRunning::Draw(
        GraphicsSystem* pGraphicsSystem,
        FontSystem* pFontSystem
        ) NN_NOEXCEPT
{
    BaseScene::Draw(pGraphicsSystem, pFontSystem);

    switch( m_curPage )
    {
    case Page_Main:
        DrawMain(pFontSystem);
        break;
    case Page_RecvActionFrame:
        DrawRecvActionFrame(pFontSystem);
        break;
    default:
        DrawMain(pFontSystem);
    }

    pGraphicsSystem->BeginDraw();
    pFontSystem->Draw();
    pGraphicsSystem->EndDraw();

    pGraphicsSystem->Synchronize(
        nn::TimeSpan::FromNanoSeconds( 1000 * 1000 * 1000 / FRAME_RATE ) );
}

void SceneMasterRunning::DrawMain(FontSystem* pFontSystem) NN_NOEXCEPT
{
    nn::gfx::util::DebugFontTextWriter&
        textWriter = pFontSystem->GetDebugFontTextWriter();

    const nn::util::Unorm8x4& textColor = Color::White;

    textWriter.SetTextColor( textColor );
    textWriter.SetFontSize( FONT_SIZE_X, FONT_SIZE_Y );

    textWriter.SetCursor( INITIAL_X, INITIAL_Y + (LINE_SPACE * 2) );
    textWriter.Print( "<<Master Information>>" );
    // SSID
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 1), MY_INITIAL_Y + (LINE_SPACE * 0) );
    textWriter.Print( "SSID" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 10), MY_INITIAL_Y + (LINE_SPACE * 0) );
    textWriter.Print( "%s", m_ssidStr );
    // BSSID
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 1), MY_INITIAL_Y + (LINE_SPACE * 1) );
    textWriter.Print( "BSSID" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 10), MY_INITIAL_Y + (LINE_SPACE * 1) );
    textWriter.Print( "%s", m_bssidStr );
    // Channel
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 1), MY_INITIAL_Y + (LINE_SPACE * 2) );
    textWriter.Print( "Channel" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 10), MY_INITIAL_Y + (LINE_SPACE * 2) );
    textWriter.Print( "%d", m_channel );
    // Security
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 1), MY_INITIAL_Y + (LINE_SPACE * 3) );
    textWriter.Print( "Security" );
    textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 10), MY_INITIAL_Y + (LINE_SPACE * 3) );
    textWriter.Print( "%s",  m_security);
    for( int i = 0; i < 3; i++ )
    {
        textWriter.SetCursor( MY_INITIAL_X + (FONT_SIZE_X * 9), MY_INITIAL_Y + (LINE_SPACE * i) );
        textWriter.Print( ":" );
    }

    textWriter.SetCursor( INITIAL_X, MY_INITIAL_Y + (LINE_SPACE * 4) );
    textWriter.Print( "<<Connected Clients>>" );
    for( int i = 0, j = 5; i < nn::wlan::ConnectableClientsCountMax; i++ )
    {
        if( m_clientStatus[i].state == nn::wlan::ConnectionState_Connected )
        {
            textWriter.SetCursor( INITIAL_X + (FONT_SIZE_X * 1), MY_INITIAL_Y + (LINE_SPACE * j) );
            char bssidStrCli[nn::wlan::MacAddress::MacStringSize];
            textWriter.Print( "%s", m_clientStatus[i].clientMacAddress.GetString(bssidStrCli) );
            j++;
        }
    }


    if( m_afParam.contentsChangeInterval != 0 )
    {
        textWriter.SetCursor( INITIAL_X + FONT_SIZE_X, INITIAL_Y + (LINE_SPACE * 15) );
        textWriter.Print( "Beacon action frame will be updated in %d ms", static_cast<int>(m_afParam.contentsChangeInterval - m_elapsedTime.GetMilliSeconds()) );
    }

    textWriter.SetCursor( INITIAL_X + FONT_SIZE_X, FOOTER_Y );
    textWriter.Print( "[+]:Quit Master mode [L/R]Change pages [X] Scan" );
}

void SceneMasterRunning::DrawRecvActionFrame(FontSystem* pFontSystem) NN_NOEXCEPT
{
    nn::gfx::util::DebugFontTextWriter&
        textWriter = pFontSystem->GetDebugFontTextWriter();

    const nn::util::Unorm8x4& textColor = Color::White;

    textWriter.SetTextColor( textColor );
    textWriter.SetFontSize( FONT_SIZE_X, FONT_SIZE_Y );

    textWriter.SetCursor( INITIAL_X, INITIAL_Y + (LINE_SPACE * 2) );
    textWriter.Print( "<<Action Frame Source Address List>>" );
    // SSID
    for( int i = m_topidOfAfSender, j = 0; j < NUM_OF_AF_SENDER_DISPLAYED; i++ )
    {
        if( i == AF_SENDER_MAX )
        {
            break;
        }
        if( m_afSenders[i].recvCnt > 0 )
        {
            textWriter.SetCursor( INITIAL_X + (FONT_SIZE_X * 1), MY_INITIAL_Y + (LINE_SPACE * j) );
            char bssidStr[nn::wlan::MacAddress::MacStringSize];
            textWriter.Print( "%s  : %d[ch] : %d[dBm] : %d",
                    m_afSenders[i].srcMac.GetString(bssidStr), m_afSenders[i].channel, m_afSenders[i].rssi, m_afSenders[i].recvCnt );
            j++;
        }
    }

    textWriter.SetCursor( INITIAL_X + FONT_SIZE_X, FOOTER_Y );
    textWriter.Print( "[+]:Quit Master mode [L/R]Change pages [B]Clear stats [X] Scan" );
}

void SceneMasterRunning::ScanThreadFunc(void* arg) NN_NOEXCEPT
{
    nn::Result result;

    while( g_exitScan == false )
    {
        if( nn::os::TryWaitEvent(&m_scanEvent) == true)
        {
            result = nn::wlan::Local::StartScan(g_scanBuffer, scanBufSize, m_scanParam);
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
}

void SceneMasterRunning::StartScan() NN_NOEXCEPT
{
    nn::os::SignalEvent(&m_scanEvent);
}
