﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_TypesPrivate.h>
#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/bsdsocket/cfg/cfg_ClientApi.h>
#include <nn/btm/btm_Api.h>

#include "../common.h"

#define STATIC_IP           false
#define STATIC_IP_ADDR      "10.254.239.17"
#define STATIC_GW_ADDR      "10.254.239.1"
#define STATIC_SUBNET_MASK  "255.255.255.224"
#define STATIC_DNS_1        "8.8.8.8"
#define STATIC_DNS_2        "8.8.8.9"
#define WLAN_INTERFACE_NAME "wl0"
#define WLAN_APNAME         "KUROSE_TEST_AP2G"
#define WLAN_PRIORITY_MODE  true

// Configure IP
void ConfigureInterface(const char* interfaceName, bool enable)
{
    nn::Result result;

    if (enable)
    {
        nn::bsdsocket::cfg::IfSettings ifcfg;
        memset(&ifcfg, 0, sizeof(ifcfg));

        #if STATIC_IP
        // static ip
        ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Static;
        ifcfg.mtu  = 1500;
        nn::socket::InetAton(STATIC_IP_ADDR, &ifcfg.u.modeStatic.addr);
        nn::socket::InetAton(STATIC_GW_ADDR, &ifcfg.u.modeStatic.gatewayAddr);
        nn::socket::InetAton(STATIC_SUBNET_MASK, &ifcfg.u.modeStatic.subnetMask);
        ifcfg.u.modeStatic.broadcastAddr.s_addr =
                (ifcfg.u.modeStatic.addr.s_addr & ifcfg.u.modeStatic.subnetMask.s_addr) |
                    ~ifcfg.u.modeStatic.subnetMask.s_addr;
        nn::socket::InetAton(STATIC_DNS_1, &ifcfg.dnsAddrs[0]);
        nn::socket::InetAton(STATIC_DNS_2, &ifcfg.dnsAddrs[1]);
        #else
        // dhcp
        ifcfg.mode = nn::bsdsocket::cfg::IfIpAddrMode_Dhcp;
        ifcfg.mtu  = 1500;
        #endif

        result = nn::bsdsocket::cfg::SetIfUp(const_cast<char*>(interfaceName), &ifcfg);
        if (result.IsFailure())
        {
            NN_LOG("failed to configure interface %s - %d:%d\n",
                       interfaceName,
                       result.GetModule(),
                       result.GetDescription());
        }
    } else {
        nn::bsdsocket::cfg::SetIfDown(const_cast<char*>(interfaceName));
    }
}

// Connect to an AP
void InitializeNetwork()
{
    nn::os::SystemEventType connectionEvent;
    nn::wlan::ConnectionStatus connectionStatus;

    nn::wlan::InitializeInfraManager();
    nn::wlan::Infra::OpenMode();
    nn::wlan::Infra::GetConnectionEvent(&connectionEvent);

    // Configure connection information for AP
    nn::wlan::Ssid mySsid(WLAN_APNAME);
    nn::wlan::Security security = {
            nn::wlan::SecurityMode_Open,
            nn::wlan::SecurityMode_Open,
            0,
            "",
    };

    // Try to connect until it succeeds
    NN_LOG("Start to connect. Please wait...\n");
    while( 1 )
    {
        // Start to connect
        nn::wlan::Infra::Connect(mySsid, nn::wlan::MacAddress::CreateBroadcastMacAddress(), -1, security, false);

        nn::os::WaitSystemEvent(&connectionEvent);

        nn::wlan::Infra::GetConnectionStatus(&connectionStatus);
        if( connectionStatus.state == nn::wlan::ConnectionState_Connected )
        {
            NN_LOG("Connected!\n");
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(3000));
    }
}

void FinalizeNetwork()
{
    nn::wlan::Infra::Disconnect();
    nn::wlan::Infra::CloseMode();
    nn::wlan::FinalizeInfraManager();
}

extern "C"
{

const int  MemoryPoolSize = 10 * 1024 * 1024;
uint8_t    mempool[MemoryPoolSize];
size_t     mempoolSize = MemoryPoolSize;
nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

int testMain(int argc, char **argv);

void* aligned_alloc()
{
    return nullptr;
}

void nninitStartup()
{
    nn::os::SetMemoryHeapSize(nn::socket::DefaultSocketMemoryPoolSize);
}

void nnMain()
{
    int    i;
    int    argC;
    char** argV;
    const  char* defaultArguments[] = { "iperf", "-s", "-i", "10" };

    argC = nn::os::GetHostArgc();

    if (argC > 1)
    {
        argV  = nn::os::GetHostArgv();
    } else {
        /* Use default arguments if command line is empty */
        argV = const_cast<char**>(defaultArguments);
        argC = sizeof(defaultArguments) / sizeof(defaultArguments[0]);
    }

    /* Print command line to the console */
    for (i = 0; i < argC; i++)
    {
        NN_LOG("%s ", argV[i]);
    }

    NN_LOG("\n");

    NN_LOG("WlanPriorityMode = bool(%d)\n", WLAN_PRIORITY_MODE);
    nn::btm::InitializeBtmInterface();
    nn::btm::EnableSlotSaving(WLAN_PRIORITY_MODE);

    WlanTest::SystemInitialize();

    InitializeNetwork();

    /* Initialize network stack, run the test */
    nn::socket::Initialize(g_SocketConfigWithMemory);
    int socket = nn::socket::Socket(nn::socket::Family::Pf_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    NN_ABORT_UNLESS(socket >= 0);

    // notify interface linkup to BSD socket process
    nn::socket::IfReq ifr = {WLAN_INTERFACE_NAME};
    ifr.ifr_flags = nn::socket::IfrFlag::Iff_Up;
    nn::socket::Ioctl(socket, static_cast<nn::socket::IoctlCommand>(nn::socket::IoctlCommandPrivate::SiocSIfFlags), &ifr, sizeof(struct ifreq));
    ConfigureInterface(WLAN_INTERFACE_NAME, true);

    // iperf
    testMain(argC, argV);

    nn::socket::Finalize();
    FinalizeNetwork();

    WlanTest::SystemFinalize();

    nn::btm::EnableSlotSaving(false);
    nn::btm::FinalizeBtmInterface();

    return;
}

}
