﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>
#include <nn/os.h>
#include <nn/hid.h>
#include "../common.h"

namespace {

uint8_t  g_rxBufferAf[2048]; // ActionFrame受信用バッファ
uint8_t  g_txBuf[1024];

const size_t ThreadStackSize = 4096;              // スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStackRxAf[ ThreadStackSize ];   // ActionFrame受信用スレッドスタック

nn::os::ThreadType  g_ThreadRxAf;
static bool g_ExitFlagRxAf = false;

nn::wlan::DetectHash g_hash = {
        {0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef} // hash
};

uint64_t g_hashList[10] = {
        0x1234567890abcdef,
        0x1111111111111111,
        0x2222222222222222,
        0x3333333333333333,
        0x4444444444444444,
        0x5555555555555555,
        0x6666666666666666,
        0x7777777777777777,
        0x8888888888888888,
        0x9999999999999999
};

nn::os::EventType g_rxAfEv;
static nn::os::Tick g_startTick;

#ifdef CHANGE_MAC_ADDR
nn::wlan::MacAddress g_ownMac(0x7a, 0x34, 0x11, 0x00, 0x2d, 0xaa);
#endif
}


void RxThreadFuncAf(void* arg)
{
    nn::Result result;
    g_ExitFlagRxAf = true;

    uint32_t* rxId = reinterpret_cast<uint32_t*>(arg);
    nn::wlan::MacAddress mac;
    char macStr[nn::wlan::MacAddress::MacStringSize];
    uint16_t channel;
    int16_t rssi;
    nn::os::Tick tick;

    while( g_ExitFlagRxAf )
    {
        size_t rxSize = 0;
        result = nn::wlan::Detect::GetActionFrame(&mac, g_rxBufferAf, sizeof(g_rxBufferAf), &rxSize, *rxId, &channel, &rssi, &tick); // ブロックされる
        if( result.IsSuccess() )
        {
            // ActionFrameを受信したタイミングのシステムチック値が取得出来るので、あらかじめ取得しておいたシステムチック値との差分を取れば、
            // 何秒後に受信していたのかが計算出来る。
            NN_LOG("Received ActionFrame from : %s (%d[ch], %d[dBm], %d[byte], %lld[ms])\n", mac.GetString(macStr), channel, rssi, rxSize,
                    (tick - g_startTick).ToTimeSpan().GetMilliSeconds());
            //WlanTest::DumpBuffer(g_rxBufferAf, rxSize);
        }
        else if( result.GetDescription() == nn::wlan::ResultGetFrameCancelled().GetDescription() )
        {
            NN_LOG("GetFrame Cancelled.\n");
            NN_LOG("%s: Wait for the event to be signaled.\n", __FUNCTION__);
            nn::os::WaitEvent(&g_rxAfEv);
        }
        else
        {
            NN_ABORT("GetActionFrame Failed\n");
        }
    }
}


void SetupRecvActionFrame(uint32_t* pRxIdAf)
{
    NN_LOG("%s enter\n", __FUNCTION__);
    uint16_t afTypes[] = {   // 受信したいActionFrameType）
            static_cast<uint16_t>(nn::wlan::ActionFrameType_Detect),
    };
    // ActionFrame用受信エントリ作成
    auto result = nn::wlan::Detect::CreateRxEntryForActionFrame(pRxIdAf, afTypes, sizeof(afTypes) / sizeof(uint16_t), 10);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_LOG("WlanTest: Created Rx Entry for action frame[%d]\n", *pRxIdAf);

    // ActionFrame受信スレッド作成
    result = nn::os::CreateThread( &g_ThreadRxAf, RxThreadFuncAf, pRxIdAf, g_ThreadStackRxAf, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::StartThread( &g_ThreadRxAf );
}

void TeardownRecvActionFrame(uint32_t rxIdAf)
{
    // 受信のキャンセルと受信スレッドの破棄
    g_ExitFlagRxAf = false;
    auto result = nn::wlan::Detect::CancelGetActionFrame(rxIdAf);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::SignalEvent(&g_rxAfEv);
    nn::os::WaitThread( &g_ThreadRxAf );
    nn::os::DestroyThread( &g_ThreadRxAf );

    // 受信エントリ削除
    result = nn::wlan::Detect::DeleteRxEntryForActionFrame(rxIdAf);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_LOG("WlanTest: Delete Rx Entry [%d]\n", rxIdAf);
}

void StartPeriodicActionFrame(uint8_t* pBuf, size_t size)
{
    // すれちがいスリープ送受信パターンのセット
    nn::wlan::DetectPeriodicAfCycle pattern = {
            100, // txInterval[ms]
            10,  // txCount
            50,  // idleCount
            0,  // rxStartCount
            10   // rxCount
    };
    nn::wlan::Detect::SetPeriodicActionFrameCycle(pattern, nn::wlan::DetectPeriodicAfCycleTarget_Hd);

    strcpy(reinterpret_cast<char*>(&pBuf[0]), "This is neighbor detect sample periodic action frame");

    // 500ms間隔で送信する
    auto result = nn::wlan::Detect::StartPeriodicActionFrame(
            nn::wlan::ActionFrameType_Detect, g_hash, pBuf, size, 100);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void StopPeriodicActoinFrame()
{
    auto result = nn::wlan::Detect::CancelPeriodicActionFrame();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void PutOneShotActionFrame(uint8_t* pBuf, size_t size, bool isEx)
{
    strcpy(reinterpret_cast<char*>(&pBuf[0]), "This is neighbor detect sample one shot action frame");

    nn::Result result;
    if( isEx == true )
    {
        nn::wlan::DetectHeader dhp = {
                1, // major
                0, // minor
                1, // cmd. 1はPeriodic
                0, // reserved
                g_hash
        };
        result = nn::wlan::Detect::PutActionFrameOneShotEx(
                nn::wlan::MacAddress::CreateBroadcastMacAddress(),  // 宛先
                nn::wlan::MacAddress::CreateBroadcastMacAddress(),  // BSSID. ZeroMacを指定すると本体MACアドレスが使われる
                nn::wlan::ActionFrameType_Detect,
                dhp,
                pBuf,  // data
                size - 40,  // size
                0  // dwell time
                );
    }
    else
    {
        // こちらのAPIではBSSIDは本体MACアドレスが自動で使われる
        result = nn::wlan::Detect::PutActionFrameOneShot(
                nn::wlan::MacAddress::CreateBroadcastMacAddress(),  // 宛先
                nn::wlan::ActionFrameType_Detect,
                g_hash,
                pBuf,  // data
                size,  // size
                0  // dwell time
                );
    }

    if( nn::wlan::ResultNoMemory().Includes(result) == true )
    {
        NN_LOG("Put action frame one shot failed due to no memory\n");
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

}

void DetectDemo()
{
    NN_LOG("%s start\n", __FUNCTION__);

    nn::Result result;
    nn::os::InitializeEvent(&g_rxAfEv, false, nn::os::EventClearMode_AutoClear);
    nn::wlan::InitializeDetectManager();

    nn::wlan::WlanState wlanState;
    nn::wlan::Detect::GetState(&wlanState);
    NN_LOG("WLAN STATE : %s\n", WlanTest::wlanStateStr[wlanState]);

    result = nn::wlan::Detect::OpenMode(6);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_LOG("Open neighbor detect mode done\n");

    nn::wlan::Detect::GetState(&wlanState);
    NN_LOG("WLAN STATE : %s\n", WlanTest::wlanStateStr[wlanState]);
#ifdef CHANGE_MAC_ADDR
    // MACアドレスの変更。I/FがUPされた後は変更できないので、StartCommunication()の前に呼ぶ必要がある。
    nn::wlan::Detect::SetMacAddress(g_ownMac);
#endif
    // I/FがUPされる
    result = nn::wlan::Detect::StartCommunication();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_LOG("HD mode start\n");

    // 自MACアドレス取得
    nn::wlan::MacAddress macAddr;
    result = nn::wlan::Detect::GetMacAddress(&macAddr);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    char macStr[nn::wlan::MacAddress::MacStringSize];
    NN_LOG("mac addr : %s\n", macAddr.GetString(macStr));

    // ActionFrameの中身作成 -------------------------
    for( uint32_t i = 0; i < sizeof(g_txBuf); i++ )
    {
        g_txBuf[i] = static_cast<uint8_t>(i & 0xFF);
    }
    // ----------------------------------------------

    // ActionFrame受信開始
    uint32_t rxIdAf; // 受信エントリ番号受取り用変数
    g_startTick = nn::os::GetSystemTick();
    SetupRecvActionFrame(&rxIdAf);

    // PeriodicActionFrameの準備と送信
    StartPeriodicActionFrame(g_txBuf, 600);
    NN_LOG("Start periodic action frame\n");

    // ランダムなタイミングでスキャンとOneShotのActionFrameを送信する
    // 1分後、ループを抜ける
    srand(static_cast<unsigned int>(nn::os::GetSystemTick().GetInt64Value()));
    nn::os::Tick sysTick = nn::os::GetSystemTick();
    while( 1 )
    {
        int id = rand() % 10;
        if( id < 2 )
        {
            NN_LOG("Put one shot action frame with current mac address\n");
            PutOneShotActionFrame(g_txBuf, sizeof(g_txBuf), true);
        }
        else if( id < 5 )
        {
            NN_LOG("Put one shot action frame\n");
            PutOneShotActionFrame(g_txBuf, sizeof(g_txBuf), false);
        }
        else
        {
            NN_LOG("Sleep 3 seconds\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
        }

        // 経過時間確認
        if( (nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetMinutes() > 1 )
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }

    // PeriodicActoinFrameの停止
    StopPeriodicActoinFrame();

    // ActionFrame受信停止と後始末
    TeardownRecvActionFrame(rxIdAf);

    // I/FがDOWNされる
    result = nn::wlan::Detect::StopCommunication();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_LOG("HD mode stop\n");

    result = nn::wlan::Detect::CloseMode();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_LOG("WlanTest: CloseClientMode done\n");

    NN_LOG("\n\n End WlanTest \n\n");
    nn::wlan::FinalizeDetectManager();
    nn::os::FinalizeEvent(&g_rxAfEv);
} //NOLINT(impl/function_size)

void SetupSleepMode(uint32_t* rxIdAf)
{
    nn::Result result;

    // ActionFrameの中身作成 -------------------------
    for( uint32_t i = 0; i < sizeof(g_txBuf); i++ )
    {
        g_txBuf[i] = static_cast<uint8_t>(i & 0xFF);
    }
    // ----------------------------------------------

    // すれちがいスリープの予約
    nn::wlan::Detect::ReserveDetectSleep();

    // すれちがいスリープ用ActionFrameのセット
    nn::wlan::Detect::SetActionFrameForSleep(nn::wlan::ActionFrameType_Detect,
            g_hash, g_txBuf, 728);

    // ハッシュリストのセット
    NN_LOG("Set hash list. Number of hash is %d\n", sizeof(g_hashList) / sizeof(g_hashList[0]));
    result = nn::wlan::Detect::SetHashList(g_hashList, sizeof(g_hashList) / sizeof(g_hashList[0]));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // すれちがいスリープ送受信パターンのセット
    nn::wlan::DetectPeriodicAfCycle pattern = {
            100, // txInterval[ms]
            5,  // txCount
            0,  // idleCount
            0,  // rxStartCount
            5   // rxCount
    };
    nn::wlan::Detect::SetPeriodicActionFrameCycle(pattern, nn::wlan::DetectPeriodicAfCycleTarget_Sa);

    // ActionFrame受信開始
    SetupRecvActionFrame(rxIdAf);
}

void DetectSleepDemo()
{
    NN_LOG("%s start\n", __FUNCTION__);

    nn::Result result;
    nn::os::InitializeEvent(&g_rxAfEv, false, nn::os::EventClearMode_AutoClear);
    nn::wlan::InitializeDetectManager();
    uint32_t rxIdAf; // 受信エントリ番号受取り用変数
    enum State {
        State_Init,
        State_SleepReady,
        State_Sleep
    };
    State wlanState = State_Init;

    result = nn::wlan::Detect::OpenMode();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_LOG("Open neighbor detect mode done\n");

    // すれちがいスリープ中に受信したすれちがいパケットの累計を0に初期化
    result = nn::wlan::Detect::ClearTotalRecvCountInSleep();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // 待ち
    nn::hid::InitializeDebugPad();
    int32_t prevPadSamplingNumber = 0;
    nn::hid::DebugPadState debugPadState[nn::hid::DebugPadStateCountMax];
    while( 1 )
    {
        // DebugPad の入力を取得します。
        // debugPad の状態を表す構造体を用意します。
        nn::hid::GetDebugPadStates(debugPadState, nn::hid::DebugPadStateCountMax);

        // 直前から現在までの Pad 入力の更新回数を取得します。
        int32_t padSamplingCount = debugPadState[0].samplingNumber - prevPadSamplingNumber;
        if (padSamplingCount >= nn::hid::DebugPadStateCountMax)
        {
            padSamplingCount = nn::hid::DebugPadStateCountMax - 1;
        }
        nn::hid::DebugPadButtonSet padButtonDown(debugPadState[0].buttons & ~debugPadState[padSamplingCount].buttons);

        // 現在までの Pad 入力の更新回数を取得します。
        prevPadSamplingNumber = debugPadState[0].samplingNumber;

        if( padButtonDown.Test<nn::hid::DebugPadButton::A>() )
        {
            if( wlanState == State_Init )
            {
                NN_LOG("SetupSleepMode\n");
                SetupSleepMode(&rxIdAf);
                wlanState = State_SleepReady;
            }
            else if( wlanState == State_SleepReady )
            {
                NN_LOG("Request Sleep\n");
                // ENABLE_DETECT_AGING_DEBUGを有効化したwlanプロセスを使っていれば、ここでRequestSleep()を呼ぶ必要はない。
                // nifmが呼ぶnn::wlan::Infra::RequestSleep()がこれの代わりになるように分岐しているので。
                /*
                result = nn::wlan::Detect::RequestSleep();
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                */
                result = nn::wlan::Detect::CancelGetActionFrame(rxIdAf);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                g_startTick = nn::os::GetSystemTick();
                wlanState = State_Sleep;
            }
            else
            {
                NN_LOG("You already did a setup.\n");
            }
        }
        else if( padButtonDown.Test<nn::hid::DebugPadButton::Y>() )
        {
            if( wlanState == State_Sleep )
            {
                NN_LOG("Wake up\n");

                // ENABLE_DETECT_AGING_DEBUGを有効化したwlanプロセスを使っていれば、ここでRequestWakeUp()を呼ぶ必要はない。
                // nifmが呼ぶnn::wlan::Infra::RequestWakeUp()がこれの代わりになるように分岐しているので。
                /*
                result = nn::wlan::Detect::RequestWakeUp();
                if( result.GetDescription() == nn::wlan::ResultInvalidState().GetDescription() )
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                }
                */
                nn::os::SignalEvent(&g_rxAfEv);
                wlanState = State_SleepReady;

                // スリープ中に受信したすれちがいActionFrame数を取得
                uint64_t count = 0;
                result = nn::wlan::Detect::GetTotalRecvCountInSleep(&count);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
                NN_LOG("Total recv packets:%lld\n", count);
            }
            else
            {
                NN_LOG("You don't do a setup. Please do a setup at first.\n");
            }
        }
        else if( padButtonDown.Test<nn::hid::DebugPadButton::B>() )
        {
            if( wlanState == State_SleepReady )
            {
                NN_LOG("Stop receiving action frame and delete rx entry for action frame.\n");
                TeardownRecvActionFrame(rxIdAf);
                wlanState = State_Init;
            }
            else
            {
                NN_LOG("You don't do a setup. Please do a setup at first.\n");
            }
        }
        else if( padButtonDown.Test<nn::hid::DebugPadButton::Start>() )
        {
            if( wlanState == State_Init )
            {
                NN_LOG("End the app.\n");
                break;
            }
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    result = nn::wlan::Detect::CloseMode();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::wlan::FinalizeDetectManager();
    nn::os::FinalizeEvent(&g_rxAfEv);

    NN_LOG("%s done.\n", __FUNCTION__);
}

extern "C" void nnMain()
{
    NN_LOG("\n\n Start DetectSample \n\n");
    WlanTest::SystemInitialize();

    //DetectDemo();     // Host Driven すれちがいデモ
    DetectSleepDemo();  // Stand Alone すれちがいデモ

    WlanTest::SystemFinalize();
}

