﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_MatrixApi.h>
#include <nn/util/util_MathTypes.h>
#include <nn/mem/mem_StandardAllocator.h>
#include "gfxutil/FontRenderer.h"

namespace
{
    //  描画できる文字の最大数
    const int DrawableCharCountMax = 2048;
    //  フォントの横幅に対する等幅描画の幅
    const float FontFixedWidth = 13.0f;
}
namespace
{
    //  DebugFontTextWriterInfo の設定をします
    void SetDebugFontTextWriterInfo(nn::gfx::util::DebugFontTextWriterInfo* info) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL( info );
        info->SetDefault();
        info->SetCharCountMax( DrawableCharCountMax );
        info->SetUserMemoryPoolEnabled( false );
    }
}

namespace gfxutil
{

//  必要なメモリ量を取得します
size_t FontRenderer::GetRequiredMemorySize(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );
    nn::gfx::util::DebugFontTextWriterInfo info;
    SetDebugFontTextWriterInfo( &info );
    return nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize( pDevice, info );
}
//  初期化します
void FontRenderer::Initialize(nn::gfx::Device* pDevice, void* pMemory, size_t memorySize, int width, int height, nn::gfx::DescriptorPool* pSamplerDescriptorPool, int samplerDescriptorPoolSlot, nn::gfx::DescriptorPool* pTextureDescriptorPool, int textureDescriptorPoolSlot) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );
    NN_SDK_ASSERT_NOT_NULL( pMemory );
    NN_SDK_ASSERT_NOT_NULL( pTextureDescriptorPool );
    NN_SDK_ASSERT_NOT_NULL( pSamplerDescriptorPool );
    NN_SDK_REQUIRES_GREATER_EQUAL( memorySize, GetRequiredMemorySize( pDevice ) );

    nn::gfx::util::DebugFontTextWriterInfo info;
    SetDebugFontTextWriterInfo( &info );
    m_DebugFont.Initialize( pDevice, info, pMemory, memorySize, nullptr, 0, 0 );

    m_DebugFont.SetDisplayWidth( width );
    m_DebugFont.SetDisplayHeight( height );
    m_DebugFont.SetTextureDescriptor( pTextureDescriptorPool, textureDescriptorPoolSlot );
    m_DebugFont.SetSamplerDescriptor( pSamplerDescriptorPool, samplerDescriptorPoolSlot );
    m_DebugFont.SetFixedWidthEnabled( true );
    m_DebugFont.SetFixedWidth( FontFixedWidth );
}
//  終了処理をします
void FontRenderer::Finalize() NN_NOEXCEPT
{
    m_DebugFont.Finalize();
}

//  文字列の描画コマンドを発行します
void FontRenderer::FlushCommand(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( pCommandBuffer );
    m_DebugFont.Draw( pCommandBuffer );
}
//  文字列を描画します
void FontRenderer::Print(const char* string) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( string );
    if( *string != '\0' )
    {
        m_DebugFont.Print( "%s", string );
    }
}

//  文字の描画位置を設定します
void FontRenderer::SetPosition(float positionX, float positionY) NN_NOEXCEPT
{
    m_DebugFont.SetCursor( positionX, positionY );
}
//  文字の色を設定します
void FontRenderer::SetColor(const nn::util::Unorm8x4& color) NN_NOEXCEPT
{
    m_DebugFont.SetTextColor( color );
}
//  文字のスケールを設定します
void FontRenderer::SetScale(float scaleX, float scaleY) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_GREATER_EQUAL( scaleX, 0.0f );
    NN_ABORT_UNLESS_GREATER_EQUAL( scaleY, 0.0f );
    m_DebugFont.SetScale( scaleX, scaleY );
    m_DebugFont.SetFixedWidth( FontFixedWidth * scaleX );
}

//  文字を描画したときの幅を計算します
float FontRenderer::CalculateWidth(const char* str) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( str );
    return m_DebugFont.CalculateStringWidth( "%s", str );
}
//  文字を描画したときの高さを計算します
float FontRenderer::CalculateHeight(const char* str) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL( str );
    return m_DebugFont.CalculateStringHeight( "%s", str );
}

void FontRenderer::EnableFixedFont(bool enable)
{
    m_DebugFont.SetFixedWidthEnabled(enable);
}

void FontRenderer::SetFixedFontWidth(float width)
{
    m_DebugFont.SetFixedWidth(width);
}

float FontRenderer::GetFixedFontWidth()
{
    return m_DebugFont.GetFixedWidth();
}

float FontRenderer::GetFontHeight()
{
    return m_DebugFont.GetFontHeight();
}


}
