﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "UI.h"
#include "App.h"
#include "Menu.h"
#include <nn/util/util_BitFlagSet.h>
#include <tuple>


namespace
{

char g_GfxHeapMemory[24 * 1024 * 1024];

}


namespace ApConnectivityTest {
namespace UI {

//----------------------------------------------------------------
// UI
//----------------------------------------------------------------

// コンストラクタ
UI::UI() :
        m_GfxContextMemoryPool(new char[16 * 1024 * 1024]),
        m_StatusConsole(0, 40, 1280, 680),
        m_ShowKeyboard(false)
{
}


// デストラクタ
UI::~UI()
{
    delete[] m_GfxContextMemoryPool;
}


// 初期化
void UI::Initialize()
{
    // グラフィックスの初期化
    m_GfxContext.Initialize(g_GfxHeapMemory, sizeof(g_GfxHeapMemory), 1280, 720);

    //m_GfxContext.CreateRenderTarget(&m_TestRenderTarget, 256, 256);

    auto& fontRenderer = m_GfxContext.GetFontRenderer();

    auto fontWidth = 14;
    fontRenderer.SetFixedFontWidth(14);
    auto fontHeight = fontRenderer.GetFontHeight();

    m_ScreenWidth = static_cast<int>(1280 + fontWidth - 1) / static_cast<int>(fontWidth);
    m_ScreenHeight =  static_cast<int>(720 + fontHeight - 1) / static_cast<int>(fontHeight);

    m_StatusConsole.Initialize();
}


// 開放
void UI::Finalize()
{
    m_StatusConsole.Finalize();

    m_GfxContext.Finalize();
}


// 毎フレーム処理
void UI::Update()
{
    static bool s_UpdateDisplay = true;

    if (m_ShowKeyboard)
    {
        auto down = App::GetHid().GetKeyDown();
        if (down.IsAnyOn())
        {
            m_Keyboard.OnKeyDown(down);
        }

        auto up = App::GetHid().GetKeyUp();
        if (up.IsAnyOn())
        {
            m_Keyboard.OnKeyUp(up);
        }

        auto press = App::GetHid().GetKeyPress();
        if (press.IsAnyOn())
        {
            m_Keyboard.OnKeyPress(press);
        }

        m_GfxContext.Begin();
        m_Keyboard.Draw();
        m_GfxContext.End();
    }
    else
    {
        // メニュー開いたり選択移動したりする処理は Menu/SubMenuItem 側に持たせたい
        // こちらはメッセージを送るだけに

        // キー押下 (特にすることはない)
        if (App::GetHid().GetKeyDown().IsAnyOn())
        {
            //s_UpdateDisplay = true;
        }

        nn::hid::DebugPadButtonSet menuMoveBackMask;
        nn::hid::DebugPadButtonSet menuMoveNextMask;
        if (m_MenuStack.size() == 1)
        {
            menuMoveBackMask = nn::hid::DebugPadButton::Left::Mask;
            menuMoveNextMask = nn::hid::DebugPadButton::Right::Mask;
        }
        else
        {
            menuMoveBackMask = nn::hid::DebugPadButton::Up::Mask;
            menuMoveNextMask = nn::hid::DebugPadButton::Down::Mask;
        }

        auto upButtons = App::GetHid().GetKeyUp();
        if (upButtons.IsAnyOn())
        {
            // A ボタンが離されたとき
            if (upButtons.Test<nn::hid::DebugPadButton::A>())
            {
                auto& activeMenu = *m_MenuStack.back();
                activeMenu.SelectItem();
            }

            // B ボタンが離されたとき
            if (upButtons.Test<nn::hid::DebugPadButton::B>())
            {
                if (m_MenuStack.size() > 1)
                {
                    m_MenuStack.back()->activeItem = m_MenuStack.back()->items.begin();
                    m_MenuStack.pop_back();
                }
            }

            s_UpdateDisplay = true;
        }

        auto pressButtons = App::GetHid().GetKeyPress();
        if (pressButtons.IsAnyOn())
        {
            // 上キーを押した(リピートあり)
            if ((pressButtons & menuMoveBackMask).IsAnyOn())
            {
                auto& activeMenu = *m_MenuStack.back();
                activeMenu.MoveFocusPrevious();
            }
            // 下キーを押した(リピートあり)
            if ((pressButtons & menuMoveNextMask).IsAnyOn())
            {
                auto& activeMenu = *m_MenuStack.back();
                activeMenu.MoveFocusNext();
            }

            s_UpdateDisplay = true;
        }

        if (s_UpdateDisplay || m_StatusConsole.GetUpdated())
        {
            auto& fontRenderer = m_GfxContext.GetFontRenderer();

            m_GfxContext.Begin();

            // メニューの描画
            fontRenderer.EnableFixedFont(false);

            for (auto& menu : m_MenuStack)
            {
                menu->Draw();
            }

            // コンソールの描画
            {
                std::vector<Rect> excludeRegions(m_MenuStack.size());
                for (auto& menu : m_MenuStack)
                {
                    excludeRegions.push_back(menu->rectangle);
                }

                m_StatusConsole.Draw(excludeRegions);
            }

            m_GfxContext.End();
        }
    }

    s_UpdateDisplay = false;
} // NOLINT(impl/function_size)


void UI::SetTopMenu(const std::shared_ptr<Menu>& menu)
{
    m_MenuStack.push_back(menu);
    menu->rectangle.x = 0;
    menu->rectangle.y = 0;
}


void UI::PopupMenu(const std::shared_ptr<Menu>& menu)
{
    m_MenuStack.push_back(menu);
}


void UI::CloseMenu()
{
    m_MenuStack.pop_back();
}


void UI::OpenKeyboard(Keyboard::KeyMap& keyMap, const std::string& initialString, size_t minLength, size_t maxLength, const std::function<void(const std::string&)>& callback)
{
    m_ShowKeyboard = true;
    m_Keyboard.Initialize(&keyMap, initialString, minLength, maxLength, callback);
}


void UI::CloseKeyboard()
{
    m_ShowKeyboard = false;
}


gfxutil::GfxContext & UI::GetGfxContext()
{
    return m_GfxContext;
}


ConsoleView& UI::GetStatusConsole()
{
    return m_StatusConsole;
}

}
}
