﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "App.h"
#include <nn/hid/hid_DebugPadMap.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/util/util_BitFlagSet.h>
#include <tuple>


namespace ApConnectivityTest {
namespace UI {

//----------------------------------------------------------------
// ConsoleView
//----------------------------------------------------------------

// コンストラクタ
ConsoleView::ConsoleView(float left, float top, float width, float height) :
        m_Left(left),
        m_Top(top),
        m_Width(width),
        m_Height(height),
        m_WriteMutex(false)
{
    Clear();
}


// デストラクタ
ConsoleView::~ConsoleView()
{
}


// 初期化
void ConsoleView::Initialize()
{
    m_ScreenWidthCharCount = static_cast<int>(m_Width) / static_cast<int>(App::GetUi().GetGfxContext().GetFontRenderer().GetFixedFontWidth());
    m_ScreenHeightCharCount = static_cast<int>(m_Height) / static_cast<int>(App::GetUi().GetGfxContext().GetFontRenderer().GetFontHeight());

    auto observer = [](const nn::diag::LogMetaData& metaInfo, const nn::diag::LogBody& logBody, void* argument) {
        auto pConsoleView = reinterpret_cast<ConsoleView*>(argument);
        if (metaInfo.moduleName[0] == '\0')
        {
            pConsoleView->ConsoleWrite(logBody.message);
        }
    };
    nn::diag::InitializeLogObserverHolder(&g_LogObserverHolder, observer, this);
    nn::diag::RegisterLogObserver(&g_LogObserverHolder);
}


// 後処理
void ConsoleView::Finalize()
{
    nn::diag::UnregisterLogObserver(&g_LogObserverHolder);
}


// コンソールをクリア
void ConsoleView::Clear()
{
    m_Lines.assign({ std::string() });
}


// 描画
void ConsoleView::Draw(const std::vector<Rect>& excludeRegions)
{
    auto& fontRenderer = App::GetUi().GetGfxContext().GetFontRenderer();

    fontRenderer.EnableFixedFont(true);
    fontRenderer.SetColor(nn::util::Color4u8::White());

    int lineCount = 0;
    for (auto line : m_Lines)
    {
        for (int i = 0; i < excludeRegions.size(); ++i)
        {
            if (excludeRegions[i].y - m_Top < (lineCount + 1) * fontRenderer.GetFontHeight() &&
                    excludeRegions[i].y - m_Top + excludeRegions[i].height > lineCount * fontRenderer.GetFontHeight())
            {
                auto begin = excludeRegions[i].x / fontRenderer.GetFixedFontWidth();
                auto end = (excludeRegions[i].x + excludeRegions[i].width) / fontRenderer.GetFixedFontWidth();

                for (int i = begin; i < line.size() && i < end; ++i)
                {
                    line[i] = ' ';
                }
            }
        }

        fontRenderer.SetPosition(m_Left, m_Top + lineCount * fontRenderer.GetFontHeight());
        fontRenderer.Print(line.c_str());
        ++lineCount;
    }

    m_Updated = false;
}


// コンソール書き込みイベントハンドラ
void ConsoleView::ConsoleWrite(const std::string& str)
{
    m_WriteMutex.Lock();

    for (size_t beginIndex = 0, currentIndex = 0; ; ++currentIndex)
    {
        if (currentIndex >= str.size())
        {
            m_Lines.back().append(str.substr(beginIndex, currentIndex - beginIndex));
            break;
        }
        if (str[currentIndex] == '\n')
        {
            m_Lines.back() += str.substr(beginIndex, currentIndex - beginIndex);
            m_Lines.push_back(std::string());

            beginIndex = currentIndex + 1;
        }
        else if (m_Lines.back().size() + currentIndex - beginIndex >= m_ScreenWidthCharCount)
        {
            m_Lines.back().append(str.substr(beginIndex, currentIndex - beginIndex));
            m_Lines.push_back(std::string());

            beginIndex = currentIndex + 1;
        }
    }

    while (m_Lines.size() > m_ScreenHeightCharCount)
    {
        m_Lines.pop_front();
    }

    m_Updated = true;

    m_WriteMutex.Unlock();
}


// アクセサ
void ConsoleView::SetLeft(float left)
{
    m_Left = left;
}

float ConsoleView::GetLeft() const
{
    return m_Left;
}

void ConsoleView::SetTop(float top)
{
    m_Top = top;
}

float ConsoleView::GetTop() const
{
    return m_Top;
}

void ConsoleView::SetWidth(float width)
{
    m_Width = width;
}

float ConsoleView::GetWidth() const
{
    return m_Width;
}

void ConsoleView::SetHeight(float height)
{
    m_Height = height;
}

float ConsoleView::GetHeight() const
{
    return m_Height;
}

bool ConsoleView::GetUpdated() const
{
    return m_Updated;
}

}
}
