﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "Utils.h"
#include "gfxutil/GfxContext.h"
#include <nn/hid/hid_DebugPad.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_MutexTypes.h>
#include <cstdio>
#include <functional>
#include <list>
#include <memory>
#include <string>
#include <vector>


namespace ApConnectivityTest {
namespace UI {

enum MenuItemType
{
    MenuItemType_Item,
    MenuItemType_Separator,
    MenuItemType_SubMenuItem,
};


class Menu;


class MenuItem
{
public:
    MenuItem(const std::string& name, bool enabled, const std::function<void()>& callback =
            std::function<void()>());
    MenuItem(const std::string& name, const std::string& value, bool enabled, const std::function<void()>& callback =
            std::function<void()>());
    virtual ~MenuItem();

    virtual void Draw(float x, float y, float nameWidth, float subNameWidth, bool nameOnly);

    virtual float GetNameWidth();
    virtual float GetValueWidth();

    void Callback();

    MenuItemType GetType() const;

    void SetName(const std::string& name);
    const std::string& GetName() const;

    void SetValue(const std::string& value);
    const std::string& GetValue() const;

    void SetEnabled(bool enabled);
    bool GetEnabled() const;

    void SetParent(Menu* parent);
    Menu* GetParent() const;

    void SetCallback(const std::function<void()>& callback);

protected:
    MenuItem(MenuItemType type, const std::string& name, bool enabled, const std::function<void()>& callback =
            std::function<void()>());

private:
    MenuItemType m_Type;
    std::string m_Name;
    std::string m_Value;
    bool m_Enabled;
    Menu* m_Parent;

    std::function<void()> m_Callback;
};


class Separator : public MenuItem
{
public:
    Separator();
    virtual ~Separator();
};


class SubMenuItem : public MenuItem
{
public:
    SubMenuItem(const std::string& name, bool enabled, const std::shared_ptr<Menu>& subMenu);
    virtual ~SubMenuItem();

    virtual void Draw(float x, float y, float nameWidth, float subNameWidth, bool nameOnly);

    std::shared_ptr<Menu> GetSubMenu() const;

private:
    std::shared_ptr<Menu> m_SubMenu;
};


class ValueSelectSubMenuItem : public SubMenuItem
{
public:
    ValueSelectSubMenuItem(const std::string& name, bool enabled, bool enableInput, int defaultIndex, const std::vector<std::string>& strList);
    ValueSelectSubMenuItem(const std::string& name, bool enabled, bool enableInput, int defaultIndex, const std::function<void(const std::string&)>& valueChangedCallback, const std::vector<std::string>& strList);
    ValueSelectSubMenuItem(const std::string& name, bool enabled, bool enableInput, size_t minLength, size_t maxLength, int defaultIndex, const std::vector<std::string>& strList);
    ValueSelectSubMenuItem(const std::string& name, bool enabled, bool enableInput, size_t minLength, size_t maxLength, int defaultIndex, const std::function<void(const std::string&)>& valueChangedCallback, const std::vector<std::string>& strList);
    virtual ~ValueSelectSubMenuItem();

    void SetMinValueLength(size_t value);
    size_t GetMinValueLength() const;

    void SetMaxValueLength(size_t value);
    size_t GetMaxValueLength() const;

private:
    void SelectItemCallback();
    void ShowKeyboardCallback();

    void KeyboardResultCallback(const std::string& str);

    void UpdateEnabledSubMenuItem();

private:
    std::shared_ptr<Menu> CreateMenu(bool enableInput, int defaultIndex, const std::vector<std::string>& strList);

private:
    size_t m_MinValueLength;
    size_t m_MaxValueLength;

    std::function<void(const std::string&)> m_ValueChangedCallback;
};


class Menu
{
public:
    Menu(bool vertiacal, int defaultIndex, const std::vector<std::shared_ptr<MenuItem>>& items);
    virtual ~Menu();

    void AddItem(const std::shared_ptr<MenuItem>& item);
    void InsertItem(size_t index, const std::shared_ptr<MenuItem>& item);
    void RemoveItem(size_t index);
    void RemoveItem(const std::shared_ptr<MenuItem>& item);

    void Draw();

    void MoveFocusPrevious();
    void MoveFocusNext();
    void SelectItem();

    std::shared_ptr<MenuItem> operator[](const std::string& name);

public:
    bool vertical;
    std::list<std::shared_ptr<MenuItem>>::iterator activeItem;
    std::list<std::shared_ptr<MenuItem>> items;

    Rect rectangle;
};

}
}
