﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{WebOfflineSimple_Main.cpp,PageSampleWebOfflineSimple}
 *
 * @brief
 *  nn::web ライブラリを使ってオフライン Web アプレットを起動し、オフライン HTML を開くシンプルなサンプルプログラム
 */

/**
 * @page PageSampleWebOfflineSimple WebOfflineSimple
 * @tableofcontents
 *
 * @brief
 *  nn::web ライブラリを使ってオフライン Web アプレットを起動し、オフライン HTML を開くシンプルなサンプルプログラム
 *
 * @section PageSampleWebOfflineSimple_SectionBrief 概要
 *  nn::web ライブラリを使ってオフライン Web アプレットを起動し、オフライン HTML を開くシンプルなサンプルプログラムです。
 *
 * @section PageSampleWebOfflineSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/WebOfflineSimple Samples/Sources/Applications/WebOfflineSimple @endlink 以下にあります。
 *
 * @section PageSampleWebOfflineSimple_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX 実機環境のみサポートしています。
 *
 * @section PageSampleWebOfflineSimple_SectionHowToOperate 操作方法
 *  画面上に操作方法が表示されます。
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> A ボタン </td><td> 指定したオフライン HTML を開く </td></tr>
 * <tr><td> B ボタン </td><td> フッターを利用しないでオフライン HTML を開く </td></tr>
 * <tr><td> X ボタン </td><td> 起動時の画面と Web ページの背景を指定してオフライン HTML を開く </td></tr>
 * <tr><td> + ボタン </td><td> サンプルアプリケーションを終了する </td></tr>
 * </table>
 *
 * @section PageSampleWebOfflineSimple_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleWebOfflineSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleWebOfflineSimple_SectionDetail 解説
 *
 * @subsection PageSampleWebOfflineSimple_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  WebOfflineSimple_Main.cpp
 *  @includelineno WebOfflineSimple_Main.cpp
 *
 * @subsection PageSampleWebOfflineSimple_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの主な処理の流れは以下の通りです。
 *  - 描画のための初期化処理と背景色の設定
 *  - メインループ
 *      - キー入力処理の更新
 *      - 文字の表示処理
 *      - 特定のキーの入力があれば、オフライン Web アプレットを起動してオフライン HTML を表示
 */

#include <stdint.h>

#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/web.h>


#include "WebOfflineSimple_Pad.hpp"
#include "WebOfflineSimple_BackgroundDrawer.hpp"

namespace {

const char Title[] = "WebOfflineSimple";

SamplePad g_SamplePad;
BackgroundDrawer g_BackgroundDrawer;

const char* DocumentPath = "sample.htdocs/index.html";
}

void PrintUsage() NN_NOEXCEPT
{
    nn::gfx::util::DebugFontTextWriter* pWriter = g_BackgroundDrawer.GetTextWriter();
    NN_ASSERT_NOT_NULL(pWriter);
    pWriter->SetCursor(20, 20);
    pWriter->SetFontSize(36);
    pWriter->Print("--------------------------------------------------------------\n");
    pWriter->Print("%s Sample\n", Title);
    pWriter->Print("--------------------------------------------------------------\n");
    pWriter->Print("[A]              Show Simple\n");
    pWriter->Print("[B]              Disabled Footer\n");
    pWriter->Print("[X]              Set BootDisplay And Background Color\n");
    pWriter->Print("[+]              Exit Application\n");
    pWriter->Print("--------------------------------------------------------------\n");
}

void ShowSimple() NN_NOEXCEPT
{
    nn::web::ShowOfflineHtmlPageArg showOfflinePageArg(DocumentPath);

    nn::web::OfflineHtmlPageReturnValue offlinePageReturnValue;
    nn::Result result = nn::web::ShowOfflineHtmlPage(&offlinePageReturnValue, showOfflinePageArg);
    NN_ABORT_UNLESS(result.IsSuccess());
    NN_LOG("Exit Reason %d\n", offlinePageReturnValue.GetOfflineExitReason());
}

void DisabledFooter() NN_NOEXCEPT
{
    nn::web::ShowOfflineHtmlPageArg showOfflinePageArg(DocumentPath);
    // フッターを無効化
    showOfflinePageArg.SetFooterEnabled(false);

    nn::web::OfflineHtmlPageReturnValue offlinePageReturnValue;
    nn::Result result = nn::web::ShowOfflineHtmlPage(&offlinePageReturnValue, showOfflinePageArg);
    NN_ABORT_UNLESS(result.IsSuccess());
    NN_LOG("Exit Reason %d\n", offlinePageReturnValue.GetOfflineExitReason());
}

void SetBootDisplayAndBackground() NN_NOEXCEPT
{
    nn::web::ShowOfflineHtmlPageArg showOfflinePageArg(DocumentPath);
    // 起動時の画面を指定します
    showOfflinePageArg.SetBootDisplayKind(nn::web::OfflineBootDisplayKind_CallerCaptureBlur);
    // 背景を指定します
    showOfflinePageArg.SetBackgroundKind(nn::web::OfflineBackgroundKind_ApplicationCaptureBlur);

    nn::web::OfflineHtmlPageReturnValue offlinePageReturnValue;
    nn::Result result = nn::web::ShowOfflineHtmlPage(&offlinePageReturnValue, showOfflinePageArg);
    NN_ABORT_UNLESS(result.IsSuccess());
    NN_LOG("Exit Reason %d\n", offlinePageReturnValue.GetOfflineExitReason());
}

bool Update() NN_NOEXCEPT
{
    if( g_SamplePad.IsTrigger<nn::hid::DebugPadButton::A>() )
    {
        // 全ての設定がデフォルトの状態で、オフライン HTML を表示します
        ShowSimple();
    }
    if( g_SamplePad.IsTrigger<nn::hid::DebugPadButton::B>() )
    {
        // フッターが非表示の状態で、オフライン HTMLを表示します
        DisabledFooter();
    }
    if( g_SamplePad.IsHold<nn::hid::DebugPadButton::X>() )
    {
        // 起動時の画面に呼び出し元のアプリケーションのキャプチャーのぼかし画像を指定し、
        // ページの背景に呼び出し元のアプリケーションのキャプチャーのぼかし画像を指定します
        SetBootDisplayAndBackground();
    }
    // Exit
    if( g_SamplePad.IsTrigger<nn::hid::DebugPadButton::Start>())
    {
        return false;
    }
    return true;
}

//
//  Main Function
//  メイン関数です。
//

extern "C" void nnMain() NN_NOEXCEPT
{
    // 描画のための初期化処理と背景色の設定
    g_BackgroundDrawer.Initialize();
    g_BackgroundDrawer.SetBackgroundColor(125, 125, 125);

    // メインループ
    for(;;)
    {
        // キー入力処理の更新
        g_SamplePad.Update();
        // 文字の表示処理
        PrintUsage();
        g_BackgroundDrawer.Draw();

        // 特定のキーの入力があれば、オフライン HTML を表示
        if( !Update() )
        {
            break;
        }
    }
    g_BackgroundDrawer.Finalize();
}
