﻿var FOCUSABLE_SELECTOR = '[tabindex]';

var KEY = {
  ENTER: 13,
  LEFT: 37,
  UP: 38,
  RIGHT: 39,
  DOWN: 40
};

var DIR_TYPE = {
  HORIZONTAL: 0,
  VERTICAL: 1,
  OBLIQUE: 2
}

var MIN_COS = {};
MIN_COS[DIR_TYPE.HORIZONTAL] = 0.734;
MIN_COS[DIR_TYPE.VERTICAL] = 0.308;
MIN_COS[DIR_TYPE.OBLIQUE] = 0.707;

var COS_WEIGHT = {};
COS_WEIGHT[DIR_TYPE.HORIZONTAL] = 0.011;
COS_WEIGHT[DIR_TYPE.VERTICAL] = 0.002;
COS_WEIGHT[DIR_TYPE.OBLIQUE] = 0.001;


var isKeyDown = {};
isKeyDown[KEY.LEFT] = false;
isKeyDown[KEY.UP] = false;
isKeyDown[KEY.RIGHT] = false;
isKeyDown[KEY.DOWN] = false;

var isDirKey = function(keyCode) {
  return KEY.LEFT <= keyCode && keyCode <= KEY.DOWN;
}

var getCenterPos = function(elem) {
  var rect = elem.getBoundingClientRect();
  return {
    x: rect.left + rect.width / 2,
    y: rect.top + rect.height / 2
  };
};

var isInRange = function(cos, dirType) {
  return cos >= MIN_COS[dirType];
};

var calculateScore = function(dist, cos, dirType) {
  return 1 / (dist + 1) + cos * COS_WEIGHT[dirType];
};

var lastFocusedNode = document.querySelector('.focused-initially');
var focusLastFocusedNodeIfNeeded = function(event) {
  if (document.activeElement === document.body) {
    lastFocusedNode.focus();
    if (window.nx && window.nx.playSystemSe) {
      window.nx.playSystemSe('SeWebNaviFocus');
    }
    return true;
  }
  return false;
};

var getCos = function(vec1, vec1Dist, vec2, vec2Dist) {
  if (vec1Dist === 0 || vec2Dist === 0) {
    return 0;
  }
  return (vec1.x * vec2.x + vec1.y * vec2.y) / vec1Dist / vec2Dist;
}

var updateFocusControl = function() {
  var inputDir = {
    x: 0,
    y: 0
  };
  if (isKeyDown[KEY.LEFT]) {
    inputDir.x -= 1;
  }
  if (isKeyDown[KEY.UP]) {
    inputDir.y -= 1;
  }
  if (isKeyDown[KEY.RIGHT]) {
    inputDir.x += 1;
  }
  if (isKeyDown[KEY.DOWN]) {
    inputDir.y += 1;
  }
  if (inputDir.x !== 0 || inputDir.y !== 0) {
    changeFocusIfNeeded(inputDir);
  }
  isKeyDown[KEY.LEFT] = false;
  isKeyDown[KEY.UP] = false;
  isKeyDown[KEY.RIGHT] = false;
  isKeyDown[KEY.DOWN] = false;

  requestAnimationFrame(updateFocusControl);
};

var changeFocusIfNeeded = function(dir) {

  var dirType, focusableElems, currentCenterPos, candidates;

  if (focusLastFocusedNodeIfNeeded(event)) {
    return;
  }

  if (dir.y === 0) {
    dirType = DIR_TYPE.HORIZONTAL;
  } else if (dir.x === 0) {
    dirType = DIR_TYPE.VERTICAL;
  } else {
    dirType = DIR_TYPE.OBLIQUE;
  }

  focusableElems = document.querySelectorAll(FOCUSABLE_SELECTOR);
  if (focusableElems.length === 0) {
    return;
  }

  currentCenterPos = getCenterPos(document.activeElement);
  candidates = [];

  Array.prototype.forEach.call(focusableElems, function(elem) {
    var candidateCenterPos, diff, dist, cos;

    if (elem === document.activeElement) {
      return;
    }

    candidateCenterPos = getCenterPos(elem);
    diff = {
      x: candidateCenterPos.x - currentCenterPos.x,
      y: candidateCenterPos.y - currentCenterPos.y
    };
    dist = Math.sqrt(Math.pow(diff.x, 2) + Math.pow(diff.y, 2));
    cos = getCos(dir, dirType === DIR_TYPE.OBLIQUE ? Math.sqrt(2) : 1, diff, dist);
    if (!isInRange(cos, dirType)) {
      return;
    }

    candidates.push({
      elem: elem,
      dist: dist,
      cos: cos
    });
  });

  if (candidates.length === 0) {
    return;
  }

  var bestCandidate = null;
  var bestScore = 0;

  candidates.forEach(function(candidate) {
    var score = calculateScore(candidate.dist, candidate.cos, dirType);
    if (bestScore < score) {
      bestCandidate = candidate;
      bestScore = score;
    }
  });

  if (bestCandidate) {
    bestCandidate.elem.focus();
    if (window.nx && window.nx.playSystemSe) {
      window.nx.playSystemSe('SeWebNaviFocus');
    }
  }
};

Array.prototype.forEach.call(
  document.querySelectorAll(FOCUSABLE_SELECTOR),
  function(elem) {
    elem.addEventListener('focus', function(event){
      lastFocusedNode = event.currentTarget;
    });
  });

document.body.addEventListener('keydown', function(event) {
  if (isDirKey(event.keyCode)) {
    event.preventDefault();
    isKeyDown[event.keyCode] = true;
  }
});

updateFocusControl();
