﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx.h>
#include <nn/vi.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

class GraphicsSystem
{
    NN_DISALLOW_COPY(GraphicsSystem);
    NN_DISALLOW_MOVE(GraphicsSystem);
public:
    // 背景の種類
    enum BackgroundKind
    {
        BackgroundKind_None,      //!< 表示なし。背景色が表示されます
        BackgroundKind_HelpGame,  //!< ヘルプ用の一枚絵
        BackgroundKind_PauseGame, //!< 一時停止用の一枚絵
        BackgroundKind_End,

        BackgroundKind_TextureMin = BackgroundKind_HelpGame,
        BackgroundKind_TextureMax = BackgroundKind_PauseGame,
        BackgroundKind_TextureCount
            = BackgroundKind_TextureMax - BackgroundKind_TextureMin + 1,
    };

    // 矩形の各頂点データを管理する構造体です。
    struct RectanglePositions
    {
        nn::util::Float2 leftTop;
        nn::util::Float2 leftBottom;
        nn::util::Float2 rightBottom;
        nn::util::Float2 rightTop;
    };

public:
    // コンストラクタ
    GraphicsSystem() NN_NOEXCEPT;
    // デストラクタ
    ~GraphicsSystem() NN_NOEXCEPT;

    // 初期化
    // ファイルシステムと Contents がマウントされている必要があります。
    // 本関数内でグラフィックスシステムのためのメモリ周りの初期化を行っています。
    void Initialize() NN_NOEXCEPT;
    // 終了処理
    void Finalize() NN_NOEXCEPT;

    // 背景色の設定(0 ～ 255 の範囲で設定してください)
    void SetBackgroundColor(int r, int g, int b) NN_NOEXCEPT;
    // 背景の種類の設定
    void SetBackgroundKind(BackgroundKind kind) NN_NOEXCEPT;

    // 矩形の表示/非表示の設定
    void SetRectangleVisible(bool value) NN_NOEXCEPT;
    // 矩形の表示/非表示の取得
    bool GetRectangleVisible() const NN_NOEXCEPT;
    // 矩形のカラー設定
    void SetRectangleColor(int r, int g, int b, int a) NN_NOEXCEPT;
    // 矩形の位置の設定
    // 画面の左上が原点(0, 0)です。右下の座標が(1280, 720)になっています。
    void SetRectanglePositions(const RectanglePositions& position) NN_NOEXCEPT;

    // 背景、矩形、デバッグフォントの順で描画を行います。
    // 描画フレームワークは Immediate を使用しています。
    void Draw() NN_NOEXCEPT;

    // デバッグフォント描画クラスの取得
    nn::gfx::util::DebugFontTextWriter* GetTextWriter() NN_NOEXCEPT;

private:
    static const int ScanBufferCount = 2;

    // 描画の種類
    enum DrawType
    {
        DrawType_Texture,  //!< テクスチャの表示
        DrawType_Constant, //!< カラーの表示

        DrawType_End
    };
private:
    void InitializeLayer() NN_NOEXCEPT;
    void InitializeDevice() NN_NOEXCEPT;
    void InitializeMemoryPool() NN_NOEXCEPT;
    void InitializeCompressedMemoryPool() NN_NOEXCEPT;
    void InitializeSwapChain() NN_NOEXCEPT;
    void InitializeQueue() NN_NOEXCEPT;
    void InitializeCommandBuffer() NN_NOEXCEPT;
    void InitializeViewport() NN_NOEXCEPT;
    void InitializeShader() NN_NOEXCEPT;
    void InitializeRasterizerState() NN_NOEXCEPT;
    void InitializeBlendState() NN_NOEXCEPT;
    void InitializeDepthStencilState() NN_NOEXCEPT;
    void InitializeVertexState() NN_NOEXCEPT;
    void InitializePositionBuffer() NN_NOEXCEPT;
    void InitializeUvBuffer() NN_NOEXCEPT;
    void InitializeIndexBuffer() NN_NOEXCEPT;
    void InitializeSamplerDescriptorPool() NN_NOEXCEPT;
    void InitializeTextureDescriptorPool() NN_NOEXCEPT;
    void InitializeFence() NN_NOEXCEPT;
    void InitializeSemaphore() NN_NOEXCEPT;
    void ResetCommandBuffer() NN_NOEXCEPT;
    void InitializeResShaderFile() NN_NOEXCEPT;
    void* ReadResource(const char* filename) NN_NOEXCEPT;
    void InitializeTexture() NN_NOEXCEPT;
    void InitializeSampler() NN_NOEXCEPT;
    void InitializeDebugFontTextWriter() NN_NOEXCEPT;
    void FinalizeResShaderFile() NN_NOEXCEPT;
    void InitializeConstantBuffer() NN_NOEXCEPT;
    void MakeCommandDrawTexture() NN_NOEXCEPT;
    void MakeCommandDrawRectangle() NN_NOEXCEPT;
private:
    nn::gfx::ClearColorValue m_BackgroundColor;             //!< 背景色
    BackgroundKind m_BackgroundKind;                        //!< 背景の種類
    nn::util::BytePtr m_pMemoryHeap;                        //!< ヒープ
    nn::util::BytePtr m_pMemory;                            //!< メモリ
    nn::gfx::Device        m_Device;                        //!< デバイス
    void*                  m_pMemoryPoolMemory;             //!< メモリプール用メモリ
    void*                  m_pMemoryPoolStart;              //!< メモリプールの開始位置
    ptrdiff_t              m_MemoryPoolOffset;              //!< メモリプールのオフセット
    nn::gfx::MemoryPool    m_MemoryPool;                    //!< メモリプール
    void*                  m_pCompressedMemoryPoolMemory;   //!< 圧縮メモリプール用メモリ
    void*                  m_pCompressedMemoryPoolStart;    //!< 圧縮メモリプールの開始位置
    ptrdiff_t              m_CompressedMemoryPoolOffset;    //!< 圧縮メモリプールのオフセット
    nn::gfx::MemoryPool    m_CompressedMemoryPool;          //!< 圧縮メモリプール
    nn::vi::Display*       m_pDisplay;                      //!< ディスプレイ
    nn::vi::Layer*         m_pLayer;                        //!< レイヤ
    nn::gfx::SwapChain     m_SwapChain;                     //!< スワップチェーン
    nn::gfx::Queue         m_Queue;                         //!< キュー
    nn::gfx::CommandBuffer m_CommandBuffer;                 //!< コマンドバッファ
    void*                  m_pCommandBufferControlMemory;   //!< コマンドバッファのコントロールメモリ
    ptrdiff_t              m_CommandBufferMemoryPoolOffset; //!< コマンドバッファのメモリプールオフセット
    nn::gfx::ViewportScissorState m_ViewportScissor;        //!< ビューポートシザー
    nn::gfx::Shader*        m_pShader[DrawType_End];        //!< シェーダのポインタ（バリエーションの数を用意）
    nn::gfx::BlendState     m_BlendState;                   //!< ブレンドステート
    nn::gfx::DepthStencilState m_DepthStencilState;         //!< デプスステンシルステート
    nn::gfx::RasterizerState m_RasterizerState;             //!< ラスタライザステート
    nn::gfx::VertexState    m_VertexState[DrawType_End];    //!< 頂点ステート（シェーダバリエーションの数だけ用意）
    int                     m_InterfaceSlot[DrawType_End];  //!< シェーダのインターフェーススロット
    nn::gfx::Buffer         m_PositionBuffer[DrawType_End]; //!< Position 用バッファ
    nn::gfx::Buffer         m_UvBuffer;                     //!< Uv 用バッファ
    nn::gfx::Buffer         m_IndexBuffer;                  //!< Index 用バッファ
    nn::gfx::DescriptorPool m_SamplerDescriptorPool;        //!< サンプラデスクリプタプール
    int                     m_SamplerDescriptorBaseIndex;   //!< サンプラデスクリプタスロットのインデックス
    nn::gfx::DescriptorSlot m_SamplerDescriptorSlot;        //!< サンプラデスクリプタスロット
    nn::gfx::DescriptorPool m_TextureDescriptorPool;        //!< テクスチャデスクリプタプール
    int                     m_TextureDescriptorBaseIndex;   //!< テクスチャデスクリプタスロットのインデックス
    nn::gfx::Fence          m_DisplayFence;                 //!< スキャンバッファ用のフェンス
    nn::gfx::Semaphore      m_DisplaySemaphore;             //!< スキャンバッファ用のセマフォ
    nn::gfx::Fence          m_GpuDoneFence;                 //!< DRAW 待ち用のフェンス
    nn::gfx::ColorTargetView* m_pScanBufferViews[ScanBufferCount]; //!< スキャンバッファビューのポインタ
    nn::gfx::ResShaderFile* m_pResShaderFile;               //!< シェーダファイル
    nn::gfx::ResTextureFile* m_pResTextureFile;             //!< テクスチャリソースファイルのポインタ
    /// テクスチャデスクリプタスロット
    nn::gfx::DescriptorSlot m_TextureDescriptorSlotList[BackgroundKind_TextureCount];
    nn::gfx::Sampler        m_Sampler;                         //!< サンプラ
    nn::gfx::util::DebugFontTextWriter m_DebugFontTextWriter;  //!< デバッグ用テキスト
    nn::util::Color4fType   m_RectangleColor;                  //!< 矩形カラー
    nn::gfx::Buffer         m_ConstantBuffer;                  //!< 定数バッファ
    bool                    m_IsRectangleVisible;              //!< 矩形の表示非表示切り替え
    /// 次のフレームのスキャンバッファのインデックス
    int                     m_NextScanBufferIndex;
    int                     m_FrameCount;                      //!< フレーム数
};
