﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{ViZoom.cpp,PageSampleViZoom}
 *
 * @brief This sample demonstrates changing magnification on the display.
 */

/**
 * @page PageSampleViZoom ViZoom
 * @tableofcontents
 *
 * @brief Sample program to instruct how to change the display magnification.
 *
 * @section ViZoom_SectionBrief Summary
 *  The display magnification can be set on the Default and External displays.  When enabled,
 *  an upscale is performed based on the rectangle specified in nn::vi::SetDisplayMagnification.
 *
 * @section ViZoom_SectionFileStructure File List
 *  Please refer to @link ../../../Samples/Sources/Applications/ViZoom Samples/Sources/Applications/ViZoom @endlink
 *
 * @section ViZoom_SectionNecessaryEnvironment Necessary Environment
 *  Joycon are required to use this sample.
 *
 * @section ViZoom_SectionHowToOperate Operating Procedure
 *  Please launch this sample with TargetManager or Visual Studio.
 *
 *  Press the Plus button to print the current magnification rectangle to the console.
 *  Press the Minus button to exit the sample.
 *
 *  Use these buttons to control the magnification rectangle position:
 *    - Up to decrease the y-coordinate
 *    - Down to increase the y-coordinate
 *    - Left to decrease the x-coordinate
 *    - Right to increase the x-coordinate
 *
 *  Use these buttons to change the magnification rectangle width and height:
 *    - X to decrease the height
 *    - B to increase the height
 *    - Y to decrease the width
 *    - A to increase the width
 *
 *  The width and height of the magnification rectangle must be adjusted before changing
 *  the position.
 *
 * @section ViZoom_SectionPrecaution Precautions
 *  No known issues.
 *
 * @section ViZoom_SectionHowToExecute How To Execute
 *  Please build and run the sample.
 *
 * @section ViZoom_SectionDetail Details
 *
 * @subsection ViZoom_SectionSampleProgram Sample Program
 *  The main sample source code is linked below.
 *
 *  ViZoom.cpp
 *  @includelineno ViZoom.cpp
 *
 * @subsection ViZoom_SectionSampleDetail Sample Program Details
 *  The nn::vi::SetDisplayMagnification functions utilizes a rectangle to specify a subset
 *  of the display to upscale.  The coordinate system is defined by nn::vi::GetDisplayLogicalResolution,
 *  which is independent of the actual display resolution.  Note the origin is the top-corner of the display.
 *
 *  Setting the magnification on the Default display will affect both TV and LCD.  Note it is possible
 *  to independently adjust the magnification on each display.
 *
 */

#include <algorithm>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/ae.h>
#include <nn/vi.private.h>
#include "Controller.h"

namespace
{
    // Mapping between duration and a value modifier
    class HeldModifier
    {
    public:
        HeldModifier(nn::TimeSpan duration, int modifier) NN_NOEXCEPT
            : m_Duration(duration)
            , m_Modifier(modifier)
        {
        };

        nn::TimeSpan GetMinimumDuration() const NN_NOEXCEPT
        {
            return m_Duration;
        }

        int GetModifier() const NN_NOEXCEPT
        {
            return m_Modifier;
        }
    private:
        nn::TimeSpan m_Duration;
        int m_Modifier;
    };

    // Helper class to determine how much to modify a value based on
    // how long the button is held
    template <typename Button>
    class ButtonManager
    {
    public:
        template <size_t Size>
        ButtonManager(HeldModifier (&modifiers)[Size])
            : m_HeldStart(0)
            , m_HeldModifiers(modifiers)
            , m_HeldModifiersSize(Size)
            , m_Modifier(0)
        {
        }

        void Update(const Controller& input) NN_NOEXCEPT
        {
            if( input.IsPressed<Button>() )
            {
                // New press this frame, start tracking held duration
                m_HeldStart = nn::os::GetSystemTick();
            }

            if( input.IsDown<Button>() )
            {
                nn::os::Tick current = nn::os::GetSystemTick();

                // Determine modifier based on held duration
                for( int i = 0; i < m_HeldModifiersSize; ++i )
                {
                    if( (current - m_HeldStart).ToTimeSpan() >= m_HeldModifiers[i].GetMinimumDuration() )
                    {
                        m_Modifier = m_HeldModifiers[i].GetModifier();
                    }
                }
            }
            else
            {
                m_Modifier = 0;
            }
        }

        int GetCurrentModifier() const NN_NOEXCEPT
        {
            return m_Modifier;
        }
    private:
        nn::os::Tick m_HeldStart;

        const HeldModifier* m_HeldModifiers;
        size_t m_HeldModifiersSize;
        int m_Modifier;
    };
}

void ViZoomMain(nn::ae::OverlayAppletParameters* param)
{
    NN_UNUSED(param);
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    Controller::Initialize();
    Controller input;

    HeldModifier increment[] =
    {
        HeldModifier(nn::TimeSpan::FromMilliSeconds(0),    1),
        HeldModifier(nn::TimeSpan::FromMilliSeconds(1000), 4),
        HeldModifier(nn::TimeSpan::FromMilliSeconds(1750), 8),
    };

    HeldModifier decrement[] =
    {
        HeldModifier(nn::TimeSpan::FromMilliSeconds(0),    -1),
        HeldModifier(nn::TimeSpan::FromMilliSeconds(1000), -4),
        HeldModifier(nn::TimeSpan::FromMilliSeconds(1750), -8),
    };

    ButtonManager<nn::hid::NpadButton::A> a(increment);
    ButtonManager<nn::hid::NpadButton::B> b(increment);
    ButtonManager<nn::hid::NpadButton::X> x(decrement);
    ButtonManager<nn::hid::NpadButton::Y> y(decrement);
    ButtonManager<nn::hid::NpadButton::Up> up(decrement);
    ButtonManager<nn::hid::NpadButton::Down> down(increment);
    ButtonManager<nn::hid::NpadButton::Left> left(decrement);
    ButtonManager<nn::hid::NpadButton::Right> right(increment);

    int xPos = 0;
    int yPos = 0;

    int width;
    int height;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::GetDisplayLogicalResolution(&width, &height, pDisplay));

    int maxWidth = width;
    int maxHeight = height;

    nn::ae::BeginOverlayUserInteraction();
    while( true )
    {
        input.Update();

        a.Update(input);
        b.Update(input);
        x.Update(input);
        y.Update(input);
        up.Update(input);
        down.Update(input);
        left.Update(input);
        right.Update(input);

        width += a.GetCurrentModifier() + y.GetCurrentModifier();
        height += b.GetCurrentModifier() + x.GetCurrentModifier();

        xPos += left.GetCurrentModifier() + right.GetCurrentModifier();
        yPos += up.GetCurrentModifier() + down.GetCurrentModifier();

        // Apply constraints to input values
        // Note: Specifying very small values for width/height may lead to content being dropped
        //       depending on the size of the textures used on the display.
        // 16 <= width <= maxWidth
        width  = std::max(16, std::min(width, maxWidth));
        //  9 <= height <= maxHeight
        height = std::max(9, std::min(height, maxHeight));

        // 0 <= xPos <= maxWidth - width
        xPos = std::min(maxWidth - width, std::max(0, xPos));
        // 0 <= yPos <= maxHeight - height
        yPos = std::min(maxHeight - height, std::max(0, yPos));

        if( input.IsPressed<nn::hid::NpadButton::Plus>() )
        {
            NN_LOG("x      : %d\n", xPos);
            NN_LOG("y      : %d\n", yPos);
            NN_LOG("width  : %d\n", width);
            NN_LOG("height : %d\n", height);
            NN_LOG("\n");
        }

        if( input.IsPressed<nn::hid::NpadButton::Minus>() )
        {
            break;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::SetDisplayMagnification(pDisplay, xPos, yPos, width, height));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
    nn::ae::EndOverlayUserInteraction();

    // Reset the display to normal magnification
    nn::vi::SetDisplayMagnification(pDisplay, 0, 0, maxWidth, maxHeight);

    nn::vi::CloseDisplay(pDisplay);
    nn::vi::Finalize();
}

extern "C" void nnMain()
{
    nn::ae::InvokeOverlayAppletMain(nn::ae::AppletId_OverlayApplet, ViZoomMain);
}
