﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <new>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/settings/system/settings_Tv.h>
#include <nn/vi.private.h>
#include <nn/fs.h>
#include <nn/os.h>

/**
 * @examplesource{ViEdid.cpp, PageSampleViEdid}
 *
 * @brief This sample demonstrates how to use the EDID display to check the capabilities
 *        of the last connected display.
 */

/**
 * @page PageSampleViEdid ViEdid
 * @tableofcontents
 *
 * @brief Sample program to instruct how to use the EDID display.
 *
 * @section ViEdid_SectionBrief Summary
 *  Private functionality in nn::vi allows clients to open an EDID display.  This display
 *  grants read-only access to some capabilities of the last-known HDMI-capable display
 *  connected to the system.
 *
 * @section ViEdid_SectionFileStructure File List
 *  Please refer to @link ../../../Samples/Sources/Applications/ViEdid Samples/Sources/Applications/ViEdid @endlink
 *
 * @section ViEdid_SectionNecessaryEnvironment Necessary Environment
 *  No additional requirements.
 *
 * @section ViEdid_SectionHowToOperate Operating Procedure
 *  This application requires a file path to be passed as a command line argument.  The Contents directory is
 *  mounted automatically for this application.
 *
 * @section ViEdid_SectionPrecaution Precautions
 *  No known issues.
 *
 * @section ViEdid_SectionHowToExecute How To Execute
 *  Please build and run the sample with one of the following command line arguments:
 *    - Contents:/480pOnly.bin
 *    - Contents:/RgbOnly.bin
 *
 * @section ViEdid_SectionDetail Details
 *
 * @subsection ViEdid_SectionSampleProgram Sample Program
 *  The main sample source code is linked below.
 *
 *  ViEdid.cpp
 *  @includelineno ViEdid.cpp
 *
 * @subsection ViEdid_SectionSampleDetail Sample Program Details
 *
 *  This sample will proceed through the following steps with nn::vi:
 *   - Explicitly initialize nn::vi for private functionality
 *   - Write the requested EDID to system settings (provided with command line argument)
 *   - Open the EDID display
 *   - Enumerate the available modes on the EDID stored in system settings
 */

void* FsAllocate(size_t size)
{
    return std::malloc(size);
}

void FsDeallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    std::free(p);
}

// The last display's EDID will be stored automatically at a
// future date.  This function's only purpose is to override
// the value stored in system settings and is not necessary
// for applications to do.
static void InitializeEdid(const char* path) NN_NOEXCEPT
{
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    char* mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(mountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("Contents", mountRomCacheBuffer, cacheSize));

    nn::fs::FileHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    nn::settings::system::Edid edid;
    std::memset(&edid, 0, sizeof(edid));

    size_t unused;
    nn::fs::ReadFile(&unused, handle, 0, edid.data0, sizeof(edid.data0));
    nn::fs::ReadFile(&unused, handle, sizeof(edid.data0), edid.data1, sizeof(edid.data1));

    nn::fs::CloseFile(handle);

    nn::fs::Unmount("Contents");
    delete[] mountRomCacheBuffer;

    nn::settings::system::SetEdid(edid);
}

static const char* ToString(nn::vi::DisplayModeInfo::StereoModeType mode) NN_NOEXCEPT
{
#define CASE(x) case nn::vi::x: return #x
    switch (mode)
    {
        CASE(StereoMode_None);
        CASE(StereoMode_FramePacking);
        CASE(StereoMode_SideBySide);
        CASE(StereoMode_TopAndBottom);
    default:
        NN_UNEXPECTED_DEFAULT;
    }
#undef CASE
}

static void PrintMode(const nn::vi::DisplayModeInfo& mode)
{
    NN_LOG("%dx%d, %.2f Hz, %s\n", mode.width,
                                   mode.height,
                                   mode.refreshRate,
                                   ToString(mode.mode));
}

extern "C" void nnMain()
{
    if( nn::os::GetHostArgc() != 2 )
    {
        NN_LOG("Please pass in a path to an EDID on the file system.\n");
        return;
    }

    InitializeEdid(nn::os::GetHostArgv()[1]);

    nn::vi::Initialize();

    nn::vi::DisplayInfo displays[3];
    // Currently three named displays on HOS.
    int displayCount = nn::vi::ListDisplays(displays, sizeof(displays) / sizeof(displays[0]));
    NN_ASSERT(3 == displayCount, "Failed to enumerate displays.");

    // There are additional named displays.  They have different policies from the Default
    // display.
    for( int i = 0; i < displayCount; ++i )
    {
        NN_LOG("Display Name    : %s\n", displays[i].name);

        if( displays[i].hasLayerLimit )
        {
            // Process can only create this many layers on this display.
            NN_LOG("Layer Limit     : %d\n", displays[i].layerCountMax);
        }

        // Max dimensions of a layer (in pixels) on the default display.
        NN_LOG("Layer Max Width : %d\n", displays[i].layerWidthPixelCountMax);
        NN_LOG("Layer Max Height: %d\n", displays[i].layerHeightPixelCountMax);

        NN_LOG("\n");
    }

    nn::vi::Display* pDisplay;
    {
        // This is a new display type for read-only access of the EDID stored in system
        // settings.  (Not all nn::vi functions will succeed on this display!)

        // The system settings are only read upon opening this display.  It must be closed
        // and reopened before it will pick up any changes in the system settings.
        nn::Result result = nn::vi::OpenDisplay(&pDisplay, "Edid");
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    // Using an array of this size will always allow all possible modes to be enumerated regardless
    // of the platform.
    nn::vi::DisplayModeInfo modes[nn::vi::DisplayModeCountMax];
    // The list returned will be filtered to only show the modes allowed on the current platform.
    int modeCount = nn::vi::ListDisplayModes(modes, sizeof(modes) / sizeof(modes[0]), pDisplay);

    for( int i = 0; i < modeCount; ++i )
    {
        PrintMode(modes[i]);
    }

    nn::vi::Finalize();
}
